/* smb-method.h - VFS modules for SMB
 *
 * Docs:
 * http://www.ietf.org/internet-drafts/draft-crhertel-smb-url-04.txt
 * http://samba.org/doxygen/samba/group__libsmbclient.html
 * http://ubiqx.org/cifs/
 *
 *  Copyright (C) 2001,2002,2003 Red Hat
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Author: Bastien Nocera <hadess@hadess.net>
 * Inspired by Alex Larsson's neato SMB vfs method
 */

#include "config.h"
#include "gnome-vfs-extras-i18n.h"

/* libgen first for basename */
#include <libgen.h>
#include <string.h>
#include <glib.h>
#include <errno.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-module.h>
#include <libgnomevfs/gnome-vfs-module-shared.h>
#include <libgnomevfs/gnome-vfs-module-callback-module-api.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include <libsmbclient.h>

static GMutex *smb_lock;

#define DEBUG_SMB_ENABLE
#define DEBUG_SMB_LOCKS

#ifdef DEBUG_SMB_ENABLE
#define DEBUG_SMB(x) g_print x
#else
#define DEBUG_SMB(x) 
#endif

#ifdef DEBUG_SMB_LOCKS
#define LOCK_SMB() 	{g_mutex_lock (smb_lock); g_print ("LOCK %s\n", G_GNUC_PRETTY_FUNCTION);}
#define UNLOCK_SMB() 	{g_print ("UNLOCK %s\n", G_GNUC_PRETTY_FUNCTION); g_mutex_unlock (smb_lock);}
#else
#define LOCK_SMB() 	g_mutex_lock (smb_lock)
#define UNLOCK_SMB() 	g_mutex_unlock (smb_lock)
#endif

static void auth_fn(const char *server, const char *share,
		char *workgroup, int wgmaxlen, char *username, int unmaxlen,
		char *password, int pwmaxlen);

static gboolean
is_hidden_entry (char *name)
{
	if (name == NULL) return TRUE;

	if (*(name + strlen (name) -1) == '$') return TRUE;

	return FALSE;

}

static gboolean
try_init (void)
{
	int err, fd;
	char *path;
	const char *msg;

	LOCK_SMB();

	/* Create an empty ~/.smb/smb.conf */
	path = g_build_filename (G_DIR_SEPARATOR_S, g_get_home_dir (),
			".smb", NULL);
	if (g_file_test (path, G_FILE_TEST_IS_DIR) == FALSE) {
		mkdir (path, 0700);
	} else {
		chmod (path, 0700);
	}
	g_free (path);

	path = g_build_filename (G_DIR_SEPARATOR_S, g_get_home_dir (),
			".smb", "smb.conf", NULL);
	if (g_file_test (path, G_FILE_TEST_IS_REGULAR) == FALSE) {
		fd = creat (path, 0600);
		if (fd > 0) {
			close (fd);
		}
	}
	g_free (path);

	err = smbc_init(auth_fn, 10);

	if (err < 0) {
		msg = gnome_vfs_result_to_string
			(gnome_vfs_result_from_errno ());
		g_warning ("Could not initialize samba client library: %s\n",
				msg);

		return FALSE;
	}

	UNLOCK_SMB();

	return TRUE;
}

static char *
_gnome_vfs_uri_get_basename (const GnomeVFSURI *uri)
{
	char *path = g_strdup (uri->text);
	char *base;

	DEBUG_SMB(("_gnome_vfs_uri_get_basename: uri: %s, basename: %s\n",
				uri->text, basename (uri->text)));

	base = g_strdup (basename (path));
	g_free (path);

	return base;
}

static char *
get_type_from_uri (GnomeVFSURI *uri, int *type)
{
	GnomeVFSURI *parent;
	char *path, *name1, *name2, *unescaped;
	struct smbc_dirent *entry;
	int fd;
	gboolean found;

	DEBUG_SMB (("get_type_from_uri: uri %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	name1 = _gnome_vfs_uri_get_basename (uri);

	/* Special case the smb://foo/ type uris, they're all servers */

	path = gnome_vfs_uri_to_string (uri, 0);
	if (strncmp (path, "smb:///", strlen ("smb:///")) == 0) {
		*type = SMBC_SERVER;
		g_free (path);
		return name1;
	}

	parent = gnome_vfs_uri_get_parent (uri);

	/* Unescape the parent's path */
	if (parent != NULL) {
		path = gnome_vfs_uri_to_string (parent, 0);
		unescaped = gnome_vfs_unescape_string (path, "/");
		g_free (path);
		path = unescaped;
	} else {
		path = NULL;
	}

	/* smb:// and smb:/// give us different results */
	if (path == NULL
			|| strcmp (path, "smb:") == 0
			|| strcmp (path, "smb:///") == 0) {
		g_free (path);
		path = g_strdup ("smb://");
	}

	DEBUG_SMB (("get_type_from_uri: opening directory '%s'\n", path));

	fd = smbc_opendir (path);
	g_free (path);
	gnome_vfs_uri_unref (parent);

	if (fd < 0) {
		gnome_vfs_uri_unref (parent);
		*type = -1;
		return NULL;
	}

	found = FALSE;
	entry = smbc_readdir (fd);

	while (entry != NULL)
	{
		unescaped = g_strndup (entry->name, entry->namelen);
		name2 = gnome_vfs_escape_string (unescaped);
		g_free (unescaped);
		DEBUG_SMB (("get_type_from_uri: comparing '%s' and '%s'\n",
					name1, name2));
		if (strcmp (name1, name2) == 0) {
			found = TRUE;
			break;
		}

		g_free (name2);
		entry = smbc_readdir (fd);
	}

	if (found) {
		*type = entry->smbc_type;
	} else {
		*type = -1;
		g_free (name1);
		name1 = NULL;
	}

	smbc_closedir (fd);

	DEBUG_SMB(("get_type_from_uri: type: %d name: %s\n",
				*type, name1));

	/* Fallback, because in some cases the network browsing doesn't
	 * work, but single hosts can still be accesses */

	return name1;
}

static char *
get_workgroup_data (const char *display_name, const char *name)
{
	return g_strdup_printf ("[Desktop Entry]\n"
			"Encoding=UTF-8\n"
			"Name=%s\n"
			"Type=Link\n"
			"URL=smb://%s/\n"
			"Icon=gnome-fs-network\n",
			display_name, name);
}
                                                                                
static char *
get_computer_data (const char *display_name, const char *name)
{
	return g_strdup_printf ("[Desktop Entry]\n"
			"Encoding=UTF-8\n"
			"Name=%s\n"
			"Type=Link\n"
			"URL=smb://%s/\n"
			"Icon=gnome-fs-server\n",
			display_name, name);
}

typedef struct {
	int fd;
	gboolean is_data;
	char *file_data;
	int fnum;
	GnomeVFSFileOffset offset;
	GnomeVFSFileOffset file_size;
} FileHandle;

static GnomeVFSResult
do_open (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle **method_handle,
	 GnomeVFSURI *uri,
	 GnomeVFSOpenMode mode,
	 GnomeVFSContext *context)
{
	FileHandle *handle = NULL;
	char *path, *name, *unescaped, *display_name;
	int type, fd, flags;

	DEBUG_SMB(("do_open() %s mode %d\n",
				gnome_vfs_uri_to_string (uri, 0), mode));

	if ((mode & GNOME_VFS_OPEN_READ) &&
			(mode & GNOME_VFS_OPEN_WRITE)) {
		flags = O_RDWR;
	} else if (mode & GNOME_VFS_OPEN_READ) {
		flags = O_RDONLY;
	} else if (mode & GNOME_VFS_OPEN_WRITE) {
		flags = O_WRONLY;
	} else {
		return GNOME_VFS_ERROR_INVALID_OPEN_MODE;
	}

	LOCK_SMB ();
	name = get_type_from_uri (uri, &type);
	UNLOCK_SMB();

	DEBUG_SMB(("do_open() get_type_from_uri () name: %s type: %d\n",
				name, type));

	if (type == -1 && (mode & GNOME_VFS_OPEN_WRITE)) {
		/* file wasn't found and we want to write must be coming 
		 * from do_create */
		type = SMBC_FILE;
		flags = O_CREAT|O_WRONLY|O_TRUNC;
	}

	switch (type) {
	case SMBC_FILE_SHARE:
	case SMBC_DIR:
		return GNOME_VFS_ERROR_IS_DIRECTORY;
	case SMBC_COMMS_SHARE:
	case SMBC_IPC_SHARE:
		return GNOME_VFS_ERROR_INVALID_URI;
	case SMBC_PRINTER_SHARE:
		handle = g_new (FileHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		handle->file_data = NULL;
		handle->file_size = 0;
		break;
	case SMBC_WORKGROUP:
		handle = g_new (FileHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		display_name = gnome_vfs_unescape_string_for_display (name);
		handle->file_data = get_workgroup_data (display_name, name);
		handle->file_size = strlen (handle->file_data);
		g_free (display_name);
		break;
	case SMBC_SERVER:
		handle = g_new (FileHandle, 1);
		handle->is_data = TRUE;
		handle->offset = 0;
		display_name = gnome_vfs_unescape_string_for_display (name);
		handle->file_data = get_computer_data (display_name, name);
		handle->file_size = strlen (handle->file_data);
		g_free (display_name);
		break;
	case SMBC_FILE:
		/* Transform the URI into a completely unescaped string */
		path = gnome_vfs_uri_to_string (uri, 0);
		unescaped = gnome_vfs_unescape_string (path, "/");
		g_free (path);

		LOCK_SMB();
		fd = smbc_open (unescaped, flags, 0);
		if (fd < 0) {
			fd = errno;
			UNLOCK_SMB();
			g_free (unescaped);
			return gnome_vfs_result_from_errno_code (fd);
		}
		UNLOCK_SMB();
		g_free (unescaped);
		handle = g_new (FileHandle, 1);
		handle->is_data = FALSE;
		handle->fd = fd;
		break;
	case SMBC_LINK:
		//FIXME
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	default:
		g_assert_not_reached ();
	}

	g_free (name);

	*method_handle = (GnomeVFSMethodHandle *)handle;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  GnomeVFSContext *context)

{
	FileHandle *handle = (FileHandle *)method_handle;
	GnomeVFSResult res;

	DEBUG_SMB(("do_close()\n"));

	res = GNOME_VFS_OK;

	if (handle->is_data) {
		g_free (handle->file_data);
	} else {
		LOCK_SMB();
		if (smbc_close (handle->fd) < 0) {
			res = gnome_vfs_result_from_h_errno ();
		}
		UNLOCK_SMB();
	}

	g_free (handle);

	return res;
}

static GnomeVFSResult
do_read (GnomeVFSMethod *method,
	 GnomeVFSMethodHandle *method_handle,
	 gpointer buffer,
	 GnomeVFSFileSize num_bytes,
	 GnomeVFSFileSize *bytes_read,
	 GnomeVFSContext *context)
{
	FileHandle *handle = (FileHandle *)method_handle;
	GnomeVFSResult res;
	ssize_t n;

	DEBUG_SMB(("do_read() %Lu bytes\n", num_bytes));

	if (handle->is_data) {
		if (handle->offset >= handle->file_size) {
			n = 0;
		} else {
			n = MIN (num_bytes, handle->file_size - handle->offset);
			memcpy (buffer, handle->file_data + handle->offset, n);
		}
	} else {
		LOCK_SMB();
		n = smbc_read (handle->fd, buffer, num_bytes);
		UNLOCK_SMB();
	}

	/* Can only happen when reading from smb: */
	if (n < 0) {
		*bytes_read = 0;
		res = gnome_vfs_result_from_h_errno ();
	} else {
		res = GNOME_VFS_OK;
	}

	*bytes_read = n;

	if (n == 0) {
		return GNOME_VFS_ERROR_EOF;
	}

	handle->offset += n;

	return res;
}

static GnomeVFSResult
do_write (GnomeVFSMethod *method,
	  GnomeVFSMethodHandle *method_handle,
	  gconstpointer buffer,
	  GnomeVFSFileSize num_bytes,
	  GnomeVFSFileSize *bytes_written,
	  GnomeVFSContext *context)


{
	GnomeVFSResult res;
	FileHandle *handle = (FileHandle *)method_handle;
	ssize_t written;

	DEBUG_SMB (("do_write() %p\n", method_handle));

	LOCK_SMB();
	written = smbc_write (handle->fd, (void *)buffer, num_bytes);
	UNLOCK_SMB();

	if (written < 0) {
		res = gnome_vfs_result_from_h_errno ();
		*bytes_written = 0;
	} else {
		res = GNOME_VFS_OK;
		*bytes_written = written;
	}

	return res;
}

static GnomeVFSResult
do_create (GnomeVFSMethod *method,
	   GnomeVFSMethodHandle **method_handle,
	   GnomeVFSURI *uri,
	   GnomeVFSOpenMode mode,
	   gboolean exclusive,
	   guint perm,
	   GnomeVFSContext *context)

{
	DEBUG_SMB (("do_create() %s mode %d\n",
				gnome_vfs_uri_to_string (uri, 0), mode));

	return do_open (method, method_handle, uri, GNOME_VFS_OPEN_WRITE,
			context);
}

static GnomeVFSResult
do_get_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  GnomeVFSFileInfo *file_info,
		  GnomeVFSFileInfoOptions options,
		  GnomeVFSContext *context)

{
	struct stat st;
	char *path, *unescaped, *name;
	int err, type;

	DEBUG_SMB (("do_get_file_info() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	if (gnome_vfs_uri_has_parent (uri) == 0) {
		file_info->name = g_strdup ("/");
		file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->mime_type = g_strdup ("x-directory/normal");
		return GNOME_VFS_OK;
	}

	LOCK_SMB();
	name = get_type_from_uri (uri, &type);
	if (type == SMBC_FILE || type == SMBC_DIR) {
		/* Transform the URI into a completely unescaped string */
		path = gnome_vfs_uri_to_string (uri, 0);
		unescaped = gnome_vfs_unescape_string (path, "/");
		g_free (path);

		err = smbc_stat (unescaped, &st);
		if (err < 0) {
			err = errno;
			UNLOCK_SMB();
			g_free (name);
			g_free (unescaped);
			return gnome_vfs_result_from_errno_code (err);
		}

		UNLOCK_SMB();
		g_free (unescaped);
		gnome_vfs_stat_to_file_info (file_info, &st);
		file_info->name = name;

		if (type  == SMBC_DIR) {
			file_info->valid_fields = file_info->valid_fields
				| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
			file_info->mime_type = g_strdup ("x-directory/normal");
		}

		if (type == SMBC_FILE) {
			file_info->valid_fields = file_info->valid_fields
				| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
			file_info->mime_type = g_strdup (
				gnome_vfs_mime_type_from_name(name));
		}

		return GNOME_VFS_OK;
	}
	UNLOCK_SMB();

	file_info->name = name;

	switch (type) {
	case SMBC_FILE_SHARE:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		//FIXME why do we need a different mimetype ?
		file_info->mime_type = g_strdup ("x-directory/normal");
		//file_info->mime_type = g_strdup ("x-directory/smb-share");
		break;
	case SMBC_WORKGROUP:
	case SMBC_SERVER:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE
			| GNOME_VFS_FILE_INFO_FIELDS_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("application/x-gnome-app-info");
		break;
	case SMBC_PRINTER_SHARE:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE
			| GNOME_VFS_FILE_INFO_FIELDS_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("application/x-smb-printer");
		break;
	case SMBC_LINK:
		//FIXME
		break;
	case -1:
		return GNOME_VFS_ERROR_NOT_FOUND;
	}

	DEBUG_SMB (("do_get_file_info()\n"
				"name: %s\n"
				"smb type: %d\n"
				"mimetype: %s\n"
				"type: %d\n",
				file_info->name, type,
				file_info->mime_type, file_info->type));

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_get_file_info_from_handle (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileInfo *file_info,
		GnomeVFSFileInfoOptions options,
		GnomeVFSContext *context)
{
	FileHandle *handle = (FileHandle *)method_handle;
	struct stat st;
	int err;

	err = smbc_fstat (handle->fd, &st);
	if (err < 0) {
		return gnome_vfs_result_from_errno_code (err);
	}

	gnome_vfs_stat_to_file_info (file_info, &st);

	return GNOME_VFS_OK;
}

static gboolean
do_is_local (GnomeVFSMethod *method,
	     const GnomeVFSURI *uri)
{
	DEBUG_SMB (("do_is_local(): %s\n", gnome_vfs_uri_to_string (uri,
					GNOME_VFS_URI_HIDE_NONE)));

	return FALSE;
}

typedef struct {
	int fd;
	char *unescaped;
} DirectoryHandle;

static GnomeVFSResult
do_open_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle **method_handle,
		   GnomeVFSURI *uri,
		   GnomeVFSFileInfoOptions options,
		   GnomeVFSContext *context)

{
	DirectoryHandle *directory_handle;
	int fd, err;
	char *path, *unescaped;

	DEBUG_SMB(("do_open_directory() %s\n",
		gnome_vfs_uri_to_string (uri, 0)));

	path = gnome_vfs_uri_to_string (uri, 0);
	unescaped = gnome_vfs_unescape_string (path, "/");
	g_free (path);

	/* smb:// and smb:/// give us different results */
	if (strcmp (unescaped, "smb:") == 0
			|| strcmp (unescaped, "smb:///") == 0) {
		g_free (unescaped);
		unescaped = g_strdup ("smb://");
	}
	DEBUG_SMB(("do_open_directory() %s\n", unescaped));

	LOCK_SMB();
	fd = smbc_opendir (unescaped);

	if (fd < 0) {
		err = errno;
		UNLOCK_SMB();
		g_free (unescaped);
		return gnome_vfs_result_from_errno_code (err);
	}

	UNLOCK_SMB();

	/* Construct the handle */
	directory_handle = g_new0 (DirectoryHandle, 1);
	directory_handle->fd = fd;
	directory_handle->unescaped = unescaped;
	*method_handle = (GnomeVFSMethodHandle *) directory_handle;

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_close_directory (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSContext *context)
{
	DirectoryHandle *directory_handle = (DirectoryHandle *) method_handle;
	GnomeVFSResult res;
	int err;

	DEBUG_SMB(("do_close_directory: %p\n", directory_handle));

	if (directory_handle == NULL)
		return GNOME_VFS_OK;

	LOCK_SMB ();
	err = smbc_closedir (directory_handle->fd);
	if (err < 0) {
		res = gnome_vfs_result_from_h_errno ();
	} else {
		res = GNOME_VFS_OK;
	}
	UNLOCK_SMB ();

	g_free (directory_handle->unescaped);
	g_free (directory_handle);

	return res;
}

static GnomeVFSResult
do_read_directory (GnomeVFSMethod *method,
		   GnomeVFSMethodHandle *method_handle,
		   GnomeVFSFileInfo *file_info,
		   GnomeVFSContext *context)
{
	DirectoryHandle *dh = (DirectoryHandle *) method_handle;
	struct smbc_dirent *entry;
	struct stat st;
	char *unescaped;
	int err;

	DEBUG_SMB (("do_read_directory()\n"));

	LOCK_SMB();
	entry = smbc_readdir (dh->fd);

	if (entry == NULL) {
		err = errno;
		UNLOCK_SMB();

		if (err < 0) {
			return gnome_vfs_result_from_errno_code (err);
		} else {
			return GNOME_VFS_ERROR_EOF;
		}
	}
	UNLOCK_SMB();

	while (entry->smbc_type == SMBC_COMMS_SHARE
			|| entry->smbc_type == SMBC_IPC_SHARE
			|| is_hidden_entry (entry->name))
	{
		LOCK_SMB ();
		entry = smbc_readdir (dh->fd);
		UNLOCK_SMB();

		if (entry == NULL) {
			break;
		}
	}

	if (entry == NULL) {
		return GNOME_VFS_ERROR_EOF;
	}

	file_info->name = g_strndup (entry->name, entry->namelen);
	DEBUG_SMB (("do_read_directory (): read %s\n", file_info->name));

	file_info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_NONE;

	switch (entry->smbc_type)
	{
	case SMBC_FILE_SHARE:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
		file_info->mime_type = g_strdup ("x-directory/smb-share");
		break;
	case SMBC_WORKGROUP:
	case SMBC_SERVER:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE
			| GNOME_VFS_FILE_INFO_FIELDS_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("application/x-gnome-app-info");
		break;
	case SMBC_PRINTER_SHARE:
		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE
			| GNOME_VFS_FILE_INFO_FIELDS_TYPE;
		file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		file_info->mime_type = g_strdup ("application/x-smb-printer");
	case SMBC_COMMS_SHARE:
	case SMBC_IPC_SHARE:
		break;
	case SMBC_DIR:
	case SMBC_FILE:
		unescaped = g_build_filename (G_DIR_SEPARATOR_S,
				dh->unescaped, file_info->name, NULL);
		/* Hack around g_build_filename adding a / at the start */
		g_message ("unescaped %s", unescaped + 1);
		if (smbc_stat (unescaped + 1, &st) == 0) {
			gnome_vfs_stat_to_file_info (file_info, &st);
		}
		g_free (unescaped);

		file_info->valid_fields = file_info->valid_fields
			| GNOME_VFS_FILE_INFO_FIELDS_TYPE;

		if (entry->smbc_type == SMBC_DIR) {
			file_info->type = GNOME_VFS_FILE_TYPE_DIRECTORY;
			file_info->mime_type = g_strdup ("x-directory/normal");
			file_info->valid_fields = file_info->valid_fields
				| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		} else {
			file_info->type = GNOME_VFS_FILE_TYPE_REGULAR;
			file_info->mime_type = g_strdup (
				gnome_vfs_mime_type_from_name(file_info->name));
			file_info->valid_fields = file_info->valid_fields
				| GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE;
		}
		break;
	case SMBC_LINK:
		//FIXME
		break;
	default:
		g_assert_not_reached ();
	}

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_seek (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSSeekPosition whence,
		GnomeVFSFileOffset offset,
		GnomeVFSContext *context)
{
	FileHandle *handle = (FileHandle *)method_handle;
	GnomeVFSResult res;
	int meth_whence;
	off_t ret;

	switch (whence) {
	case GNOME_VFS_SEEK_START:
		meth_whence = SEEK_SET;
		break;
	case GNOME_VFS_SEEK_CURRENT:
		meth_whence = SEEK_CUR;
		break;
	case GNOME_VFS_SEEK_END:
		meth_whence = SEEK_END;
		break;
	default:
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	LOCK_SMB();
	ret = smbc_lseek (handle->fd, (off_t) offset, meth_whence);
	if (ret == (off_t) -1) {
		res = gnome_vfs_result_from_h_errno ();
	} else {
		res = GNOME_VFS_OK;
	}
	UNLOCK_SMB();

	return res;
}

static GnomeVFSResult
do_tell (GnomeVFSMethod *method,
		GnomeVFSMethodHandle *method_handle,
		GnomeVFSFileOffset *offset_return)
{
	FileHandle *handle = (FileHandle *)method_handle;
	GnomeVFSResult res;
	off_t ret;

	LOCK_SMB();
	ret = smbc_lseek (handle->fd, (off_t) 0, SEEK_CUR);
	if (ret == (off_t) -1) {
		res = gnome_vfs_result_from_h_errno ();
	} else {
		res = GNOME_VFS_OK;
	}
	UNLOCK_SMB();

	*offset_return = (GnomeVFSFileOffset) ret;

	return res;
}

static GnomeVFSResult
do_unlink (GnomeVFSMethod *method,
	   GnomeVFSURI *uri,
	   GnomeVFSContext *context)
{
	char *path, *unescaped;
	int type, err;

	DEBUG_SMB (("do_unlink() %s\n",
				gnome_vfs_uri_to_string (uri, 0)));

	LOCK_SMB();
	path = get_type_from_uri (uri, &type);
	UNLOCK_SMB();
	g_free (path);

	switch (type) {
	case SMBC_FILE_SHARE:
	case SMBC_PRINTER_SHARE:
	case SMBC_COMMS_SHARE:
	case SMBC_IPC_SHARE:
	case SMBC_WORKGROUP:
	case SMBC_SERVER:
		return GNOME_VFS_ERROR_NOT_PERMITTED;
	case SMBC_DIR:
	case SMBC_FILE:
		/* Transform the URI into a completely unescaped string */
		path = gnome_vfs_uri_to_string (uri, 0);
		unescaped = gnome_vfs_unescape_string (path, "/");
		g_free (path);

		err = smbc_unlink (unescaped);
		g_free (unescaped);

		if (err < 0) {
			err = errno;
			g_free (unescaped);
			return gnome_vfs_result_from_errno_code (err);
		}
		return GNOME_VFS_OK;
	case SMBC_LINK:
	default:
		g_assert_not_reached ();
	}

	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_check_same_fs (GnomeVFSMethod *method,
		  GnomeVFSURI *a,
		  GnomeVFSURI *b,
		  gboolean *same_fs_return,
		  GnomeVFSContext *context)
{
	char *server1;
	char *server2;
	char *path1;
	char *path2;
	char *p1, *p2;

	DEBUG_SMB (("do_check_same_fs()\n"));

	server1 =
		gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (a),
					   NULL);
	server2 =
		gnome_vfs_unescape_string (gnome_vfs_uri_get_host_name (b),
					   NULL);
	path1 =
		gnome_vfs_unescape_string (gnome_vfs_uri_get_path (a), NULL);
	path2 =
		gnome_vfs_unescape_string (gnome_vfs_uri_get_path (b), NULL);
                                                    
	if (!server1 || !server2 || !path1 || !path2 ||
	    (strcmp (server1, server2) != 0)) {
		g_free (server1);
		g_free (server2);
		g_free (path1);
		g_free (path2);
		*same_fs_return = FALSE;
		return GNOME_VFS_OK;
	}
                             
	p1 = path1;
	p2 = path2;
	if (*p1 == '/') {
		p1++;
	}
	if (*p2 == '/') {
		p2++;
	}

	/* Make sure both URIs are on the same share: */
	while (*p1 && *p2 && *p1 == *p2 && *p1 != '/') {
		p1++;
		p2++;
	}
	if (*p1 == 0 || *p2 == 0 || *p1 != *p2) {
		*same_fs_return = FALSE;
	} else {
		*same_fs_return = TRUE;
	}
                                            
	g_free (server1);
	g_free (server2);
	g_free (path1);
	g_free (path2);

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_move (GnomeVFSMethod *method,
	 GnomeVFSURI *old_uri,
	 GnomeVFSURI *new_uri,
	 gboolean force_replace,
	 GnomeVFSContext *context)
{
	GnomeVFSResult res;
	char *old_path, *new_path;
	char *old_unescaped, *new_unescaped;
	int err;

	DEBUG_SMB (("do_move() %s %s\n",
				gnome_vfs_uri_to_string (old_uri, 0),
				gnome_vfs_uri_to_string (new_uri, 0)));

	res = GNOME_VFS_OK;

	/* Transform the URI into a completely unescaped string */
	old_path = gnome_vfs_uri_to_string (old_uri, 0);
	old_unescaped = gnome_vfs_unescape_string (old_path, "/");
	g_free (old_path);

	new_path = gnome_vfs_uri_to_string (new_uri, 0);
	new_unescaped = gnome_vfs_unescape_string (new_path, "/");
	g_free (new_path);

	LOCK_SMB();
	err = smbc_rename (old_unescaped, new_unescaped);
	if (err < 0) {
		err = errno;
		if (err == EXDEV) {
			res = GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM;
		} else if (err == EEXIST && force_replace != FALSE) {
			/* If the target exists and force_replace is TRUE */
			err = smbc_unlink (new_unescaped);
			if (err < 0) {
				err = errno;
				res = gnome_vfs_result_from_errno_code (err);
			} else {
				err = smbc_rename (old_unescaped,
						new_unescaped);
				if (err < 0) {
					err = errno;
					res = gnome_vfs_result_from_errno_code
						(err);
				}
			}
		} else {
			res = gnome_vfs_result_from_errno_code (err);
		}
	}
	UNLOCK_SMB();

	g_free (old_unescaped);
	g_free (new_unescaped);

	return res;
}

static GnomeVFSResult
do_truncate_handle (GnomeVFSMethod *method,
		    GnomeVFSMethodHandle *method_handle,
		    GnomeVFSFileSize where,
		    GnomeVFSContext *context)

{
	DEBUG_SMB(("do_truncate_handle\n"));
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
}

static GnomeVFSResult
do_make_directory (GnomeVFSMethod *method,
		   GnomeVFSURI *uri,
		   guint perm,
		   GnomeVFSContext *context)
{
	char *path, *unescaped;
	int err;

	if (gnome_vfs_uri_has_parent (uri) == 0) {
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}

	/* Transform the URI into a completely unescaped string */
	path = gnome_vfs_uri_to_string (uri, 0);
	unescaped = gnome_vfs_unescape_string (path, "/");
	g_free (path);

	LOCK_SMB();
	if (smbc_mkdir (unescaped, perm) < 0) {
		err = errno;
		UNLOCK_SMB();
		g_free (unescaped);
		return gnome_vfs_result_from_errno_code (err);
	}

	UNLOCK_SMB();
	g_free (unescaped);

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_remove_directory (GnomeVFSMethod *method,
		     GnomeVFSURI *uri,
		     GnomeVFSContext *context)
{
	char *name, *path, *unescaped;
	int type, err;

	if (gnome_vfs_uri_has_parent (uri) == 0) {
		return GNOME_VFS_ERROR_ACCESS_DENIED;
	}


	LOCK_SMB();
	name = get_type_from_uri (uri, &type);
	UNLOCK_SMB();
	g_free (name);

	if (type != SMBC_DIR) {
		if (type == SMBC_FILE_SHARE) {
			return GNOME_VFS_ERROR_ACCESS_DENIED;
		}

		return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	}

	/* Transform the URI into a completely unescaped string */
	path = gnome_vfs_uri_to_string (uri, 0);
	unescaped = gnome_vfs_unescape_string (path, "/");
	g_free (path);

	LOCK_SMB();
	if (smbc_rmdir (unescaped) < 0) {
		err = errno;
		UNLOCK_SMB();
		g_free (unescaped);
		return gnome_vfs_result_from_errno_code (err);
	}

	UNLOCK_SMB();

	return GNOME_VFS_OK;
}

static GnomeVFSResult
do_set_file_info (GnomeVFSMethod *method,
		  GnomeVFSURI *uri,
		  const GnomeVFSFileInfo *info,
		  GnomeVFSSetFileInfoMask mask,
		  GnomeVFSContext *context)
{
	char *old_path, *old_unescaped;
	int err;

	DEBUG_SMB (("do_set_file_info: mask %x\n", mask));

	if (mask & GNOME_VFS_SET_FILE_INFO_TIME) {
		return GNOME_VFS_ERROR_NOT_SUPPORTED;
	}

	old_path = gnome_vfs_uri_to_string (uri, 0);
	old_unescaped = gnome_vfs_unescape_string (old_path, "/");
	g_free (old_path);

	if (mask & GNOME_VFS_SET_FILE_INFO_OWNER) {
		LOCK_SMB();
		err = smbc_chown (old_unescaped, info->uid, info->gid);
		if (err < 0) {
			err = errno;
			UNLOCK_SMB();
			g_free (old_unescaped);
			return gnome_vfs_result_from_errno_code (err);
		}
		UNLOCK_SMB();
	}

	if (mask & GNOME_VFS_SET_FILE_INFO_PERMISSIONS) {
		LOCK_SMB();
		err = smbc_chmod (old_unescaped, info->permissions);
		if (err < 0) {
			err = errno;
			UNLOCK_SMB();
			g_free (old_unescaped);
			return gnome_vfs_result_from_errno_code (err);
		}
		UNLOCK_SMB();
	}

	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
		GnomeVFSURI *parent_uri;
		char *new_unescaped, *new_name, *parent_unescaped, *parent;

		/* Construct the new URL */
		new_name = gnome_vfs_unescape_string(info->name, "/");
		DEBUG_SMB (("set_info: set new name: %s\n", new_name));

		parent_uri = gnome_vfs_uri_get_parent (uri);
		parent = gnome_vfs_uri_to_string (parent_uri, 0);
		gnome_vfs_uri_unref (parent_uri);
		parent_unescaped = gnome_vfs_unescape_string (parent, "/");
		g_free (parent);

		new_unescaped = g_build_filename (G_DIR_SEPARATOR_S,
				parent_unescaped, new_name, NULL);

		g_free (parent_unescaped);
		g_free (new_name);

		LOCK_SMB();
		err = smbc_rename (old_unescaped, new_unescaped);
		if (err < 0) {
			err = errno;
			UNLOCK_SMB();
			g_free (old_unescaped);
			g_free (new_unescaped);
			return gnome_vfs_result_from_errno_code (err);
		}

		UNLOCK_SMB();

		g_free (new_unescaped);
	}

	g_free (old_unescaped);

	return GNOME_VFS_OK;
}

//FIXME caching?
static void
auth_fn(const char *server, const char *share,
		char *workgroup, int wgmaxlen, char *username, int unmaxlen,
		char *password, int pwmaxlen)
{
	GnomeVFSModuleCallbackAuthenticationIn in_args;
	GnomeVFSModuleCallbackAuthenticationOut out_args;

	DEBUG_SMB (("auth_fn called: server: %s share: %s wgroup %s\n",
			server, share, workgroup));

	memset (&in_args, 0, sizeof (in_args));
	in_args.uri = g_strdup_printf ("smb://%s", server);

	memset (&out_args, 0, sizeof (out_args));

	gnome_vfs_module_callback_invoke
		(GNOME_VFS_MODULE_CALLBACK_AUTHENTICATION,
		 &in_args, sizeof (in_args),
		 &out_args, sizeof (out_args));

	if (out_args.username != NULL)
		strncpy(username, out_args.username, unmaxlen);
	if (out_args.password != NULL)
		strncpy(password, out_args.password, pwmaxlen);

	g_free (out_args.username);
	g_free (out_args.password);
	g_free (in_args.uri);
}

static GnomeVFSMethod method = {
	sizeof (GnomeVFSMethod),
	do_open,
	do_create,
	do_close,
	do_read,
	do_write,
	do_seek,
	do_tell,
	do_truncate_handle,
	do_open_directory,
	do_close_directory,
	do_read_directory,
	do_get_file_info,
	do_get_file_info_from_handle,
	do_is_local,
	do_make_directory,
	do_remove_directory,
	do_move,
	do_unlink,
	do_check_same_fs,
	do_set_file_info,
	NULL, /* do_truncate */
	NULL, /* do_find_directory */
	NULL  /* do_create_symbolic_link */
};

GnomeVFSMethod *
vfs_module_init (const char *method_name, const char *args)
{
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	smb_lock = g_mutex_new ();

	DEBUG_SMB (("<-- smb module init called -->\n"));

	if (try_init ()) return &method;
	else return NULL;
}

void
vfs_module_shutdown (GnomeVFSMethod *method)
{
	g_mutex_free (smb_lock);

	DEBUG_SMB (("<-- smb module shutdown called -->\n"));
}

