/*      xmms - esound output plugin
 *    Copyright (C) 1999      Galex Yen
 *
 *      this program is free software
 *
 *      Description:
 *              This program is an output plugin for xmms v0.9 or greater.
 *              The program uses the esound daemon to output audio in order
 *              to allow more than one program to play audio on the same
 *              device at the same time.
 *
 *              Contains code Copyright (C) 1998-1999 Mikael Alm, Olle Hallnas,
 *              Thomas Nillson and 4Front Technologies
 */

#include "artsout.h"
#include "config.h"

static gpointer buffer;
static gboolean going = FALSE, prebuffer, paused = FALSE, remove_prebuffer = FALSE;
static gint buffer_size, prebuffer_size, blk_size = 4096;
static gint rd_index = 0, wr_index = 0;
static gint output_time_offset = 0;
static guint64 written = 0, output_bytes = 0;
static gint bps, ebps;
static gint sample_width=8;
static gint flush;
static gint format, channels, frequency, latency;
static gint input_bps, input_format, input_frequency, input_channels;
static arts_stream_t stream = 0;
static pthread_t buffer_thread;
static gboolean realtime = FALSE;


static void artsd_setup_format(AFormat fmt,gint rate, gint nch)
{
	format = fmt;
	frequency = rate;
	channels = nch;
	bps = rate * nch;
	latency = arts_stream_get(stream,ARTS_P_SERVER_LATENCY)  * frequency / 44100 * channels;

	switch(fmt) {
	        case FMT_U8:
		case FMT_S8:
			latency *= 2;
			sample_width = 8;
			break;
		case FMT_U16_LE:
		case FMT_U16_BE:
		case FMT_U16_NE:
		case FMT_S16_LE:
		case FMT_S16_BE:
		case FMT_S16_NE:
			sample_width = 16;
			bps *= 2;
			break;
	}
}


gint artsd_get_written_time(void)
{
	if (!going)
		return 0;
	return (gint) ((written * 1000) / input_bps);
}

gint artsd_get_output_time(void)
{
	guint64 bytes;

	if (!stream || !going)
		return 0;

	bytes = output_bytes;
	if (!paused)
		bytes -= (bytes < latency ? bytes : latency);

	return output_time_offset + (gint) ((bytes * 1000) / ebps);
}

gint artsd_used(void)
{
	if (realtime)
		return 0;
	else
	{
		if (wr_index >= rd_index)
			return wr_index - rd_index;
		return buffer_size - (rd_index - wr_index);
	}
}

gint artsd_playing(void)
{
	if (!going)
		return FALSE;
	if (!artsd_used())
		return FALSE;

	return TRUE;
}

gint artsd_free(void)
{
	if (!realtime)
	{
		if (remove_prebuffer && prebuffer)
		{
			prebuffer = FALSE;
			remove_prebuffer = FALSE;
		}
		if (prebuffer)
			remove_prebuffer = TRUE;

		if (rd_index > wr_index)
			return (rd_index - wr_index) - 1;
		return (buffer_size - (wr_index - rd_index)) - 1;
	}
	else
	{
		if (paused)
			return 0;
		else
			return 1000000;
	}
}

static void artsd_write_audio(gpointer data,gint length)
{
        int errorcode;
	AFormat new_format;
	gint new_frequency,new_channels;
	EffectPlugin *ep;

	new_format = input_format;
	new_frequency = input_frequency;
	new_channels = input_channels;

	ep = get_current_effect_plugin();
	if(effects_enabled() && ep && ep->query_format)
	{
		ep->query_format(&new_format,&new_frequency,&new_channels);
	}

	if(new_format != format || new_frequency != frequency || new_channels != channels)
	{
		output_time_offset += (gint) ((output_bytes * 1000) / ebps);
		output_bytes = 0;
		artsd_setup_format(new_format, new_frequency, new_channels);
		frequency = new_frequency;
		channels = new_channels;
		arts_close_stream(stream);
		stream = NULL;
  		artsd_set_audio_params();
	}
	if(effects_enabled() && ep && ep->mod_samples)
		length = ep->mod_samples(&data,length, input_format, input_frequency, input_channels);

	errorcode = arts_write(stream,data,length);
	if(errorcode < 0)
		fprintf(stderr,"arts_write error: %s\n",arts_error_text(errorcode));
	else
  		output_bytes += errorcode;
}


void artsd_write(gpointer ptr, gint length)
{

	gint cnt, off = 0;

	if (!realtime)
	{
		remove_prebuffer = FALSE;

		written += length;
		while (length > 0)
		{
			cnt = MIN(length, buffer_size - wr_index);
			memcpy((gchar *)buffer + wr_index, (gchar *)ptr + off, cnt);
			wr_index = (wr_index + cnt) % buffer_size;
			length -= cnt;
			off += cnt;

		}
	}
	else
	{
		if (paused)
			return;
		artsd_write_audio(ptr,length);
		written += length;

	}

}

void artsd_close(void)
{
	if (!going)
		return;
	wr_index = 0;
	rd_index = 0;
	going = 0;
	if (!realtime)
		pthread_join(buffer_thread, NULL);
	else
	{
		arts_close_stream(stream);
		stream = NULL;
		arts_free();
	}
}

void artsd_flush(gint time)
{
	if (!realtime)
	{
		flush = time;
		while (flush != -1)
			xmms_usleep(10000);
	}
	else
	{
		arts_close_stream(stream);
		stream = NULL;
		artsd_set_audio_params();
		output_time_offset = time;
		written = (guint64)(time / 10) * (guint64)(input_bps / 100);
		output_bytes = 0;
	}
}

void artsd_pause(short p)
{
	paused = p;
}

void *artsd_loop(void *arg)
{
	gint length, cnt;


	while (going)
	{
		if (artsd_used() > prebuffer_size)
			prebuffer = FALSE;
	        if (paused && stream != NULL) {
			arts_close_stream(stream);
			stream = NULL;
		}
		if (artsd_used() > 0 && !paused && !prebuffer)
		{
			if (stream == NULL)
				artsd_set_audio_params();

			length = MIN(blk_size, artsd_used());
			while (length > 0)
			{
				cnt = MIN(length,buffer_size-rd_index);
				artsd_write_audio((gchar *)buffer + rd_index, cnt);
				rd_index=(rd_index+cnt)%buffer_size;
				length-=cnt;
			}
		}
		else
			xmms_usleep(10000);

		if (flush != -1)
		{
			output_time_offset = flush;
			written = (guint64)(flush / 10) * (guint64)(input_bps / 100);
			rd_index = wr_index = output_bytes = 0;
			flush = -1;
			prebuffer = TRUE;
		}

	}

	arts_close_stream(stream);
	arts_free();
	stream=NULL;
	g_free(buffer);
	pthread_exit(NULL);
}

void artsd_set_audio_params(void)
{
	/* frequency = GUINT16_SWAP_LE_BE( frequency ); */
		ebps = channels * frequency;
		switch(format) {
		        case FMT_U8:
			case FMT_S8:
				sample_width = 8;
				break;
			case FMT_U16_LE:
			case FMT_U16_BE:
			case FMT_U16_NE:
			case FMT_S16_LE:
			case FMT_S16_BE:
			case FMT_S16_NE:
				sample_width = 16;
				ebps *= 2;
				break;
		}
        stream = arts_play_stream(frequency, sample_width, channels, "xmms - plugin");
}

gint artsd_open(AFormat fmt, gint rate, gint nch)
{
	int errorcode;
	errorcode = arts_init();
	if (errorcode < 0) {
		fprintf(stderr,"arts_init error: %s\n", arts_error_text(errorcode));
		return 1;
	}
        artsd_setup_format(fmt,rate,nch);

	input_format = format;
	input_channels = channels;
	input_frequency = frequency;
	input_bps = bps;

	realtime = xmms_check_realtime_priority();

	if(!realtime)
	{
		buffer_size = (arts_cfg.buffer_size * input_bps) / 1000;
		if (buffer_size < 8192)
			buffer_size = 8192;
		prebuffer_size = (buffer_size * arts_cfg.prebuffer) / 100;
		if (buffer_size - prebuffer_size < 4096)
			prebuffer_size = buffer_size - 4096;

		buffer = g_malloc0(buffer_size);
	}
	flush = -1;
	prebuffer = 1;
	wr_index = rd_index = output_time_offset = written = output_bytes = 0;
	paused = FALSE;
	remove_prebuffer = FALSE;


	artsd_set_audio_params();
	if ((int)stream == -1)
	{
                arts_close_stream(stream);
                stream = NULL;
                arts_free();
                g_free(buffer);
                return 0;
	}
        latency = arts_stream_get(stream,ARTS_P_SERVER_LATENCY)  * frequency / 44100 * channels;
	going = 1;

	if (!realtime)
		pthread_create(&buffer_thread, NULL, artsd_loop, NULL);
	return 1;
}







