/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ksimview.h"
#include "ksimview.moc"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ksimsysinfo.h"
#include "ksim.h"
#include <ksimframe.h>
#include <label.h>
#include <common.h>
#include <themeloader.h>
#include <pluginmodule.h>
#include <pluginloader.h>
#include <ksimconfig.h>
#include <ksimpref.h>
#include "baselist.h"
#include <themetypes.h>

#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kdesktopfile.h>
#include <kwin.h>
#include <krun.h>

#include <qfile.h>
#include <qbitmap.h>
#include <qtimer.h>
#include <qlayout.h>
#include <qobjectlist.h>
#include <qpainter.h>
#include <qcursor.h>

#include <unistd.h>
#include <sys/param.h>

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

KSim::MainView::MainView(KConfig *config,
   bool loadPlugins, QWidget *parent, const char *name)
   : QWidget(parent, name), DCOPObject("KSim")
{
  // create the local "themes" and "monitors" dirs
  if (!makeDirs())
    KMessageBox::sorry(0, i18n("There was an error while trying to create "
       "the local dirs. This could be caused by permission problems."));

  setBackgroundMode(PaletteBackground);

  if (parent->isA("KSim::MainWindow"))
    m_topLevel = static_cast<KSim::MainWindow *>(parent);
  else
    m_topLevel = 0;

  m_moving = false;
  m_oldLocation = 1;
  m_prefDialog = 0L;

  m_config = new KSim::Config(config);
  m_pluginMenu = new QPopupMenu(this, "m_pluginMenu");

  // Make sure the current theme is valid
  // if not then revert to the default theme
  KSim::ThemeLoader::self().validate();

  // parse the current theme in case it doesnt have the correct dir structure
  if (KSim::ThemeLoader::currentName() != "ksim")
    KSim::ThemeLoader::self().parseDir(KSim::ThemeLoader::currentUrl(),
       KSim::ThemeLoader::currentAlternative());

  kdDebug() << "loadPlugins: " << loadPlugins << endl;

  m_subLayout = new QVBoxLayout(this);

  m_topFrame = new KSim::Frame(m_topLevel, KSim::Types::TopFrame, this);
  m_subLayout->addWidget(m_topFrame);

  m_sizeLayout = new QHBoxLayout;
  m_subLayout->addLayout(m_sizeLayout);

  m_leftFrame = new KSim::Frame(m_topLevel, KSim::Types::LeftFrame, this);
  m_sizeLayout->addWidget(m_leftFrame);

  m_pluginLayout = new QVBoxLayout;
  m_sizeLayout->addLayout(m_pluginLayout);

  m_hostLabel = new KSim::Label(KSim::Types::Host, this);
  m_hostLabel->installEventFilter(this);
  m_pluginLayout->addWidget(m_hostLabel);

  char hostName[MAXHOSTNAMELEN];
  if (gethostname(hostName, sizeof(hostName)))
    m_hostLabel->setText(i18n("Unknown"));
  else {
    QCString host(hostName);
    int dotLocation = host.find(".");
    if (!m_config->displayFqdn() && dotLocation != -1)
      host.truncate(dotLocation);

    m_hostLabel->setText(host);
  }

  m_sysinfo = new KSim::Sysinfo(m_config, this);
  m_pluginLayout->addWidget(m_sysinfo);

  m_bottomFrame = new KSim::Frame(m_topLevel, KSim::Types::BottomFrame, this);
  m_subLayout->addWidget(m_bottomFrame);

  m_rightFrame = new KSim::Frame(m_topLevel, KSim::Types::RightFrame, this);
  m_sizeLayout->addWidget(m_rightFrame);

  connect(&KSim::PluginLoader::self(),
     SIGNAL(pluginLoaded(const KSim::Plugin &)),
     this, SLOT(addMonitor(const KSim::Plugin &)));

  KSim::ThemeLoader::self().themeColours(this);

  // load the plugins and create the plugin menu
  if (loadPlugins) {
    addPlugins();
    createPluginMenu();
  }
}

KSim::MainView::~MainView()
{
  delete m_config;
}

void KSim::MainView::show()
{
  QWidget::show();
  maskMainView();
}

void KSim::MainView::cleanup()
{
  // delete our configuration dialog and remove
  // the plugin config pages from the configuration
  // dialog before we unload our plugins
  delete m_prefDialog;

  // Clean up the singleton instances
  KSim::PluginLoader::cleanup();
}

KSim::Config *KSim::MainView::config() const
{
  return m_config;
}

QPopupMenu *KSim::MainView::pluginMenu() const
{
  return m_pluginMenu;
}

bool KSim::MainView::makeDirs() const
{
  QString homeDir = locateLocal("data", "ksim");
  QString themeDir = homeDir + QString::fromLatin1("/themes");
  QString monitorDir = homeDir + QString::fromLatin1("/monitors");

  // return true if the dirs already exist
  if (QFile::exists(themeDir) && QFile::exists(monitorDir))
    return true;

  bool themeCreated = KStandardDirs::makeDir(themeDir);
  bool monitorCreated = KStandardDirs::makeDir(monitorDir);

  if (!themeCreated || !monitorCreated)
    return false;

  return true;
}

const QString &KSim::MainView::hostname() const
{
  return m_hostLabel->text();
}

void KSim::MainView::maskMainView()
{
  parentWidget()->clearMask();
  if (!m_topFrame->background()->mask() ||
      !m_leftFrame->background()->mask() ||
      !m_rightFrame->background()->mask() ||
      !m_bottomFrame->background()->mask())
    return;

  QBitmap topPixmap(*m_topFrame->background()->mask());
  QBitmap leftPixmap(*m_leftFrame->background()->mask());
  QBitmap rightPixmap(*m_rightFrame->background()->mask());
  QBitmap bottomPixmap(*m_bottomFrame->background()->mask());

  QSize insideSize(m_pluginLayout->geometry().size());

  // make a cleared bigrect where we can put our pixmap masks on
  QBitmap bigBitmap(m_subLayout->geometry().size(), true);

  // better return if our bitmap is null so we can avoid crashes
  if (bigBitmap.isNull())
    return;

  QPainter painter;
  painter.begin(&bigBitmap);
  painter.setBrush(color1);
  painter.setPen(color1);
  painter.drawRect(m_pluginLayout->geometry());
  painter.drawPixmap(0, 0, topPixmap);
  painter.drawPixmap(0, topPixmap.height(), leftPixmap);
  painter.drawPixmap(insideSize.width() + leftPixmap.width(),
     topPixmap.height(), rightPixmap);
  painter.drawPixmap(0, height() - bottomPixmap.height(),
     bottomPixmap);

  painter.end();
  parentWidget()->setMask(bigBitmap);
}

void KSim::MainView::reparseConfig(bool emitReload,
   const KSim::ChangedPluginList &list)
{
  kdDebug() << "creating a new view" << endl;

  // Update the theme information for the theme classes
  bool themeChanged = KSim::ThemeLoader::self().isDifferent();
  if (themeChanged)
    KSim::ThemeLoader::self().reload();

  const KSim::PluginList &pluginList = KSim::PluginLoader::self().pluginList();
  KSim::PluginList::ConstIterator plugin;
  for (plugin = pluginList.begin(); plugin != pluginList.end(); ++plugin) {
    if ((*plugin).configPage()) {
      (*plugin).configPage()->saveConfig();
      (*plugin).configPage()->config()->sync();
    }

    if (!(*plugin).isDifferent() && (*plugin).isEnabled()) {
      kdDebug() << "Recreating " << (*plugin).name() << "'s view" << endl;
      kapp->processEvents();
      // Make sure the plugins background colour is
      // set to the correct theme colour
      if (themeChanged && (*plugin).view())
        KSim::ThemeLoader::self().themeColours((*plugin).view());

      // Reload the config file and and recreate the plugins view
      if ((*plugin).configPage() && (*plugin).configPage()->config())
        (*plugin).configPage()->config()->reparseConfiguration();

      if ((*plugin).view()) {
        (*plugin).view()->config()->reparseConfiguration();
        (*plugin).view()->reparseConfig();
      }
    }
  }

  ChangedPluginList::ConstIterator it;
  for (it = list.begin(); it != list.end(); ++it) {
    if ((*it).isDifferent()) {
      if ((*it).isEnabled()) { // Go through the added/removed plugins and load/unload them
        addPlugin(KDesktopFile((*it).filename()));
        m_prefDialog->createPage((*it).libName());
      }
      else {
        m_prefDialog->removePage((*it).libName());
        removePlugin(KDesktopFile((*it).filename()));
      }
    }
  }

  // reparse various non-plugin specific code
  // recreate the plugin menu and the dock icon
  createPluginMenu();
  KSim::BaseList::configureObjects(themeChanged);
  if (themeChanged)
    KSim::ThemeLoader::self().themeColours(this);

  m_sysinfo->createView();
  maskMainView();

  if (m_topLevel)
    m_topLevel->reparse();

  if (emitReload)
    emit reload();

  kdDebug() << "new view created" << endl;
}

void KSim::MainView::addPlugins()
{
  QStringList locatedFiles = KGlobal::dirs()->findAllResources("data", "ksim/monitors/*.desktop");
  QStringList::ConstIterator it;
  for (it = locatedFiles.begin(); it != locatedFiles.end(); ++it)
    addPlugin(KDesktopFile(*it, true));
}

void KSim::MainView::addPlugin(const KDesktopFile &file, bool force)
{
  if (force || m_config->enabledMonitor(file.readEntry("X-KSIM-LIBRARY")))
    KSim::PluginLoader::self().loadPlugin(file);
}

// provided for convenience
void KSim::MainView::removePlugin(const KDesktopFile &file)
{
  KSim::PluginLoader::self().unloadPlugin(file.readEntry("X-KSIM-LIBRARY").local8Bit());
}

void KSim::MainView::createPluginMenu()
{
  m_pluginMenu->clear();
  KSim::PluginList::ConstIterator it;
  const KSim::PluginList &pluginList = KSim::PluginLoader::self().pluginList();
  for (it = pluginList.begin(); it != pluginList.end(); ++it) {
    if (!(*it).view())
      continue;

    m_pluginMenu->insertItem((*it).icon(), (*it).name(),
       (*it).view()->menu());
  }
}

void KSim::MainView::addMonitor(const KSim::Plugin &plugin)
{
  if (!plugin.view())
    return;

  plugin.view()->reparent(this, 0, QPoint(0, 0), true);
  KSim::ThemeLoader::self().themeColours(plugin.view());

//  int location = m_config->monitorLocation(plugin.libName());
//  if (location == -1)
//    location = m_oldLocation + 1;
//  else {
//    location += 1;
//    if (location > m_oldLocation)
//      location = m_oldLocation + 1;
//  }

//  kdDebug() << "m_oldLocation: " << m_oldLocation << endl;
//  kdDebug() << "location: " << location << endl;
  m_pluginLayout->addWidget(plugin.view());
  connect(plugin.view(), SIGNAL(runCommand(const QCString &)),
     SLOT(runCommand(const QCString &)));

//  if (location > m_oldLocation)
//    m_oldLocation = location;
}

void KSim::MainView::runCommand(const QCString &name)
{
  if (name.isNull())
    return;

  kdDebug() << "runCommand(" << name.mid(5) << ")" << endl;
  QString exec = m_config->monitorCommand(name.mid(5));
  kdDebug() << "exec is " << exec << endl;
  KRun::runCommand(exec);
}

void KSim::MainView::preferences()
{
  if (m_prefDialog == 0L) {
    m_prefDialog = new KSim::ConfigDialog(m_config, this, "m_prefDialog");
    connect(m_prefDialog, SIGNAL(reparse(bool, const KSim::ChangedPluginList &)),
       this, SLOT(reparseConfig(bool, const KSim::ChangedPluginList &)));
  }

  m_prefDialog->exec();
  destroyPref();
}

void KSim::MainView::mouseMoveEvent(QMouseEvent *e)
{
  if (m_moving)
    m_topLevel->move(QCursor::pos() - m_mousePoint);

  QWidget::mouseMoveEvent(e);
}

void KSim::MainView::mouseReleaseEvent(QMouseEvent *)
{
  m_moving = false;
}

bool KSim::MainView::eventFilter(QObject *o, QEvent *e)
{
  if ((o == m_hostLabel) && e->type() == QEvent::MouseButtonPress) {
    m_mousePoint = mapFromGlobal(QCursor::pos());
    m_mousePoint.setX(m_mousePoint.x() + x());
    m_mousePoint.setY(m_mousePoint.y() + y());

    QMouseEvent *mouseEvent = static_cast<QMouseEvent *>(e);
    if (mouseEvent->button() == LeftButton) {
      m_moving = true;
      return true;
    }

    if (mouseEvent->button() == RightButton) {
      m_topLevel->contextMenu();
      return true;
    }
  }

  return QWidget::eventFilter(o, e);
}

void KSim::MainView::resizeEvent(QResizeEvent *)
{
  maskMainView();
}

void KSim::MainView::paletteChange(const QPalette &)
{
  // Call true here to fool the KSim::Base pointers to
  // think our theme changed, we can afford todo this
  // since the palette doesnt change often.
  KSim::BaseList::configureObjects(true);

  // Probably find a better way to do this
  KSim::PluginList::ConstIterator it;
  const KSim::PluginList &pluginList = KSim::PluginLoader::self().pluginList();
  for (it = pluginList.begin(); it != pluginList.end(); ++it)
    KSim::ThemeLoader::self().themeColours((*it).view());
}

void KSim::MainView::destroyPref()
{
  if (m_prefDialog != 0L) {
    kdDebug() << "deleting KSimPref" << endl;
    delete m_prefDialog;
    m_prefDialog = 0L;
  }
}
