/*
   This file is part of KDE/aRts (Noatun) - xine integration
   Copyright (C) 2002 Ewald Snel <ewald@rambo.its.tudelft.nl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <xine/xine_internal.h>
#include <xine/video_out.h>
#include <xine/video_out_x11.h>
#include "video_fifo_out.h"


/*
 * copy snapshot (ARGB format).
 */
Pixmap vo_fifo_snapshot( xine_arts_video *video, xine_t *xine )
{
    uint8_t *src0, *src1, *src2;
    uint8_t *base[3];
    uint32_t *dst;
    uint32_t *pixels;
    uint8_t clip8[1024];
    int i, x, y, pitch, cy, cu, cv;
    Pixmap pixmap = -1;
    int width, height, ratio_code, format;

    if (!xine_get_current_frame( xine, &width, &height, &ratio_code,
				 &format, &base[0], &base[1], &base[2] ))
    {
	return (Pixmap)-1;
    }

    pixels = (uint32_t *)malloc( width * height * sizeof(uint32_t) );
    pitch  = (format == IMGFMT_YV12) ? width : 2*width;
    dst	   = pixels;

    for (i=0; i < 1024; i++)
    {
	clip8[i] = (i < 384) ? 0 : ((i >= 640) ? 255 : (i - 384));
    }

    for (y=0; y < height; y++)
    {
	src0 = base[0] + y*pitch;

	if (format == IMGFMT_YV12)
	{
	    src1 = base[1] + (y >> 1)*(pitch / 2);
	    src2 = base[2] + (y >> 1)*(pitch / 2);
	}

	for (x=0; x < width; x++)
	{
	    if (format == IMGFMT_YV12)
	    {
		cy = 9535*(src0[x] - 16) + (384 << 13);
		cu = src1[x >> 1] - 128;
		cv = src2[x >> 1] - 128;
	    }
	    else
	    {
		cy = 9535*(src0[2*x] - 16) + (384 << 13);
		cu = src0[4*(x >> 1) + 1] - 128;
		cv = src0[4*(x >> 1) + 3] - 128;
	    }

	    *dst++ = (clip8[(cy + 13074*cv) >> 13] << 16) |
		     (clip8[(cy -  3203*cu - 6660*cv) >> 13] << 8) |
		      clip8[(cy + 16531*cu) >> 13];
	}
    }

    /* encode 32 bit RGBA data as 1bpp pixmap for easy transport */
    pixmap = XCreateBitmapFromData( video->display, video->xcom_window,
				    (char *)pixels, 32*width, height );
    XFlush( video->display );

    free( pixels );

    return pixmap;
}

/*
 * xine frame output.
 */
void vo_fifo_resize_notify( xine_arts_video *video )
{
    XEvent event;

    /* Resize notify signal for front-ends */
    memset( &event, 0, sizeof(event) );

    event.type			= ClientMessage;
    event.xclient.window	= video->window;
    event.xclient.message_type	= video->xcom_resize;
    event.xclient.format	= 32;
    event.xclient.data.l[0]	= video->width;
    event.xclient.data.l[1]	= video->height;

    XSendEvent( video->display, video->window, True, 0, &event );

    XFlush( video->display );
}

static void frame_output_cb( void *user_data,
			     int video_width, int video_height,
			     int *dest_x, int *dest_y,
			     int *dest_width, int *dest_height,
			     int *win_x, int *win_y )
{
    xine_arts_video *video = (xine_arts_video *)user_data;
    Window child, root;
    unsigned int u;
    int n;

    XLockDisplay( video->display );

    XGetGeometry( video->display, video->window, &root, &n, &n,
		  (unsigned int *)dest_width, (unsigned int *)dest_height,
		  &u, &u );

    XTranslateCoordinates( video->display, video->window,
			   root, 0, 0, win_x, win_y, &child );

    *dest_x	 = 0;
    *dest_y	 = 0;

    if ((!video->ds_counter || (--video->ds_counter == 0)) &&
	(video_width != video->width || video_height != video->height))
    {
	video->width  = video_width;
	video->height = video_height;

	vo_fifo_resize_notify( video );
    }
    XUnlockDisplay( video->display );
}

static void dest_size_cb( void *user_data, int video_width, int video_height,
			  int *dest_width, int *dest_height )
{
    xine_arts_video *video = (xine_arts_video *)user_data;
    Window root;
    unsigned int u;
    int n;

    XLockDisplay( video->display );

    XGetGeometry( video->display, video->window, &root, &n, &n,
		  (unsigned int *)dest_width, (unsigned int *)dest_height,
		  &u, &u );

    if (video_width != video->width || video_height != video->height)
    {
	video->width  = video_width;
	video->height = video_height;

	vo_fifo_resize_notify( video );
    }
    XUnlockDisplay( video->display );

    /* Reset 'seen dest_size_cb' counter */
    video->ds_counter = 25;
}

vo_driver_t *init_video_out_plugin( config_values_t *config, xine_arts_video *video )
{
    x11_visual_t vis;
    vo_driver_t *vo;
    int screen;
    char *video_driver_id;

    vo			  = 0;
    video->width	  = 0;
    video->height	  = 0;
    video->ds_counter	  = 0;
    video->shm_completion = (XShmQueryExtension( video->display ) == True)
			    ? XShmGetEventBase( video->display ) + ShmCompletion
			    : -1;

    screen = DefaultScreen( video->display );

    vis.display		  = video->display;
    vis.screen		  = screen;
    vis.d		  = video->window;
    vis.dest_size_cb	  = &dest_size_cb;
    vis.frame_output_cb	  = &frame_output_cb;
    vis.user_data	  = video;
    vis.display_ratio	  = (double)(DisplayWidth( video->display, screen ) *
				     DisplayHeightMM( video->display, screen ))
				  / (DisplayWidthMM( video->display, screen ) *
				     DisplayHeight( video->display, screen ));

    video_driver_id = config->register_string( config,
					       (char *)"video.driver",
					       (char *)"auto",
					       (char *)"video driver to use",
					       NULL, NULL, NULL );

    if (!strcasecmp( video_driver_id, "XShm" ))
    {
	vo = xine_load_video_output_plugin( config, video_driver_id,
					    VISUAL_TYPE_X11, (void *)&vis );
    }
    if (!vo)
    {
	vo = xine_load_video_output_plugin( config, (char *)"Xv",
					    VISUAL_TYPE_X11, (void *)&vis );
    }
    if (!vo)
    {
	vo = xine_load_video_output_plugin( config, (char *)"XShm",
					    VISUAL_TYPE_X11, (void *)&vis );
    }
    if (!vo)
    {
	vo = xine_load_video_output_plugin( config, (char *)"OpenGL",
					    VISUAL_TYPE_X11, (void *)&vis );
    }
    if (!vo)
    {
	fprintf( stderr, "Error: xine video output plugin not available\n" );
    }

    return vo;
}
