/*
   This file is part of KDE/aRts (Noatun) - xine integration
   Copyright (C) 2002 Ewald Snel <ewald@rambo.its.tudelft.nl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <noatun/app.h>
#include <noatun/engine.h>
#include <noatun/player.h>
#include <noatun/stdaction.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kurldrag.h>
#include <kmenubar.h>
#include <kpopupmenu.h>
#include <kstatusbar.h>
#include <kstdaction.h>
#include <kurl.h>
#include <qaccel.h>
#include <qdragobject.h>
#include <qhbox.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qstrlist.h>
#include <qtooltip.h>
#include <qvbox.h>
#include "userinterface.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "back.xpm"
#include "eject.xpm"
#include "forward.xpm"
#include "pause.xpm"
#include "play.xpm"
#include "stop.xpm"


SimpleUI::SimpleUI()
     : KMainWindow(), UserInterface()
{
    setAcceptDrops( true );
    setCaption( i18n("Noatun") );
    setIcon( SmallIcon( "noatun" ) );

    setupCentralWidget();
    setupActions();

    // Setup menu items and default menus
    QPopupMenu *file = new QPopupMenu( this );
    action( "file_open" )->plug( file );
    file->insertSeparator();
    action( "file_properties" )->plug( file );
    file->insertSeparator();
    action( "file_quit" )->plug( file );
    menuBar()->insertItem( i18n("&File"), file );
    QPopupMenu *view = new QPopupMenu( this );
    video->action( "half_size" )->plug( view );
    video->action( "normal_size" )->plug( view );
    video->action( "double_size" )->plug( view );
    view->insertSeparator();
    video->action( "fullscreen_mode" )->plug( view );
    menuBar()->insertItem( i18n("&View"), view );
    QPopupMenu *settings = new QPopupMenu( this );
    action( "show_playlist" )->plug( settings );
    settings->insertSeparator();
    action( "options_configure" )->plug( settings );
    action( "effects" )->plug( settings );
    action( "equalizer" )->plug( settings );
    menuBar()->insertItem( i18n("&Settings"), settings );
    menuBar()->insertItem( i18n("&Actions"), napp->pluginMenu() );
    menuBar()->insertItem( i18n("&Help"), helpMenu() );

    contextMenu = video->popupMenu( this );

    statusBar()->show();

    connect( napp->player(), SIGNAL(playing()), this, SLOT(slotPlaying()) );
    connect( napp->player(), SIGNAL(stopped()), this, SLOT(slotStopped()) );
    connect( napp->player(), SIGNAL(paused()), this, SLOT(slotPaused()) );
    connect( napp->player(), SIGNAL(timeout()), this, SLOT(slotTimeout()) );
    connect( napp->player(), SIGNAL(newSong()), this, SLOT(slotChanged()) );
    connect( napp, SIGNAL(hideYourself()), this, SLOT(hide()) );
    connect( napp, SIGNAL(showYourself()), this, SLOT(show()) );

    napp->player()->handleButtons();

    resize( minimumSize() );

    // Show UI and calculate video widget frame
    show();

    extra_width  = (width() - video->width());
    extra_height = (height() - video->height());

    // Load configuration
    KConfig &config = *KGlobal::config();
    config.setGroup( "Simple" );
    QString str = config.readEntry( "View", "NormalSize" );

    if (str == "HalfSize")
	video->setHalfSize();
    else if (str == "NormalSize")
	video->setNormalSize();
    else if (str == "DoubleSize")
	video->setDoubleSize();
    else
	applyMainWindowSettings( &config, "Simple" );

    // PlayObject could be running, update video widget
    slotChanged();

    video->give();
}

SimpleUI::~SimpleUI()
{
    KConfig &config = *KGlobal::config();
    saveMainWindowSettings( &config, "Simple" );
    config.setGroup( "Simple" );
    QString str;

    if (video->isHalfSize())
	str = "HalfSize";
    else if (video->isNormalSize())
	str = "NormalSize";
    else if (video->isDoubleSize())
	str = "DoubleSize";
    else
	str = "CustomSize";

    config.writeEntry( "View", str );
}

void SimpleUI::setupActions()
{
    KStdAction::open( napp, SLOT(fileOpen()), actionCollection() );
    KStdAction::quit( napp, SLOT(quit()), actionCollection() );
    NoatunStdAction::playlist( actionCollection(), "show_playlist" );
    KStdAction::preferences( napp, SLOT(preferences()), actionCollection() );
    NoatunStdAction::effects( actionCollection(), "effects" );
    NoatunStdAction::equalizer( actionCollection(), "equalizer" );

    new KAction( i18n("&Properties..."), 0,
		 propertiesDialog, SLOT(show()),
		 actionCollection(), "file_properties" );
}

void SimpleUI::setupCentralWidget()
{
    QVBox *npWidget = new QVBox( this );
    npWidget->setMargin( 0 );
    npWidget->setSpacing( 0 );

    QHBox *topFrame = new QHBox( npWidget );
    topFrame->setFixedHeight( 20 );
    topFrame->setFrameShape( QFrame::StyledPanel );
    topFrame->setFrameShadow( QFrame::Raised );
    topFrame->setMargin( 1 );
    topFrame->setSpacing( 0 );
    stateLabel = new QLabel( topFrame );
    stateLabel->setFixedSize( 16, 16 );
    stateLabel->setPixmap( QPixmap( stop_xpm ) );
    stateLabel->setScaledContents( false );
    positionLabel = new QLabel( topFrame );
    positionLabel->setAlignment( AlignVCenter | AlignRight );

    video = new VideoFrame( npWidget );
    connect( video, SIGNAL(adaptSize(int,int)),
	     this, SLOT(slotAdaptSize(int,int)) );
    connect( video, SIGNAL(rightButtonPressed(const QPoint &)),
	     this, SLOT(slotContextMenu(const QPoint &)) );

    QHBox *ctlFrame = new QHBox( npWidget );
    ctlFrame->setFixedHeight( 38 );
    ctlFrame->setFrameShape( QFrame::StyledPanel );
    ctlFrame->setFrameShadow( QFrame::Raised );
    ctlFrame->setMargin( 6 );
    ctlFrame->setSpacing( 6 );

    QPushButton *backButton = new QPushButton( ctlFrame );
    backButton->setFixedSize( 24, 24 );
    backButton->setPixmap( QPixmap( back_xpm ) );
    QToolTip::add( backButton, i18n("Back") );
    connect( backButton, SIGNAL(clicked()), napp->player(), SLOT(back()) );

    stopButton = new QPushButton( ctlFrame );
    stopButton->setFixedSize( 24, 24 );
    stopButton->setPixmap( QPixmap( stop_xpm ) );
    QToolTip::add( stopButton, i18n("Stop") );
    connect( stopButton, SIGNAL(clicked()), napp->player(), SLOT(stop()) );

    playButton = new QPushButton( ctlFrame );
    playButton->setFixedSize( 24, 24 );
    playButton->setPixmap( QPixmap( play_xpm ) );
    QToolTip::add( playButton, i18n("Play / Pause") );
    connect( playButton, SIGNAL(clicked()), napp->player(), SLOT(playpause()) );

    QPushButton *forwButton = new QPushButton( ctlFrame );
    forwButton->setFixedSize( 24, 24 );
    forwButton->setPixmap( QPixmap( forward_xpm ) );
    QToolTip::add( forwButton, i18n("Forward") );
    connect( forwButton, SIGNAL(clicked()), napp->player(), SLOT(forward()) );

    slider = new L33tSlider( 0, 1000, 10, 0, L33tSlider::Horizontal, ctlFrame );
    slider->setFixedHeight( 24 );
    slider->setMinimumWidth( 100 );
    slider->setTickmarks( QSlider::NoMarks );
    connect( slider, SIGNAL(userChanged(int)), this, SLOT(slotSkipTo(int)) );
    connect( slider, SIGNAL(sliderMoved(int)), this, SLOT(slotSliderMoved(int)) );

    QPushButton *ejectButton = new QPushButton( ctlFrame );
    ejectButton->setFixedSize( 24, 24 );
    ejectButton->setPixmap( QPixmap( eject_xpm ) );
    QToolTip::add( ejectButton, i18n("Eject") );
    connect( ejectButton, SIGNAL(clicked()), napp->player(), SLOT(toggleListView()) );

    setCentralWidget( npWidget );

    video->setMinimumSize( minimumSizeHint().width(), 1 );

    // Create properties dialog
    propertiesDialog = new PropertiesDialog( this );
    propertiesDialog->resize( 375, 285 );
}

void SimpleUI::closeEvent( QCloseEvent * )
{
    unload();
}

void SimpleUI::dragEnterEvent( QDragEnterEvent *event )
{
    event->accept( QUriDrag::canDecode( event ) );
}

void SimpleUI::dropEvent( QDropEvent *event )
{
	KURL::List uri;
	if (KURLDrag::decode(event, uri))
	{
		for (KURL::List::Iterator i = uri.begin(); i != uri.end(); ++i)
			napp->player()->openFile(*i, false);
	}
}

void SimpleUI::slotAdaptSize( int width, int height )
{
    resize( width + extra_width, height + extra_height );
}

void SimpleUI::slotPlaying()
{
    playButton->setPixmap( QPixmap( pause_xpm ) );
    stateLabel->setPixmap( QPixmap( play_xpm ) );
    stopButton->setEnabled( true );
    slider->setEnabled( true );

    if (napp->player()->current())
	statusBar()->message( napp->player()->current().title() );
}

void SimpleUI::slotStopped()
{
    playButton->setPixmap( QPixmap( play_xpm ) );
    stateLabel->setPixmap( QPixmap( stop_xpm ) );
    stopButton->setEnabled( false );
    slider->setEnabled( false );
    slider->setValue( 0 );
    positionLabel->setText( "" );
    statusBar()->message( "" );
}

void SimpleUI::slotPaused()
{
    playButton->setPixmap( QPixmap( play_xpm ) );
    stateLabel->setPixmap( QPixmap( pause_xpm ) );
    slider->setEnabled( true );
}

void SimpleUI::slotVolumeUp()
{
    napp->player()->setVolume( QMIN( 100, (napp->player()->volume() + 8) ) );
}

void SimpleUI::slotVolumeDown()
{
    napp->player()->setVolume( QMAX( 0, (napp->player()->volume() - 8) ) );
}

void SimpleUI::slotTimeout()
{
    if (napp->player()->current() && !slider->currentlyPressed())
    {
	positionLabel->setText( napp->player()->lengthString() );
	slider->setRange( 0, (int)napp->player()->getLength() / 1000 );
	slider->setValue( (int)napp->player()->getTime() / 1000 );
    }
}

void SimpleUI::slotSkipTo( int sec )
{
    napp->player()->skipTo( sec * 1000 );
}

void SimpleUI::slotChanged()
{
    propertiesDialog->setPlayObject( napp->player()->current(),
				     napp->player()->engine()->playObject() );
}

void SimpleUI::slotContextMenu( const QPoint &pos )
{
    contextMenu->exec( pos );
}

void SimpleUI::slotSliderMoved( int sec )
{
    if (napp->player()->current())
	positionLabel->setText( napp->player()->lengthString( sec * 1000 ) );
}

#include "userinterface.moc"
