namespace Binge.Generators.CSharp
{
	using System;
	using System.Collections;
	using System.IO;
	using System.Text;
	using Binge.Bits;
	using Binge.Generators;

	public class Generator: Binge.Generators.Generator
	{
		public Generator (): base () {}
		public Generator (string tdir, string sdir): base (tdir, sdir) {}
		public Generator (string tdir, string sdir, Hashtable ns): base (tdir, sdir, ns) {}

		// Foo (): base () and Foo (): this ()
		enum BaseCallTypes
		{
			None,
			Base,
			This
		}

		public override void Generate ()
		{
			IDictionaryEnumerator de = Namespaces.GetEnumerator ();

			while (de.MoveNext ())
				Generate (de.Value as Namespace);
		}

		void Start (Namespace ns)
		{
			Prnl ("namespace {0}", ns.Name);
			Prnl ("{");
			tab++;
			Prnl ("using System;");

			foreach (string use in ns.Using)
				Prnl ("using {0};", use);
		}

		void End (Namespace ns)
		{
			tab--;
			Prnl ("}");
			Prnl ("");
		}

		void Generate (Namespace ns)
		{
			IDictionaryEnumerator de = ns.Classes.GetEnumerator ();
			while (de.MoveNext ())
			{
				Class klass = de.Value as Class;
				string name = (klass.TargetName != String.Empty ?
					klass.TargetName : klass.NativeName) + ".cs";

				writer = new StreamWriter (TargetDirectory + name, false, FileEncoding);
				Start (ns);
				Generate (klass);
				End (ns);
				writer.Close ();
			}

			de = ns.Interfaces.GetEnumerator ();
			while (de.MoveNext ())
			{
				Interface iface = de.Value as Interface;
				string name = iface.TargetName != String.Empty ?
					iface.TargetName : iface.NativeName + ".cs";

				writer = new StreamWriter (TargetDirectory + name, false, FileEncoding);
				Start (ns);
				Generate (iface);
				End (ns);
				writer.Close ();
			}

			if (ns.Enums.Count == 0)
				return;

			// Store top level enums in one file

			writer = new StreamWriter (TargetDirectory + ns.Name + "Enums.cs", false, FileEncoding);
			Start (ns);
			Prnl ("");

			de = ns.Enums.GetEnumerator ();
			while (de.MoveNext ())
			{
				Binge.Bits.Enum enm = de.Value as Binge.Bits.Enum;
				Generate (enm);
			}

			End (ns);
			writer.Close ();
		}

		void Generate (Binge.Bits.Attribute attr)
		{
			StringBuilder sb = new StringBuilder ();

			Prn ("[{0} ", attr.Name);

			if (! (attr.Args.Count > 0 && attr.KwArgs.Count > 0))
			{
				Prn ("]");
				return;
			}

			Prn (" (");

			for (int i=0; i<attr.Args.Count; i++)
			{
				Prn ((string)attr.Args[i]);

				if (i == attr.Args.Count - 1)
				{
					if (attr.KwArgs.Count == 0)
						Prn (")]");
					else
						Prn (", ");
				}
				else
					Prn (", ");
			}

			IDictionaryEnumerator de = attr.KwArgs.GetEnumerator ();

			for (int i=0; de.MoveNext (); i++)
			{
				Prn ("{0}={1}", (string)de.Key, (string)de.Value);

				if (i == attr.KwArgs.Count - 1)
					Prn (")]");
				else
					Prn (", ");
			}
		}

		void Generate (Binge.Bits.Enum enm)
		{
			Prnl ("enum {0}", enm.NativeName);
			Prnl ("{");
			tab++;

			for (int i=0; i < enm.Items.Count; i++)
			{
				EnumItem item = enm.Items[i] as EnumItem;
				string name = item.TargetName != String.Empty ?
					item.TargetName : item.NativeName;
				string value = item.TargetValue != String.Empty ?
					item.TargetValue : item.NativeValue;

				Prn ("{0}{1}", tab, name);

				if (value != String.Empty)
					Prn (" = {0}", value);

				if (enm.Items.Count - i != 1)
					Prn (",");

				Prnl ("");
			}

			tab--;
			Prnl ("}");
			Prnl ("");
		}

		string MakeMemberAccess (MemberAccess access)
		{
			if (access == MemberAccess.Public)
				return "public";
			if (access == MemberAccess.Protected)
				return "protected";
			if (access == MemberAccess.Private)
				return "private";
			if (access == MemberAccess.Internal)
				return "internal";
			if (access == MemberAccess.InternalProtected)
				return "internal protected";
			if (access == MemberAccess.Default)
				return "public";

			return "";
		}

		void Generate (Interface iface)
		{
			// FIXME
			Prnl ("// Interfaces not supported yet");
		}

		void Generate (Class klass)
		{
			foreach (string use in klass.Using)
				Prnl ("using {0};", use);

			Prnl ("");

			foreach (Binge.Bits.Attribute attr in klass.Attributes)
			{
				Generate (attr);
				Prnl ("");
			}

			string name = klass.TargetName != String.Empty ?
				klass.TargetName : klass.NativeName;

			Prn ("{0}{1} class {2}", tab, MakeMemberAccess (klass.Access), name);

			if (klass.Ancestors.Count > 0 || klass.Implements.Count > 0)
				Prn (": ");

			if (klass.Ancestors.Count == 0)
				Prn (": QtSharp");

			for (int i=0; i<klass.Ancestors.Count; i++)
			{
				Prn ((string)klass.Ancestors[i]);

				if (klass.Ancestors.Count - i != 1)
					Prn (", ");
			}

			for (int i=0; i<klass.Implements.Count; i++)
			{
				Prn ((string)klass.Implements[i]);

				if (klass.Implements.Count - i != 1)
					Prn (", ");
			}

			Prnl ("");
			Prnl ("{");
			tab++;

			foreach (Field field in klass.Fields)
				Generate (field);

			foreach (Property property in klass.Properties)
				Generate (property);

			foreach (Binge.Bits.Enum enm in klass.Enums)
				Generate (enm);

			foreach (Constructor ctor in klass.Constructors)
				Generate (ctor);

			if (klass.Destructors.Count > 0)
			{
				if (klass.Destructors.Count > 1)
					Dbg ("Warning: Multiple destructors in class '{0}', using first in list.", klass.NativeName);

				Generate (klass.Destructors[0] as Destructor);
			}

			foreach (Method method in klass.Methods)
				Generate (method);

			foreach (Method method in klass.ImportedMethods)
				Generate (method);

			tab--;
			Prnl ("}");
		}

		void Generate (MemberBase member)
		{
			string mt = member.ToString ();

			foreach (Binge.Bits.Attribute attr in member.Attributes)
			{
				Generate (attr);
				Prnl ("");
			}

			Prn ("{0} ", MakeMemberAccess (member.Access));

			if (member.IsStatic)
				Prn ("static ");

			if (member.IsReadOnly && mt == "Field")
				Prn ("readonly ");

			Field field = (Field)member; // FIXME

			string name = member.TargetName != String.Empty ?
				member.TargetName : member.NativeName;
			string type = field.TargetType != String.Empty ?
				field.TargetType : field.NativeType;

			Prn ("{0} {1}", type, name);
		}

		void Generate (Parameter param)
		{
			string type = param.TargetType != String.Empty ?
				param.TargetType : param.NativeType;
			string name = param.TargetName != String.Empty ?
				param.TargetName : param.NativeName;

			Prn ("{0} {1}", type, name);
		}

		void Generate (Field field)
		{
			Generate (field as MemberBase);
		}

		void Generate (Property prop)
		{
			if (prop.IsWrapper)
				Generate (prop.Underlying);

			Prnl ("{0} {1} {2} {3}", MakeMemberAccess (prop.Access),
				prop.IsStatic ? "static" : String.Empty,
				prop.TargetType,
				prop.TargetName != String.Empty ?
					prop.TargetName : prop.NativeName);

			Prnl ("{");
			tab++;

			Prnl ("get");
			Prnl ("{");
			tab++;

			// FIXME
			Prnl ("// Properties not implemented yet.");
			Prnl ("return null;");

			tab--;
			Prnl ("}");

			if (! prop.IsReadOnly)
			{
				Prnl ("set");
				Prnl ("{");
				tab++;

				// FIXME
				Prnl ("// Properties not implemented yet.");
				Prnl ("return null;");

				tab--;
				Prnl ("}");
			}

			tab--;
			Prnl ("}");
			Prnl ("");
		}

		private static int counter = 0;

		void Generate (Constructor ctor)
		{
			foreach (Constructor octor in ctor.Overloads)
				Generate (octor);

			string access = MakeMemberAccess (ctor.Access);
			string name = ctor.TargetName != String.Empty ?
				ctor.TargetName : ctor.NativeName;

			Prn ("{0}{1} {2} ", tab, access, name);
			PrintParams (ctor);

			if (ctor.CallType == Constructor.CallTypes.This)
				Prn (": this (");
			else if (ctor.CallType == Constructor.CallTypes.Base)
				Prn (": base (");

			if (ctor.CallType != Constructor.CallTypes.Normal)
			{
				for (int i=0; i<ctor.CallParameters.Count; i++)
				{
					Parameter p = ctor.CallParameters[i] as Parameter;
					string o;
					
					if (p.Default != String.Empty)
						o = p.Default;
					else if (p.TargetName != String.Empty)
						o = p.TargetName;
					else
						o = p.NativeName;

					Prn (o);

					if (ctor.CallParameters.Count - i > 1)
						Prn (", ");
				}

				Prn (")");
			}

			Prn ("\n");
			Prnl ("{");
			tab++;

			foreach (string line in ctor.Glue)
				Prnl (line);

			tab--;
			Prnl ("}");
			Prnl ("");
		}

		void Generate (Destructor dtor)
		{
			string name = dtor.TargetName != String.Empty ?
				dtor.TargetName : dtor.NativeName;

			Prnl ("~{1} ()", tab, name);
			Prnl ("{");
			tab++;

			foreach (string line in dtor.Glue)
				Prnl (line);

			tab--;
			Prnl ("}");
			Prnl ("");
		}

		void Generate (Method method)
		{
			foreach (Method omethod in method.Overloads)
				Generate (omethod);

			foreach (Binge.Bits.Attribute attr in method.Attributes)
			{
				Prn (tab.ToString ());
				Generate (attr);
				Prn ("\n");
			}

			string access = MakeMemberAccess (method.Access);
			string name = method.TargetName != String.Empty ?
				method.TargetName : method.NativeName;
			string rtype = method.Returns.TargetType != String.Empty ?
				method.Returns.TargetType : method.Returns.NativeType;

			StringBuilder sb = new StringBuilder ();

			// Preprocessing should ensure there are no conflicts here.
			sb.Append (method.IsExtern ? "extern " : "");
			sb.Append (method.IsStatic ? "static " : "");
			sb.Append (method.IsVirtual ? "virtual " : "");
			sb.Append (method.IsOverride ? "override " : "");
			sb.Append (method.IsNew ? "new " : "");

			string modifiers = sb.ToString ();

			Prn ("{0}{1} {2}{3} {4} ", tab, access, modifiers, rtype, name);
			PrintParams (method);

			if (method.IsExtern)
				Prn (";\n");
			else
			{
				Prnl ("");
				Prnl ("{");
				tab++;

				foreach (string line in method.Glue)
					Prnl (line);

				tab--;
				Prnl ("}");
			}

			Prnl ("");
		}

		void PrintParams (MethodBase method)
		{
			Prn ("(");

			for (int i=0; i<method.Parameters.Count; i++)
			{
				Parameter p = method.Parameters[i] as Parameter;

				string type;
				if (p.MarshallType != String.Empty) // FIXME Not such a hot idea.
					type = p.MarshallType;
				else if (p.TargetType != String.Empty)
					type = p.TargetType;
				else
					type = p.NativeType;

				string name = p.TargetName != String.Empty ?
					p.TargetName : p.NativeName;

				Prn ("{0} {1}", type, name);

				if (method.Parameters.Count - i > 1)
					Prn (", ");
			}

			Prn (")");
		}
	}
}
