/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   MODULE_NAME:		fsckpfs.c
 *
 *   COMPONENT_NAME: 	jfs utilities
 *
 *   FUNCTIONS:
 *              ait_node_get
 *              ait_node_put
 *              ait_special_read_ext1
 *              alloc_high_mem
 *              blkmap_find_bit
 *              blkmap_flush
 *              blkmap_get_ctl_page
 *              blkmap_get_page
 *              blkmap_put_ctl_page
 *              blkmap_put_page
 *              blktbl_ctl_page_put
 *              blktbl_dmap_get
 *              blktbl_dmap_put
 *              blktbl_dmaps_flush
 *              blktbl_Ln_page_get
 *              blktbl_Ln_page_put
 *              blktbl_Ln_pages_flush
 *              close_volume
 *              default_volume
 *              dnode_get
 *              ea_get
 *              fscklog_put_buffer
 *              iag_get
 *              iag_get_first
 *              iag_get_next
 *              iag_put
 *              iags_flush
 *              inodes_flush
 *              inode_get
 *              inode_get_first_fs
 *              inode_get_next
 *              inode_put
 *              inotbl_get_ctl_page
 *              inotbl_put_ctl_page
 *              mapctl_flush
 *              mapctl_get
 *              mapctl_put
 *              node_get
 *              open_volume
 *              readwrite_device
 *              recon_dnode_assign
 *              recon_dnode_get
 *              recon_dnode_put
 *              recon_dnode_release
 *              imapleaf_get
 *              open_device_read
 *              open_device_rw_exclusive
 *
*/
/*
 * defines and includes common among the xfsck modules
 */
#include "xfsckint.h"
#include "jfs_byteorder.h"

#include <fcntl.h>
#include <unistd.h>

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * superblock buffer pointer
 *
 *      defined in xchkdsk.c
 */
extern struct superblock *sb_ptr;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * fsck aggregate info structure pointer
 *
 *      defined in xchkdsk.c
 */
extern struct fsck_agg_record *agg_recptr;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * For message processing
 *
 *      defined in xchkdsk.c
 */
extern char *verbose_msg_ptr;
extern char *msgs_txt_ptr;
extern unsigned long msgs_txt_maxlen;

extern char *MsgText[];

extern char *Vol_Label;

extern char message_parm_0[];
extern char message_parm_1[];
extern char message_parm_2[];
extern char message_parm_3[];
extern char message_parm_4[];
extern char message_parm_5[];
extern char message_parm_6[];
extern char message_parm_7[];
extern char message_parm_8[];
extern char message_parm_9[];

extern char *msgprms[];
extern int16_t  msgprmidx[];

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * Device information.
 *
 *      defined in xchkdsk.c
 */
extern HFILE Dev_IOPort;
extern uint32_t Dev_blksize;
extern int32_t Dev_SectorSize;
extern agg_byteidx_t ondev_jlog_byte_offset;

/* + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + + +
 *
 * The current processing phase.
 *
 *      defined in xchkdsk.c
 *      constants defining possible values are defined in xfsck.h
 */
extern int current_fsck_phase;

/* VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
 *
 * The following are internal to this file
 *
 */

retcode_t imapleaf_get ( agg_byteidx_t, xtpage_t ** );

retcode_t open_device_read ( char  * );

retcode_t open_device_rw_exclusive ( char  * );

uint32_t checksum( uint8_t *, uint32_t );

/* VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV */


/*****************************************************************************
 * NAME: ait_node_get
 *
 * FUNCTION:  Read the specified AIT xTree node into the specified buffer
 *
 * PARAMETERS:
 *      node_fsblk_offset  - input - offset, in aggregate blocks, into the
 *                                   aggregate, of the xTree node wanted
 *      xtpage_ptr            - input - pointer an fsck buffer into which the
 *                                   xTree node should be read.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t ait_node_get ( fsblkidx_t  node_fsblk_offset,
                         xtpage_t    *xtpage_ptr         )
{
  retcode_t      anodg_rc = FSCK_OK;
  agg_byteidx_t  node_start_byte;
  uint32_t       bytes_read;

  node_start_byte = node_fsblk_offset * sb_ptr->s_bsize;

  if ( (agg_recptr->ondev_wsp_fsblk_offset != 0) &&
       (node_fsblk_offset > agg_recptr->ondev_wsp_fsblk_offset) ) { /*
      * the offset is beyond the range
      * valid for fileset objects
      */
    /*
     * This case is not caused by an I/O error, but by
     * invalid data in an inode.  Let the caller handle
     * the consequences.
     */
    anodg_rc = FSCK_BADREADTARGET2;
  } else {
    anodg_rc = readwrite_device( node_start_byte,
                                 XTPAGE_SIZE,
                                 &(bytes_read),
                                 (void *) xtpage_ptr,
                                 fsck_READ );
    if ( anodg_rc == FSCK_OK ) {  /* read appears successful */
      if ( bytes_read < XTPAGE_SIZE ) {  /*
         * didn't get the minimum number of bytes
         */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_NODE1 );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", anodg_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)node_start_byte );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", XTPAGE_SIZE );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_read );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        anodg_rc = FSCK_FAILED_BADREAD_NODE1;
      } else { /* got the minimum number of bytes */

        /* swap if on big endian machine */
        ujfs_swap_xtpage_t( xtpage_ptr );
      }
    } else {  /* bad return code from read */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_READ_NODE );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", anodg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)node_start_byte );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", XTPAGE_SIZE );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_read );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      anodg_rc = FSCK_FAILED_READ_NODE;
    }  /* end else bad return code from read */
  }  /* end else we'll have to read it from the disk */

  return( anodg_rc );
}  /* end of ait_node_get () */


/*****************************************************************************
 * NAME: ait_node_put
 *
 * FUNCTION:  Write the specified buffer into the specified AIT xTree node
 *
 * PARAMETERS:
 *      node_fsblk_offset  - input - offset, in aggregate blocks, to which
 *                                           the buffer is to be written
 *      xtpage_ptr         - input - pointer to the buffer to write
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t ait_node_put ( fsblkidx_t  node_fsblk_offset,
                         xtpage_t    *xtpage_ptr         )
{
  retcode_t      anodp_rc = FSCK_OK;
  agg_byteidx_t  node_start_byte;
  uint32_t       bytes_written;

  node_start_byte = node_fsblk_offset * sb_ptr->s_bsize;

  ujfs_swap_xtpage_t(xtpage_ptr);
  anodp_rc = readwrite_device( node_start_byte, PSIZE,
                               &bytes_written, (void *) xtpage_ptr,
                               fsck_WRITE                           );
  ujfs_swap_xtpage_t(xtpage_ptr);

  if ( anodp_rc == FSCK_OK ) {
    if ( bytes_written != PSIZE ) {  /* didn't write correct number of bytes */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 2 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_BADWRITE_FBLKMP );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", anodp_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)node_start_byte );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", PSIZE );
      msgprms[4] = message_parm_4;
      msgprmidx[3] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );

      anodp_rc= FSCK_BADWRITE_FBLKMP;
    } 
  } else {  /* else the write was not successful */
    /*
     * message to user
     */
    msgprms[0] = message_parm_0;
    msgprmidx[0] = fsck_metadata;
    msgprms[1] = Vol_Label;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", 3 );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    fsck_send_msg( fsck_URCVWRT, 3 );
    /*
     * message to debugger
     */
    sprintf( message_parm_0, "%d", FSCK_BADWRITE_FBLKMP );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    sprintf( message_parm_1, "%d", anodp_rc );
    msgprms[1] = message_parm_1;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", fsck_WRITE );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    sprintf( message_parm_3, "%lld", (long long)node_start_byte );
    msgprms[3] = message_parm_3;
    msgprmidx[3] = 0;
    sprintf( message_parm_4, "%d", PSIZE );
    msgprms[4] = message_parm_4;
    msgprmidx[4] = 0;
    sprintf( message_parm_5, "%d", bytes_written );
    msgprms[5] = message_parm_5;
    msgprmidx[5] = 0;
    fsck_send_msg( fsck_ERRONAGG, 6 );

    anodp_rc= FSCK_BADWRITE_FBLKMP;
  }  /* end else the write was not successful */

  return( anodp_rc );
}  /* end of ait_node_put () */


/*****************************************************************************
 * NAME: ait_special_read_ext1
 *
 * FUNCTION:  Reads the first extent of either the Primary or Secondary
 *            Aggregate Inode Table into the fsck inode buffer.
 *
 * PARAMETERS:
 *      which_ait  - input - { fsck_primary | fsck_secondary }
 *
 * NOTES:  This routine is used during the early stages of fsck processing
 *         when the normal mechanisms for reading inodes have not yet been
 *         established.
 *
 *         This routine may also be used later in fsck processing as a fast
 *         read routine for the inodes in the first extent of the AIT.
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t ait_special_read_ext1( int  which_ait )
{
  retcode_t      aree_rc = FSCK_OK;
  retcode_t      intermed_rc = FSCK_OK;
  agg_byteidx_t  offset_1stext;
  dinode_t       *bp;
  int            i;

  aree_rc = inodes_flush();

  /*
   * calculate the byte offset of the first extent
   */
  if ( (which_ait == fsck_primary) ) {
    offset_1stext = AITBL_OFF;
  } else {  /* must be secondary */
    offset_1stext = addressPXD(&(sb_ptr->s_ait2)) *
                    sb_ptr->s_bsize;
  }  /* end else must be secondary */

  if ( agg_recptr->ino_buf_agg_offset != offset_1stext ) {  
    /* we don't already have the one we want */
    intermed_rc = readwrite_device( offset_1stext,
                                    INODE_IO_BUFSIZE,
                                    &(agg_recptr->ino_buf_data_len),
                                    (void *) agg_recptr->ino_buf_ptr,
                                    fsck_READ                        );
    if ( intermed_rc != FSCK_OK ) {  /* didn't get anything */
      aree_rc = FSCK_CANTREADAITEXT1;
      msgprms[0] = message_parm_0;
      msgprmidx[0] = which_ait;
      fsck_send_msg( fsck_CANTREADAITEXT1, 1 );
    } else {  /* got something */

      /* swap if on big endian machine */
      ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, GET,
		       sb_ptr->s_flag);

      agg_recptr->ino_for_aggregate = -1;
      agg_recptr->ino_which_it = which_ait;
      agg_recptr->ino_buf_1st_ino = 0;
      agg_recptr->ino_fsnum = 0;
      agg_recptr->ino_buf_agg_offset = offset_1stext;
      PXDaddress(&(agg_recptr->ino_ixpxd), offset_1stext / sb_ptr->s_bsize );
      PXDlength(&(agg_recptr->ino_ixpxd),
                INODE_EXTENT_SIZE / sb_ptr->s_bsize );

      if ( agg_recptr->ino_buf_data_len < INODE_EXTENT_SIZE ) {  /*
         * didn't get enough
         */
        aree_rc = FSCK_CANTREADAITEXT1;
        msgprms[0] = message_parm_0;
        msgprmidx[0] = which_ait;
        fsck_send_msg( fsck_CANTREADEAITEXT1, 1 );
      }  /* end didn't get enough */
    }  /* end else got something */
  }  /* we don't already have the one we want */

  return( aree_rc );
}  /* end of ait_special_read_ext1 () */


/*****************************************************************************
 * NAME: blkmap_find_bit
 *
 * FUNCTION:  Calculate the position, in the fsck workspace block map,
 *            of the bit representing the given aggregate block.
 *
 * PARAMETERS:
 *      blk_number   - input - ordinal number of the aggregate block whose
 *                             bit is to be located
 *      page_number  - input - pointer to a variable in which to return
 *                             the ordinal number of the page, in the fsck
 *                             workspace block map, containing the bit
 *                             for the given block
 *      byte_offset  - input - pointer to a variable in which to return
 *                             the ordinal number of the byte, in page_number
 *                             page, containing the bit for the given block
 *      mask_ptr     - input - pointer to a variable in which to return
 *                             a mask to apply to the byte at byte_offset
 *                             in order to reference the bit for the given
 *                             block
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blkmap_find_bit( fsblkidx_t        blk_number,
                           mappgidx_t        *page_number,
                           reg_idx_t         *byte_offset,
                           fsck_bitmask_ptr  mask_ptr    )
{
  retcode_t  bfb_rc = FSCK_OK;
  uint64_t   remainder;
  uint32_t   bit_position;

  *page_number = blk_number >> log2BITSPERPAGE;
  remainder = blk_number - ((*page_number) << log2BITSPERPAGE);
  *byte_offset = (remainder >> log2BITSPERDWORD) * BYTESPERDWORD;
  bit_position = remainder - ((*byte_offset) << log2BITSPERBYTE);
  *mask_ptr = 0x80000000u >> bit_position;

  return( bfb_rc );
}  /* end of blkmap_find_bit () */


/*****************************************************************************
 * NAME: blkmap_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current block map buffer has been updated since
 *            the most recent read operation, write the buffer contents to
 *            the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blkmap_flush ( )
{
  retcode_t  bmpf_rc = FSCK_OK;
  uint32_t   bytes_written;
  fsck_blk_map_page_t  *fsck_bmpt_ptr;
  int        i;

  if ( agg_recptr->blkmp_buf_write ) {  /* buffer has been updated since
    * most recent write
    */

    /* swap if on big endian machine */
    fsck_bmpt_ptr = agg_recptr->blkmp_buf_ptr;
    swap_multiple(ujfs_swap_fsck_blk_map_page, fsck_bmpt_ptr, 4);
    bmpf_rc = readwrite_device( agg_recptr->blkmp_agg_offset,
                                agg_recptr->blkmp_buf_data_len,
                                &bytes_written,
                                (void *) agg_recptr->blkmp_buf_ptr,
                                fsck_WRITE                          );
    fsck_bmpt_ptr = agg_recptr->blkmp_buf_ptr;
    swap_multiple(ujfs_swap_fsck_blk_map_page, fsck_bmpt_ptr, 4);

    if ( bmpf_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->blkmp_buf_data_len ) {
        agg_recptr->blkmp_buf_write = 0; /* buffer has been written
                               * to the device and won't need to be
                               * written again unless/until the
                               * buffer contents have been altered.
                               */
      } else {  /* didn't write correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 2 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_FBMAP_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", bmpf_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->blkmp_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->blkmp_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[3] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONWSP, 6 );

        bmpf_rc = FSCK_FAILED_FBMAP_BADFLUSH;
      }  /* end else didn't write correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 3 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_FBMAP_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", bmpf_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->blkmp_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->blkmp_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONWSP, 6 );

      bmpf_rc = FSCK_FAILED_FBMAP_FLUSH;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */

  return( bmpf_rc );
}  /* end of blkmap_flush () */


/*****************************************************************************
 * NAME: blkmap_get_ctl_page
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            write the contents of the given buffer over the current fsck
 *            fsck workspace block map control page on the device.
 *
 * PARAMETERS:
 *      blk_ctlptr  - input -  pointer to the buffer into the current fsck
 *                              workspace block map control page should be read.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t  blkmap_get_ctl_page ( blk_cntrlptr  blk_ctlptr )
{
  retcode_t  bmgcp_rc = FSCK_OK;
  uint32_t   bytes_read;

  if ( agg_recptr->processing_readwrite ) {  /* have write access */
    bmgcp_rc = readwrite_device( agg_recptr->ondev_wsp_byte_offset,
                                 BYTESPERPAGE,
                                 &bytes_read,
                                 (void *) agg_recptr->blkmp_ctlptr,
                                 fsck_READ                          );
    if ( bmgcp_rc == FSCK_OK ) {

      /* swap if on big endian machine */
      ujfs_swap_fsck_blk_map_hdr( agg_recptr->blkmp_ctlptr );

      if ( bytes_read != (uint32_t) BYTESPERPAGE ) {  /*
        * didn't read correct number of bytes
        */
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_BADREAD_FBLKMP );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", bmgcp_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->ondev_wsp_byte_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", BYTESPERPAGE );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_read );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONWSP, 6 );

        bmgcp_rc = FSCK_BADREAD_FBLKMP;
      } /* end else didn't write correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_BADREAD_FBLKMP );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", bmgcp_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->ondev_wsp_byte_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", BYTESPERPAGE );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_read );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONWSP, 6 );

      bmgcp_rc = FSCK_BADREAD_FBLKMP;
    }  /* end else the write was not successful */
  }  /* end have write access */

  return( bmgcp_rc );
}  /* end of blkmap_get_ctl_page () */


/*****************************************************************************
 * NAME: blkmap_get_page
 *
 * FUNCTION:  Read the requested fsck workspace block map page into and/or
 *            locate the requested fsck workspace block map page in the
 *            fsck block map buffer.
 *
 * PARAMETERS:
 *      page_num       - input - ordinal number of the fsck workspace
 *                               block map page which is needed
 *      addr_page_ptr  - input - pointer to a variable in which to return
 *                               the address of the page in an fsck buffer
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blkmap_get_page( mappgidx_t   page_num,
                           blk_pageptr  *addr_page_ptr )
{
  retcode_t      bgp_rc = FSCK_OK;
  agg_byteidx_t  page_start_byte, page_end_byte;
  fsck_blk_map_page_t  *fsck_bmpt_ptr;
  int            i;

  page_start_byte = page_num * BYTESPERPAGE;
  page_end_byte = page_start_byte + BYTESPERPAGE -1;
  if ( (page_start_byte >= agg_recptr->blkmp_blkmp_offset) &&
       (page_end_byte <= (agg_recptr->blkmp_blkmp_offset +
                          agg_recptr->blkmp_buf_data_len   ))  ) {  /*
    * the desired page is already in the buffer
    */
    *addr_page_ptr = (blk_pageptr) ( (char *) agg_recptr->blkmp_buf_ptr + 
                     (page_start_byte - agg_recptr->blkmp_blkmp_offset) );
  } else {  /* else need to read it in from dasd */
    if ( !agg_recptr->processing_readwrite ) {  
      /* this isn't supposed
       * to happen.  If we don't have write access
       * to the aggregate then we're always supposed
       * to get a hit in the buffer!
       */
      bgp_rc = FSCK_INTERNAL_ERROR_6;
    } else {  /* we have read/write access */
      bgp_rc = blkmap_flush();   /* if the buffer has been modified,
                                  * write it to dasd
                                  */
      if ( bgp_rc == FSCK_OK ) {  /* successful write */
        agg_recptr->blkmp_blkmp_offset = page_start_byte;
        agg_recptr->blkmp_agg_offset =
        page_start_byte + (BYTESPERPAGE * 1) +
        agg_recptr->ondev_wsp_byte_offset; /*
          * The byte offset in the fsck block map plus
          * one page of control information plus the
          * aggregate bytes which precede the on-dasd
          * fsck workspace
          */
        bgp_rc = readwrite_device( agg_recptr->blkmp_agg_offset,
                                   agg_recptr->blkmp_buf_length,
                                   &(agg_recptr->blkmp_buf_data_len),
                                   (void *) agg_recptr->blkmp_buf_ptr,
                                   fsck_READ                          );

        if ( bgp_rc == FSCK_OK ){  /* successful read */
          /* swap if on big endian machine */
	  fsck_bmpt_ptr = agg_recptr->blkmp_buf_ptr;
          swap_multiple(ujfs_swap_fsck_blk_map_page, fsck_bmpt_ptr, 4);

          if ( agg_recptr->blkmp_buf_data_len >= BYTESPERPAGE ) {
            *addr_page_ptr = agg_recptr->blkmp_buf_ptr;
          } else {  /* but didn't get enough to continue */
            /*
             * message to user
             */
            msgprms[0] = message_parm_0;
            msgprmidx[0] = fsck_metadata;
            msgprms[1] = Vol_Label;
            msgprmidx[1] = 0;
            fsck_send_msg( fsck_URCVREAD, 2 );
            /*
             * message to debugger
             */
            sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_FBLKMP );
            msgprms[0] = message_parm_0;
            msgprmidx[0] = 0;
            sprintf( message_parm_1, "%d", bgp_rc );
            msgprms[1] = message_parm_1;
            msgprmidx[1] = 0;
            sprintf( message_parm_2, "%d", fsck_READ );
            msgprms[2] = message_parm_2;
            msgprmidx[2] = 0;
            sprintf( message_parm_3, "%lld", (long long)agg_recptr->blkmp_agg_offset );
            msgprms[3] = message_parm_3;
            msgprmidx[3] = 0;
            sprintf( message_parm_4, "%d", agg_recptr->blkmp_buf_length );
            msgprms[4] = message_parm_4;
            msgprmidx[4] = 0;
            sprintf( message_parm_5, "%d", agg_recptr->blkmp_buf_data_len );
            msgprms[5] = message_parm_5;
            msgprmidx[5] = 0;
            fsck_send_msg( fsck_ERRONWSP, 6 );

            bgp_rc = FSCK_FAILED_BADREAD_FBLKMP;
          }  /* end else but didn't get enough to continue */
        } else {  /* read failed */
           /*
            * message to user
            */
          msgprms[0] = message_parm_0;
          msgprmidx[0] = fsck_metadata;
          msgprms[1] = Vol_Label;
          msgprmidx[1] = 0;
          fsck_send_msg( fsck_URCVREAD, 2 );
          /*
           * message to debugger
           */
          sprintf( message_parm_0, "%d", FSCK_FAILED_READ_FBLKMP );
          msgprms[0] = message_parm_0;
          msgprmidx[0] = 0;
          sprintf( message_parm_1, "%d", bgp_rc );
          msgprms[1] = message_parm_1;
          msgprmidx[1] = 0;
          sprintf( message_parm_2, "%d", fsck_READ );
          msgprms[2] = message_parm_2;
          msgprmidx[2] = 0;
          sprintf( message_parm_3, "%lld", (long long)agg_recptr->blkmp_agg_offset );
          msgprms[3] = message_parm_3;
          msgprmidx[3] = 0;
          sprintf( message_parm_4, "%d", agg_recptr->blkmp_buf_length );
          msgprms[4] = message_parm_4;
          msgprmidx[4] = 0;
          sprintf( message_parm_5, "%d", agg_recptr->blkmp_buf_data_len );
          msgprms[5] = message_parm_5;
          msgprmidx[5] = 0;
          fsck_send_msg( fsck_ERRONWSP, 6 );

          bgp_rc = FSCK_FAILED_READ_FBLKMP;
        }  /* end else read failed */
      }  /* end successful flush */
    }  /* end else we have read/write access */
  }  /* end else need to read it in from dasd */

  return( bgp_rc );
}  /* end of blkmap_get_page () */


/*****************************************************************************
 * NAME: blkmap_put_ctl_page
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            write the contents of the given buffer over the current fsck
 *            fsck workspace block map control page on the device.
 *
 * PARAMETERS:
 *      blk_ctlptr  - input -  pointer to the buffer which should be written
 *                             over the current fsck workspace block map
 *                             control page.
 *
 * NOTES:  Unlike most _put_ routines in this module, blkmap_put_ctl_page
 *         actually writes to the device.  This is done because the block
 *         map control page contains state and footprint information which
 *         provide crucial serviceability should the fsck session be
 *         interrupted.
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t  blkmap_put_ctl_page ( blk_cntrlptr  blk_ctlptr )
{
  retcode_t  bmpcp_rc = FSCK_OK;
  uint32_t   bytes_written;

  if ( agg_recptr->processing_readwrite ) {  /* have write access */

    /* swap if on big endian machine */
    ujfs_swap_fsck_blk_map_hdr(agg_recptr->blkmp_ctlptr);
    bmpcp_rc = readwrite_device(agg_recptr->ondev_wsp_byte_offset,
                                BYTESPERPAGE,
                                &bytes_written,
                                (void *) agg_recptr->blkmp_ctlptr,
                                fsck_WRITE                        );
    ujfs_swap_fsck_blk_map_hdr(agg_recptr->blkmp_ctlptr);

    if ( bmpcp_rc == FSCK_OK ) {
      if ( bytes_written != (uint32_t) BYTESPERPAGE ) {  /*
         * didn't write correct number of bytes
         */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 4 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BADWRITE_FBLKMP );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", bmpcp_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->ondev_wsp_byte_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", BYTESPERPAGE );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONWSP, 6 );

        bmpcp_rc = FSCK_FAILED_BADWRITE_FBLKMP;
      }  /* end else didn't write correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 5 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_WRITE_FBLKMP );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", bmpcp_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->ondev_wsp_byte_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", BYTESPERPAGE );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONWSP, 6 );

      bmpcp_rc = FSCK_FAILED_WRITE_FBLKMP;
    }  /* end else the write was not successful */
  }  /* end have write access */

  return( bmpcp_rc );
}   /* end of blkmap_put_ctl_page () */


/*****************************************************************************
 * NAME: blkmap_put_page
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current fsck workspace
 *            block map buffer has been modified and should be written to
 *            the device in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      page_num  - input - ordinal number of the page in the fsck workspace
 *                          block map to write from the buffer to the device
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blkmap_put_page( mappgidx_t  page_num )
{
  retcode_t  bpp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {
    agg_recptr->blkmp_buf_write = 1;
  }

  return( bpp_rc );
}  /* end of blkmap_put_page () */


/*****************************************************************************
 * NAME: blktbl_ctl_page_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current JFS Block Map
 *            control page buffer has been modified and should be written
 *            to the device in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      ctlpage_ptr  - input - the address, in an fsck buffer, of the page
 *                             which has been modified.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_ctl_page_put ( dbmap_t  *ctlpage_ptr )
{
  retcode_t  bcpp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {
    /* swap if on big endian machine */
    ujfs_swap_dbmap_t( ctlpage_ptr );
    bcpp_rc = mapctl_put( (void *) ctlpage_ptr );
  }

  return( bcpp_rc );
}  /* end of blktbl_ctl_page_put () */


/*****************************************************************************
 * NAME: blktbl_dmap_get
 *
 * FUNCTION: Read the JFS Block Table dmap page describing the specified
 *           aggregate block into and/or locate the JFS Block Table dmap
 *           locate the requested page describing the specified
 *           aggregate block in the fsck dmap buffer.
 *
 * PARAMETERS:
 *      for_block           - input - ordinal number of the aggregate block
 *                                    whose dmap page is needed
 *      addr_dmap_page_ptr  - input - pointer to a variable in which to return
 *                                    the address of the found dmap page in
 *                                    an fsck buffer
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_dmap_get ( fsblkidx_t  for_block,
                            dmap_t      **addr_dmap_page_ptr )
{
  retcode_t      bdg_rc = FSCK_OK;
  fsblkidx_t     dmp_logical_fsblk_offset;
  agg_byteidx_t  dmp_logical_byte_offset;
  agg_byteidx_t  ext_logical_byte_offset;
  agg_byteidx_t  ext_byte_offset;
  agg_byteidx_t  last_in_buffer;
  dinode_t       *bmap_inoptr;
  xad_t          *xad_ptr;
  int8_t         offset_found;
  int            which_it;
  reg_idx_t      bytes_read, ext_bytes_left;
  agg_byteidx_t  ext_bytes, offset_into_extent;
  int            i;
  dmap_t         *dmap_t_ptr;

  dmp_logical_fsblk_offset = BLKTODMAP( for_block, agg_recptr->log2_blksperpg );
  dmp_logical_byte_offset = dmp_logical_fsblk_offset * sb_ptr->s_bsize;
  *addr_dmap_page_ptr = NULL;

  if ( dmp_logical_byte_offset >= agg_recptr->bmapdm_logical_offset ) {
    last_in_buffer = agg_recptr->bmapdm_logical_offset +
                     agg_recptr->bmapdm_buf_data_len - 1;
    if ( (dmp_logical_byte_offset+(agg_byteidx_t)sizeof(dmap_t)-1) <= last_in_buffer ) {  
        /*
         * the one we want is already in the buffer
         */
      *addr_dmap_page_ptr = (dmap_t *) (agg_recptr->bmapdm_buf_ptr +
                                        dmp_logical_byte_offset -
                                        agg_recptr->bmapdm_logical_offset);
    }  /* end the one we want is already in the buffer */
  }

  if ( *addr_dmap_page_ptr == NULL ) {  /* we have to read it in */
    bdg_rc = blktbl_dmaps_flush();      /* perform any pending writes */
    if ( bdg_rc == FSCK_OK ) {  /* flush worked ok */
      if ( agg_recptr->primary_ait_4part1 ) {
        which_it = fsck_primary;
      } else {
        which_it = fsck_secondary;
      }
      bdg_rc = ait_special_read_ext1( which_it );
      if ( bdg_rc != FSCK_OK ) {
        report_readait_error( bdg_rc,
                              FSCK_FAILED_CANTREADAITEXTF,
                              which_it );
        bdg_rc = FSCK_FAILED_CANTREADAITEXTF;
      } else {  /* got the first agg extent */
        bmap_inoptr = (dinode_t *)
                      (agg_recptr->ino_buf_ptr + BMAP_I * sizeof(dinode_t));

        bdg_rc = xTree_search( bmap_inoptr, dmp_logical_fsblk_offset,
                               &xad_ptr, &offset_found );
        if ( bdg_rc == FSCK_OK ) {  /* nothing extraordinary happened */
          if ( !offset_found ) {
            bdg_rc = FSCK_INTERNAL_ERROR_51;
          } else {  /* we have the xad which describes the dmap */
            ext_logical_byte_offset = offsetXAD(xad_ptr) * sb_ptr->s_bsize;
            ext_byte_offset = addressXAD(xad_ptr) * sb_ptr->s_bsize;
            agg_recptr->bmapdm_agg_offset = ext_byte_offset +
                                            dmp_logical_byte_offset -
                                            ext_logical_byte_offset;

            bdg_rc = readwrite_device( agg_recptr->bmapdm_agg_offset,
                                       agg_recptr->bmapdm_buf_length, &bytes_read,
                                       (void *) agg_recptr->bmapdm_buf_ptr,
                                       fsck_READ );
            if ( bdg_rc == FSCK_OK ) {  /* got the extent containing the dmap */

              /* swap if on big endian machine */
	      dmap_t_ptr = (dmap_t *)agg_recptr->bmapdm_buf_ptr;
	      swap_multiple(ujfs_swap_dmap_t, dmap_t_ptr, 4);

              agg_recptr->bmapdm_logical_offset = dmp_logical_byte_offset;
              *addr_dmap_page_ptr = (dmap_t *) agg_recptr->bmapdm_buf_ptr;
              /*
               * we need to set the buffer data length to the number of
               * bytes with actual bmap data.  That is, we may have read
               * beyond the end of the extent, and if so, we need to ignore
               * the tag-along data.
               */
              ext_bytes = lengthXAD(xad_ptr) * sb_ptr->s_bsize;
              offset_into_extent = dmp_logical_byte_offset - ext_logical_byte_offset;
              ext_bytes_left = ext_bytes - offset_into_extent;
              agg_recptr->bmapdm_buf_data_len = MIN( bytes_read, ext_bytes_left );
            } else {
              /*
               * message to user
               */
              msgprms[0] = message_parm_0;
              msgprmidx[0] = fsck_metadata;
              msgprms[1] = Vol_Label;
              msgprmidx[1] = 0;
              fsck_send_msg( fsck_URCVREAD, 2 );
              /*
               * message to debugger
               */
              sprintf( message_parm_0, "%d", FSCK_FAILED_READ_BMPDM );
              msgprms[0] = message_parm_0;
              msgprmidx[0] = 0;
              sprintf( message_parm_1, "%d", bdg_rc );
              msgprms[1] = message_parm_1;
              msgprmidx[1] = 0;
              sprintf( message_parm_2, "%d", fsck_READ );
              msgprms[2] = message_parm_2;
              msgprmidx[2] = 0;
              sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmapdm_agg_offset );
              msgprms[3] = message_parm_3;
              msgprmidx[3] = 0;
              sprintf( message_parm_4, "%d", agg_recptr->bmapdm_buf_length );
              msgprms[4] = message_parm_4;
              msgprmidx[4] = 0;
              sprintf( message_parm_5, "%d", agg_recptr->bmapdm_buf_data_len );
              msgprms[5] = message_parm_5;
              msgprmidx[5] = 0;
              fsck_send_msg( fsck_ERRONAGG, 6 );

              bdg_rc = FSCK_FAILED_READ_BMPDM;
            }
          }  /* end else we have the xad which describes the dmap */
        }  /* end nothing extraordinary happened */
      }  /*  end else got the first agg extent */
    }  /* end flush worked ok */
  }  /* end we have to read it in */

  return( bdg_rc );
}  /* end of blktbl_dmap_get () */


/*****************************************************************************
 * NAME: blktbl_dmap_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current fsck dmap
 *            buffer has been modified and should be written to the device
 *            in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      dmap_page_ptr  - input - address of the dmap page, in the fsck buffer,
 *                               which has been modified.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_dmap_put( dmap_t  *dmap_page_ptr )
{
  retcode_t  bdp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) { 
      agg_recptr->bmapdm_buf_write = 1;
  }

  return( bdp_rc );
}  /* end of blktbl_dmap_put () */


/*****************************************************************************
 * NAME: blktbl_dmaps_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current dmap buffer has been updated since the most
 *            recent read operation, write the buffer contents to the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_dmaps_flush ( )
{
  retcode_t  bdf_rc = FSCK_OK;
  uint32_t   bytes_written;
  int        i;
  dmap_t     *dmap_t_ptr;

  if ( agg_recptr->bmapdm_buf_write ) {  
      /* buffer has been updated since
       * most recent write
       */

    /* swap if on big endian machine */
    dmap_t_ptr = (dmap_t *)agg_recptr->bmapdm_buf_ptr;
    swap_multiple(ujfs_swap_dmap_t, dmap_t_ptr, 4);
    bdf_rc = readwrite_device ( agg_recptr->bmapdm_agg_offset,
                                agg_recptr->bmapdm_buf_data_len,
                                &bytes_written,
                                (void *) agg_recptr->bmapdm_buf_ptr,
                                fsck_WRITE                          );
    dmap_t_ptr = (dmap_t *)agg_recptr->bmapdm_buf_ptr;
    swap_multiple(ujfs_swap_dmap_t, dmap_t_ptr, 4);

    if ( bdf_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->bmapdm_buf_data_len ) {
        agg_recptr->bmapdm_buf_write = 0; /* buffer has been written to
                               * the device and won't need to be
                               * written again unless/until the
                               * buffer contents have been altered again.
                               */
      } else {  /* didn't write the correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 6 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BMPDM_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", bdf_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmapdm_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->bmapdm_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );

        bdf_rc = FSCK_FAILED_BMPDM_BADFLUSH;
      }  /* end else didn't write the correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 7 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_BMPDM_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", bdf_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmapdm_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->bmapdm_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );

      bdf_rc = FSCK_FAILED_BMPDM_FLUSH;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */

  return( bdf_rc );
}  /* end of blktbl_dmaps_flush () */


/*****************************************************************************
 * NAME: blktbl_Ln_page_get
 *
 * FUNCTION: Read the JFS Block Map page describing the specified aggregate
 *           block at the specified summary level into and/or locate the
 *           locate the requested JFS Block Map page describing the specified
 *           aggregate block at the specified summary level in the
 *           fsck Level n page buffer.
 *
 * PARAMETERS:
 *      level             - input - Summary level of the page to get
 *      for_block         - input - ordinal number of the aggregate block
 *                                  whose summary page is needed
 *      addr_Ln_page_ptr  - input - pointer to a variable in which to return
 *                                  the address of the requested page in an
 *                                  fsck buffer
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_Ln_page_get ( int8_t      level,
                               fsblkidx_t  for_block,
                               dmapctl_t   **addr_Ln_page_ptr )
{
  retcode_t      blpg_rc = FSCK_OK;
  fsblkidx_t     Lnpg_logical_fsblk_offset;
  agg_byteidx_t  Lnpg_logical_byte_offset;
  agg_byteidx_t  ext_logical_byte_offset;
  agg_byteidx_t  ext_byte_offset;
  agg_byteidx_t  last_in_buffer;
  dinode_t       *bmap_inoptr;
  xad_t          *xad_ptr;
  int8_t         offset_found;
  int            which_it;

  Lnpg_logical_fsblk_offset =
  BLKTOCTL( for_block, agg_recptr->log2_blksperpg, level );
  Lnpg_logical_byte_offset = Lnpg_logical_fsblk_offset * sb_ptr->s_bsize;

  *addr_Ln_page_ptr = NULL;

  if ( Lnpg_logical_byte_offset >= agg_recptr->bmaplv_logical_offset ) {
    last_in_buffer = agg_recptr->bmaplv_logical_offset +
                     agg_recptr->bmaplv_buf_data_len;
    if ( (Lnpg_logical_byte_offset+(agg_byteidx_t)sizeof(dmapctl_t)) <= last_in_buffer ) {  
        /*
         * the one we want is already in the buffer
         */
      *addr_Ln_page_ptr = (dmapctl_t *) ( agg_recptr->bmaplv_buf_ptr +
                                          (Lnpg_logical_byte_offset -
                                           agg_recptr->bmaplv_logical_offset) );
    }  /* end the one we want is already in the buffer */
  }

  if ( *addr_Ln_page_ptr == NULL ) {  /* we have to read it in */
    blpg_rc = blktbl_Ln_pages_flush();     /* perform any pending writes */
    if ( blpg_rc == FSCK_OK ) {  /* flush worked ok */
      if ( agg_recptr->primary_ait_4part1 ) {
        which_it = fsck_primary;
      } else {
        which_it = fsck_secondary;
      }
      blpg_rc = ait_special_read_ext1( which_it );
      if ( blpg_rc != FSCK_OK ) {
        report_readait_error( blpg_rc,
                              FSCK_FAILED_CANTREADAITEXTG,
                              which_it );
        blpg_rc = FSCK_FAILED_CANTREADAITEXTG;
      } else {  /* got the first agg extent */
        bmap_inoptr = (dinode_t *)
                      (agg_recptr->ino_buf_ptr + BMAP_I * sizeof(dinode_t));
        blpg_rc = xTree_search( bmap_inoptr, Lnpg_logical_fsblk_offset,
                                &xad_ptr, &offset_found );
        if ( blpg_rc == FSCK_OK ) {  /* nothing extraordinary happened */
          if ( !offset_found ) {  /* didn't find it! */
            blpg_rc = FSCK_INTERNAL_ERROR_52;
          } else {  /* we have the xad which describes the page */
            ext_logical_byte_offset = offsetXAD(xad_ptr) * sb_ptr->s_bsize;
            ext_byte_offset = addressXAD(xad_ptr) * sb_ptr->s_bsize;
            agg_recptr->bmaplv_agg_offset = ext_byte_offset +
                                            Lnpg_logical_byte_offset -
                                            ext_logical_byte_offset;

            blpg_rc = readwrite_device( agg_recptr->bmaplv_agg_offset,
                                        agg_recptr->bmaplv_buf_length,
                                        &(agg_recptr->bmaplv_buf_data_len),
                                        (void *) agg_recptr->bmaplv_buf_ptr,
                                        fsck_READ );
            if ( blpg_rc == FSCK_OK ) {  /* got the page */

              /* swap if on big endian machine */
              ujfs_swap_dmapctl_t( (dmapctl_t *) agg_recptr->bmaplv_buf_ptr );

              agg_recptr->bmaplv_current_level = level;
              agg_recptr->bmaplv_logical_offset = Lnpg_logical_byte_offset;
              *addr_Ln_page_ptr = (dmapctl_t *) agg_recptr->bmaplv_buf_ptr;
            } else {
              /*
               * message to user
               */
              msgprms[0] = message_parm_0;
              msgprmidx[0] = fsck_metadata;
              msgprms[1] = Vol_Label;
              msgprmidx[1] = 0;
              fsck_send_msg( fsck_URCVREAD, 2 );
              /*
               * message to debugger
               */
              sprintf( message_parm_0, "%d", FSCK_FAILED_READ_BMPLV );
              msgprms[0] = message_parm_0;
              msgprmidx[0] = 0;
              sprintf( message_parm_1, "%d", blpg_rc );
              msgprms[1] = message_parm_1;
              msgprmidx[1] = 0;
              sprintf( message_parm_2, "%d", fsck_READ );
              msgprms[2] = message_parm_2;
              msgprmidx[2] = 0;
              sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmaplv_agg_offset );
              msgprms[3] = message_parm_3;
              msgprmidx[3] = 0;
              sprintf( message_parm_4, "%d", agg_recptr->bmaplv_buf_length );
              msgprms[4] = message_parm_4;
              msgprmidx[4] = 0;
              sprintf( message_parm_5, "%d", agg_recptr->bmaplv_buf_data_len );
              msgprms[5] = message_parm_5;
              msgprmidx[5] = 0;
              fsck_send_msg( fsck_ERRONAGG, 6 );

              blpg_rc = FSCK_FAILED_READ_BMPLV;
            }
          }  /* end else we have the xad which describes the page */
        }  /* end nothing extraordinary happened */
      }  /*  end else got the first agg extent */
    }  /* end flush worked ok */
  }  /* end we have to read it in */

  return( blpg_rc );
}  /* end of blktbl_Ln_page_get () */


/*****************************************************************************
 * NAME: blktbl_Ln_page_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current fsck Level n
 *            buffer has been modified and should be written to the device
 *            in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      Ln_page_ptr  - input - Address, in an fsck buffer, of the block map
 *                             summary page which has been modified.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_Ln_page_put ( dmapctl_t  *Ln_page_ptr )
{
  retcode_t  blpp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite )
  { agg_recptr->bmaplv_buf_write = 1;
  }

  return( blpp_rc );
}  /* end of blktbl_Ln_page_put () */


/*****************************************************************************
 * NAME: blktbl_Ln_pages_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current Level n Page buffer has been updated since
 *            the most recent read operation, write the buffer contents to
 *            the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t blktbl_Ln_pages_flush ( )
{
  retcode_t  blpf_rc = FSCK_OK;
  uint32_t   bytes_written;

  if ( agg_recptr->bmaplv_buf_write ) {  
      /* buffer has been updated since
       * most recent write
       */

    /* swap if on big endian machine */
    ujfs_swap_dmapctl_t((dmapctl_t *)agg_recptr->bmaplv_buf_ptr); 
    blpf_rc = readwrite_device ( agg_recptr->bmaplv_agg_offset,
                                 agg_recptr->bmaplv_buf_data_len,
                                 &bytes_written,
                                 (void *) agg_recptr->bmaplv_buf_ptr,
                                 fsck_WRITE                          );
    ujfs_swap_dmapctl_t((dmapctl_t *)agg_recptr->bmaplv_buf_ptr); 

    if ( blpf_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->bmaplv_buf_data_len ) {
        agg_recptr->bmaplv_buf_write = 0; /* buffer has been written to
                               * the device and won't need to be
                               * written again unless/until the
                               * buffer contents have been altered again.
                               */
      } else {  /* didn't write the correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 8 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BMPLV_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", blpf_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmaplv_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->bmaplv_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );

        blpf_rc = FSCK_FAILED_BMPLV_BADFLUSH;
      }  /* end else didn't write the correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 9 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_BMPLV_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", blpf_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->bmaplv_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->bmaplv_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );

      blpf_rc = FSCK_FAILED_BMPLV_FLUSH;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */

  return( blpf_rc );
}  /* end of blktbl_Ln_pages_flush () */


/**************************************************************************
 * NAME: close_volume
 *
 * FUNCTION:  If fsck holds a lock on the device containing the aggregate,
 *            release that lock.
 *
 *            Close the device containing the aggregate.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t close_volume( )
{
  if (close(Dev_IOPort))
    return ERROR_INVALID_HANDLE;
  return NO_ERROR;
}


/**************************************************************************
 * NAME: default_volume
 *
 * FUNCTION:  If fsck holds a lock on the device containing the aggregate,
 *            release that lock.
 *
 *            Close the device containing the aggregate.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t default_volume( )
{
  retcode_t  dv_rc = FSCK_OK;
  int        buf_len;
  char       *buf_ptr;

  buf_len = msgs_txt_maxlen;
  buf_ptr = msgs_txt_ptr;

  getcwd( buf_ptr, buf_len );

  Vol_Label[0] = msgs_txt_ptr[0];
  Vol_Label[1] = msgs_txt_ptr[1];

  msgprms[0] = Vol_Label;
  msgprmidx[0] = 0;
  fsck_send_msg( fsck_DEFAULTVOL, 1 );

  return( dv_rc );
}  /* end default_volume () */


/*****************************************************************************
 * NAME: dnode_get
 *
 * FUNCTION: Read the requested dnode page into and/or locate the requested
 *           dnode page in the fsck dnode buffer.
 *
 * PARAMETERS:
 *      dnode_fsblk_offset  - input - offset, in aggregate blocks, into the
 *                                    aggregate of the dnode to read
 *      dnode_length        - input - number of bytes in the dnode
 *      addr_dtpage_ptr     - input - pointer to a variable in which to return
 *                                    the address of the requested dnode in an
 *                                    fsck buffer
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t dnode_get ( fsblkidx_t  dnode_fsblk_offset,
                      reg_idx_t   dnode_length,
                      dtpage_t    **addr_dtpage_ptr    )
{
  retcode_t      dnodg_rc = FSCK_OK;
  agg_byteidx_t  dnode_start_byte, dnode_end_byte;
  dtpage_t       *dtp;

  dnode_start_byte = dnode_fsblk_offset * sb_ptr->s_bsize;
  dnode_end_byte = dnode_start_byte + dnode_length - 1;

  if ( (agg_recptr->ondev_wsp_fsblk_offset != 0) &&
       (dnode_fsblk_offset > agg_recptr->ondev_wsp_fsblk_offset) ) { /*
     * the offset is beyond the range
     * valid for fileset objects
     */
    /*
     * This case is not caused by an I/O error, but by
     * invalid data in an inode.  Let the caller handle
     * the consequences.
     */
    dnodg_rc = FSCK_BADREADTARGET;
  } else if ( (dnode_start_byte >= agg_recptr->node_agg_offset) &&
              (dnode_end_byte <= (agg_recptr->node_agg_offset +
              agg_recptr->node_buf_data_len))  ) {  /*
     * the target dir node is already in
     * the buffer
     */
    dtp = *addr_dtpage_ptr = (dtpage_t *)
                             (agg_recptr->node_buf_ptr +
                              dnode_start_byte - agg_recptr->node_agg_offset);

    /* swap if on big endian machine */
    if (!(dtp->header.flag & BT_SWAPPED) ) {
        ujfs_swap_dtpage_t( dtp, sb_ptr->s_flag );
        dtp->header.flag |= BT_SWAPPED;
    }
  } else {  /* else we'll have to read it from the disk */
    agg_recptr->node_agg_offset = dnode_start_byte;
    dnodg_rc = readwrite_device( agg_recptr->node_agg_offset,
                                 agg_recptr->node_buf_length,
                                 &(agg_recptr->node_buf_data_len),
                                 (void *) agg_recptr->node_buf_ptr,
                                 fsck_READ                         );
    if ( dnodg_rc == FSCK_OK ) {  /* read appears successful */
      if ( agg_recptr->node_buf_data_len >= dnode_length ) {  /*
         * we may not have gotten all we asked for,
         * but we got enough to cover the dir node
         * we were after
         */

        dtp = *addr_dtpage_ptr = (dtpage_t *) agg_recptr->node_buf_ptr;

        /* swap if on big endian machine */
        ujfs_swap_dtpage_t( dtp, sb_ptr->s_flag );
        dtp->header.flag |= BT_SWAPPED;

      } else {  /* didn't get the minimum number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_DNODE );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", dnodg_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );

        dnodg_rc = FSCK_FAILED_BADREAD_DNODE;
      }  /* end else didn't get the minimum number of bytes */
    } else {  /* bad return code from read */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_READ_DNODE );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", dnodg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );

      dnodg_rc = FSCK_FAILED_READ_DNODE;
    }  /* end else bad return code from read */
  }  /* end else we'll have to read it from the disk */

  return( dnodg_rc );
}  /* end of dnode_get () */


/*****************************************************************************
 * NAME: ea_get
 *
 * FUNCTION: Read the specified Extended Attributes data (ea) into
 *           the specified buffer.
 *
 * PARAMETERS:
 *      ea_fsblk_offset  - input - offset, in aggregate blocks, into the
 *                                    aggregate of the ea to read
 *      ea_byte_length   - input - length, in bytes, of the ea to read
 *      eabuf_ptr        - input - the address (in dynamic storage) of the
 *                                 buffer into which to read the ea
 *      eabuf_length     - input - pointer to a variable in which contains
 *                                 the length of the buffer at eabuf_ptr
 *      ea_data_length   - input - pointer to a variable in which to return
 *                                 the number of bytes actually read from the
 *                                 device
 *      ea_agg_offset    - input - pointer to a variable in which to return
 *                                 the offset, in bytes, into the aggregate
 *                                 of the ea to read
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t ea_get ( fsblkidx_t     ea_fsblk_offset,
                   reg_idx_t      ea_byte_length,
                   char           *eabuf_ptr,
                   reg_idx_t      *eabuf_length,
                   reg_idx_t      *ea_data_length,
                   agg_byteidx_t  *ea_agg_offset    )
{
  retcode_t      ea_rc = FSCK_OK;
  agg_byteidx_t  start_byte, end_byte;

  start_byte = ea_fsblk_offset * sb_ptr->s_bsize;
  end_byte = start_byte + ea_byte_length - 1;
  ea_rc = readwrite_device( start_byte, ea_byte_length, ea_data_length,
                            (void *) eabuf_ptr, fsck_READ );

  /* swap if on big endian machine, currently unused */

  if ( ea_rc == FSCK_OK ) {  /* read appears successful */
    *ea_agg_offset = start_byte;
    if ( (*ea_data_length) < ea_byte_length ) {  /* we didn't get enough */
      *ea_agg_offset = 0;
      *ea_data_length = 0;
      ea_rc = FSCK_BADEADESCRIPTOR;
    }  /* end we didn't get enough */
  } else {  /* bad return code from read */
    ea_rc = FSCK_CANTREADEA;
  }  /* end else bad return code from read */
  return( ea_rc );
}  /* end of ea_get () */


/*****************************************************************************
 * NAME: fscklog_put_buffer
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            and if the in-aggregate fsck log is not full, write the
 *            contents of the current fscklog buffer into the in-aggregate
 *            fsck log.
 *
 * PARAMETERS:  none
 *
 * NOTES:  o Unlike most _put_ routines in this module, _buffer
 *           actually writes to the device.  This is done because the fsck
 *           log contains information which provides crucial serviceability
 *           should the fsck session be interrupted.
 *
 *         o Errors here are recorded in the control page of the fsck
 *           in-aggregate workspace but never affect other fsck processing.
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t  fscklog_put_buffer ( )
{
  retcode_t  flpb_rc = FSCK_OK;
  retcode_t  io_rc = FSCK_OK;
  uint32_t   bytes_written = 0;
  uint32_t   log_bytes_left;
  int32_t    num_log_errors;
  fscklog_error_t  *log_error_recptr;

  if ( (!agg_recptr->fscklog_full) &&
       (agg_recptr->processing_readwrite) ) {  /* have write access */

    /* swap if on big endian machine in proper format */

    io_rc = readwrite_device( agg_recptr->fscklog_agg_offset,
                              agg_recptr->fscklog_buf_length,
                              &bytes_written,
                              (void *) agg_recptr->fscklog_buf_ptr,
                              fsck_WRITE
                            );
    if ( (io_rc != FSCK_OK) ||
         (bytes_written != (uint32_t) agg_recptr->fscklog_buf_length) ) {  /*
       * write failed or didn't write correct
       * number of bytes
       */
      agg_recptr->fscklog_full = 1; /* This prevents infinite recursion */
      if (agg_recptr->blkmp_ctlptr) {
        num_log_errors = agg_recptr->blkmp_ctlptr->hdr.num_logwrite_errors;
        if ( num_log_errors < 120 ) {
          log_error_recptr =
                &(agg_recptr->blkmp_ctlptr->hdr.logerr[num_log_errors]);
          log_error_recptr->err_offset = agg_recptr->fscklog_agg_offset;
          log_error_recptr->bytes_written = bytes_written;
          log_error_recptr->io_retcode = io_rc;
        }
        agg_recptr->blkmp_ctlptr->hdr.num_logwrite_errors += 1;
      }
      /*
       * message to debugger
       *
       * N.B. This is NOT a fatal condition!
       */
      sprintf( message_parm_0, "%d", FSCK_BADWRITE_FSCKLOG );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", io_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->fscklog_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->fscklog_buf_length );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONLOG, 6 );
    }  /* end write failed or didn't write correct number of bytes */
  }  /* end have write access */
  /*
   * We want to reset the buffer no matter what.
   * It is useful to refill the buffer even if logging is not
   * active because it may provide diagnostic information in
   * a dump.
   */
  agg_recptr->fscklog_agg_offset += agg_recptr->fscklog_buf_length;
  agg_recptr->fscklog_log_offset += agg_recptr->fscklog_buf_length;
  agg_recptr->fscklog_buf_data_len = 0;
  log_bytes_left = (agg_recptr->ondev_fscklog_byte_length / 2) -
                   agg_recptr->fscklog_log_offset;
  if ( log_bytes_left < agg_recptr->fscklog_buf_length ) {  /*
     * can't fit another buffer full
     * into the log
     */
    if ( !agg_recptr->initializing_fscklog ) {  
        /* this is a false
         * condition if doing log initialization
         */
      agg_recptr->fscklog_full = -1;
      agg_recptr->blkmp_ctlptr->hdr.fscklog_full = -1;
    }
  }  /* end can't fit another buffer full into the log */
  return( flpb_rc );
}  /* end of fscklog_put_buffer () */


/*****************************************************************************
 * NAME: iag_get
 *
 * FUNCTION: Read the requested iag into and/or locate the requested iag
 *           in the fsck iag buffer.
 *
 * PARAMETERS:
 *      is_aggregate  - input -  0 => the iag is owned by the fileset
 *                              !0 => the iag is owned by the aggregate
 *      which_it      - input - ordinal number of the aggregate inode
 *                              representing the inode table to which the
 *                              iag belongs.
 *      which_ait     - input - the aggregate inode table { fsck_primary |
 *                              fsck_secondary } containing the version of
 *                              which_it to use for this operation
 *      iag_num       - input - ordinal number of the iag needed
 *      addr_iag_ptr  - input - pointer to a variable in which to return
 *                              the address, in an fsck buffer, of the
 *                              requested iag.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t iag_get ( int      is_aggregate,
                    int      which_it,
                    int      which_ait,
                    iagno_t  iag_num,
                    iag_t    **addr_iag_ptr  )
{
  retcode_t   iagg_rc = FSCK_OK;
  fsblkidx_t  imap_logical_block, extent_offset;
  fsblkidx_t  offset_in_extent = 0;
  dinode_t    *imap_inoptr;
  xad_t       *xad_ptr;
  int8_t      offset_found;
  int         which_agg_inode = 0;

  *addr_iag_ptr = NULL;
  if ( (agg_recptr->iag_buf_1st_inode == (iag_num << L2INOSPERIAG)) &&
       (agg_recptr->iag_for_aggregate == is_aggregate)              &&
       (agg_recptr->iag_which_it == which_it)                         ) { 
      /*
       * the target iag is already in the buffer
       */
    *addr_iag_ptr = (iag_t *) agg_recptr->iag_buf_ptr;
  } else {  /* need to get the iag */
    iagg_rc = iags_flush();         /* perform any pending writes */
    if ( iagg_rc == FSCK_OK ) {  /* flush worked ok */
      if ( is_aggregate ) {  /* this is an IAG describing aggregate inodes */
        if ( iag_num < agg_recptr->agg_imap.num_iags ) {  /* in bounds */
          which_agg_inode = AGGREGATE_I;
          iagg_rc = ait_special_read_ext1( which_ait );
          if ( iagg_rc != FSCK_OK ) { /* read ait failed */
            report_readait_error( iagg_rc,
                                  FSCK_FAILED_CANTREADAITEXTH,
                                  which_ait );
            iagg_rc = FSCK_FAILED_CANTREADAITEXTH;
          }  /* end read ait failed */
        } else {  /* invalid request */
          iagg_rc = FSCK_IAGNOOOAGGBOUNDS;
        }  /* end invalid request */
      } else {  /* an IAG describing fileset inodes */
        if ( iag_num < agg_recptr->fset_imap.num_iags ) {  /* in bounds */
          which_agg_inode = FILESYSTEM_I;
          iagg_rc = ait_special_read_ext1( which_ait );
          if ( iagg_rc != FSCK_OK ) { /* read ait failed */
            report_readait_error( iagg_rc,
                                  FSCK_FAILED_CANTREADAITEXTK,
                                  which_ait );
            iagg_rc = FSCK_FAILED_CANTREADAITEXTK;
          }  /* end read ait failed */
        } else {  /* invalid request */
          iagg_rc = FSCK_IAGNOOOFSETBOUNDS;
        }  /* end invalid request */
      }  /* end else an IAG describing fileset inodes */
      if ( iagg_rc == FSCK_OK ) {  /* got the extent */
        imap_inoptr = (dinode_t *)
                      (agg_recptr->ino_buf_ptr +
                      (which_agg_inode * sizeof(dinode_t)) );
        imap_logical_block = IAGTOLBLK( iag_num, agg_recptr->log2_blksperpg );
        iagg_rc = xTree_search( imap_inoptr, imap_logical_block,
                                &xad_ptr, &offset_found );
        if ( iagg_rc == FSCK_OK ) {  /* nothing extraordinary happened */
          if ( !offset_found ) {
            iagg_rc = FSCK_INTERNAL_ERROR_50;
          } else {  /* we have the xad which describes the iag */
            extent_offset = offsetXAD(xad_ptr);
            if ( extent_offset != imap_logical_block ) {
              offset_in_extent = imap_logical_block - extent_offset;
            }
            agg_recptr->iag_agg_offset = sb_ptr->s_bsize *
                                         (addressXAD(xad_ptr) + offset_in_extent);
            iagg_rc = readwrite_device( agg_recptr->iag_agg_offset,
                                        agg_recptr->iag_buf_length,
                                        &(agg_recptr->iag_buf_data_len),
                                        (void *) agg_recptr->iag_buf_ptr,
                                        fsck_READ );
            if ( iagg_rc == FSCK_OK ) {  /* got the iag */

              /* swap if on big endian machine */
              ujfs_swap_iag_t( (iag_t *) agg_recptr->iag_buf_ptr );

              agg_recptr->iag_buf_1st_inode = iag_num << L2INOSPERIAG;
              agg_recptr->iag_fsnum = imap_inoptr->di_fileset;
              agg_recptr->iag_for_aggregate = is_aggregate;
              agg_recptr->iag_which_it = which_it;
              *addr_iag_ptr = (iag_t *) agg_recptr->iag_buf_ptr;
            } else {
              /*
               * message to user
               */
              msgprms[0] = message_parm_0;
              msgprmidx[0] = fsck_metadata;
              msgprms[1] = Vol_Label;
              msgprmidx[1] = 0;
              fsck_send_msg( fsck_URCVREAD, 2 );
              /*
               * message to debugger
               */
              sprintf( message_parm_0, "%d", FSCK_FAILED_READ_IAG );
              msgprms[0] = message_parm_0;
              msgprmidx[0] = 0;
              sprintf( message_parm_1, "%d", iagg_rc );
              msgprms[1] = message_parm_1;
              msgprmidx[1] = 0;
              sprintf( message_parm_2, "%d", fsck_READ );
              msgprms[2] = message_parm_2;
              msgprmidx[2] = 0;
              sprintf( message_parm_3, "%lld", (long long)agg_recptr->iag_agg_offset );
              msgprms[3] = message_parm_3;
              msgprmidx[3] = 0;
              sprintf( message_parm_4, "%d", agg_recptr->iag_buf_length );
              msgprms[4] = message_parm_4;
              msgprmidx[4] = 0;
              sprintf( message_parm_5, "%d", agg_recptr->iag_buf_data_len );
              msgprms[5] = message_parm_5;
              msgprmidx[5] = 0;
              fsck_send_msg( fsck_ERRONAGG, 6 );
              iagg_rc = FSCK_FAILED_READ_IAG;
            }
          }  /* end else we have the xad which describes the iag */
        }  /* end nothing extraordinary happened */
      }  /*  end got the inode */
    }  /* end flush worked ok */
  }  /* end else need to get the iag */
  return( iagg_rc );
}  /* end of iag_get () */


/*****************************************************************************
 * NAME: iag_get_first
 *
 * FUNCTION: Read the first iag in the specified inode table into and/or
 *           locate the first iag in the specified inode table in the
 *           fsck iag buffer.  Set up for sequential access on the iag's
 *           in this table.
 *
 * PARAMETERS:
 *      is_aggregate  - input -  0 => the iag is owned by the fileset
 *                              !0 => the iag is owned by the aggregate
 *      which_it      - input - ordinal number of the aggregate inode
 *                              representing the inode table to which the
 *                              iag belongs.
 *      which_ait     - input - the aggregate inode table { fsck_primary |
 *                              fsck_secondary } containing the version of
 *                              which_it to use for this operation
 *      addr_iag_ptr  - input - pointer to a variable in which to return
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t iag_get_first ( int    is_aggregate,
                          int    it_number,
                          int    which_ait,
                          iag_t  **addr_iag_ptr  )
{
  retcode_t  iaggf_rc = FSCK_OK;
  dinode_t   *imap_inoptr;
  xad_t      *xadptr;
  int        it_tbl_is_agg_owned = -1;  /* the table may not describe the
                                         * aggregate inodes, but the inode
                                         * describing the table is ALWAYS
                                         * an aggregate inode
                                         */
  agg_recptr->fais.this_iagnum = 0;
  agg_recptr->fais.this_inoidx = 0;
  agg_recptr->fais.rootleaf_imap = 0;  /* assume it's not a rootleaf */
  /*
   * get the first leaf (xtpage) of the imap into the mapleaf buffer
   *
   * Note: if the imap B+ Tree has a root-leaf then the
   *       first (and only) leaf is located in the dinode.
   */
  iaggf_rc = ait_special_read_ext1( which_ait );  /*
                             * save inode_get from having to read an
                             * iag as an intermediate step.
                             */
  if ( iaggf_rc != FSCK_OK ) { /* read ait failed */
    report_readait_error( iaggf_rc,
                          FSCK_FAILED_CANTREADAITEXTJ,
                          which_ait );
    iaggf_rc = FSCK_FAILED_CANTREADAITEXTJ;
  } else {  /* got the inode extent */
    if ( !is_aggregate ) {  /* after the fileset inode table */
      if ( agg_recptr->fset_imap.imap_is_rootleaf ) { /* it's a root-leaf */
        agg_recptr->fais.rootleaf_imap = -1;
        iaggf_rc = inode_get( it_tbl_is_agg_owned, which_ait,
                              it_number, &imap_inoptr ); /*
                             * read the imap inode into the inode buffer
                             */
        if ( iaggf_rc != FSCK_OK ) { /* something went wrong */
          iaggf_rc = FSCK_FAILED_AGFS_READ5;
        }  /* end something went wrong */
      }  /* it's a root-leaf */
    } else {  /* want the aggregate inode table */
      if ( agg_recptr->agg_imap.imap_is_rootleaf ) {  /* it's a rootleaf */
        agg_recptr->fais.rootleaf_imap = -1;
        iaggf_rc = inode_get( it_tbl_is_agg_owned, it_number,
                              AGGREGATE_I, &imap_inoptr ); /*
                             * read the imap inode into the inode buffer
                             */
        if ( iaggf_rc != FSCK_OK ) { /* something went wrong */
          iaggf_rc = FSCK_FAILED_AGFS_READ5;
        }  /* end something went wrong */
      }  /* it's a root-leaf */
    }  /* end else want the aggregate inode table */
  }  /* end else got the inode extent */
  if ( (iaggf_rc == FSCK_OK) && (agg_recptr->fais.rootleaf_imap) ) {  
      /*
       * root-leaf imap and we have the inode
       */
    memcpy( (void *) (agg_recptr->mapleaf_buf_ptr),
            (void *) imap_inoptr,
            sizeof(dinode_t) );  /* copy the inode into the imap leaf buf */
    agg_recptr->mapleaf_buf_data_len = 0;
    agg_recptr->mapleaf_agg_offset = 0;
    agg_recptr->mapleaf_for_aggregate = is_aggregate;
    agg_recptr->mapleaf_which_it = it_number;
    imap_inoptr = (dinode_t *) (agg_recptr->mapleaf_buf_ptr);
    agg_recptr->fais.this_mapleaf = (xtpage_t *) &(imap_inoptr->di_btroot);
  }  /* end root-leaf imap and we have the inode */
  if ( (iaggf_rc == FSCK_OK) && (!agg_recptr->fais.rootleaf_imap) ) {  
      /*
       * something below the root
       */
    if ( !is_aggregate ) {  /* after the fileset inode table */
      iaggf_rc = imapleaf_get( agg_recptr->fset_imap.first_leaf_offset,
                               &(agg_recptr->fais.this_mapleaf) ); /* get
                              * the first leaf into the mapleaf buffer
                              */
    } else {  /* must want the aggregate inode table */
      iaggf_rc = imapleaf_get( agg_recptr->agg_imap.first_leaf_offset,
                               &(agg_recptr->fais.this_mapleaf) ); /* get
                              * the first leaf into the mapleaf buffer
                              */
    }  /* end else must want the aggregate inode table */
  }  /* end else something below the root */
  if ( iaggf_rc == FSCK_OK ) {  /* the first imap leaf is in the buf */
    agg_recptr->fais.iagidx_now = XTENTRYSTART;   /* first in the leaf */
    agg_recptr->fais.iagidx_max =
    agg_recptr->fais.this_mapleaf->header.nextindex - 1;
    /*
     * get the first iag of the imap into the iag buffer
     */
    iaggf_rc = iags_flush();
    if ( iaggf_rc == FSCK_OK ) {  /* flushed ok */
      xadptr =
          &(agg_recptr->fais.this_mapleaf->xad[agg_recptr->fais.iagidx_now]);
      agg_recptr->iag_agg_offset = (sb_ptr->s_bsize * addressXAD(xadptr)) +
                                   sizeof(dinomap_t);  /*
                              * the first iag is preceded by the IT control
                              * page.
                              */
      agg_recptr->iag_buf_1st_inode = agg_recptr->fais.this_inoidx;
      agg_recptr->iag_fsnum = it_number;
      agg_recptr->iag_for_aggregate = is_aggregate;
      agg_recptr->iag_fsnum = it_number;
      iaggf_rc = readwrite_device( agg_recptr->iag_agg_offset,
                                   agg_recptr->iag_buf_length,
                                   &(agg_recptr->iag_buf_data_len),
                                   (void *) (agg_recptr->iag_buf_ptr),
                                   fsck_READ );

      /* swap if on big endian machine */
      ujfs_swap_iag_t( (iag_t *) agg_recptr->iag_buf_ptr );

    }  /* end flushed ok */
  }  /* end the first imap leaf is in the buf */
  if ( iaggf_rc == FSCK_OK ) {  /* first iag is in iag buffer */
    agg_recptr->fais.iagptr = (iag_t *) agg_recptr->iag_buf_ptr;
    agg_recptr->fais.extidx_now = 0;
    agg_recptr->fais.extidx_max = EXTSPERIAG - 1;
    *addr_iag_ptr = agg_recptr->fais.iagptr;
  } else {
    /*
     * message to user
     */
    msgprms[0] = message_parm_0;
    msgprmidx[0] = fsck_metadata;
    msgprms[1] = Vol_Label;
    msgprmidx[1] = 0;
    fsck_send_msg( fsck_URCVREAD, 2 );
    /*
     * message to debugger
     */
    sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_IAG );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    sprintf( message_parm_1, "%d", iaggf_rc );
    msgprms[1] = message_parm_1;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", fsck_READ );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    sprintf( message_parm_3, "%lld", (long long)agg_recptr->iag_agg_offset );
    msgprms[3] = message_parm_3;
    msgprmidx[3] = 0;
    sprintf( message_parm_4, "%d", agg_recptr->iag_buf_length );
    msgprms[4] = message_parm_4;
    msgprmidx[4] = 0;
    sprintf( message_parm_5, "%d", agg_recptr->iag_buf_data_len );
    msgprms[5] = message_parm_5;
    msgprmidx[5] = 0;
    fsck_send_msg( fsck_ERRONAGG, 6 );
    iaggf_rc = FSCK_FAILED_BADREAD_IAG;
  }
  return( iaggf_rc );
}  /* end of iag_get_first () */


/*****************************************************************************
 * NAME: iag_get_next
 *
 * FUNCTION: Read the next iag in the specified inode table into and/or
 *           locate the next iag in the specified inode table in the
 *           fsck iag buffer.
 *
 * PARAMETERS:
 *      addr_iag_ptr  - input - pointer to a variable in which to return
 *                              the address, in an fsck buffer, of the next
 *                              iag in the aggregate inode table currently
 *                              being traversed.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t iag_get_next ( iag_t  **addr_iag_ptr )
{
  retcode_t      iaggn_rc = FSCK_OK;
  int8_t         iag_found = 0;
  int8_t         end_of_imap =0;
  agg_byteidx_t  leaf_byteaddr;
  xad_t          *xadptr;

  agg_recptr->fais.this_iagnum++;
  iaggn_rc = iags_flush();
  if ( iaggn_rc == FSCK_OK ) {  /* flushed the iags ok */
    while ( (!iag_found) && (!end_of_imap) && (iaggn_rc == FSCK_OK) ) {
      agg_recptr->fais.iagidx_now++;
      if ( agg_recptr->fais.iagidx_now <= agg_recptr->fais.iagidx_max ) {  
          /*
           * the imap leaf is in the buffer already
           */
        xadptr = 
            &(agg_recptr->fais.this_mapleaf->xad[agg_recptr->fais.iagidx_now]);
        agg_recptr->iag_agg_offset = sb_ptr->s_bsize * addressXAD(xadptr);
        agg_recptr->iag_buf_1st_inode = agg_recptr->fais.this_inoidx;
        iaggn_rc = readwrite_device( agg_recptr->iag_agg_offset,
                                     agg_recptr->iag_buf_length,
                                     &(agg_recptr->iag_buf_data_len),
                                     (void *) (agg_recptr->iag_buf_ptr),
                                     fsck_READ );
        if ( iaggn_rc == FSCK_OK ) {  /* got the iag */

          /* swap if on big endian machine */
          ujfs_swap_iag_t( (iag_t *) agg_recptr->iag_buf_ptr );

          agg_recptr->fais.iagptr = (iag_t *) (agg_recptr->iag_buf_ptr);
          agg_recptr->fais.extidx_now = 0;
          agg_recptr->fais.this_inoidx = agg_recptr->fais.this_iagnum *
                                         NUM_INODE_PER_IAG;
          agg_recptr->iag_buf_1st_inode = agg_recptr->fais.this_inoidx;
          agg_recptr->fais.extidx_max = EXTSPERIAG - 1;
          iag_found = -1;
        } else {
          /*
           * message to user
           */
          msgprms[0] = message_parm_0;
          msgprmidx[0] = fsck_metadata;
          msgprms[1] = Vol_Label;
          msgprmidx[1] = 0;
          fsck_send_msg( fsck_URCVREAD, 2 );
          /*
           * message to debugger
           */
          sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD1_IAG );
          msgprms[0] = message_parm_0;
          msgprmidx[0] = 0;
          sprintf( message_parm_1, "%d", iaggn_rc );
          msgprms[1] = message_parm_1;
          msgprmidx[1] = 0;
          sprintf( message_parm_2, "%d", fsck_READ );
          msgprms[2] = message_parm_2;
          msgprmidx[2] = 0;
          sprintf( message_parm_3, "%lld", (long long)agg_recptr->iag_agg_offset );
          msgprms[3] = message_parm_3;
          msgprmidx[3] = 0;
          sprintf( message_parm_4, "%d", agg_recptr->iag_buf_length );
          msgprms[4] = message_parm_4;
          msgprmidx[4] = 0;
          sprintf( message_parm_5, "%d", agg_recptr->iag_buf_data_len );
          msgprms[5] = message_parm_5;
          msgprmidx[5] = 0;
          fsck_send_msg( fsck_ERRONAGG, 6 );
          iaggn_rc = FSCK_FAILED_BADREAD1_IAG;
        }
      } else {  /* we need to get the next imap leaf (if any) */
        if ( agg_recptr->fais.rootleaf_imap ) {  
            /*
             * there aren't any more imap leafs
             */
          end_of_imap = -1;
        }  /* end there aren't any more imap leafs */
        else if ( agg_recptr->fais.this_mapleaf->header.next == ((int64_t)0)) { 
            /* 
             * there aren't any more imap leafs
             */
          end_of_imap = -1;
        } else {  /* there is another leaf */
          leaf_byteaddr = sb_ptr->s_bsize *
                          agg_recptr->fais.this_mapleaf->header.next;
          iaggn_rc = imapleaf_get( leaf_byteaddr,
                                   &(agg_recptr->fais.this_mapleaf) );
          if ( iaggn_rc == FSCK_OK ) {  /* got the imap leaf */
            agg_recptr->fais.iagidx_now = XTENTRYSTART - 1;
            agg_recptr->fais.iagidx_max =
            agg_recptr->fais.this_mapleaf->header.nextindex - 1;
          }  /* end got the imap leaf */
        }  /* end else there is another leaf */
      }  /* end else we need to get the next imap leaf (if any) */
    }  /* end while !iag_found and !end_of_imap */
  }  /* end flushed the iags ok */
  if ( end_of_imap ) {  /* there aren't any more iags */
    agg_recptr->fais.iagptr = NULL;
    agg_recptr->fais.iagidx_now = -1;
  }  /* end there aren't any more iags */
  if ( iaggn_rc == FSCK_OK ) {  /* everything worked! */
    *addr_iag_ptr = agg_recptr->fais.iagptr;
  }  /* end everything worked! */
  return( iaggn_rc );
}  /* end of iag_get_next () */


/*****************************************************************************
 * NAME: iag_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current iag buffer
 *            has been modified and should be written to the device in the
 *            next flush operation on this buffer.
 *
 * PARAMETERS:
 *      iagptr  - input -  pointer to the iag, in an fsck buffer, which has
 *                         been modified.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t iag_put ( iag_t  *iagptr )
{
  retcode_t  iagp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {  /* we have write access */
    agg_recptr->iag_buf_write = 1;
  } /* end we have write access */
  return( iagp_rc );
}  /* end of iag_put () */


/*****************************************************************************
 * NAME: iags_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current iag buffer has been updated since
 *            the most recent read operation, write the buffer contents to
 *            the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t iags_flush ( )
{
  retcode_t  iagf_rc = FSCK_OK;
  uint32_t   bytes_written;

  if ( agg_recptr->iag_buf_write ) {  
      /* buffer has been updated since
       * most recent write
       */

    /* swap if on big endian machine */
    ujfs_swap_iag_t((iag_t *)agg_recptr->iag_buf_ptr); 
    iagf_rc = readwrite_device ( agg_recptr->iag_agg_offset,
                                 agg_recptr->iag_buf_data_len,
                                 &bytes_written,
                                 (void *) agg_recptr->iag_buf_ptr,
                                 fsck_WRITE  );
    ujfs_swap_iag_t((iag_t *)agg_recptr->iag_buf_ptr); 

    if ( iagf_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->iag_buf_data_len ) {
        agg_recptr->iag_buf_write = 0; /* buffer has been written to
                               * the device and won't need to be
                               * written again unless/until the
                               * buffer contents have been altered again.
                               */
      } else {  /* didn't write the correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 10 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_IAG_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", iagf_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->iag_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->iag_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        iagf_rc = FSCK_FAILED_IAG_BADFLUSH;
      }  /* end else didn't write the correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 11 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_IAG_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", iagf_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->iag_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->iag_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      iagf_rc = FSCK_FAILED_IAG_FLUSH;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */
  return( iagf_rc );
}  /* end of iags_flush () */


/*****************************************************************************
 * NAME: imapleaf_get
 *
 * FUNCTION: Read the specified inode map leaf node into and/or 
 *           locate the specified inode map leaf node in the
 *           fsck inode map leaf buffer.
 *
 * PARAMETERS:
 *      leaf_start_byte  - input - offset, in bytes, in the aggregate, of
 *                                 the needed map leaf node.
 *      addr_leaf_ptr    - input - pointer to a variable in which to return
 *                                 the address of the map leaf in an fsck
 *                                 buffer.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t imapleaf_get ( agg_byteidx_t  leaf_start_byte,
                         xtpage_t       **addr_leaf_ptr   )
{
  retcode_t      imlfg_rc = FSCK_OK;
  agg_byteidx_t  leaf_end_byte;
  int            i;
  xtpage_t       *xtpage_t_ptr;

  leaf_end_byte = leaf_start_byte + sizeof(xtpage_t) - 1;
  if ( (leaf_start_byte >= agg_recptr->mapleaf_agg_offset) &&
       (leaf_end_byte <= (agg_recptr->mapleaf_agg_offset +
                          agg_recptr->mapleaf_buf_data_len))  ) {  
      /*
       * the target leaf is already in
       * the buffer
       */
    *addr_leaf_ptr = (xtpage_t *) (agg_recptr->mapleaf_buf_ptr +
                                   leaf_start_byte -
                                   agg_recptr->mapleaf_agg_offset);
  } else {  /* else we'll have to read it from the disk */
    agg_recptr->mapleaf_agg_offset = leaf_start_byte;
    imlfg_rc = readwrite_device( agg_recptr->mapleaf_agg_offset,
                                 agg_recptr->mapleaf_buf_length,
                                 &(agg_recptr->mapleaf_buf_data_len),
                                 (void *) agg_recptr->mapleaf_buf_ptr,
                                 fsck_READ );

    if ( imlfg_rc == FSCK_OK ) {  /* read appears successful */
      if ( agg_recptr->mapleaf_buf_data_len >= (sizeof(xtpage_t)) ) {  
          /*
           * we may not have gotten all we asked for,
           * but we got enough to cover the node we
           * were after
           */

        /* swap if on big endian machine */
	xtpage_t_ptr = (xtpage_t *)agg_recptr->mapleaf_buf_ptr;
	swap_multiple(ujfs_swap_xtpage_t, xtpage_t_ptr, 4);

        *addr_leaf_ptr = (xtpage_t *) agg_recptr->mapleaf_buf_ptr;
      } else {  /* didn't get the minimum number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_IMPLF );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", imlfg_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        imlfg_rc = FSCK_FAILED_BADREAD_IMPLF;
      }  /* end else didn't get the minimum number of bytes */
    } else {  /* bad return code from read */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_READ_IMPLF );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", imlfg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      imlfg_rc = FSCK_FAILED_READ_IMPLF;
    }  /* end else bad return code from read */
  }  /* end else we'll have to read it from the disk */
  return( imlfg_rc );
}  /* end of imapleaf_get () */


/*****************************************************************************
 * NAME: inode_get
 *
 * FUNCTION: Read the specified inode into and/or located the specified inode
 *           in the fsck inode buffer.
 *
 * PARAMETERS:
 *      is_aggregate  - input -  0 => the inode is owned by the fileset
 *                              !0 => the inode is owned by the aggregate
 *      which_it      - input - ordinal number of the aggregate inode
 *                              representing the inode table to which the
 *                              inode belongs.
 *      ino_idx       - input - ordinal number of the inode to read
 *      addr_ino_ptr  - input - pointer to a variable in which to return
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inode_get ( int       is_aggregate,
                      int       it_number,
                      inoidx_t  ino_idx,
                      dinode_t  **addr_ino_ptr  )
{
  retcode_t      inog_rc = FSCK_OK;
  int            which_ait;
  iagno_t        iag_num;
  iag_t          *iag_ptr;
  int32_t        iag_extidx;
  fsblkidx_t     inoext_fsblk_addr;
  agg_byteidx_t  inoext_byte_addr;
  dinode_t       *din_t_ptr;
  int            i;

  *addr_ino_ptr = NULL;
  /*
   * see if the requested inode is already in the buffer
   */
  if ( (is_aggregate && agg_recptr->ino_for_aggregate)             ||
       ( ((!is_aggregate) && (!agg_recptr->ino_for_aggregate)) &&
         (it_number == agg_recptr->ino_fsnum)                    )   ) { 
      /*
       * buffer contains inodes from the proper table
       */
    if ( (ino_idx >= agg_recptr->ino_buf_1st_ino) &&
         (ino_idx <= (agg_recptr->ino_buf_1st_ino + INOSPEREXT - 1)) ) { 
        /*
         * the requested inode number is in the
         * range for the extent in the buffer now
         */
      *addr_ino_ptr = (dinode_t *)
                      ( (ino_idx - agg_recptr->ino_buf_1st_ino) * DISIZE +
                        agg_recptr->ino_buf_ptr
                      );
    }  /* end the requested inode number is in the range for the ... */
  }  /* end buffer contains inodes from the proper table */
  if ( *addr_ino_ptr == NULL ) {  /* it isn't in the buffer */
    inog_rc = inodes_flush();  /* handle any pending deferred writes */
    /*
     * get the iag describing its extent
     */
    if ( inog_rc == FSCK_OK ) {  /* flushed cleanly */
      if ( is_aggregate ) {
        if ( ino_idx < FIRST_FILESET_INO ) {  /* in part 1 of AIT */
          if ( agg_recptr->primary_ait_4part1 ) {
            which_ait = fsck_primary;
          } else {
            which_ait = fsck_secondary;
          }
        } else {  /* in part 2 of AIT */
          if ( agg_recptr->primary_ait_4part2 ) {
            which_ait = fsck_primary;
          } else {
            which_ait = fsck_secondary;
          }
        }
      } else { /* fileset inode */
        if ( agg_recptr->primary_ait_4part2 ) {
          which_ait = fsck_primary;
        } else {
          which_ait = fsck_secondary;
        }
      }  /* end else fileset inode */
      iag_num = INOTOIAG(ino_idx);  /* figure out which IAG describes the
                          * extent containing the requested inode
                          */
      inog_rc = iag_get(is_aggregate, it_number, which_ait, iag_num, &iag_ptr);
      if ( inog_rc == FSCK_OK ) {  /* got the iag */
        iag_extidx = (ino_idx-agg_recptr->iag_buf_1st_inode) >> L2INOSPEREXT;/*
                          * this is the index into the ixpxd array (in the
                          * iag) for the entry describing the extent with
                          * the desired inode.
                          */
        inoext_fsblk_addr = addressPXD(&(iag_ptr->inoext[iag_extidx]));
        if ( inoext_fsblk_addr == 0 ) {  /* the extent isn't allocated */
          inog_rc = FSCK_INOEXTNOTALLOC;
        } else {  /* the inode extent is allocated */
          inoext_byte_addr = inoext_fsblk_addr * sb_ptr->s_bsize;
          inog_rc = readwrite_device( inoext_byte_addr,
                                      agg_recptr->ino_buf_length,
                                      &(agg_recptr->ino_buf_data_len),
                                      (void *) agg_recptr->ino_buf_ptr,
                                      fsck_READ );

          if ( inog_rc == FSCK_OK ) {  /* looks like it worked */
            agg_recptr->ino_buf_agg_offset = inoext_byte_addr;
            agg_recptr->ino_buf_1st_ino = (iag_num << L2INOSPERIAG) +
                                          (iag_extidx << L2INOSPEREXT);
            agg_recptr->ino_fsnum = it_number;
            memcpy( (void *) &(agg_recptr->ino_ixpxd),
                    (void *) &(iag_ptr->inoext[iag_extidx]),
                    sizeof(pxd_t) );
            is_aggregate ? (agg_recptr->ino_for_aggregate = -1)
            : (agg_recptr->ino_for_aggregate = 0);
            agg_recptr->ino_which_it = it_number;

            *addr_ino_ptr = (dinode_t *)
                            ( (ino_idx - agg_recptr->ino_buf_1st_ino) * DISIZE +
                              agg_recptr->ino_buf_ptr               );
            /* swap if on big endian machine */
	    ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, GET,
			     sb_ptr->s_flag);

          } else {  /* bad return code from read */
            /*
             * message to user
             */
            msgprms[0] = message_parm_0;
            msgprmidx[0] = fsck_metadata;
            msgprms[1] = Vol_Label;
            msgprmidx[1] = 0;
            fsck_send_msg( fsck_URCVREAD, 2 );
            /*
             * message to debugger
             */
            sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_INODE1 );
            msgprms[0] = message_parm_0;
            msgprmidx[0] = 0;
            sprintf( message_parm_1, "%d", inog_rc );
            msgprms[1] = message_parm_1;
            msgprmidx[1] = 0;
            sprintf( message_parm_2, "%d", fsck_READ );
            msgprms[2] = message_parm_2;
            msgprmidx[2] = 0;
            sprintf( message_parm_3, "%lld", (long long)inoext_byte_addr );
            msgprms[3] = message_parm_3;
            msgprmidx[3] = 0;
            sprintf( message_parm_4, "%d", agg_recptr->ino_buf_length );
            msgprms[4] = message_parm_4;
            msgprmidx[4] = 0;
            sprintf( message_parm_5, "%d", agg_recptr->ino_buf_data_len );
            msgprms[5] = message_parm_5;
            msgprmidx[5] = 0;
            fsck_send_msg( fsck_ERRONAGG, 6 );
            inog_rc = FSCK_FAILED_BADREAD_INODE1;
          }  /* end else bad return code from read */
        }  /* end else the inode extent is allocated */
      }  /* end got the iag */
    }  /* end flushed cleanly */
  }  /* end it isn't in the buffer */
  return( inog_rc );
}  /* end of inode_get () */


/*****************************************************************************
 * NAME: inode_get_first_fs
 *
 * FUNCTION: Read the first inode in the specified fileset into and/or
 *           locate the first inode in the specified fileset in the
 *           fsck inode buffer.
 *
 * PARAMETERS:
 *      it_number     - input - ordinal number of the aggregate inode
 *                              representing the inode table to which the
 *                              inode belongs.
 *      ino_idx       - input - pointer to a variable in which to return
 *                              the ordinal number of the first inode in
 *                              use in the aggregate.
 *      addr_ino_ptr  - input - pointer to a variable in which to return
 *                              the address, in an fsck buffer, of the first
 *                              inode now being used in the aggregate.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inode_get_first_fs ( int       it_number,
                               inoidx_t  *ino_idx,
                               dinode_t  **addr_ino_ptr )
{
  retcode_t  igff_rc = FSCK_OK;
  int        which_ait;
  int        is_aggregate = 0;
  pxd_t      *pxdptr;
  iag_t      *iagptr;
  int        i;
  dinode_t   *din_t_ptr;

  if ( agg_recptr->primary_ait_4part2 ) {
    which_ait = fsck_primary;
  } else {
    which_ait = fsck_secondary;
  }
  /*
   * get the first iag of the imap into the iag buffer
   */
  igff_rc = iag_get_first( is_aggregate, it_number, which_ait, &iagptr);
  if ( igff_rc == FSCK_OK ) {  /* first iag is in iag buffer */
    /*
     * get the first inode extent described by the iag into the inode buf
     *
     * note: the very first one must be allocated since it contains
     *       the root directory inode (which has been verified correct)
     */
    igff_rc = inodes_flush();
    if ( igff_rc == FSCK_OK ) {  /* flushed ok */
      pxdptr = &(agg_recptr->fais.iagptr->inoext[agg_recptr->fais.extidx_now]);
      agg_recptr->ino_buf_agg_offset = sb_ptr->s_bsize * addressPXD(pxdptr);
      agg_recptr->ino_buf_1st_ino = agg_recptr->fais.this_inoidx;
      agg_recptr->ino_fsnum = it_number;
      agg_recptr->ino_for_aggregate = is_aggregate;
      agg_recptr->ino_which_it = it_number;
      memcpy( (void *) &(agg_recptr->ino_ixpxd),
              (void *) pxdptr,
              sizeof(pxd_t) );
      igff_rc = readwrite_device( agg_recptr->ino_buf_agg_offset,
                                  agg_recptr->ino_buf_length,
                                  &(agg_recptr->ino_buf_data_len),
                                  (void *) (agg_recptr->ino_buf_ptr),
                                  fsck_READ );

      /* swap if on big endian machine */
      ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, GET,
		       sb_ptr->s_flag);
    }  /* end flushed ok */
  }  /* end first iag is in iag buffer */
  if ( igff_rc == FSCK_OK ) {  /* the first inode extent is in buf */
    agg_recptr->fais.extptr = (dinode_t *) (agg_recptr->ino_buf_ptr);
    /*
     * FILESET_OBJECT_I is the inode number for the first
     * fileset inode not reserved for metadata (excepting the
     * root directory which is handled as a special case)
     */
    agg_recptr->fais.this_inoidx = FILESET_OBJECT_I;
    agg_recptr->fais.inoidx_now = FILESET_OBJECT_I;
    agg_recptr->fais.inoidx_max = INOSPEREXT - 1;
    /*
     * locate the first (regular fileset object) inode in the inode buf
     */
    agg_recptr->fais.inoptr = (dinode_t *)
                              ( (char *) agg_recptr->fais.extptr +
                                (agg_recptr->fais.inoidx_now * DISIZE) );
  } else {  /* bad return code from read */
    /*
     * message to user
     */
    msgprms[0] = message_parm_0;
    msgprmidx[0] = fsck_metadata;
    msgprms[1] = Vol_Label;
    msgprmidx[1] = 0;
    fsck_send_msg( fsck_URCVREAD, 2 );
    /*
     * message to debugger
     */
    sprintf( message_parm_0, "%d", FSCK_FAILED_READ_INODE );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    sprintf( message_parm_1, "%d", igff_rc );
    msgprms[1] = message_parm_1;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", fsck_READ );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    sprintf( message_parm_3, "%lld", (long long)agg_recptr->ino_buf_agg_offset );
    msgprms[3] = message_parm_3;
    msgprmidx[3] = 0;
    sprintf( message_parm_4, "%d", agg_recptr->ino_buf_length );
    msgprms[4] = message_parm_4;
    msgprmidx[4] = 0;
    sprintf( message_parm_5, "%d", agg_recptr->ino_buf_data_len );
    msgprms[5] = message_parm_5;
    msgprmidx[5] = 0;
    fsck_send_msg( fsck_ERRONAGG, 6 );
    igff_rc = FSCK_FAILED_READ_INODE;
  }  /* end else bad return code from read */
  if ( igff_rc == FSCK_OK ) { /* everything worked! */
    *ino_idx = agg_recptr->fais.this_inoidx;
    *addr_ino_ptr = agg_recptr->fais.inoptr;
  }  /* end everything worked! */
  return( igff_rc );
}  /* end of inode_get_first_fs () */


/*****************************************************************************
 * NAME: inode_get_next
 *
 * FUNCTION: Read the next inode in the specified fileset into and/or
 *           locate the next inode in the specified fileset in the
 *           fsck inode buffer.
 *
 * PARAMETERS:
 *      ino_idx       - input - pointer to a variable in which to return
 *                              the ordinal number of the next inode in the
 *                              current inode table traversal.
 *      addr_ino_ptr  - input - pointer to a variable in which to return
 *                              the address, in an fsck buffer, of the next
 *                              inode in the current inode table traversal.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inode_get_next ( inoidx_t  *ino_idx,
                           dinode_t  **addr_ino_ptr )
{
  retcode_t   ign_rc = FSCK_OK;
  int8_t      ext_found, end_of_fileset;
  fsblkidx_t  ext_blkaddr;
  pxd_t       *pxdptr;
  iag_t       *iagptr;
  int         i;
  dinode_t    *din_t_ptr;

  agg_recptr->fais.this_inoidx++;   /* increment inode ordinal number. If
                               * this one is allocated then it's the one
                               * we want
                               */
  agg_recptr->fais.inoidx_now++;    /* increment index into current extent */
  if ( agg_recptr->fais.inoidx_now <= agg_recptr->fais.inoidx_max ) {  
      /*
       * the inode is in the buffer already
       */
    agg_recptr->fais.inoptr = (dinode_t *)
                              ( (char *) agg_recptr->fais.inoptr + DISIZE );
  } else {  /* we need the next allocated inode extent */
    ign_rc = inodes_flush();
    if ( ign_rc == FSCK_OK ) {  /* flushed inodes ok */
      ext_found = 0;
      end_of_fileset = 0;
      agg_recptr->fais.extidx_now++;  /* increment index into current iag */
      while ( (!ext_found) &&
              (!end_of_fileset) &&
              (ign_rc == FSCK_OK) ) {
        if ( agg_recptr->fais.extidx_now <= agg_recptr->fais.extidx_max ) { 
            /*
             * the iag is in the buffer already
             */
          pxdptr =
          &(agg_recptr->fais.iagptr->inoext[agg_recptr->fais.extidx_now]);
          ext_blkaddr = addressPXD(pxdptr);
          if ( ext_blkaddr == 0 ) {  /* this extent isn't allocated */
            agg_recptr->fais.extidx_now++;
            agg_recptr->fais.this_inoidx += INOSPEREXT;
          } else {  /* this extent is allocated */
            agg_recptr->ino_buf_agg_offset = sb_ptr->s_bsize * ext_blkaddr;
            agg_recptr->ino_buf_1st_ino = agg_recptr->fais.this_inoidx;
            memcpy( (void *) &(agg_recptr->ino_ixpxd),
                    (void *) pxdptr,
                    sizeof(pxd_t) );
            ign_rc = readwrite_device( agg_recptr->ino_buf_agg_offset,
                                       agg_recptr->ino_buf_length,
                                       &(agg_recptr->ino_buf_data_len),
                                       (void *) (agg_recptr->ino_buf_ptr),
                                       fsck_READ );

            if ( ign_rc == FSCK_OK ) {  /* got the extent */

              /* swap if on big endian machine */
	      ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, GET,
			       sb_ptr->s_flag);

              ext_found = -1;
              agg_recptr->fais.extptr = (dinode_t *) (agg_recptr->ino_buf_ptr);
              agg_recptr->fais.inoidx_now = 0;
              agg_recptr->fais.inoidx_max = INOSPEREXT - 1;
              ext_found = -1;
              agg_recptr->fais.inoptr = (dinode_t *) (agg_recptr->ino_buf_ptr);
            } else {  /* bad return code from read */
              /*
               * message to user
               */
              msgprms[0] = message_parm_0;
              msgprmidx[0] = fsck_metadata;
              msgprms[1] = Vol_Label;
              msgprmidx[1] = 0;
              fsck_send_msg( fsck_URCVREAD, 2 );
              /*
               * message to debugger
               */
              sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_INODE );
              msgprms[0] = message_parm_0;
              msgprmidx[0] = 0;
              sprintf( message_parm_1, "%d", ign_rc );
              msgprms[1] = message_parm_1;
              msgprmidx[1] = 0;
              sprintf( message_parm_2, "%d", fsck_READ );
              msgprms[2] = message_parm_2;
              msgprmidx[2] = 0;
              sprintf( message_parm_3, "%lld", (long long)agg_recptr->ino_buf_agg_offset );
              msgprms[3] = message_parm_3;
              msgprmidx[3] = 0;
              sprintf( message_parm_4, "%d", agg_recptr->ino_buf_length );
              msgprms[4] = message_parm_4;
              msgprmidx[4] = 0;
              sprintf( message_parm_5, "%d", agg_recptr->ino_buf_data_len );
              msgprms[5] = message_parm_5;
              msgprmidx[5] = 0;
              fsck_send_msg( fsck_ERRONAGG, 6 );
              ign_rc = FSCK_FAILED_BADREAD_INODE;
            }  /* end else bad return code from read */
          }  /* end else this extent is allocated */
        } else { /* we need to get the next iag (if any) */
          ign_rc = iag_get_next( &iagptr );
          if ( (ign_rc == FSCK_OK) && (iagptr == NULL) ) {
            end_of_fileset = -1;
          }
        }  /* end else we need to get the next iag (if any) */
      }  /* end while !ext_found and !end_of_fileset */
      if ( end_of_fileset ) {  /* there aren't any more extents */
        agg_recptr->fais.inoptr = NULL;
        agg_recptr->fais.this_inoidx = -1;
      }  /* end there aren't any more extents */
    }  /* end flushed inodes ok */
  }  /* end we need the next allocated inode extent */
  if ( ign_rc == FSCK_OK ) {  /* everything worked! */
    *ino_idx = agg_recptr->fais.this_inoidx;
    *addr_ino_ptr = agg_recptr->fais.inoptr;
  }  /* end everything worked! */
  return( ign_rc );
}  /* end of inode_get_next () */


/*****************************************************************************
 * NAME: inode_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current inode buffer
 *            has been modified and should be written to the device in the
 *            next flush operation on this buffer.
 *
 * PARAMETERS:
 *      ino_ptr  - input - address, in an fsck buffer, of the inode which has
 *                         been updated.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inode_put ( dinode_t  *ino_ptr )
{
  retcode_t  ip_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {
    agg_recptr->ino_buf_write = -1;  /* buffer has been updated and needs
                                    * to be written to the device
                                    */
  }
  return( ip_rc );
}  /* end of inode_put () */


/*****************************************************************************
 * NAME: inodes_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current inode buffer has been updated since the most
 *            recent read operation, write the buffer contents to the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inodes_flush ( )
{
  retcode_t  inof_rc = FSCK_OK;
  uint32_t   bytes_written;
  int        i;
  dinode_t   *dinode_t_ptr;


  if ( agg_recptr->ino_buf_write ) {  
      /* buffer has been updated since
       * most recent write
       */

    /* swap if on big endian machine */
    ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, PUT, sb_ptr->s_flag);
    inof_rc = readwrite_device( agg_recptr->ino_buf_agg_offset,
                                agg_recptr->ino_buf_data_len,
                                &bytes_written,
                                (void *) agg_recptr->ino_buf_ptr,
                                fsck_WRITE );
    ujfs_swap_inoext((dinode_t *)agg_recptr->ino_buf_ptr, GET, sb_ptr->s_flag);

    if ( inof_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->ino_buf_data_len ) {
        agg_recptr->ino_buf_write = 0; /* buffer has been written
                               * to the device and won't need to be
                               * written again unless/until the
                               * buffer contents have been altered.
                               */
      } else {  /* didn't write the correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 12 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_INODE_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", inof_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->ino_buf_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->ino_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        inof_rc = FSCK_FAILED_INODE_BADFLUSH;
      }  /* end else didn't write the correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 14 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_INODE_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", inof_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->ino_buf_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->ino_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      inof_rc = FSCK_FAILED_INODE_FLUSH;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */
  return( inof_rc );
}  /* end of inodes_flush () */


/*****************************************************************************
 * NAME: inotbl_get_ctl_page
 *
 * FUNCTION: Read the control page for the specified inode table into and/or
 *           locate the control page for the specified inode table in the
 *           fsck inode table control page buffer.
 *
 * PARAMETERS:
 *      is_aggregate      - input -  0 => the inode table is fileset owned
 *                                  !0 => the inode table is aggregate owned
 *      addr_ctlpage_ptr  - input - pointer to a variable in which to return
 *                                  the address of the control page in an
 *                                  fsck buffer
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inotbl_get_ctl_page ( int        is_aggregate,
                                dinomap_t  **addr_ctlpage_ptr )
{
  retcode_t   igcp_rc = FSCK_OK;
  retcode_t   intermed_rc = FSCK_OK;
  fsblkidx_t  offset_ctlpage = 0;
  int         which_it;
  int         which_ait;
  dinode_t    *inoptr;
  xtpage_t    *xtpg_ptr;
  /*
   * get the byte offset of the control page
   */
  if ( is_aggregate ) {  /* aggregate inode table wanted */
    if ( agg_recptr->primary_ait_4part1 ) {
      offset_ctlpage = AIMAP_OFF / sb_ptr->s_bsize;
    } else {
      offset_ctlpage = addressPXD(&(sb_ptr->s_aim2));
    }
  } else {  /* fileset inode table wanted */
    which_it = FILESYSTEM_I;
    if ( agg_recptr->primary_ait_4part2 ) {
      which_ait = fsck_primary;
    } else {
      which_ait = fsck_secondary;
    }
    if ( agg_recptr->fset_imap.imap_is_rootleaf ) { /* root leaf inode */
      intermed_rc = ait_special_read_ext1( which_ait );
      if ( intermed_rc != FSCK_OK ) {
        report_readait_error( intermed_rc,
                              FSCK_FAILED_AGFS_READ4,
                              which_ait );
        igcp_rc = FSCK_FAILED_AGFS_READ4;
      } else {  /* got the agg inode extent */
        inoptr = (dinode_t *)
                 (agg_recptr->ino_buf_ptr + which_it * sizeof(dinode_t));
        xtpg_ptr = (xtpage_t *) &(inoptr->di_btroot);
        offset_ctlpage = addressXAD(&(xtpg_ptr->xad[XTENTRYSTART]));
      }  /* end else got the agg inode extent */
    } else {  /* tree doesn't have a root-leaf */
      intermed_rc =
      imapleaf_get( agg_recptr->fset_imap.first_leaf_offset, &xtpg_ptr );
      if ( intermed_rc != FSCK_OK ) {
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_READ_NODE3 );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", igcp_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->ino_buf_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->ino_buf_length );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", agg_recptr->ino_buf_data_len );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        igcp_rc = FSCK_FAILED_READ_NODE3;
      } else {  /* got the first leaf node */
        offset_ctlpage = addressXAD(&(xtpg_ptr->xad[XTENTRYSTART]));
      }  /* end else got the first leaf node */
    }  /* end else tree doesn't have a root-leaf */
  }  /* end else fileset inode table wanted */
  /*
   * read the control page into the buffer
   */
  if ( igcp_rc == FSCK_OK ) {  /* we have an offset */
    igcp_rc = mapctl_get( offset_ctlpage, (void **) &xtpg_ptr );
    if ( igcp_rc != FSCK_OK ) {  /* this is fatal */
      igcp_rc = FSCK_FAILED_CANTREADAITCTL;
    } else {  /* the control page is in the buffer */
      *addr_ctlpage_ptr = (dinomap_t *) xtpg_ptr;

      /* swap if on big endian machine */
      ujfs_swap_dinomap_t( *addr_ctlpage_ptr );

    }  /* end else the control page is in the buffer */
  }  /* end we have an offset */
  return( igcp_rc );
}  /* end of inotbl_get_ctl_page () */


/*****************************************************************************
 * NAME: inotbl_put_ctl_page
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current fsck inode table
 *            control page buffer has been modified and should be written
 *            to the device in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      is_aggregate  - input -  0 => the inode table is fileset owned
 *                              !0 => the inode table is aggregate owned
 *      ctlpage_ptr   - input - the address, in an fsck buffer, of the
 *                               control page which has been updated
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t inotbl_put_ctl_page ( int        is_aggregate,
                                dinomap_t  *ctlpage_ptr  )
{
  retcode_t  ipcp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {
    /* swap if on big endian machine */
    ujfs_swap_dinomap_t( ctlpage_ptr );
    ipcp_rc = mapctl_put( (void *) ctlpage_ptr );
  }
  return( ipcp_rc );
}  /* end of inotbl_put_ctl_page () */


/*****************************************************************************
 * NAME: mapctl_get
 *
 * FUNCTION: Read the specified map control page into and/or locate the
 *           requested specified map control page in the fsck map control
 *           page buffer.
 *
 * PARAMETERS:
 *      mapctl_fsblk_offset  - input - offset, in aggregate blocks, into the
 *                                     aggregate, of the needed map control
 *                                     page.
 *      addr_mapctl_ptr      - input - pointer to a variable in which to return
 *                                     the address, in an fsck buffer, of the
 *                                     map control page which has been read.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t mapctl_get ( fsblkidx_t  mapctl_fsblk_offset,
                       void        **addr_mapctl_ptr      )
{
  retcode_t      mg_rc = FSCK_OK;
  agg_byteidx_t  start_byte, end_byte;

  start_byte = mapctl_fsblk_offset * sb_ptr->s_bsize;
  end_byte = start_byte + BYTESPERPAGE - 1;
  if ( start_byte == agg_recptr->mapctl_agg_offset ) {  
      /*
       * the target control page is already in
       * the buffer
       */
    *addr_mapctl_ptr = (void *) (agg_recptr->mapctl_buf_ptr);
  } else {  /* else we'll have to read it from the disk */
    mg_rc = mapctl_flush();  /* handle any pending deferred writes */
    if ( mg_rc == FSCK_OK ) { /* flushed ok */
      agg_recptr->mapctl_agg_offset = start_byte;
      mg_rc = readwrite_device( agg_recptr->mapctl_agg_offset,
                                agg_recptr->mapctl_buf_length,
                                &(agg_recptr->mapctl_buf_data_len),
                                (void *) agg_recptr->mapctl_buf_ptr,
                                fsck_READ );

/* endian - can't swap here, don't know if it's dmap_t or dinomap_t.
            swap before calling this routine.                          */

      if ( mg_rc == FSCK_OK ) {  /* read appears successful */
        if ( agg_recptr->mapctl_buf_data_len >= BYTESPERPAGE ) {  
            /*
             * we may not have gotten all we asked for,
             * but we got enough to cover the page we
             * were after
             */
          *addr_mapctl_ptr = (void *) agg_recptr->mapctl_buf_ptr;
        } else {  /* didn't get the minimum number of bytes */
          /*
           * message to user
           */
          msgprms[0] = message_parm_0;
          msgprmidx[0] = fsck_metadata;
          msgprms[1] = Vol_Label;
          msgprmidx[1] = 0;
          fsck_send_msg( fsck_URCVREAD, 2 );
          /*
           * message to debugger
           */
          sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_MAPCTL );
          msgprms[0] = message_parm_0;
          msgprmidx[0] = 0;
          sprintf( message_parm_1, "%d", mg_rc );
          msgprms[1] = message_parm_1;
          msgprmidx[1] = 0;
          sprintf( message_parm_2, "%d", fsck_READ );
          msgprms[2] = message_parm_2;
          msgprmidx[2] = 0;
          sprintf( message_parm_3, "%lld", (long long)agg_recptr->mapctl_agg_offset );
          msgprms[3] = message_parm_3;
          msgprmidx[3] = 0;
          sprintf( message_parm_4, "%d", agg_recptr->mapctl_buf_length );
          msgprms[4] = message_parm_4;
          msgprmidx[4] = 0;
          sprintf( message_parm_5, "%d", agg_recptr->mapctl_buf_data_len );
          msgprms[5] = message_parm_5;
          msgprmidx[5] = 0;
          fsck_send_msg( fsck_ERRONAGG, 6 );
          mg_rc = FSCK_FAILED_BADREAD_MAPCTL;
        }  /* end else didn't get the minimum number of bytes */
      } else {  /* bad return code from read */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_READ_MAPCTL );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", mg_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->mapctl_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->mapctl_buf_length );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", agg_recptr->mapctl_buf_data_len );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        mg_rc = FSCK_FAILED_READ_MAPCTL;
      }  /* end else bad return code from read */
    }  /* end flushed ok */
  }  /* end else we'll have to read it from the disk */
  return( mg_rc );
}  /* end of mapctl_get () */


/*****************************************************************************
 * NAME: mapctl_put
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate,
 *            note, in the fsck workspace, that the current fsck map control
 *            page buffer has been modified and should be written to the
 *            device in the next flush operation on this buffer.
 *
 * PARAMETERS:
 *      mapctl_ptr  - input - address, in an fsck buffer, of the map control
 *                            page which has been updated.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t mapctl_put ( void  *mapctl_ptr )
{
  retcode_t  mp_rc = FSCK_OK;

  if ( agg_recptr->processing_readwrite ) {
    agg_recptr->mapctl_buf_write = -1; /* buffer has been updated and needs
                                      * to be written to the device
                                      */
  }
  return( mp_rc );
}  /* end of mapctl_put () */


/*****************************************************************************
 * NAME: mapctl_flush
 *
 * FUNCTION:  If the current fsck session has write access to the aggregate
 *            and the current map control buffer has been updated since
 *            the most recent read operation, write the buffer contents to
 *            the device.
 *
 * PARAMETERS:  none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t mapctl_flush ( )
{
  retcode_t  mf_rc = FSCK_OK;
  uint32_t   bytes_written;

  if ( agg_recptr->mapctl_buf_write ) {  
      /* buffer has been updated since
       * most recent write
       */

      /* endian - swap before calling this routine */

      mf_rc = readwrite_device( agg_recptr->mapctl_agg_offset,
                                agg_recptr->mapctl_buf_data_len,
                                &bytes_written,
                                (void *) agg_recptr->mapctl_buf_ptr,
                                fsck_WRITE );
    if ( mf_rc == FSCK_OK ) {
      if ( bytes_written == agg_recptr->mapctl_buf_length ) {
        agg_recptr->mapctl_buf_write = 0; /* buffer has been written to the
                               * device and won't need to be written again
                               * unless/until it the buffer contents have
                               * been altered.
                               */
      } else {  /* didn't write the correct number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", 15 );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        fsck_send_msg( fsck_URCVWRT, 3 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_MAPCTL_BADFLUSH );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", mf_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_WRITE );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->mapctl_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->mapctl_buf_data_len );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", bytes_written );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        mf_rc = FSCK_FAILED_MAPCTL_BADFLUSH;
      }  /* end else didn't write the correct number of bytes */
    } else {  /* else the write was not successful */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", 16 );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      fsck_send_msg( fsck_URCVWRT, 3 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_MAPCTL_FLUSH );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", mf_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_WRITE );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->mapctl_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->mapctl_buf_data_len );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_written );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      mf_rc = FSCK_FAILED_MAPCTL_FLUSH ;
    }  /* end else the write was not successful */
  }  /* end buffer has been updated since most recent write */
  return( mf_rc );
}  /* end of mapctl_flush () */


/*****************************************************************************
 * NAME: node_get
 *
 * FUNCTION: Read the specified xTree node into and/or locate the specified
 *           xTree node in the fsck xTree node buffer.
 *
 * PARAMETERS:
 *      node_fsblk_offset  - input - offset, in aggregate blocks, into the
 *                                   aggregate, of the xTree node wanted
 *      addr_xtpage_ptr    - input - pointer to a variable in which to return
 *                                   the address, in an fsck buffer, of the
 *                                   of the xTree node which has been read.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t node_get ( fsblkidx_t  node_fsblk_offset,
                     xtpage_t    **addr_xtpage_ptr    )
{
  retcode_t      nodg_rc = FSCK_OK;
  agg_byteidx_t  node_start_byte, node_end_byte;
  int            i;
  xtpage_t       *xtpage_t_ptr;

  node_start_byte = node_fsblk_offset * sb_ptr->s_bsize;
  node_end_byte = node_start_byte + sizeof(xtpage_t) - 1;

  if ( (agg_recptr->ondev_wsp_fsblk_offset != 0) &&
       (node_fsblk_offset > agg_recptr->ondev_wsp_fsblk_offset) ) { 
      /*
       * the offset is beyond the range
       * valid for fileset objects
       */
    /*
     * This case is not caused by an I/O error, but by
     * invalid data in an inode.  Let the caller handle
     * the consequences.
     */
    nodg_rc = FSCK_BADREADTARGET1;
  } else if ( (node_start_byte >= agg_recptr->node_agg_offset) &&
              (node_end_byte <= (agg_recptr->node_agg_offset +
                                agg_recptr->node_buf_data_len))  ) {  
      /*
       * the target node is already in
       * the buffer
       */
    *addr_xtpage_ptr = (xtpage_t *)
                       (agg_recptr->node_buf_ptr +
                        node_start_byte - agg_recptr->node_agg_offset);
  } else {  /* else we'll have to read it from the disk */
    agg_recptr->node_agg_offset = node_start_byte;
    nodg_rc = readwrite_device( agg_recptr->node_agg_offset,
                                agg_recptr->node_buf_length,
                                &(agg_recptr->node_buf_data_len),
                                (void *) agg_recptr->node_buf_ptr,
                                fsck_READ );
    if ( nodg_rc == FSCK_OK ) {  /* read appears successful */
      if ( agg_recptr->node_buf_data_len >= (sizeof(xtpage_t)) ) {  
          /*
           * we may not have gotten all we asked for,
           * but we got enough to cover the node we
           * were after
           */

        /* swap if on big endian machine */
	xtpage_t_ptr = (xtpage_t *)agg_recptr->node_buf_ptr;
	swap_multiple(ujfs_swap_xtpage_t, xtpage_t_ptr, 4);

        *addr_xtpage_ptr = (xtpage_t *) agg_recptr->node_buf_ptr;
      } else {  /* didn't get the minimum number of bytes */
        /*
         * message to user
         */
        msgprms[0] = message_parm_0;
        msgprmidx[0] = fsck_metadata;
        msgprms[1] = Vol_Label;
        msgprmidx[1] = 0;
        fsck_send_msg( fsck_URCVREAD, 2 );
        /*
         * message to debugger
         */
        sprintf( message_parm_0, "%d", FSCK_FAILED_BADREAD_NODE );
        msgprms[0] = message_parm_0;
        msgprmidx[0] = 0;
        sprintf( message_parm_1, "%d", nodg_rc );
        msgprms[1] = message_parm_1;
        msgprmidx[1] = 0;
        sprintf( message_parm_2, "%d", fsck_READ );
        msgprms[2] = message_parm_2;
        msgprmidx[2] = 0;
        sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
        msgprms[3] = message_parm_3;
        msgprmidx[3] = 0;
        sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
        msgprms[4] = message_parm_4;
        msgprmidx[4] = 0;
        sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
        msgprms[5] = message_parm_5;
        msgprmidx[5] = 0;
        fsck_send_msg( fsck_ERRONAGG, 6 );
        nodg_rc = FSCK_FAILED_BADREAD_NODE;
      }  /* end else didn't get the minimum number of bytes */
    } else {  /* bad return code from read */
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_FAILED_READ_NODE );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", nodg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)agg_recptr->node_agg_offset );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", agg_recptr->node_buf_length );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", agg_recptr->node_buf_data_len );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      nodg_rc = FSCK_FAILED_READ_NODE;
    }  /* end else bad return code from read */
  }  /* end else we'll have to read it from the disk */
  return( nodg_rc );
}  /* end of node_get () */


/*****************************************************************************
 * NAME: open_device_read
 *
 * FUNCTION:  Open the specified device for read access.
 *
 * PARAMETERS:
 *      Device  - input - the device specification
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t open_device_read ( char  *Device )
{
  Dev_IOPort = open(Device, O_RDONLY, 0);

  sprintf( message_parm_0, "%d", (Dev_IOPort < 0) ? ERROR_FILE_NOT_FOUND : 0 );
  msgprms[0] = message_parm_0;
  msgprmidx[0] = 0;
  fsck_send_msg( fsck_DEVOPENRDRC, 1 );
  if (Dev_IOPort < 0)
    return ERROR_FILE_NOT_FOUND;
  Dev_blksize = Dev_SectorSize = PBSIZE;
  return 0;
}


/***************************************************************************
 * NAME: open_device_rw_exclusive
 *
 * FUNCTION:  Open the device for read/write access.  Lock the device
 *            to prevent others from reading from or writing to the
 *            device.
 *
 * PARAMETERS:
 *      Device  - input -  The device specification
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t open_device_rw_exclusive ( char  *Device )
{
  Dev_IOPort = open(Device, O_RDWR|O_EXCL, 0);

  sprintf( message_parm_0, "%d", (Dev_IOPort < 0) ? ERROR_FILE_NOT_FOUND : 0 );
  msgprms[0] = message_parm_0;
  msgprmidx[0] = 0;
  fsck_send_msg( fsck_DEVOPENRDWRRC, 1 );
  if (Dev_IOPort < 0)
    return ERROR_FILE_NOT_FOUND;
  Dev_blksize = Dev_SectorSize = PBSIZE;
  return 0;
}


/*****************************************************************************
 * NAME: open_volume
 *
 * FUNCTION:  Open the device on which the aggregate resides.
 *
 * PARAMETERS:
 *      volname  - input - The device specifier
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t open_volume ( char  *volname )
{
  retcode_t  opnvol_rc = FSCK_OK;

  if ( agg_recptr->parm_options[UFS_CHKDSK_LEVEL0] ) { 
      /*
       * read-only request
       */
    opnvol_rc = open_device_read ( volname );
    if ( opnvol_rc == FSCK_OK ) {  /* successfully opened for Read */
      agg_recptr->processing_readonly = 1;
    }
  } else { /* read-write request */
    opnvol_rc = open_device_rw_exclusive ( volname );
    if ( opnvol_rc == 0 ) {  /* successfully opened for Read/Write */
      agg_recptr->processing_readwrite = 1;
    } else {  /* unable to open for exclusive Read/Write */
      opnvol_rc = open_device_read ( volname );
      if ( opnvol_rc == FSCK_OK ) {  /* successfully opened for Read */
        agg_recptr->processing_readonly = 1;
      }
    }  /* end unable to open for exclusive Read/Write */
  }  /* end read-write request */

  if ( opnvol_rc != FSCK_OK ) {
    agg_recptr->fsck_is_done = 1;
  }

  return( opnvol_rc );
}  /* end of open_volume () */


/*****************************************************************************
 * NAME: readwrite_device
 *
 * FUNCTION:  Read data from or write data to the device on which the
 *            aggregate resides.
 *
 * PARAMETERS:
 *      dev_offset           - input - the offset, in bytes, into the aggregate
 *                                     of the data to read or to which to write
 *                                     the data.
 *      requested_data_size  - input - the number of bytes requested
 *      actual_data_size     - input - pointer to a variable in which to return
 *                                     the number of bytes actually read or
 *                                     written
 *      data_buffer          - input - the address of the buffer in which to
 *                                     put the data or from which to write
 *                                     the data
 *      mode                 - input - { fsck_READ | fsck_WRITE }
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t readwrite_device( agg_byteidx_t  dev_offset,
                            uint32_t       requested_data_size,
                            uint32_t       *actual_data_size,
                            void           *data_buffer,
                            int            mode                 )
{
  retcode_t  rwdb_rc = FSCK_OK;

  if ( (ondev_jlog_byte_offset > 0) &&
       (ondev_jlog_byte_offset <= dev_offset) ) {  /* into jlog! */
#ifdef _JFS_DEBUG
    printf( "READ/WRITE target is in JOURNAL!!\n");
    printf("       read/write offset = x%llx\r\n", dev_offset );
    printf("       jlog offset       = x%llx\r\n", ondev_jlog_byte_offset );
    abort();
#endif
    rwdb_rc = FSCK_IOTARGETINJRNLLOG;
  } else {
    /* not trying to access the journal log */
    if ( (dev_offset % Dev_SectorSize) || (requested_data_size % Dev_SectorSize ) ) {
      rwdb_rc = FSCK_FAILED_SEEK;
    } else {  /* offset seems ok */
      switch (mode) {
        case fsck_READ :
          rwdb_rc = ujfs_rw_diskblocks(Dev_IOPort, dev_offset, requested_data_size, data_buffer, GET);
          break;
        case fsck_WRITE :
          rwdb_rc = ujfs_rw_diskblocks(Dev_IOPort, dev_offset, requested_data_size, data_buffer, PUT);
          break;
        default:
          rwdb_rc =  FSCK_INTERNAL_ERROR_3;
          break;
      }
    }  /* end else offset seems ok */

    if ( rwdb_rc == FSCK_OK ) {
      *actual_data_size = requested_data_size;
    } else {
      *actual_data_size = 0;
    } 

  }  /* end else not trying to access the journal log */

  return( rwdb_rc );

}  /* end of readwrite_device () */


/*****************************************************************************
 * NAME: recon_dnode_assign
 *
 * FUNCTION:  Allocate a buffer for a new dnode at the specified aggregate
 *            offset.
 *
 * PARAMETERS:
 *      fsblk_offset  - input - The offset, in aggregate blocks, into the
 *                              aggregate, of the new dnode.
 *      addr_buf_ptr  - input - pointer to a variable in which to return
 *                              the address of the buffer allocated.
 *
 *
 * NOTES: The offset of the dnode being created is a required input because
 *        these buffers have a trailer record which contains the offset
 *         at which to write the dnode.  If this is not stored when the
 *         buffer is allocated for a new dnode, the other routines would
 *         have to treat a new dnode as a special case.
 *
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t recon_dnode_assign ( fsblkidx_t  fsblk_offset,
                               dtpage_t    **addr_buf_ptr  )
{
  retcode_t         rda_rc = FSCK_OK;
  recon_bufrec_ptr  bufrec_ptr;

  rda_rc = dire_buffer_alloc( addr_buf_ptr );

  if ( rda_rc != FSCK_OK ) {
    sprintf( message_parm_0, "%d", 1 );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    fsck_send_msg( fsck_CANTRECONINSUFSTG, 1 );
  } else {
    bufrec_ptr = (recon_bufrec_ptr) *addr_buf_ptr;
    bufrec_ptr->dnode_blkoff = fsblk_offset;
    bufrec_ptr->dnode_byteoff = fsblk_offset * sb_ptr->s_bsize;
  }
  return( rda_rc );
}  /* end of recon_dnode_assign () */


/*****************************************************************************
 * NAME: recon_dnode_get
 *
 * FUNCTION: Allocate an fsck dnode buffer and read the JFS dnode page at
 *           the specified offset into it.
 *
 * PARAMETERS:
 *      fsblk_offset  - input - the offset, in aggregate blocks, into the
 *                              aggregate, of the desired dnode.
 *      addr_buf_ptr  - input - pointer to a variable in which to return
 *                              the fsck dnode buffer which has been allocated
 *                              and into which the requested dnode has been
 *                              read.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t recon_dnode_get ( fsblkidx_t  fsblk_offset,
                            dtpage_t    **addr_buf_ptr  )
{
  retcode_t         rdg_rc = FSCK_OK;
  reg_idx_t         bytes_read;
  recon_bufrec_ptr  bufrec_ptr;
  dtpage_t          *dtp;

  rdg_rc = dire_buffer_alloc( addr_buf_ptr );

  if ( rdg_rc != FSCK_OK ) {
    sprintf( message_parm_0, "%d", 2 );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    fsck_send_msg( fsck_CANTRECONINSUFSTG, 1 );
  } else {  /* got the buffer */
    bufrec_ptr = (recon_bufrec_ptr) *addr_buf_ptr;
    bufrec_ptr->dnode_blkoff = fsblk_offset;
    bufrec_ptr->dnode_byteoff = fsblk_offset * sb_ptr->s_bsize;
    rdg_rc = readwrite_device( bufrec_ptr->dnode_byteoff,
                               BYTESPERPAGE,
                               &bytes_read,
                               (void *) (*addr_buf_ptr),
                               fsck_READ                );
    if ( rdg_rc != FSCK_OK ) {
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_CANTREADRECONDNODE );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", rdg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)bufrec_ptr->dnode_byteoff );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", BYTESPERPAGE );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_read );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      rdg_rc = FSCK_CANTREADRECONDNODE;
      dire_buffer_release( *addr_buf_ptr );
    } else if ( bytes_read != BYTESPERPAGE ) {
      /*
       * message to user
       */
      msgprms[0] = message_parm_0;
      msgprmidx[0] = fsck_metadata;
      msgprms[1] = Vol_Label;
      msgprmidx[1] = 0;
      fsck_send_msg( fsck_URCVREAD, 2 );
      /*
       * message to debugger
       */
      sprintf( message_parm_0, "%d", FSCK_CANTREADRECONDNODE1 );
      msgprms[0] = message_parm_0;
      msgprmidx[0] = 0;
      sprintf( message_parm_1, "%d", rdg_rc );
      msgprms[1] = message_parm_1;
      msgprmidx[1] = 0;
      sprintf( message_parm_2, "%d", fsck_READ );
      msgprms[2] = message_parm_2;
      msgprmidx[2] = 0;
      sprintf( message_parm_3, "%lld", (long long)bufrec_ptr->dnode_byteoff );
      msgprms[3] = message_parm_3;
      msgprmidx[3] = 0;
      sprintf( message_parm_4, "%d", BYTESPERPAGE );
      msgprms[4] = message_parm_4;
      msgprmidx[4] = 0;
      sprintf( message_parm_5, "%d", bytes_read );
      msgprms[5] = message_parm_5;
      msgprmidx[5] = 0;
      fsck_send_msg( fsck_ERRONAGG, 6 );
      rdg_rc = FSCK_CANTREADRECONDNODE1;
      dire_buffer_release( *addr_buf_ptr );
    } else {
      /* swap if on big endian machine */
      dtp = *addr_buf_ptr;
      ujfs_swap_dtpage_t( dtp, sb_ptr->s_flag );
      dtp->header.flag |= BT_SWAPPED;
    }
  }  /* end got the buffer */
  return( rdg_rc );
}  /* end of recon_dnode_get () */


/*****************************************************************************
 * NAME: recon_dnode_put
 *
 * FUNCTION:  Write the dnode in the specified buffer into the aggregate
 *            and then release the buffer.
 *
 * PARAMETERS:
 *      buf_ptr  - input -  the address of the buffer containing the
 *                          dnode to write.
 *
 * NOTES:  Unlike most _put_ routines in this module, blkmap_put_ctl_page
 *         actually writes to the device.
 *
 *         The buffer has a trailer record which contains the offset
 *         at which to write the dnode.
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t recon_dnode_put ( dtpage_t  *buf_ptr )
{
  retcode_t         rdp_rc = FSCK_OK;
  reg_idx_t         bytes_written;
  recon_bufrec_ptr  bufrec_ptr;

  bufrec_ptr = (recon_bufrec_ptr) buf_ptr;

  /* swap if on big endian machine */
  ujfs_swap_dtpage_t(buf_ptr, sb_ptr->s_flag ); 
  buf_ptr->header.flag &= ~BT_SWAPPED;
  rdp_rc = readwrite_device( bufrec_ptr->dnode_byteoff,
                             BYTESPERPAGE,
                             &bytes_written,
                             (void *) buf_ptr,
                             fsck_WRITE  );
  ujfs_swap_dtpage_t(buf_ptr, sb_ptr->s_flag ); 
  buf_ptr->header.flag |= BT_SWAPPED;

  if ( rdp_rc != FSCK_OK ) {
    /*
     * message to user
     */
    msgprms[0] = message_parm_0;
    msgprmidx[0] = fsck_metadata;
    msgprms[1] = Vol_Label;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", 17 );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    fsck_send_msg( fsck_URCVWRT, 3 );
    /*
     * message to debugger
     */
    sprintf( message_parm_0, "%d", FSCK_CANTWRITRECONDNODE );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    sprintf( message_parm_1, "%d", rdp_rc );
    msgprms[1] = message_parm_1;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", fsck_WRITE );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    sprintf( message_parm_3, "%lld", (long long)bufrec_ptr->dnode_byteoff );
    msgprms[3] = message_parm_3;
    msgprmidx[3] = 0;
    sprintf( message_parm_4, "%d", BYTESPERPAGE );
    msgprms[4] = message_parm_4;
    msgprmidx[4] = 0;
    sprintf( message_parm_5, "%d", bytes_written );
    msgprms[5] = message_parm_5;
    msgprmidx[5] = 0;
    fsck_send_msg( fsck_ERRONAGG, 6 );
    rdp_rc = FSCK_CANTWRITRECONDNODE;
  } else if ( bytes_written != BYTESPERPAGE ) {
    /*
     * message to user
     */
    msgprms[0] = message_parm_0;
    msgprmidx[0] = fsck_metadata;
    msgprms[1] = Vol_Label;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", 18 );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    fsck_send_msg( fsck_URCVWRT, 3 );
    /*
     * message to debugger
     */
    sprintf( message_parm_0, "%d", FSCK_CANTWRITRECONDNODE1 );
    msgprms[0] = message_parm_0;
    msgprmidx[0] = 0;
    sprintf( message_parm_1, "%d", rdp_rc );
    msgprms[1] = message_parm_1;
    msgprmidx[1] = 0;
    sprintf( message_parm_2, "%d", fsck_WRITE );
    msgprms[2] = message_parm_2;
    msgprmidx[2] = 0;
    sprintf( message_parm_3, "%lld", (long long)bufrec_ptr->dnode_byteoff );
    msgprms[3] = message_parm_3;
    msgprmidx[3] = 0;
    sprintf( message_parm_4, "%d", BYTESPERPAGE );
    msgprms[4] = message_parm_4;
    msgprmidx[4] = 0;
    sprintf( message_parm_5, "%d", bytes_written );
    msgprms[5] = message_parm_5;
    msgprmidx[5] = 0;
    fsck_send_msg( fsck_ERRONAGG, 6 );
    rdp_rc = FSCK_CANTWRITRECONDNODE1;
  }
  bufrec_ptr->dnode_blkoff = 0;
  bufrec_ptr->dnode_byteoff = 0;
  dire_buffer_release( buf_ptr );
  return( rdp_rc );
}  /* end of recon_dnode_put () */


/*****************************************************************************
 * NAME: recon_dnode_release
 *
 * FUNCTION:  Release the specified fsck dnode buffer without writing its
 *            contents to the aggregate.
 *
 * PARAMETERS:
 *      buf_ptr  - input -  Address of the fsck dnode buffer to release.
 *
 * NOTES:
 *
 * RETURNS:
 *      success: FSCK_OK
 *      failure: something else
 */
retcode_t recon_dnode_release ( dtpage_t  *buf_ptr )
{
  retcode_t         rdr_rc = FSCK_OK;
  recon_bufrec_ptr  bufrec_ptr;

  bufrec_ptr = (recon_bufrec_ptr) buf_ptr;
  bufrec_ptr->dnode_blkoff = 0;
  bufrec_ptr->dnode_byteoff = 0;
  rdr_rc = dire_buffer_release( buf_ptr );
  return( rdr_rc );
}  /* end of recon_dnode_release () */
