// $Id: kpaint.cpp,v 1.77 2001/02/18 06:12:43 waba Exp $

#include <string.h>
#include <unistd.h>
#include <time.h>

#include <qlayout.h>
#include <qregexp.h>
#include <qwidget.h>
#include <qsignalmapper.h>
#include <config.h>

#include <kapp.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kimageio.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmainwindow.h>
#include <ktoolbar.h>
#include <kedittoolbar.h>
#include <kurl.h>
#include <kcolorbtn.h>
#include <kcolorcombo.h>
#include <kcolordlg.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kaccel.h>
#include <kstdaccel.h>
#include <kio/netaccess.h>

#include <qscrollview.h>
#include "canvas.h"
#include "version.h"
#include "kpaint.h"
#include "manager.h"
#include "canvassize.h"
#include "palettedialog.h"
#include "infodialog.h"
#include "depthdialog.h"
#include "view.h"
#include "mainview.h"
#include "navwidget.h"


extern int openwins;

KPaint::KPaint(const QString & url) : KMainWindow(0)
{
    modified = false;
  
    if(url == 0)
    {
        filename = i18n("untitled") 
            + "." + KImageIO::suffix(KImageIO::types(KImageIO::Writing).first());
    }
    else
    {
        filename = url;
    }

    mv = new MainView(this);
    v = mv->getViewport();
    c = mv->getCanvas();

    v->resize(c->size());
    setCentralWidget( mv );

    man= new Manager(c);
    //connect(c, SIGNAL(sizeChanged()), v, SLOT(resizeScrollBars()));
    connect(c, SIGNAL(modified()), this, SLOT(updateCommands()));
    connect(c, SIGNAL(selection(bool)), SLOT(enableEditCutCopy(bool)));
    // connect(c, SIGNAL(clipboard(bool)), SLOT(enableEditPaste(bool)));

    zoom= 100;
    openwins++;

    initMenus();
    initSideBar(); 
    initStatus();
    readOptions();

    if (url != 0)
    {
        loadFile(url);
    }

    allowEditPalette = c->pixmap()->depth() == 8;
    updateCommands();
    resize(640,480);

    c->clearModified();
}


KPaint::~KPaint()
{
    delete man;
}


void KPaint::setPixmap(QPixmap *p)
{
    modified= false;

    url = KURL();
    c->setPixmap(p);
}


int KPaint::exit()
{
    int die= 0;

    if (!(c->isModified())) 
    {
        die= 1;
    }
    else 
    {
        if (KMessageBox::warningContinueCancel(this, 
	    i18n("You have unsaved changes, you "
             "will lose them if you close "
	         "this window now."),
        i18n("Unsaved Changes"),
	    i18n("Close")) == KMessageBox::Cancel) 
        {
            die= 0;
        }
        else 
        {
            die= 1;
        }
  }
  
    return die;
}


void KPaint::closeEvent(QCloseEvent *e)
{
    if (exit())
    {
        e->accept();
        openwins--;
    }

    if (openwins == 0)
        kapp->exit(0);
}


void KPaint::updateCommands()
{
    copyAction->setEnabled(c->hasSelection());
    cutAction->setEnabled(c->hasSelection());
    //pasteAction->setEnabled(c->hasClipboardData());
    // use checkClipboardData signal sent from kapp->clipboard()
    // need a local slot for that
    pasteAction->setEnabled(true);
    maskAction->setEnabled(false);
    saveAction->setEnabled(c->isModified());
    saveAsAction->setEnabled(c->isModified());
    paletteAction->setEnabled(allowEditPalette);
    //pasteImageAction->setEnabled(c->hasClipboardData());
}

void KPaint::readOptions()
{
    KConfig *config = KApplication::kApplication()->config();
  
    config->setGroup( "Appearance" );

    // Read the entries
    toolbarAction->setChecked(config->readBoolEntry("ShowToolsToolBar", true));
    statusbarAction->setChecked(config->readBoolEntry("ShowStatusBar", true));

    updateControls();
}


void KPaint::writeOptions()
{
    KConfig *config = KApplication::kApplication()->config();

    config->setGroup( "Appearance" );

    // Write the entries
    config->writeEntry("ShowToolsToolBar", toolbarAction->isChecked());
    config->writeEntry("ShowStatusBar", statusbarAction->isChecked());
    config->sync();
}


void KPaint::enableEditCutCopy(bool e)
{
    kdDebug() << "kpaint: " << e << " cut/copy" << endl;
    cutAction->setEnabled(e);
    copyAction->setEnabled(e);
}


void KPaint::enableEditPaste(bool e)
{
    kdDebug() << "kpaint: " << e << " paste" << endl;
    pasteAction->setEnabled(e);
    pasteImageAction->setEnabled(e);
}


void KPaint::updateControls()
{
    KToolBar *bar = (KToolBar *)child("mainToolBar", "KToolBar");
    if (bar)
    {
        if (toolbarAction->isChecked())
            bar->show();
        else
            bar->hide();
    }

    bar = (KToolBar *)child("toolsToolBar", "KToolBar");
    if (bar)
    {
        if (toolbarAction->isChecked())
            bar->show();
        else
            bar->hide();
    }
            
    if (statusbarAction->isChecked())
        statusbar->show();
    else
        statusbar->hide();
}


void KPaint::canvasSizeChanged()
{
    QString size;

    size.sprintf("%d x %d", c->width(), c->height());
    statusbar->changeItem(size, ID_FILE_SIZE);
    modified = true;
}


void KPaint::initStatus()
{
    QString size;
    //  QFontMetrics fm; (below)
    ToolList tl;
    Tool *t;
    Tool *maxt = 0;
    uint maxtlen= 0, tmp;

    statusbar= statusBar();

    /* write the image size */
    size.sprintf("%d x %d", c->width(), c->height());
    statusbar->insertItem(size, ID_FILE_SIZE);

    /* write the color depth */
    size.sprintf(" %d bpp", c->getDepth());
    statusbar->insertItem(size, ID_COLOR_DEPTH);

    /* write the zoomfactor */
    statusbar->insertItem("100%", ID_ZOOM_FACTOR);

    /* write the toolname */
    /* get the max. font length of the toolnames */
    tl = man->getToolList();
    QFontMetrics fm = statusbar->fontMetrics();

    for (t= tl.first(); t != NULL; t= tl.next())
    {
        tmp = fm.width(t->getName());
        /* check the tool with the max. name len */
        if ( maxtlen < tmp )
        {
            maxtlen = tmp;
            maxt = t;
        }
    }
    /* write the maxlen toolname */
    if (maxt)
      size = maxt->getName();
    else
      size = "XXXX";
    statusbar->insertItem(size, ID_TOOL_NAME);

    /* write the filename */
    statusbar->insertItem(url.path(), ID_FILE_NAME);

    /* update to the current toolname */
    statusbar->changeItem(man->getCurrentTool().getName(), ID_TOOL_NAME);
    man->setStatusBar(statusbar);

    statusbar->show();
}


void KPaint::slotSideToolbarMoved()
{
    kdDebug() << "Sidebar moved" << endl;
}

/*
    this sidebar is actually a toolbar with custom widgets inserted
    instead of the usual buttons.  It can be docked to either side,
    top or bottom of window but starts on right side
*/
bool KPaint::initSideBar()
{
    int iw = 66;
    int ih = 66;
        
    // the toolbar itself
    sideToolbar = new KToolBar(this, 0, iw+8);
    CHECK_PTR(sideToolbar);
    addToolBar(sideToolbar);
    sideToolbar->show();
    sideToolbar->setBarPos(KToolBar::Right);

    connect(sideToolbar, SIGNAL(orientationChanged()),
	    this, SLOT(slotSideToolbarMoved()));

    // the preview/navigation widget with container
    QWidget *navWidget = new QWidget(sideToolbar);
    
    navWidget->setMinimumWidth(iw);
    navWidget->setMinimumHeight(ih);
    navWidget->setMaximumWidth(120);
    navWidget->setMaximumHeight(120);
        
    QBoxLayout *ml = new QVBoxLayout(navWidget);
    //QLabel *navlabel = new QLabel(i18n("Image Preview"), navWidget);
    
    nav = new NavWidget(navWidget, "navwidget");
    nav->setMinimumSize(120, 80);
    nav->setMaximumSize(120, 120);
    nav->setPixmap(c->pixmap());
    
    connect(c, SIGNAL(pixmapChanged(QPixmap *)),
	    nav, SLOT(setPixmap(QPixmap *)));
    connect(c, SIGNAL(modified()), 
        nav, SLOT(pixmapChanged()));
    //ml->addWidget(navlabel);
    ml->addWidget(nav);

    QWidget *mouseWidget = new QWidget(sideToolbar);
    mouseWidget->setFixedSize(66, 96);
    QBoxLayout *vl = new QVBoxLayout(mouseWidget);

    // The colour selector widgets with container widget
    QWidget *colorsWidget = new QWidget(mouseWidget);
 
    colorsWidget->setMinimumWidth(32);
    colorsWidget->setMinimumHeight(32);    
    colorsWidget->setMaximumWidth(66);
    colorsWidget->setMaximumHeight(66);    
    
    ml = new QHBoxLayout(colorsWidget);
    //QLabel *lmbLabel = new QLabel(i18n("LeftButton Color"), colorsWidget);
    //QLabel *rmbLabel = new QLabel(i18n("RightButton Color"), colorsWidget);
    lmbColourBtn = new KColorButton(red, colorsWidget, "lmbColour");
    rmbColourBtn = new KColorButton(green, colorsWidget, "rmbColour");
    QPushButton *mmbBtn = new QPushButton(colorsWidget, "mmbButton");
    
    lmbColourBtn->setFixedSize(22, 30);
    mmbBtn->setFixedSize(22, 30);    
    rmbColourBtn->setFixedSize(22, 30);
    
    man->setLMColorButton(lmbColourBtn);
    man->setRMColorButton(rmbColourBtn);

    connect(lmbColourBtn, SIGNAL(changed(const QColor &)),
	    man, SLOT(setLMBcolour(const QColor &)));
    connect(rmbColourBtn, SIGNAL(changed(const QColor &)),
	    man, SLOT(setRMBcolour(const QColor &)));

    //ml->addWidget(rmbLabel);  
    ml->addWidget(lmbColourBtn);      
    ml->addWidget(mmbBtn);    
    ml->addWidget(rmbColourBtn);    
    //ml->addWidget(lmbLabel);

    vl->addWidget(colorsWidget);        
    
    QWidget *mouseBodyWidget = new QWidget(mouseWidget);    
    ml = new QHBoxLayout(mouseBodyWidget);
    QPushButton *mouseBtn = new QPushButton(mouseBodyWidget, "MouseButton");
    mouseBtn->setFixedSize(66, 60);
    ml->addWidget(mouseBtn);
    vl->addWidget(mouseBodyWidget);        
        
  
    KPaletteTable *paletteTable = new KPaletteTable( sideToolbar , 120 , 8 );
    paletteTable->setMaximumHeight(66);    

    connect(paletteTable, SIGNAL(colorSelected(const QColor &, const QString &)),
	  man, SLOT(setLMBcolour(const QColor &)));
    
    // insert containter widgets so they show up as toolbar items
    sideToolbar->insertWidget(ID_NAV_WIDGET, iw, navWidget);
    sideToolbar->insertSeparator();
    sideToolbar->insertWidget(ID_COLORS_WIDGET, iw, mouseWidget);
    sideToolbar->insertSeparator();    
    sideToolbar->insertWidget(ID_COLOR_BAR, iw, paletteTable);
    
    return true;
}


void KPaint::initMenus()
{
    KAction *action;

    // file menu
    action = KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
    action = KStdAction::open(this, SLOT(fileOpen()), actionCollection());
    saveAction = KStdAction::save(this, SLOT(fileSave()), actionCollection());
    saveAsAction = KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
    action = KStdAction::quit(this, SLOT(fileExit()), actionCollection());

    // edit menu
    copyAction = KStdAction::copy(this, SLOT(editCopy()), actionCollection());
    cutAction = KStdAction::cut(this, SLOT(editCut()), actionCollection());
    pasteAction = KStdAction::paste(this, SLOT(editPaste()), actionCollection());
    pasteImageAction = new KAction(i18n("Paste &image"), 0, this,
                       SLOT(editPasteImage()), actionCollection(),
                       "edit_paste_image");

    maskAction = new KAction(i18n("&Mask..."), 0, this,
                       SLOT(editMask()), actionCollection(),
                       "edit_mask");

    // view menu
    zoomInAction = KStdAction::zoomIn(this, SLOT(editZoomIn()), actionCollection());
    zoomOutAction = KStdAction::zoomOut(this, SLOT(editZoomOut()), actionCollection());
    zoomAction = new KSelectAction( i18n( "Zoom" ), 0,
                                        actionCollection(), "view_zoom" );
    connect( zoomAction, SIGNAL( activated( const QString & ) ),
             this, SLOT( viewZoom( const QString & ) ) );
    QStringList lst;
    lst << "10%";
    lst << "20%";
    lst << "30%";
    lst << "40%";
    lst << "50%";
    lst << "60%";
    lst << "70%";
    lst << "80%";
    lst << "90%";
    lst << "100%";
    lst << "200%";
    lst << "300%";
    lst << "400%";
    lst << "500%";
    lst << "600%";
    lst << "700%";
    lst << "800%";
    lst << "900%";
    lst << "1000%";
    zoomAction ->setItems( lst );
    zoomAction ->setCurrentItem(9);
    // image menu
    action = new KAction(i18n("&Information..."), 0, this,
                       SLOT(imageInfo()), actionCollection(),
                       "image_info");
    action = new KAction(i18n("&Resize..."), 0, this,
                       SLOT(imageResize()), actionCollection(),
                       "image_resize");
    paletteAction = new KAction(i18n("&Edit palette..."), 0, this,
                       SLOT(imageEditPalette()), actionCollection(),
                       "image_palette");
    action = new KAction(i18n("&Change color depth..."), 0, this,
                       SLOT(imageChangeDepth()), actionCollection(),
                       "image_color");

    // settings menu
    toolbarAction = KStdAction::showToolbar(this, SLOT(updateControls()), actionCollection());
    statusbarAction = KStdAction::showStatusbar(this, SLOT(updateControls()), actionCollection());
    action = KStdAction::saveOptions(this, SLOT(writeOptions()), actionCollection());

    // window menu
    action = new KAction(i18n("&New window"), 0, this,
                       SLOT(newWindow()), actionCollection(),
                       "window_new");                                       
    action = new KAction(i18n("&Close window"), 0, this,
                       SLOT(closeWindow()), actionCollection(),
                       "window_close");                                       

    // tool menu
    action = new KAction(i18n("&Tool properties..."), 0, this,
                   SLOT(toolProperties()), actionCollection(),
                   "tool_config");                                       

    QSignalMapper *mapper = new QSignalMapper(this);

    ToolList tl = man->getToolList();
    Tool *t;
    int id=0;
    
    for (t= tl.first(); t != NULL; t= tl.next()) 
    {
        action = new KAction(t->getName(), QIconSet(t->pixmap()), 0, this,
		   SLOT(dummy()), actionCollection(),
		   QCString().sprintf("tool_%d", id));  

        mapper->setMapping(action, id);
        id++;

        connect(action, SIGNAL(activated()), mapper, SLOT(map()));
    }
  
    connect(mapper, SIGNAL(mapped(int)), this, SLOT(setTool(int)));

    KStdAction::configureToolbars(this, SLOT(editToolbars()), actionCollection());
    createGUI("kpaint.rc");
}


void KPaint::editToolbars()
{
  KEditToolbar dlg(actionCollection(),"kpaint.rc");
  if (dlg.exec())
    createGUI("kpaint.rc");
}


void KPaint::fileOpen()
{
    kdDebug(4400) << "fileOpen()" << endl;

    KURL url = KFileDialog::getOpenURL(QString::null,
        KImageIO::pattern(KImageIO::Reading), this);

    if (url.isEmpty())
        return;
  
    loadFile(url.path());
}


bool KPaint::loadFile(const QString& file)
{
    KURL turl(file);
    
    /* file loaded from konqueror file association, opewith, or
    one of several kde file open dialogs */
        
    if (c->load(turl.path())) 
    {
	    filename = turl.path();
	    modified = false;
	    format = KImageIO::type(file);
	    url = turl;
	    statusbar->changeItem(url.path(), ID_FILE_NAME);
	    canvasSizeChanged();
        c->clearModified();
    }

    /* brain-dead KURL needs absolute path for local file 
    loaded from command line */

    else if(file.left(1) != "/")
    {
        char abspath[256];

        getcwd(abspath, 256); 
        QString relative = file;
        QString fullPath = QString(abspath) + '/' + relative;        

        KURL purl(fullPath);
            
        if (c->load(purl.path())) 
        {
            filename = purl.path();
	        modified = false;
	        format = KImageIO::type(fullPath);
	        url = purl;
	        statusbar->changeItem(purl.path(), ID_FILE_NAME);
	        canvasSizeChanged();
            c->clearModified();
        }
        else
        {
            QString msg 
                = i18n("KPaint could not open %1 \n").arg(turl.path());
	        KMessageBox::error(0, msg);
	        return false;
        }
    }    
    else
    {
            QString msg 
                = i18n("KPaint could not open %1 \n").arg(turl.path());
	    KMessageBox::error(0, msg);
	    return false;
    }

    return true;
}


void KPaint::fileSave()
{
    if (!saveFile(filename, format, url))
        return;
  
    modified = false;
}


bool KPaint::saveFile(QString filename_, QString format_, KURL &url_)
{
    if (!c->save(filename_, format_.ascii()))
        return false;
        
    filename = filename_;
    format = format_;

    if (!url_.isEmpty())
    {
        if (!KIO::NetAccess::upload(filename_, url_))
	    return false;

        url = url_;
    }

    statusbar->changeItem(filename, ID_FILE_NAME);
    c->clearModified();
    return true;
}


void KPaint::fileSaveAs()
{
    kdDebug(4400) << "fileSaveAsCommand" << endl;

    KURL file = KFileDialog::getSaveURL(QString::null, KImageIO::pattern(KImageIO::Writing), this);
  
    if(file.isEmpty())
        return;

    // delete tempfile of not locally loaded file
    if (!url.isEmpty())
    {
        if( !url.isLocalFile() )
        {
            kdDebug(4400) << "KPaint: Deleting temp file \'" << filename << "\'" << endl;
            KIO::NetAccess::removeTempFile(filename);
        }
    }

    QString tmpfile;
    if (file.isLocalFile())
        tmpfile = file.path();
    else
        tmpfile = tmpnam(0);

    if (saveFile(tmpfile, KImageIO::type(file.path()).ascii(), file))
        c->clearModified();
}


void KPaint::fileNew()
{
    int w, h;
    QPixmap p;
    QString proto;
    canvasSizeDialog sz(0, "canvassizedlg",c->width(), c->height());

    kdDebug(4400) << "File New" << endl;

    if (sz.exec())
    {
        w= sz.getWidth();
        h= sz.getHeight();
        p.resize(w, h);
        p.fill(QColor("white"));
        c->setPixmap(&p);
        man->setCurrentTool(0);
        format = KImageIO::suffix(KImageIO::types(KImageIO::Writing).first());
        filename = i18n("untitled") + "." + format;
        url=KURL();

        statusbar->changeItem(filename, ID_FILE_NAME);
        canvasSizeChanged();
        repaint(0);
    }
}


void KPaint::fileExit()
{
    kdDebug(4400) << "fileExit()" << endl;

    if (exit()) 
    {
        kapp->exit(0);
    }
}


void KPaint::newWindow()
{
   kdDebug(4400) << "newWindow()" << endl;
   KPaint *kp;
   
   kp = new KPaint();
   kp->show();
}


void KPaint::closeWindow()
{
    kdDebug(4400) << "closeWindow()" << endl;
    if (exit()) close();
}


// Edit
void KPaint::editCopy()
{
    kdDebug(4400) << "editCopy()\n" << endl;
    c->copy(false);
}

void KPaint::editCut()
{
    kdDebug(4400) << "editCut()\n" << endl;
    c->cut();
}

/*
    paste is a tool now. It's in brush.cpp and brush.h,
    and can later be extended for painting with clip 
    pixmap as a brush as well as pasting down
*/
void KPaint::editPaste()
{
    kdDebug(4400) << "editPaste()\n" << endl;
    man->setCurrentTool(9);
    // c->paste();
}

/* This is intended to copy clipboard to a
   file and open it in a new window - */
   
void KPaint::editPasteImage()
{
    kdDebug(4400) << "editPasteImage()" << endl;

    if (!c->clipboardData()->isNull()) 
    {
        QPixmap *p = new QPixmap( *(c->clipboardData()) ); 
        CHECK_PTR(p);
        KPaint *kp = new KPaint();
        kp->setPixmap(p);
        kp->show();
    }
}


void KPaint::editZoomIn()
{
    kdDebug(4400) << "editZoomIn()" << endl;

    if (zoom >= 100)
    {
        zoom += 100;
        if (zoom == 1000)
        {
            zoomInAction->setEnabled(false);
            zoomOutAction->setEnabled(true);
        }
    }
    else
    {
        zoom += 10;
        zoomInAction->setEnabled(true);
        zoomOutAction->setEnabled(true);
    }

    c->setZoom(zoom);

    QString zoomstr;
    zoomstr.setNum(zoom);
    zoomstr += '%';

    statusbar->changeItem(zoomstr, ID_ZOOM_FACTOR);
    QStringList list=zoomAction ->items();
    zoomAction->setCurrentItem(list.findIndex(zoomstr));
}

void KPaint::viewZoom( const QString &_strZoom )
{
    QString z( _strZoom );
    z = z.replace( QRegExp( "%" ), "" );
    z = z.simplifyWhiteSpace();
    int new_zoom = z.toInt();
    
    if(new_zoom==10)
    {
        zoomOutAction->setEnabled(false);
        zoomInAction->setEnabled(true);
    }
    else if(new_zoom>=1000)
    {
        zoomInAction->setEnabled(false);
        zoomOutAction->setEnabled(true);
    }
    else
    {
        zoomInAction->setEnabled(true);
        zoomOutAction->setEnabled(true);
    }

    if(zoom!=new_zoom)
    {
        c->setZoom(new_zoom);
        zoom=new_zoom;
        statusbar->changeItem(_strZoom, ID_ZOOM_FACTOR);
    }
}

void KPaint::editZoomOut()
{
    kdDebug(4400) << "editZoomOut()" << endl;

    if (zoom > 100)
    {
        zoom -= 100;
        zoomOutAction->setEnabled(true);
        zoomInAction->setEnabled(true);
    }
    else
    {
        zoom -= 10;
        if (zoom <= 10)
        {
            zoomOutAction->setEnabled(false);
            zoomInAction->setEnabled(true);
        }
    }

    c->setZoom(zoom);

    QString zoomstr;
    zoomstr.setNum(zoom);
    zoomstr += '%';
    QStringList list=zoomAction ->items();
    zoomAction->setCurrentItem(list.findIndex(zoomstr));

    statusbar->changeItem(zoomstr, ID_ZOOM_FACTOR);
}


void KPaint::editMask()
{
    kdDebug(4400) << "editMask()" << endl;
}


void KPaint::editOptions()
{
    kdDebug(4400) << "editOptions()" << endl;
    /* obsolet (jha)
    KKeyDialog::configureKeys(keys); */
}


void KPaint::imageInfo()
{
  imageInfoDialog info(c, 0, "Image Information");
  kdDebug(4400) << "imageInfo()" << endl;
  info.exec();
}


void KPaint::imageResize()
{
    kdDebug(4400) << "imageResize()" << endl;

    canvasSizeDialog sz(this,"imageResize",c->width(), c->height());
    if (sz.exec())
    {
        int w= sz.getWidth();
        int h= sz.getHeight();
        if( w!=0 && h!=0)
        {
            c->resizeImage(w, h);
            canvasSizeChanged();
        }
    }
}

void KPaint::imageEditPalette()
{
    kdDebug(4400) << "imageEditPalette()" << endl;
    paletteDialog pal(c->pixmap());

    if (pal.exec()) 
    {
        c->setPixmap(pal.pixmap());
        c->repaint(0);
    }
}

void KPaint::imageChangeDepth()
{
    QString depthstr;
    KStatusBar *sb = statusBar();
    depthDialog d(c);

    kdDebug(4400) << "imageChangeDepth()" << endl;
  
    if (d.exec()) 
    {
        switch (d.depthBox->currentItem()) 
        {

    case ID_COLOR_1:
      kdDebug(4400) << "setDepth to 1" << endl;
      c->setDepth(1);
      depthstr.sprintf(" %d bpp", 1);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;

    case ID_COLOR_4:
      kdDebug(4400) << "setDepth to 4" << endl;
      c->setDepth(4);
      depthstr.sprintf(" %d bpp", 4);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;

    case ID_COLOR_8:
      kdDebug(4400) << "setDepth to 8" << endl;
      c->setDepth(8);
      depthstr.sprintf(" %d bpp", 8);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= true;
      break;

    case ID_COLOR_15:
      kdDebug(4400) << "setDepth to 15" << endl;
      c->setDepth(15);
      depthstr.sprintf(" %d bpp", 15);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;

    case ID_COLOR_16:
      kdDebug(4400) << "setDepth to 16" << endl;
      c->setDepth(16);
      depthstr.sprintf(" %d bpp", 16);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;

    case ID_COLOR_24:
      kdDebug(4400) << "setDepth to 24" << endl;
      c->setDepth(24);
      depthstr.sprintf(" %d bpp", 24);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;

    case ID_COLOR_32:
      kdDebug(4400) << "setDepth to 32" << endl;
      c->setDepth(32);
      depthstr.sprintf("%d bpp", 32);
      sb->changeItem(depthstr, ID_COLOR_DEPTH);
      allowEditPalette= false;
      break;
      
    default:
      break;
      
        }
    }
}

// Tool
void KPaint::setTool(int t)
{
    man->setCurrentTool(t);
}

void KPaint::toolProperties()
{
    kdDebug(4400) << "toolProperties()" << endl;
    man->showPropertiesDialog();
}

void KPaint::dummy()
{
  // just here to make KAction happy
}

#include "kpaint.moc"
