/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "spinner.h"
#include "misc.h"

#include <string.h>
#include <gtk/gtkbox.h>

/* GNOME includes */
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-animator.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/* local function prototypes */
static void free_spinner_list (void);
static void spinner_fill_iconlist_from_dir (GnomeIconList *icon_list,
					    gchar *base);

/**
 * spinner_create: create a spinner and add it to an hbox
 */
void
spinner_create (GaleonWindow *window)
{
	GtkWidget *spinner_button;
	GtkWidget *spinner;
	int toolbar, i;
	gchar *anim_dir;
	gchar *filename;
	GdkImlibImage *first;
	gint width, height;

	/* check window */
	return_if_not_window (window);

	/* initially no spinner */
	window->spinner = NULL;

	/* get the directory of the animations from configuration space,
 	 * default to installed share directory */
	anim_dir = gnome_config_get_string("/galeon/Appearance/spinner_dir="
					   SHARE_DIR "/spinners/pipeon/");
	if (anim_dir == NULL || strlen(anim_dir) == 0)
	{
		anim_dir = g_strdup(SHARE_DIR);
	}
	
	/* try to load the first file to find the size */
	filename = g_strdup_printf ("%s/rest.png", anim_dir);
	first = gdk_imlib_load_image (filename);
	if (first == NULL)
	{
		g_warning ("unable to load first spinner frame: %s\n",
			   filename);
		g_free (filename);
		return;
	}
	width = first->rgb_width;
	height = first->rgb_height;
	gdk_imlib_destroy_image (first);
	g_free (filename);

	/* set the size of the pixmap appropriately to whether 
	 * text is turned on in the toolbars, so it resizes okay */
	toolbar = gnome_config_get_int("/galeon/Appearance/toolbar_type=0");

	/* downsize the image if it would bloat the toolbar */
	if (toolbar == 0 && height > 24)
	{
		/* preserve aspect ratio */
		width = (width * 24) / height;
		height = 24;
	}

	if (toolbar == 1 && height > 22)
	{
		/* preserve aspect ratio */
		width = (width * 22) / height;
		height = 22;
	}

	/* build a button to hold the spinner */
	spinner_button = gtk_button_new ();	
	gtk_button_set_relief (GTK_BUTTON (spinner_button), GTK_RELIEF_NONE);

	/* build the spinner and insert it into the button */
	spinner = gnome_animator_new_with_size (width, height);
	gtk_container_add (GTK_CONTAINER(spinner_button), spinner);

	/* should always loop continously when loading */
	gnome_animator_set_loop_type (GNOME_ANIMATOR (spinner),
				      GNOME_ANIMATOR_LOOP_RESTART);

	/* 999 is an arbitrary upper limit, in case something
	 * goes wrong with testing the existence of the files */
	for (i = 0; i < 1000; i++)
	{
		/* build the filename */
		if (i == 0)
		{
			filename = g_strdup_printf("%s/rest.png", anim_dir);
		}
		else
		{
			filename = g_strdup_printf("%s/%03d.png", anim_dir, i);
		}

		if ((g_file_exists(filename) & (G_FILE_TEST_ISFILE |
						G_FILE_TEST_ISLINK)) == 0)
		{
			/* free the string */
			g_free(filename);

			/* check if this is the first one and warn */
			if (i == 0)
			{
				g_warning("no frames of animation found in "
					  "directory `%s'", anim_dir);
				return;
			}

			/* jump out of loading */
			break;
		}

		/* otherwise add it in */
		gnome_animator_append_frame_from_file_at_size
			(GNOME_ANIMATOR(spinner),
			 filename,       /* filename */
			 0, 0,           /* x_offset, y_offset */
			 100,            /* interval */
			 width, height); /* x_size, y_size */
 
		/* free allocated filename */
		g_free(filename);
	}

	/* add it to the hbox */
	g_return_if_fail (GTK_IS_WIDGET (window->main_toolbar));
	gtk_box_pack_end (GTK_BOX (window->main_toolbar->parent), 
			  GTK_WIDGET (spinner_button), FALSE, FALSE, 0);

	/* attach click signal */
	gtk_signal_connect (GTK_OBJECT (spinner_button), "clicked",
			    window_spinner_button_clicked_cb, window);

	/* show the spinner */
	gtk_widget_show (spinner_button);
	gtk_widget_show (spinner);

	/* free directory string */
	g_free (anim_dir);

	/* set in window */
	window->spinner = spinner;
}

/**
 * spinner_destroy: destroy the spinner of a window
 */
void
spinner_destroy (GaleonWindow *window)
{
	if (window->spinner != NULL)
	{
		gnome_animator_stop (GNOME_ANIMATOR (window->spinner));
		gtk_widget_destroy (window->spinner->parent);
		window->spinner = NULL;
	}
}

/**
 * spinner_start: start the spinner in a window, the spinner runs as
 * long as there are any active embeds within the window
 */
void
spinner_start (GaleonWindow *window)
{
	/* start the spinner animation */
	if (window->spinner && window->n_running == 0)
	{
		gnome_animator_start (GNOME_ANIMATOR (window->spinner));
	}

	/* keep track of the number of embeds running */
	window->n_running++;
}

/**
 * spinner_stop: stop the spinner (unless there are still active embeds)
 */
void
spinner_stop (GaleonWindow *window)
{
	/* keep track of the number of embeds running */
	window->n_running--;

	/* start the spinner animation */
	if (window->spinner && window->n_running == 0)
	{
		gnome_animator_stop (GNOME_ANIMATOR (window->spinner));
		gnome_animator_goto_frame (GNOME_ANIMATOR (window->spinner), 0);
	}
}

static GList *spinner_list = NULL;

/**
 * Free any existing spinner list.
 */
static void
free_spinner_list (void)
{
	GList *node;

	for (node = spinner_list; node; node = node->next)
		g_free(node->data);
	g_list_free(spinner_list);
	spinner_list = NULL;
}

/**
 * spinner_fill_iconlist: fill a gnome icon list with icons of available spinners
 */
void
spinner_fill_iconlist (GnomeIconList *icon_list)
{
	/* clear spinner list */
	free_spinner_list ();
	gnome_icon_list_clear (GNOME_ICON_LIST (icon_list));
	
	/* load all the galeon spinners */
	spinner_fill_iconlist_from_dir (icon_list, SHARE_DIR "/spinners");

	/* load all the nautilus spinners */
	/* NB: we assume nautilus is installed with the same prefix as galeon */
	spinner_fill_iconlist_from_dir (icon_list, 
					SHARE_DIR "/../pixmaps/nautilus");
}

static void
spinner_fill_iconlist_from_dir (GnomeIconList *icon_list, gchar *base)
{
	int index;
	gchar *icon;
	gchar *pref_spinner_path; 	
	GList *node;
	GList *list = NULL;
	GnomeVFSResult rc;

	pref_spinner_path = 
		gnome_config_get_string("/galeon/Appearance/spinner_dir="
					SHARE_DIR "/spinners/pipeon/");	

	rc = gnome_vfs_directory_list_load
		(&list, base, (GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FOLLOW_LINKS), NULL);
	
	index = icon_list->icons;
	for (node = list; node; node = node->next)
	{
		GnomeVFSFileInfo *info;
		gchar *path;
		gchar *ext_path;

		info = node->data;
		if (info->name[0] == '.')
			continue;
		if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
			continue;

		path = g_concat_dir_and_file (base, info->name);
		ext_path = g_concat_dir_and_file (path, "throbber");
		if (g_file_exists (ext_path))
		{
			g_free (path);
			path = ext_path;
		}
		else
		{
			g_free (ext_path);
		}

		/* Check to see if there is a rest.png */
		icon = g_concat_dir_and_file (path, "rest.png");
		if (g_file_exists (icon))
		{
			spinner_list = g_list_append (spinner_list, 
						      g_strdup (path));

			/* Name defaults to the directory name */
			gnome_icon_list_append (icon_list, icon, info->name);

			/* Select the icon configured in prefs */
			if (strcmp (pref_spinner_path, path) == 0)
				gnome_icon_list_select_icon (icon_list, index);

			index++;
		}

		g_free (path);
		g_free (icon);
	}

	gnome_vfs_file_info_list_free(list);
	g_free(pref_spinner_path);
}

/**
 * spinner_get_path_from_index: used in prefs_callbacks.c to get the 
 * path of selected icon
 */
const gchar *
spinner_get_path_from_index (gint index)
{
	gchar *path;

	path = g_list_nth_data (spinner_list, index);

	return path;
}
