/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "window.h"
#include "mozilla.h"
#include "PromptService.h"

#include "nsCOMPtr.h"
#include "nsIFactory.h"
#include "nsString.h"
#include "nsIServiceManager.h"
#include "nsXPComFactory.h"

#include "nsIPromptService.h"
#include "nsIUnicodeEncoder.h"

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnome/gnome-i18n.h>

/* local function prototypes */
static void set_title (GtkWidget *dialog, const PRUnichar *title);
static void set_label_text (GtkWidget *label, const PRUnichar *text);
static void set_check_button_text (GtkWidget *check_button, 
				   const PRUnichar *text);
static void set_check_button (GtkWidget *button, PRBool *value);
static void set_editable (GtkWidget *entry, PRUnichar **text);
static void get_check_button (GtkWidget *button, PRBool *value);
static void get_editable (GtkWidget *editable, PRUnichar **text);

/**
 * class CPromptService: an GNOME implementation of prompt dialogs for
 * Mozilla
 */
class CPromptService: public nsIPromptService
{
	public:
		CPromptService();
  		virtual ~CPromptService();

		NS_DECL_ISUPPORTS
		NS_DECL_NSIPROMPTSERVICE
  
};

NS_IMPL_ISUPPORTS1 (CPromptService, nsIPromptService)

/**
 * CPromptService::CPromptService: constructor
 */
CPromptService::CPromptService ()
{
	NS_INIT_REFCNT();
}

/**
 * CPromptService::~CPromptService: destructor
 */
CPromptService::~CPromptService ()
{
}

/**
 * CPromptService::Alert: show an alert box
 */
NS_IMETHODIMP CPromptService::Alert (nsIDOMWindow *parent, 
				     const PRUnichar *dialogTitle,
				     const PRUnichar *text)
{    
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *dialog;
	GladeXML *gxml;

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", 
				 "alert_dialog", &dialog, NULL);
	label       = glade_xml_get_widget (gxml, "label");
	title_label = glade_xml_get_widget (gxml, "title_label");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::AlertCheck: show an alert box with a checkbutton,
 * (typically for things like "dont show this warning again")
 */
NS_IMETHODIMP CPromptService::AlertCheck (nsIDOMWindow *parent, 
					  const PRUnichar *dialogTitle,
					  const PRUnichar *text,
					  const PRUnichar *checkMsg, 
					  PRBool *checkValue)
{
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GtkWidget *dialog;
	GladeXML *gxml;

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", "alert_check_dialog", 
				 &dialog, NULL);
	label        = glade_xml_get_widget (gxml, "label");
	title_label  = glade_xml_get_widget (gxml, "title_label");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);
	set_check_button_text (check_button, checkMsg);
	set_check_button (check_button, checkValue);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	get_check_button (check_button, checkValue);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Confirm: for simple yes/no dialogs
 */
NS_IMETHODIMP CPromptService::Confirm (nsIDOMWindow *parent, 
				       const PRUnichar *dialogTitle,
				       const PRUnichar *text,
				       PRBool *_retval)
{
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *dialog;
	GladeXML *gxml;
	gint ret;

	/* build the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirm_dialog", 
				 &dialog, NULL);
	label       = glade_xml_get_widget (gxml, "label");
	title_label = glade_xml_get_widget (gxml, "title_label");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Confirm: for simple yes/no dialogs, with an additional
 * check button
 */
NS_IMETHODIMP CPromptService::ConfirmCheck (nsIDOMWindow *parent, 
					    const PRUnichar *dialogTitle,
					    const PRUnichar *text,
					    const PRUnichar *checkMsg, 
					    PRBool *checkValue,
					    PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GladeXML *gxml;
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirm_check_dialog", 
				 &dialog, NULL);
	label        = glade_xml_get_widget (gxml, "label");
	title_label  = glade_xml_get_widget (gxml, "title_label");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);
	set_check_button_text (check_button, checkMsg);
	set_check_button (check_button, checkValue);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::ConfirmEx: For fancy confirmation dialogs
 */
NS_IMETHODIMP CPromptService::ConfirmEx (nsIDOMWindow *parent, 
					 const PRUnichar *dialogTitle,
					 const PRUnichar *text,
					 PRUint32 buttonFlags, 
#if MOZILLA_VERSION > VERSION2(0,9)
					 const PRUnichar *button0Title,
					 const PRUnichar *button1Title,
#endif
					 const PRUnichar *button2Title,
					 const PRUnichar *checkMsg, 
					 PRBool *checkValue,
					 PRInt32 *buttonPressed)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GladeXML *gxml;
	gint ret;
	gint i;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirmex_dialog", 
				 &dialog, NULL);
	label        = glade_xml_get_widget (gxml, "label");
	title_label  = glade_xml_get_widget (gxml, "title_label");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);
	set_check_button_text (check_button, checkMsg);
	set_check_button (check_button, checkValue);

	/* create buttons */
	for (i = 0; i < 4; i++)
	{
		switch (buttonFlags & 0xff)
		{
		case BUTTON_TITLE_OK:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_OK);
			break;
		case BUTTON_TITLE_CANCEL:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_CANCEL);
			break;
		case BUTTON_TITLE_YES:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_YES);
			break;
		case BUTTON_TITLE_NO:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    GNOME_STOCK_BUTTON_NO);
			break;
		case BUTTON_TITLE_SAVE:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Save"));
			break;
		case BUTTON_TITLE_REVERT:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Revert"));
			break;
#if MOZILLA_VERSION > VERSION2(0,9)
		case BUTTON_TITLE_DONT_SAVE:
			gnome_dialog_append_button (GNOME_DIALOG(dialog),
						    _("Don't save"));
			break;
		case BUTTON_TITLE_IS_STRING:
			{
				gchar *tmpstr;
				if (i==0)
					tmpstr = mozilla_unicode_to_locale (
							button0Title);
				else if (i==1)
					tmpstr = mozilla_unicode_to_locale (
							button1Title);
				else if (i==2)
					tmpstr = mozilla_unicode_to_locale (
							button2Title);
				else
					break;
				gnome_dialog_append_button (
						GNOME_DIALOG(dialog), tmpstr);
				g_free (tmpstr);
			}
			break;
#endif
		default:
			break;
		}	

		buttonFlags >>= 8;
	}

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	*buttonPressed = ret;

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::Prompt: show a prompt for text, with a checkbutton
 */
NS_IMETHODIMP CPromptService::Prompt (nsIDOMWindow *parent, 
				      const PRUnichar *dialogTitle,
				      const PRUnichar *text, 
				      PRUnichar **value,
				      const PRUnichar *checkMsg, 
				      PRBool *checkValue,
				      PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *entry;
	GtkWidget *label;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GladeXML *gxml;
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_dialog", 
				 &dialog, NULL);
	entry        = glade_xml_get_widget (gxml, "entry");
	label        = glade_xml_get_widget (gxml, "label");
	title_label  = glade_xml_get_widget (gxml, "title_label");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_label_text (label, text);
	set_check_button_text (check_button, checkMsg);
	set_editable (entry, value);
	set_check_button (check_button, checkValue);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (entry, value);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::PromptUsernameAndPassword: show a prompt for username
 * and password with an additional check button.
 */
NS_IMETHODIMP CPromptService::PromptUsernameAndPassword 
                                        (nsIDOMWindow *parent, 
					 const PRUnichar *dialogTitle,
					 const PRUnichar *text,
					 PRUnichar **username,
					 PRUnichar **password,
					 const PRUnichar *checkMsg,
					 PRBool *checkValue,
					 PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GtkWidget *username_entry;
	GtkWidget *password_entry;
	GladeXML *gxml;
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_user_pass_dialog", 
				 &dialog, NULL);
	title_label    = glade_xml_get_widget (gxml, "title_label");
	check_button   = glade_xml_get_widget (gxml, "check_button");
	username_entry = glade_xml_get_widget (gxml, "username_entry");
	password_entry = glade_xml_get_widget (gxml, "password_entry");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_check_button_text (check_button, checkMsg);
	set_editable (username_entry, username);
	set_editable (password_entry, password);
	set_check_button (check_button, checkValue);

	/* make an enter keypress activate the "ok" button */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (password_entry));

	/* set the keyboard focus to the username entry */
	gtk_widget_grab_focus (username_entry);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (username_entry, username);
	get_editable (password_entry, password);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}

/**
 * CPromptService::PromptPassword: show a prompt for just
 * a password with an additional check button.
 */
NS_IMETHODIMP CPromptService::PromptPassword (nsIDOMWindow *parent, 
					      const PRUnichar *dialogTitle,
					      const PRUnichar *text,
					      PRUnichar **password,
					      const PRUnichar *checkMsg, 
					      PRBool *checkValue,
					      PRBool *_retval)
{
	GtkWidget *dialog;
	GtkWidget *title_label;
	GtkWidget *check_button;
	GtkWidget *password_entry;
	GladeXML *gxml;
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "prompt_pass_dialog", 
				 &dialog, NULL);
	title_label    = glade_xml_get_widget (gxml, "title_label");
	check_button   = glade_xml_get_widget (gxml, "check_button");
	password_entry = glade_xml_get_widget (gxml, "password_entry");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set dynamic attributes */
	set_title (dialog, dialogTitle);
	set_label_text (title_label, dialogTitle);
	set_check_button_text (check_button, checkMsg);
	set_editable (password_entry, password);
	set_check_button (check_button, checkValue);

	GtkWidget *parentWidget = mozilla_find_gtk_parent (parent);
	if (parentWidget)
	{
		gnome_dialog_set_parent (GNOME_DIALOG (dialog),
					 GTK_WINDOW(parentWidget));
	}
	else
	{
	        gtk_widget_show(dialog);
	        window_set_layer(dialog);
	};

	/* run dialog and capture return values */
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	get_check_button (check_button, checkValue);
	get_editable (password_entry, password);
	*_retval = (ret == 0);

	/* done */
	gtk_widget_destroy (dialog);
	return NS_OK;
}


/**
 * CPromptService::Select: FIXME
 */
NS_IMETHODIMP CPromptService::Select (nsIDOMWindow *parent, 
				      const PRUnichar *dialogTitle,
				      const PRUnichar *text, PRUint32 count,
				      const PRUnichar **selectList, 
				      PRInt32 *outSelection,
				      PRBool *_retval)
{
	g_warning ("CPromptService::Select: NOT IMPLEMENTED\n");
	return NS_ERROR_NOT_IMPLEMENTED;
}

NS_DEF_FACTORY (CPromptService, CPromptService);

/**
 * NS_NewPromptServiceFactory:
 */ 
nsresult NS_NewPromptServiceFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsCPromptServiceFactory *result = new nsCPromptServiceFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	}
    
	NS_ADDREF(result);
	*aFactory = result;
  
	return NS_OK;
}

/**
 * set_title: set a dialog title to a unicode string
 */
static void
set_title (GtkWidget *dialog, const PRUnichar *title)
{
	gchar *gtitle;

	/* convert the title */
	gtitle = mozilla_unicode_to_locale (title);

	/* set it */
	gtk_window_set_title (GTK_WINDOW (dialog), 
			      (gtitle == NULL ? N_("Galeon") : gtitle));

	/* and free */
	g_free (gtitle);
}

/**
 * set_label_text: set a labels text to a unicode string
 */
static void
set_label_text (GtkWidget *label, const PRUnichar *text)
{
	gchar *gtext;

	/* convert the title */
	gtext = mozilla_unicode_to_locale (text);

	/* set it */
	gtk_label_set_text (GTK_LABEL (label), gtext);

	/* and free */
	g_free (gtext);
}

/**
 * set_check_button_text: set a check buttons text to a unicode string
 */
static void
set_check_button_text (GtkWidget *check_button, const PRUnichar *text)
{
	gchar *gtext;

	/* convert the title */
	gtext = mozilla_unicode_to_locale (text);

	/* set it */
	gtk_label_set_text (GTK_LABEL (GTK_BIN (check_button)->child), gtext);

	/* and free */
	g_free (gtext);
}
	
/**
 * set_check_button: set a togglebutton to an initial state
 */
static void
set_check_button (GtkWidget *button, PRBool *value)
{
	/* check pointer is valid */
	if (value == NULL)
	{
		gtk_widget_hide (GTK_WIDGET (button));
		return;
	}

	/* set the value of the check button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), *value);
}

/**
 * set_editable: set an editable to a unicode string
 */
static void
set_editable (GtkWidget *entry, PRUnichar **text)
{
	gchar *gtext;

	/* check there's something set */
	if (text == NULL || *text == NULL)
	{
		return;
	}

	/* convert the title */
	gtext = mozilla_unicode_to_locale (*text);

	/* set this string value in the widget */
	gtk_entry_set_text (GTK_ENTRY (entry), gtext);

	/* and free */
	g_free (gtext);
}

/**
 * get_check_button: get value of a toggle button and store it in a PRBool
 */
static void
get_check_button (GtkWidget *button, PRBool *value)
{
	/* check we can write */
	if (value == NULL)
	{
		return;
	}

	/* set the value from the check button */
	*value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
}

/**
 * get_editable: get a string from an editable and store it as unicode
 */
static void
get_editable (GtkWidget *editable, PRUnichar **text)
{
	gchar *edited;

	/* check we can write */
	if (text == NULL)
	{
		return;
	}

	/* get the text */
	edited = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);

	/* decode and set it as the return value */
	*text = mozilla_locale_to_unicode (edited);

	/* free */
	g_free (edited);
}
