/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mime.h"
#include "misc.h"
#include "window.h"
#include "mozilla.h"

#include <gtk/gtkmain.h>
#include <gtk/gtkfilesel.h>
#include <libgnome/gnome-config.h>

#include "nsIFilePicker.h"

#include "nsCRT.h"
#include "nsCOMPtr.h"
#include "nsIFactory.h"
#include "nsISupportsArray.h"
#include "nsIServiceManager.h"
#include "nsXPComFactory.h"

#include "nsString.h"
#include "nsIPref.h"
#include "nsIURI.h"
#include "nsIChannel.h"
#include "nsILocalFile.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"

class GFilePicker;

extern "C" {
void filePicker_save_as_ok_cb(GtkButton *button, GFilePicker *aHandler);
void filePicker_save_as_cancel_cb(GtkButton *button, GFilePicker *aHandler);
gboolean filePicker_save_as_close_cb (GtkWidget *dialog, GdkEvent *event,
				      GFilePicker *aHandler);
}

/* Header file */
class GFilePicker : public nsIFilePicker
{
	friend void filePicker_save_as_ok_cb(GtkButton *button,
					     GFilePicker *aHandler);
	friend void filePicker_save_as_cancel_cb(GtkButton *button,
						 GFilePicker *aHandler);
	friend gboolean filePicker_save_as_close_cb (GtkWidget *dialog,
						     GdkEvent *event,
						     GFilePicker *aHandler);
  public:
	NS_DECL_ISUPPORTS
	NS_DECL_NSIFILEPICKER

	GFilePicker();
	virtual ~GFilePicker();
	/* additional members */
	NS_IMETHODIMP InitWithGtkWidget (GtkWidget *parentWidget, 
					 const char *title,PRInt16 mode);

  private:
	nsCOMPtr<nsIDOMWindowInternal> mParentInternal;
	nsString mTitle;
	PRInt16 mMode;

	nsString mFilter;
	nsString mDefaultString;
	nsCOMPtr<nsILocalFile> mFile;
	nsCOMPtr<nsILocalFile> mDisplayDirectory;

	PRBool mFileSelectorClosed;
	PRBool mFileSelectorOk;

	GtkWidget *mParentWidget;	
	GtkWidget *mFileSelector;
};

/* Implementation file */
NS_IMPL_ISUPPORTS1(GFilePicker, nsIFilePicker)

GFilePicker::GFilePicker()
{
	NS_INIT_ISUPPORTS();
	/* member initializers and constructor code */
	nsCOMPtr<nsIPref> prefs = do_GetService (NS_PREF_CONTRACTID);
	nsresult rv = prefs->GetFileXPref ("browser.download.dir",
					   getter_AddRefs(mDisplayDirectory));

	if (NS_FAILED(rv))
	{
		char *dirName = g_strdup (gnome_config_get_string 
				("/galeon/Downloading/download_dir"));

		if (!dirName)
		{
			dirName = g_strdup (g_get_home_dir());
		};
		mDisplayDirectory = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
		mDisplayDirectory->InitWithPath (dirName);
		g_free (dirName);
		rv = prefs->SetFileXPref ("browser.download.dir",
					  mDisplayDirectory);
	}


}

GFilePicker::~GFilePicker()
{
  /* destructor code */
}

////////////////////////////////////////////////////////////////////////////////
// begin nsIFilePicker impl
////////////////////////////////////////////////////////////////////////////////

/* void init (in nsIDOMWindowInternal parent, in wstring title, in short mode); */
NS_IMETHODIMP GFilePicker::Init(nsIDOMWindowInternal *parent, 
				 const PRUnichar *title, PRInt16 mode)
{
	mParentInternal = parent;
	nsCOMPtr<nsIDOMWindow> pWindow = do_QueryInterface (mParentInternal);
	mParentWidget = mozilla_find_gtk_parent (pWindow);
	
	mTitle = title;
	mMode = mode;
	
	mFile = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	
	return NS_OK;
}

/* void appendFilters (in long filterMask); */
NS_IMETHODIMP GFilePicker::AppendFilters(PRInt32 filterMask)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

/* void appendFilter (in wstring title, in wstring filter); */
NS_IMETHODIMP GFilePicker::AppendFilter(const PRUnichar *title,
					const PRUnichar *filter)
{
	//GtkFileSelection is crippled, so we can't provide a short-list
	//of filters to choose from. We provide minimal functionality
	//by using the most recent AppendFilter call as the active filter.
	mFilter = filter;
	return NS_OK;
}

/* attribute wstring defaultString; */
NS_IMETHODIMP GFilePicker::GetDefaultString(PRUnichar * *aDefaultString)
{
	*aDefaultString = mDefaultString.ToNewUnicode ();
	return NS_OK;
}
NS_IMETHODIMP GFilePicker::SetDefaultString(const PRUnichar * aDefaultString)
{
	if (aDefaultString)
		mDefaultString = aDefaultString;
	else
		mDefaultString = NS_LITERAL_STRING ("");
	return NS_OK;
}

/* attribute nsILocalFile displayDirectory; */
NS_IMETHODIMP GFilePicker::GetDisplayDirectory(nsILocalFile * *aDisplayDirectory)
{
	NS_IF_ADDREF (*aDisplayDirectory = mDisplayDirectory);
	return NS_OK;
}
NS_IMETHODIMP GFilePicker::SetDisplayDirectory(nsILocalFile * aDisplayDirectory)
{
	mDisplayDirectory = aDisplayDirectory;
	return NS_OK;
}

/* readonly attribute nsILocalFile file; */
NS_IMETHODIMP GFilePicker::GetFile(nsILocalFile * *aFile)
{
	NS_IF_ADDREF (*aFile = mFile);
	return NS_OK;
}

/* readonly attribute nsIFileURL fileURL; */
NS_IMETHODIMP GFilePicker::GetFileURL(nsIFileURL * *aFileURL)
{
	nsCOMPtr<nsIFileURL> fileURL = 
		do_CreateInstance ("@mozilla.org/network/standard-url;1");
	fileURL->SetFile (mFile);
	NS_IF_ADDREF (*aFileURL = fileURL);
	return NS_OK;
}

/* short show (); */
NS_IMETHODIMP GFilePicker::Show(PRInt16 *_retval)
{
	char *cTitle = mozilla_unicode_to_locale (mTitle.get());

	mFileSelector = gtk_file_selection_new (cTitle);
	g_free (cTitle);

	char *cFileName;
	if (mMode == nsIFilePicker::modeGetFolder)
	{
		cFileName = NULL;
	}
	else
	{
		cFileName = mozilla_unicode_to_locale (mDefaultString.get());
	}

	char *cDirName;
	mDisplayDirectory->GetPath (&cDirName);

	char *cFullPath = g_strconcat (cDirName, "/", cFileName, NULL);
	g_free (cFileName);
	nsMemory::Free (cDirName);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION(mFileSelector),
				 	 cFullPath);
	g_free (cFullPath);

	if (!mFilter.IsEmpty())
	{
		char *cFilter = mozilla_unicode_to_locale (mFilter.get());
		gtk_file_selection_complete (GTK_FILE_SELECTION(mFileSelector),
					     cFilter);
		g_free (cFilter);
	}

	if (mParentWidget)
		gtk_window_set_transient_for (GTK_WINDOW(mFileSelector),
					      GTK_WINDOW(mParentWidget));

	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)
			    ->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(filePicker_save_as_ok_cb),
			    (gpointer)this);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)
			    ->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(filePicker_save_as_cancel_cb),
			    (gpointer)this);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)),
			    "delete-event",
			    GTK_SIGNAL_FUNC(filePicker_save_as_close_cb),
			    (gpointer)this);

	if (mMode == nsIFilePicker::modeGetFolder)
	{
		gtk_widget_set_sensitive (GTK_FILE_SELECTION(mFileSelector)
					  ->file_list, FALSE);
	}

	gtk_window_set_modal (GTK_WINDOW(mFileSelector), TRUE);

	mFileSelectorClosed = PR_FALSE;

	gtk_widget_show (mFileSelector);
	window_set_layer (mFileSelector);

	while (mFileSelectorClosed == PR_FALSE)
	{
		while (gtk_events_pending())
			gtk_main_iteration();
		usleep (10000);
	}

	if (mFileSelectorOk == PR_TRUE)
	{
		*_retval = nsIFilePicker::returnOK;
	}
	else
	{
		*_retval = nsIFilePicker::returnCancel;
	}

	return (mFileSelectorOk == PR_TRUE) ? NS_OK : NS_ERROR_FAILURE;
}

////////////////////////////////////////////////////////////////////////////////
// begin local public methods impl
////////////////////////////////////////////////////////////////////////////////

NS_IMETHODIMP GFilePicker::InitWithGtkWidget (GtkWidget *parentWidget, 
					      const char *title,PRInt16 mode)
{
	mParentWidget = parentWidget;
	
	PRUnichar *uTitle = mozilla_locale_to_unicode (title);
	mTitle = uTitle;
	g_free (uTitle);

	mMode = mode;

	mFile = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	
	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local private methods impl
////////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------

NS_DEF_FACTORY (GFilePicker, GFilePicker);

/**
 * NS_NewFilePickerFactory:
 */ 
nsresult NS_NewFilePickerFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsGFilePickerFactory *result = new nsGFilePickerFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	}
    
	NS_ADDREF(result);
	*aFactory = result;

	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin FileSelector callbacks.
////////////////////////////////////////////////////////////////////////////////

void 
filePicker_save_as_ok_cb (GtkButton *button, GFilePicker *aFilePicker)
{
	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_TRUE;

	gchar *aFileName = 
		g_strdup (gtk_file_selection_get_filename(GTK_FILE_SELECTION(
						aFilePicker->mFileSelector)));

	aFilePicker->mFile->InitWithPath (aFileName);

	if (aFilePicker->mMode == nsIFilePicker::modeGetFolder)
	{
		aFilePicker->mDisplayDirectory->InitWithPath (aFileName);
		aFilePicker->mDefaultString = NS_LITERAL_STRING("");
	}
	else
	{
		nsCOMPtr<nsIFile> aDirectory;
		aFilePicker->mFile->GetParent (getter_AddRefs(aDirectory));
		aFilePicker->mDisplayDirectory = do_QueryInterface (aDirectory);
		PRUnichar *leafName;
		aFilePicker->mFile->GetUnicodeLeafName (&leafName);
		aFilePicker->mDefaultString = leafName;
		nsMemory::Free (leafName);
	}

	g_free (aFileName);

	gtk_widget_destroy (GTK_WIDGET(aFilePicker->mFileSelector));
}

void
filePicker_save_as_cancel_cb (GtkButton *button, GFilePicker *aFilePicker)
{
	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_FALSE;

	gtk_widget_destroy (GTK_WIDGET(aFilePicker->mFileSelector));
}

gboolean
filePicker_save_as_close_cb (GtkWidget *dialog, GdkEvent *event,
			     GFilePicker *aFilePicker)
{
	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_FALSE;

	return FALSE;
}

////////////////////////////////////////////////////////////////////////////////
/**
 * show_file_picker: Shows a file picker. Can be configured to select a
 * file or a directory.
 * @parentWidget: Parent Widget for file picker.
 * @title: Title for file picker.
 * @directory: Initial directory to start in. 
 * @file: Initial filename to show in filepicker.
 * @mode: Mode to run filepicker in (modeOpen, modeSave, modeGetFolder)
 * @ret_fullpath: On a successful return, will hold the full path to selected
 *		file or directory.
 * returns: TRUE for success, FALSE for failure.
 */
////////////////////////////////////////////////////////////////////////////////

extern "C" gboolean
show_file_picker (GtkWidget *parentWidget, const char* title,
		  const char* directory, const char* file, FilePickerMode mode,
		  char * *ret_fullpath)
{
	GFilePicker *aFilePicker = new GFilePicker;
	PRUnichar *uDefaultString = mozilla_locale_to_unicode (file);

	if (!directory)
		directory = g_strdup (g_get_home_dir());
	nsCOMPtr<nsILocalFile> aDir = 
				do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	aDir->InitWithPath (directory);

	aFilePicker->InitWithGtkWidget (parentWidget, title, mode);
	aFilePicker->SetDefaultString (uDefaultString);
	aFilePicker->SetDisplayDirectory (aDir);
	
	PRInt16 retval;
	aFilePicker->Show (&retval);
	
	if (retval == nsIFilePicker::returnCancel)
	{
		delete aFilePicker;
		return FALSE;
	}
	else
	{
		if (*ret_fullpath)
			g_free (*ret_fullpath);
		nsCOMPtr<nsILocalFile> aFile;
		aFilePicker->GetFile (getter_AddRefs(aFile));
		char *tempFullPathStr;
		aFile->GetPath (&tempFullPathStr);
		*ret_fullpath = g_strdup (tempFullPathStr);
		nsMemory::Free (tempFullPathStr);
		delete aFilePicker;
		return TRUE;
	}
}
