/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999,2000  Hvard Kvlen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include <time.h>
#include "libxmms/util.h"
#include <sys/stat.h>
#include <unistd.h>

GList *playlist = NULL;
GList *shuffle_list = NULL;
static gboolean playlist_get_info_scan_active = FALSE, playlist_get_info_going = FALSE;
static pthread_t playlist_get_info_thread;
pthread_mutex_t playlist_mutex = PTHREAD_MUTEX_INITIALIZER;
static PlaylistEntry *playlist_position;

extern PlayList_List *playlistwin_list;
extern Vis *mainwin_vis;
extern SVis *mainwin_svis;

static gboolean playlist_get_info_entry(PlaylistEntry *entry);

static GList *find_playlist_position_list(void)
{
	/* Caller should hold playlist-mutex */
	if (cfg.shuffle)
		return g_list_find(shuffle_list, playlist_position);
	return g_list_find(playlist, playlist_position);
}

void playlist_clear(void)
{
	GList *node;
	PlaylistEntry *entry;
	
	if (get_input_playing())
		input_stop();

	pthread_mutex_lock(&playlist_mutex);
	if (playlist)
	{
		node = playlist;
		while (node)
		{
			entry = (PlaylistEntry *) node->data;
			if (entry->filename)
				g_free(entry->filename);
			if (entry->title)
				g_free(entry->title);
			g_free(entry);
			node = node->next;
		}
		g_list_free(playlist);
		playlist = NULL;
		playlist_position = NULL;
	}

	pthread_mutex_unlock(&playlist_mutex);
	playlist_generate_shuffle_list();
	playlistwin_update_list();
}

void playlist_delete_node(GList *node, gboolean *set_info_text, gboolean *restart_playing)
{
	/* Caller should hold playlist mutex */
	PlaylistEntry *entry;
	GList *playing_song;
	/*
	 * We call g_list_find manually here because
	 * we don't want an item in the shuffle_list
	 */
	playing_song = g_list_find(playlist, playlist_position);
	entry = (PlaylistEntry *) node->data;
	if (playing_song == node)
	{
		*set_info_text = TRUE;
		if (get_input_playing())
		{
			PL_UNLOCK();
			input_stop();
			PL_LOCK();
			*restart_playing = TRUE;
		}
		playing_song = find_playlist_position_list();
		if (g_list_next(playing_song))
			playlist_position = g_list_next(playing_song)->data;
		else if (g_list_previous(playing_song))
			playlist_position = g_list_previous(playing_song)->data;
		else
			playlist_position = NULL;
		/* Make sure the entry did not disappear under us */
		if (g_list_index(get_playlist(), entry) == -1)
			return;
	}
	if (entry->filename)
		g_free(entry->filename);
	if (entry->title)
		g_free(entry->title);
	shuffle_list = g_list_remove(shuffle_list, entry);
	playlist = g_list_remove_link(playlist, node);
	g_free(entry);
	g_list_free_1(node);
}

void playlist_delete_index(glong index)
{
	gboolean restart_playing = FALSE, set_info_text = FALSE;
	GList *node;
	
	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return;
	}

	node = g_list_nth(playlist, index);
	if(!node)
	{
		PL_UNLOCK();
		return;
	}

	playlist_delete_node(node, &set_info_text, &restart_playing);

	PL_UNLOCK();

	playlistwin_update_list();
	if (restart_playing)
	{
		if (playlist_position)
			playlist_play();
		else
			mainwin_set_song_info(0, 0, 0);
	}
	else if (set_info_text)
		mainwin_set_info_text();

}

void playlist_delete_filenames(GList *filenames)
{
	GList *node, *fnode;
	gboolean set_info_text = FALSE, restart_playing = FALSE;

	PL_LOCK();
	for (fnode = filenames; fnode; fnode = g_list_next(fnode))
	{
		node = playlist;
		while (node)
		{
			GList *next = g_list_next(node);
			if (!strcmp(((PlaylistEntry *) node->data)->filename, fnode->data))
				playlist_delete_node(node, &set_info_text, &restart_playing);
			node = next;
		}
	}

	PL_UNLOCK();

	playlistwin_update_list();
	if (restart_playing)
	{
		if (playlist_position)
			playlist_play();
		else
			mainwin_set_song_info(0, 0, 0);
	}
	else if (set_info_text)
		mainwin_set_info_text();	
}

void playlist_delete(gboolean crop)
{
	gboolean restart_playing = FALSE, set_info_text = FALSE;
	GList *node, *next;
	PlaylistEntry *entry;

	pthread_mutex_lock(&playlist_mutex);

	node = playlist;

	while (node)
	{
		entry = (PlaylistEntry *) node->data;
		next = g_list_next(node);
		if ((entry->selected && !crop) || (!entry->selected && crop))
			playlist_delete_node(node, &set_info_text, &restart_playing);
		node = next;
	}
	pthread_mutex_unlock(&playlist_mutex);
	
	playlistwin_update_list();
	if (set_info_text)
		mainwin_set_info_text();
	if (restart_playing)
	{
		if (playlist_position)
			playlist_play();
		else
			mainwin_set_song_info(0, 0, 0);
	}
}

static void __playlist_ins(gchar * filename, glong pos)
{
	PlaylistEntry *entry;

	entry = g_malloc0(sizeof (PlaylistEntry));
	entry->filename = g_strdup(filename);
	entry->length = -1;

	pthread_mutex_lock(&playlist_mutex);
	playlist = g_list_insert(playlist, entry, pos);
	pthread_mutex_unlock(&playlist_mutex);		

	playlist_get_info_scan_active = TRUE;
}

void playlist_ins(gchar * filename, glong pos)
{
	gchar *ext;
	gboolean ispl = FALSE;
	
	ext = strrchr(filename, '.');
	if (ext && ((!strcasecmp(ext, ".m3u") || !strcasecmp(ext, ".pls"))))
		ispl = TRUE;
	else if (!input_check_file(filename))
	{
		/*
		 * Some files (typically produced by some cgi-scripts)
		 * don't have the correct extension.  Try to recognize
		 * these files by looking at their content.  We only
		 * check for http entries since it does not make sense
		 * to have file entries in a playlist fetched from the
		 * net.
		 */
		gchar buf[64], *p;
		FILE *file;
		gint r;
		struct stat stat_buf;
	    
		/*
		 * Some strange people put fifo's with the .mp3 extension,
		 * so we need to make sure it's a real file.  This is not
		 * supposed to be for security, else we'd open then do
		 * fstat()
		 */
		if (!stat(filename, &stat_buf) && S_ISREG(stat_buf.st_mode))
		{
			if ((file = fopen(filename, "r")) != NULL)
			{
				r = fread(buf, 1, sizeof(buf), file);
				fclose(file);

				for ( p = buf; r-- > 0 && (*p=='\r' || *p=='\n'); p++ )
					;
				if ( r > 5 && !strncasecmp(p, "http:", 5))
					ispl = TRUE;
			}
		}

	}


	if (ispl)
		playlist_load_ins(filename, pos);
	else
		if (input_check_file(filename))
		{
			__playlist_ins(filename, pos);
			playlist_generate_shuffle_list();
			playlistwin_update_list();
		}
}

static guint playlist_ins_dir_real(gchar * path, glong pos)
{
	DIR *dir;
	struct dirent *dirent;
	struct stat statbuf;
	gchar *temp;
	guint entries = 0;
	GList *list;

	if (path[strlen(path) - 1] != '/')
		temp = g_strconcat(path, "/", NULL);
	else
		temp = g_strdup(path);

	if ((list = input_scan_dir(temp)) != NULL)
	{
		GList *node;
		g_free(temp);
		node = list;
		while (node)
		{
			__playlist_ins(node->data, pos);
			entries++;
			if (pos >= 0)
				pos++;
			g_free(node->data);
			node = g_list_next(node);
		}
		g_list_free(list);
		return entries;
	}

	if ((dir = opendir(path)) != NULL)
	{
		while ((dirent = readdir(dir)) != NULL)
		{
			if (dirent->d_name[0] != '.')
			{
				gchar *filename = g_strconcat(temp, dirent->d_name, NULL);
				/* Why don't we follow symlinks? */
				lstat(filename, &statbuf);
				if (S_ISDIR(statbuf.st_mode))
				{
					guint i;
					i = playlist_ins_dir_real(filename, pos);
					entries += i;
					if (pos >= 0)
						pos += i;
				}
				else if (input_check_file(filename))
				{
					__playlist_ins(filename, pos);
					entries++;
					if (pos >= 0)
						pos++;
				}
				g_free(filename);
			}
		}
		closedir(dir);
	}
	g_free(temp);
	return entries;
}

guint playlist_ins_dir(gchar * path, glong pos)
{
	guint entries;

	entries = playlist_ins_dir_real(path, pos);
	playlist_generate_shuffle_list();
	playlistwin_update_list();
	return entries;
}

guint playlist_ins_url_string(gchar * string, glong pos)
{
	gchar *temp, *ext;
	struct stat statbuf;
	gint i = 1, entries = 0;

	while (*string)
	{
		temp = strchr(string, '\n');
		if (temp)
		{
			if (*(temp - 1) == '\r')
				*(temp - 1) = '\0';
			*temp = '\0';
		}
		if (!strncasecmp(string, "file:", 5)) {
			if(!xmms_urldecode_path(string))
				return 0;
		}

		stat(string, &statbuf);
		if (S_ISDIR(statbuf.st_mode))
			i = playlist_ins_dir(string, pos);
		else
		{
			ext = strrchr(string, '.');
			if (ext && (!strcasecmp(ext, ".m3u") || !strcasecmp(ext, ".pls")))
				i = playlist_load_ins(string, pos);
			else
			{
				playlist_ins(string, pos);
				i = 1;
			}
		}
		entries += i;
		if (pos >= 0)
			pos += i;
		if (!temp)
			break;
		string = temp + 1;
	}
	playlist_generate_shuffle_list();
	playlistwin_update_list();

	return entries;
}

void playlist_play(void)
{
	char *filename = NULL;

	if (get_playlist_length() == 0)
		return;

	/* If the user wants a skin randomly selected on play */
	if (cfg.random_skin_on_play)
	{
		/* Get the current skin list */
		scan_skins();
		/* If there are entries */
		if (g_list_length(skinlist)) {
			/* Get a random value to select the skin to use */
			int randval = random() % (g_list_length(skinlist) + 1);
			/* If the random value is 0, use the default skin */
			/* Otherwise subtract 1 from the random value and */
			/* select the skin */
			load_skin(randval == 0 ? NULL : ((struct SkinNode *) g_list_nth(skinlist, randval - 1)->data)->path);
		}
		/* Call scan_skins() again to make sure skin selector */
		/* is up to date */
		scan_skins();
	}

	if (get_input_playing())
		input_stop();

	vis_clear_data(mainwin_vis);
	vis_clear_data(playlistwin_vis);
	svis_clear_data(mainwin_svis);
	mainwin_disable_seekbar();

	PL_LOCK();
	filename = playlist_position->filename;
	PL_UNLOCK();

	if (!filename)
		return;
	
	input_play(filename);

	if (input_get_time() != -1)
	{
		equalizerwin_load_auto_preset(filename);
		input_set_eq(cfg.equalizer_active, cfg.equalizer_preamp,
			     cfg.equalizer_bands);
	}
	playlist_check_pos_current();
}

void playlist_set_info(gchar * title, gint length, gint rate, gint freq, gint nch)
{
	PL_LOCK();
	if (playlist_position)
	{
		g_free(playlist_position->title);
		playlist_position->title = g_strdup(title);
		playlist_position->length = length;
	}
	PL_UNLOCK();
	mainwin_set_song_info(rate, freq, nch);
	mainwin_set_info_text();
	playlistwin_update_list();
}

void playlist_check_pos_current(void)
{
	int pos, row, bottom;

	PL_LOCK();
	if (!playlist || !playlist_position || !playlistwin_list)
	{
		PL_UNLOCK();
		return;
	}

	pos = g_list_index(playlist, playlist_position);

	if (playlistwin_item_visible(pos))
	{
		PL_UNLOCK();
		return;
	}

	bottom = MAX(0, __get_playlist_length() -
		     playlistwin_list->pl_num_visible);
	row = CLAMP(pos - playlistwin_list->pl_num_visible / 2, 0, bottom);
	PL_UNLOCK();
	playlistwin_set_toprow(row);
}

void playlist_next(void)
{
	GList *plist_pos_list;
	gboolean restart_playing = FALSE;

	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return;
	}

	plist_pos_list = find_playlist_position_list();

	if (!cfg.repeat && !g_list_next(plist_pos_list))
	{
		PL_UNLOCK();
		return;
	}
	
	if (get_input_playing())
	{
		/* We need to stop before changing playlist_position */
		PL_UNLOCK();
		input_stop();
		PL_LOCK();
		restart_playing = TRUE;
	}

	if (g_list_next(plist_pos_list))
		playlist_position = plist_pos_list->next->data;
	else if (cfg.repeat)
	{
		playlist_position = NULL;
		PL_UNLOCK();
		playlist_generate_shuffle_list();
		PL_LOCK();
		if (cfg.shuffle)
			playlist_position = shuffle_list->data;
		else
			playlist_position = playlist->data;
	}
	PL_UNLOCK();
	playlist_check_pos_current();

	if (restart_playing)
		playlist_play();
	else
	{
		mainwin_set_info_text();
		playlistwin_update_list();
	}
}

void playlist_prev(void)
{
	GList *plist_pos_list;
	gboolean restart_playing = FALSE;
	
	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return;
	}

	plist_pos_list = find_playlist_position_list();

	if (!cfg.repeat && !g_list_previous(plist_pos_list))
	{
		PL_UNLOCK();
		return;
	}

	if (get_input_playing())
	{
		/* We need to stop before changing playlist_position */
		PL_UNLOCK();
		input_stop();
		PL_LOCK();
		restart_playing = TRUE;
	}
	
	if (g_list_previous(plist_pos_list))
		playlist_position = g_list_previous(plist_pos_list)->data;
	PL_UNLOCK();
	playlist_check_pos_current();

	if (restart_playing)
		playlist_play();
	else
	{
		mainwin_set_info_text();
		playlistwin_update_list();
	}
}

void playlist_set_position(gint pos)
{
	GList *node;
	gboolean restart_playing = FALSE;
	
	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return;
	}

	node = g_list_nth(playlist, pos);
	if(!node)
	{
		PL_UNLOCK();
		return;
	}
	
	if (get_input_playing())
	{
		/* We need to stop before changing playlist_position */
		PL_UNLOCK();
		input_stop();
		PL_LOCK();
		restart_playing = TRUE;
	}

	playlist_position = node->data;
	PL_UNLOCK();
	playlist_check_pos_current();

	if (restart_playing)
		playlist_play();
	else
	{
		mainwin_set_info_text();
		playlistwin_update_list();
	}
}

void playlist_eof_reached(void)
{
	GList *plist_pos_list;

	input_stop();

	PL_LOCK();
	plist_pos_list = find_playlist_position_list();

	if (cfg.no_playlist_advance)
	{
		PL_UNLOCK();
		mainwin_set_song_info(0, 0, 0);
		if (cfg.repeat)
			playlist_play();
		return;
	}
	if (!g_list_next(plist_pos_list))
	{
		if (cfg.shuffle)
		{
			playlist_position = NULL;
			PL_UNLOCK();
  			playlist_generate_shuffle_list();
			PL_LOCK();
			playlist_position = shuffle_list->data;
		}
		else
			playlist_position = playlist->data;
		if (!cfg.repeat)
		{
			PL_UNLOCK();
			mainwin_set_song_info(0, 0, 0);
			mainwin_set_info_text();
			return;
		}
	}
	else
		playlist_position = g_list_next(plist_pos_list)->data;
	PL_UNLOCK();
	playlist_check_pos_current();
	playlist_play();
	mainwin_set_info_text();
	playlistwin_update_list();
}

gint get_playlist_length(void)
{
	gint retval;

	PL_LOCK();
	retval = __get_playlist_length();
	PL_UNLOCK();
	
	return retval;
}

gint __get_playlist_length(void)
{
	/* Caller should hold playlist_mutex */
	if (playlist)
		return(g_list_length(playlist));
	return 0;
}

gchar *playlist_get_info_text(void)
{
	gchar *text, *title, *tmp, *tmp2;

	PL_LOCK();
	if (!playlist_position)
	{
		PL_UNLOCK();
		return NULL;
	}

	if (playlist_position->title)
		title = playlist_position->title;
	else
		title = g_basename(playlist_position->filename);
	
	text = g_malloc(strlen(title) + 20);
	if (playlist_position->length != -1)
		text = g_strdup_printf("%d. %s (%d:%-2.2d)",
				       __get_playlist_position() + 1,
				       title, playlist_position->length / 60000,
				       (playlist_position->length / 1000) % 60);
	else
		text = g_strdup_printf("%d. %s", __get_playlist_position() + 1,
				       title);
	if (cfg.convert_underscore)
		while ((tmp = strchr(text, '_')) != NULL)
			*tmp = ' ';
	if (cfg.convert_twenty)
		while ((tmp = strstr(text, "%20")) != NULL)
		{
			tmp2 = tmp + 3;
			*(tmp++) = ' ';
			while (*tmp2)
				*(tmp++) = *(tmp2++);
			*tmp = '\0';
		}
	
	PL_UNLOCK();
	return text;
}

int playlist_get_current_length(void)
{
	gint retval = 0;

	PL_LOCK();
	if (playlist && playlist_position)
		retval = playlist_position->length;
	PL_UNLOCK();
	return retval;
}

gboolean playlist_save(gchar * filename)
{
	PlaylistEntry *entry;
	GList *node;
	FILE *file;
	gchar *ext;
	gboolean is_pls = FALSE;

	if ((file = fopen(filename, "w")) != NULL)
	{
		ext = strrchr(filename, '.');
		if (ext)
		{
			if (!strcasecmp(ext, ".pls"))
			{
				is_pls = TRUE;
				fprintf(file, "[playlist]\n");
				fprintf(file, "NumberOfEntries=%d\n", get_playlist_length());
			}
		}
		PL_LOCK();
		node = playlist;
		while (node)
		{
			entry = (PlaylistEntry *) node->data;
			if (is_pls)
				fprintf(file, "File%d=%s\n", g_list_position(playlist, node) + 1, entry->filename);
			else
				fprintf(file, "%s\n", entry->filename);
			node = node->next;
		}
		PL_UNLOCK();
		fclose(file);
		return TRUE;
	}
	return FALSE;
}

gboolean playlist_load(gchar * filename)
{
	return playlist_load_ins(filename, -1);
}

static void playlist_load_ins_file(gchar * filename, gchar * playlist_name, glong pos)
{
	gchar *temp, *path;

	filename = g_strstrip(filename);

	if(cfg.use_backslash_as_dir_delimiter)
	{
		while ((temp = strchr(filename, '\\')) != NULL)
		      *temp = '/';
	}

	if (filename[0] != '/' && !strstr(filename, "://"))
	{
		path = g_strdup(playlist_name);
		temp = strrchr(path, '/');
		if (temp)
			*temp = '\0';
		else
		{
			__playlist_ins(filename, pos);
			return;
		}
		temp = g_strdup_printf("%s/%s", path, filename);
		__playlist_ins(temp, pos);
		g_free(temp);
		g_free(path);
	}
	else
		__playlist_ins(filename, pos);
}

guint playlist_load_ins(gchar * filename, glong pos)
{
	FILE *file;
	gchar *line, *ext, key[10];
	gint i, noe;
	guint entries = 0;
	int linelen = 1024;

	ext = strrchr(filename, '.');
	if (ext && !strcasecmp(ext, ".pls"))
	{
		if ((line = read_ini_string(filename, "playlist", "NumberOfEntries")) != NULL)
		{
			noe = atoi(line);
			g_free(line);
		}
		else
			return FALSE;
		for (i = 1; i <= noe; i++)
		{
			g_snprintf(key, 10, "File%d", i);
			if ((line = read_ini_string(filename, "playlist", key)) != NULL)
			{
				playlist_load_ins_file(line, filename, pos);
				entries ++;
				if (pos >= 0)
					pos++;
				g_free(line);
			}
		}
		playlist_generate_shuffle_list();
		playlistwin_update_list();
		return entries;
	}
	else
	{
		if ((file = fopen(filename, "r")) != NULL)
		{
			line = g_malloc(linelen);
			while (fgets(line, linelen, file))
			{
				while (strlen(line) == linelen - 1 && line[strlen(line) - 1] == '\n')
				{
					linelen += 1024;
					line = (gchar *) g_realloc(line, linelen);
					fgets(&line[strlen(line)], 1024, file);
				}
				while (line[strlen(line) - 1] == '\r' || line[strlen(line) - 1] == '\n')
					line[strlen(line) - 1] = '\0';
				if (line[0] == '#')
					continue;
				playlist_load_ins_file(line, filename, pos);
				entries++;
				if (pos >= 0)
					pos++;
			}
			fclose(file);
			playlist_generate_shuffle_list();
			playlistwin_update_list();
			return entries;
		}
	}
	return 0;
}

GList *get_playlist(void)
{
	/* Caller should hold playlist_mutex */
	return playlist;
}

gint __get_playlist_position(void)
{
	/* Caller should hold playlist_mutex */
	if (playlist && playlist_position)
		return g_list_index(playlist, playlist_position);
	return 0;
}

gint get_playlist_position(void)
{
	gint retval;

	PL_LOCK();
	retval = __get_playlist_position();
	PL_UNLOCK();

	return retval;
}

gchar * playlist_get_filename(gint pos)
{
	gchar *ret;
	PlaylistEntry *entry;
	GList *node;
	
	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return NULL;
	}
	node = g_list_nth(playlist, pos);
	if (!node)
	{
		PL_UNLOCK();
		return NULL;
	}
	entry = node->data;
	
	ret = g_strdup(entry->filename);
	PL_UNLOCK();

	return ret;
}

gchar * playlist_get_songtitle(gint pos)
{
	gchar *title = NULL, *filename;
	PlaylistEntry *entry;
	GList *node;

	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return NULL;
	}
	node = g_list_nth(playlist, pos);
	if (!node)
	{
		PL_UNLOCK();
		return NULL;
	}
	entry = node->data;

	filename = g_strdup(entry->filename);
	
	if (entry->title == NULL && entry->length == -1)
	{
		if (playlist_get_info_entry(entry))
			title = g_strdup(entry->title);

		PL_UNLOCK();
	}
	else
	{
		title = g_strdup(entry->title);
		PL_UNLOCK();
	}

	if (title == NULL)
		title = g_strdup(g_basename(filename));

	g_free(filename);

	return title;
}

gint playlist_get_songtime(gint pos)
{
	gint retval = -1;
	PlaylistEntry *entry;
	GList *node;
	
	PL_LOCK();
	if (!playlist)
	{
		PL_UNLOCK();
		return -1;
	}
	node = g_list_nth(playlist, pos);
	if (!node)
	{
		PL_UNLOCK();
		return -1;
	}
	entry = node->data;

	if (entry->title == NULL && entry->length == -1)
	{
		if (playlist_get_info_entry(entry))
			retval = entry->length;

		PL_UNLOCK();
	}
	else
	{
		retval = entry->length;
		PL_UNLOCK();
	}

	return retval;
}

gint playlist_sort_by_title_cmpfunc(PlaylistEntry * a, PlaylistEntry * b)
{
	gchar *a_title = NULL, *b_title = NULL;

	if (a->title)
		a_title = a->title;
	else
	{
		if (strrchr(a->filename, '/'))
			a_title = strrchr(a->filename, '/') + 1;
		else
			a_title = a->filename;
	}

	if (b->title)
		b_title = b->title;
	else
	{
		if (strrchr(a->filename, '/'))
			b_title = strrchr(b->filename, '/') + 1;
		else
			b_title = b->filename;

	}
	return strcasecmp(a_title, b_title);
}

void playlist_sort_by_title(void)
{
	PL_LOCK();
	playlist = g_list_sort(playlist, (GCompareFunc) playlist_sort_by_title_cmpfunc);
	PL_UNLOCK();
}

gint playlist_sort_by_filename_cmpfunc(PlaylistEntry * a, PlaylistEntry * b)
{
	gchar *a_filename = NULL, *b_filename = NULL;

	if (strrchr(a->filename, '/'))
		a_filename = strrchr(a->filename, '/') + 1;
	else
		a_filename = a->filename;

	if (strrchr(b->filename,'/'))
		b_filename = strrchr(b->filename, '/') + 1;
	else
		b_filename = b->filename;

	return strcasecmp(a_filename, b_filename);
}

void playlist_sort_by_filename(void)
{
	PL_LOCK();
	playlist = g_list_sort(playlist, (GCompareFunc) playlist_sort_by_filename_cmpfunc);
	PL_UNLOCK();
}

gint playlist_sort_by_path_cmpfunc(PlaylistEntry * a, PlaylistEntry * b)
{
	return strcasecmp(a->filename, b->filename);
}

void playlist_sort_by_path(void)
{
	PL_LOCK();
	playlist = g_list_sort(playlist, (GCompareFunc) playlist_sort_by_path_cmpfunc);
	PL_UNLOCK();
}

static GList* playlist_sort_selected(GList *list, GCompareFunc cmpfunc)
{
	GList *list1, *list2;
	GList *temp_list = NULL;
	GList *index_list = NULL;

	/*
	 * We take all the selected entries out of the playlist,
	 * sorts them, and then put them back in again.
	 */

	list1 = g_list_last(list);

	while(list1)
	{
		list2 = g_list_previous(list1);
		if(((PlaylistEntry *) list1->data)->selected)
		{
			index_list = g_list_prepend(index_list, GINT_TO_POINTER(g_list_position(list, list1)));
			list = g_list_remove_link(list, list1);
			temp_list = g_list_concat(list1, temp_list);
		}
		list1 = list2;
	}

	temp_list = g_list_sort(temp_list, cmpfunc);
	list1 = temp_list;
	list2 = index_list;

	while(list2)
	{
		if(!list1)
		{
			g_log(NULL, G_LOG_LEVEL_CRITICAL, "%s: Error during list sorting. Possibly dropped some playlist-entries.", PACKAGE);
			break;
		}
		list = g_list_insert(list, list1->data, GPOINTER_TO_INT(list2->data));
		list2 = g_list_next(list2);
		list1 = g_list_next(list1);
	}
	g_list_free(index_list);
	g_list_free(temp_list);

	return list;
}


void playlist_sort_selected_by_title(void)
{
	PL_LOCK();	
	playlist = playlist_sort_selected(playlist, (GCompareFunc) playlist_sort_by_title_cmpfunc);
	PL_UNLOCK();
}

void playlist_sort_selected_by_filename(void)
{
	PL_LOCK();
	playlist = playlist_sort_selected(playlist, (GCompareFunc) playlist_sort_by_filename_cmpfunc);
	PL_UNLOCK();
}

void playlist_sort_selected_by_path(void)
{
	PL_LOCK();
	playlist = playlist_sort_selected(playlist, (GCompareFunc) playlist_sort_by_path_cmpfunc);
	PL_UNLOCK();
}

void playlist_reverse(void)
{
	PL_LOCK();
	playlist = g_list_reverse(playlist);
	PL_UNLOCK();
}

static GList *playlist_shuffle_list(GList *list)
{
	/* Caller should holde playlist mutex */
	/*
	 * Note that this doesn't make a copy of the original list.
	 * The pointer to the original list is not valid after this
	 * fuction is run.
	 */
	gint len = g_list_length(list);
	gint i, j;
	GList *node, **ptrs;

	if (!len)
		return NULL;

	ptrs = g_new(GList *, len);

	for (node = list, i = 0; i < len; node = g_list_next(node), i++)
		ptrs[i] = node;

	j = random() % len;
	list = ptrs[j];
	ptrs[j]->next = NULL;
	ptrs[j] = ptrs[0];

	for (i = 1; i < len; i++)
	{
		j = random() % (len - i);
		list->prev = ptrs[i + j];
		ptrs[i + j]->next = list;
		list = ptrs[i + j];
		ptrs[i + j] = ptrs[i];
	}
	list->prev = NULL;

	g_free(ptrs);

	return list;
}

void playlist_random(void)
{
	PL_LOCK();

	playlist = playlist_shuffle_list(playlist);

	PL_UNLOCK();
}

GList * playlist_get_selected_list(void)
{
	GList *node, *list = NULL;
	gint i = 0;

	PL_LOCK();
	for (node = get_playlist(); node != 0; node = g_list_next(node), i++)
	{
		if (((PlaylistEntry *) node->data)->selected)
		{
			list = g_list_prepend(list, GINT_TO_POINTER(i));
		}
	}
	PL_UNLOCK();
	return g_list_reverse(list);
}
	

void playlist_generate_shuffle_list(void)
{
	GList *node;
	gint numsongs;
	gboolean update_mainwin = FALSE;

	PL_LOCK();

	if (shuffle_list)
	{
		g_list_free(shuffle_list);
		shuffle_list = NULL;
	}
	if (playlist)
	{
		shuffle_list = playlist_shuffle_list(g_list_copy(playlist));
		numsongs = g_list_length(shuffle_list);

		if (!numsongs)
		{
			PL_UNLOCK();
			return;
		}

		if(playlist_position)
		{
			node = g_list_nth(shuffle_list, g_list_index(shuffle_list, playlist_position));
			shuffle_list = g_list_remove_link(shuffle_list, node);
			shuffle_list = g_list_prepend(shuffle_list, node->data);
		}
		else
		{
			if (cfg.shuffle)
				playlist_position = shuffle_list->data;
			else
				playlist_position = playlist->data;
			update_mainwin = TRUE;
		}
	}
	PL_UNLOCK();
	if (update_mainwin)
		mainwin_set_info_text();
}

void playlist_fileinfo(gint pos)
{
	gchar *path = NULL;
	PL_LOCK();
	path = g_strdup(((PlaylistEntry *) g_list_nth(get_playlist(), pos)->data)->filename);
	PL_UNLOCK();
	if (path)
	{
		input_file_info_box(path);
		g_free(path);
	}
}

void playlist_fileinfo_current(void)
{
	gchar *path = NULL;
	PL_LOCK();
	
	if (get_playlist())
		path = g_strdup(playlist_position->filename);
	PL_UNLOCK();
	if (path)
	{
		input_file_info_box(path);
		g_free(path);
	}
}

static gboolean playlist_get_info_entry(PlaylistEntry *entry)
{
	/*
	 * Caller need to hold playlist mutex.
	 * Note that this function temporary drops the playlist mutex.
	 * If it returns false, the entry might not be valid.
	 */
	gchar *temp_filename, *temp_title;
	gint temp_length;

	temp_filename = g_strdup(entry->filename);
	temp_title = NULL;
	temp_length = -1;

	/* We don't want to lock the playlist while reading info */
	PL_UNLOCK();
	input_get_song_info(temp_filename, &temp_title, &temp_length);
	PL_LOCK();
	g_free(temp_filename);

	if (!temp_title && temp_length == -1)
		return FALSE;

	/* Make sure entry is still around */
	if (g_list_index(get_playlist(), entry) == -1)
		return FALSE;

	/* entry is still around */
	entry->title = temp_title;
	entry->length = temp_length;

	return TRUE;
}

void *playlist_get_info_func(void *arg)
{
	GList *node;
	gboolean update_playlistwin = FALSE, update_mainwin = FALSE;
	PlaylistEntry *entry;

	while (playlist_get_info_going)
	{
		if (cfg.get_info_on_load && playlist_get_info_scan_active)
		{
			PL_LOCK();
			for (node = get_playlist(); node; node = g_list_next(node))
			{
				entry = node->data;
				if (entry->title || entry->length != -1)
					continue;
				if (!playlist_get_info_entry(entry))
				{
					if (g_list_index(get_playlist(), entry) == -1)
						/* Entry disapeared while we
						   looked it up.  Restart. */
						node = get_playlist();
				}
				else if (entry->title || entry->length != -1)
				{
					int pos;
					pos = g_list_position(get_playlist(), node);
					if (playlistwin_item_visible(pos))
						update_playlistwin = TRUE;
					if (entry == playlist_position)
						update_mainwin = TRUE;
					break;
				}
			}
			PL_UNLOCK();

			if (!node)
				playlist_get_info_scan_active = FALSE;
		}
		else if (!cfg.get_info_on_load && cfg.get_info_on_demand &&
			 cfg.playlist_visible && !cfg.playlist_shaded)
		{
			gboolean found = FALSE;

			PL_LOCK();
			if (!get_playlist())
			{
				PL_UNLOCK();
				xmms_usleep(1000000);
				continue;
			}
			
			for (node = g_list_nth(get_playlist(), playlistwin_get_toprow());
			     node && playlistwin_item_visible(g_list_position(get_playlist(), node));
			     node = g_list_next(node))
			{
				entry = node->data;
				if (entry->title || entry->length != -1)
					continue;

				if (!playlist_get_info_entry(entry))
				{
					if (g_list_index(get_playlist(), entry) == -1)
						/* Entry disapeared while we
						   looked it up.  Restart. */
						node = g_list_nth(get_playlist(), playlistwin_get_toprow());
				}
				else if (entry->title || entry->length != -1)
				{
					int pos;
					pos = g_list_position(get_playlist(), node);
					if (playlistwin_item_visible(pos))
						update_playlistwin = TRUE;
					if (entry == playlist_position)
						update_mainwin = TRUE;
					found = TRUE;
					break;
				}
			}
			PL_UNLOCK();
			if (!found)
			{
				xmms_usleep(500000);
				continue;
			}
		}
		else
			xmms_usleep(500000);

		if (update_playlistwin)
		{
			playlistwin_update_list();
			update_playlistwin = FALSE;
		}
		if (update_mainwin)
		{
			mainwin_set_info_text();
			update_mainwin = FALSE;
		}
	}
	pthread_exit(NULL);
}

void playlist_remove_dead_files(void)
{
	/* FIXME? Does virtual directories work well? */
	GList *node, *next_node, *curr_pos, *temp = NULL;
	PlaylistEntry *entry;
	gboolean list_changed = FALSE;
	
	PL_LOCK();
	node = playlist;
	while (node)
	{
		/* A dead file is a file that is not readable. */
		next_node = g_list_next(node);
		entry = (PlaylistEntry *) node->data;

		if (entry && entry->filename &&
		    !strstr(entry->filename, "://") && /* Don't kill URL's */
		    ((temp = input_scan_dir(entry->filename)) == NULL) &&
		    access(entry->filename, R_OK))
		{
			list_changed = TRUE;
			curr_pos = g_list_find(playlist, playlist_position);
			if (node == curr_pos)
			{
				if (get_input_playing())
				{
					/* Don't remove the currently
                                           playing song */
					node = next_node;
					continue;
				}
				if (g_list_next(curr_pos))
					playlist_position = curr_pos->next->data;
				else if (g_list_previous(curr_pos))
					playlist_position = curr_pos->prev->data;
				else if (node != playlist)
					playlist_position = playlist->data;
				else
					playlist_position = NULL;
			}

			playlist = g_list_remove_link(playlist, node);
			g_free(entry->title);
			g_free(entry->filename);
			g_free(node->data);
			g_list_free_1(node);
		}
		else if (temp)
		{
			g_list_free(temp);
			temp = NULL;
		}
		node = next_node;
	}
	PL_UNLOCK();

	if (list_changed)
	{
		playlist_generate_shuffle_list();
		playlistwin_update_list();
	}
}

void playlist_start_get_info_thread(void)
{
	playlist_get_info_going = TRUE;
	pthread_create(&playlist_get_info_thread, NULL,
		       playlist_get_info_func, NULL);
}

void playlist_stop_get_info_thread(void)
{
	playlist_get_info_going = FALSE;
	pthread_join(playlist_get_info_thread, NULL);
}

void playlist_start_get_info_scan(void)
{
	playlist_get_info_scan_active = TRUE;
}

void playlist_selection_enqueue(GtkSelectionData *selection_data)
{
	gchar *str;

	if (selection_data->type == GDK_SELECTION_TYPE_STRING && selection_data->length > 0)
	{
		str = g_malloc(selection_data->length + 1);
		memcpy(str, selection_data->data, selection_data->length);
		str[selection_data->length] = 0;
		playlist_ins_url_string(str, -1);
		playlistwin_update_list();
		g_free(str);
	}
}
