/*
 *  Copyright 2000 Hvard Kvlen <havardk@sol.no>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xmms/plugin.h"
#include "xmms/i18n.h"
#include "libxmms/util.h"
#include "config.h"
#include <glib.h>
#include <stdlib.h>
#include <pthread.h>
#include <string.h>
#include <math.h>
#include <stdio.h>

#define MIN_FREQ 10
#define MAX_FREQ 20000

#ifndef PI
#define PI 3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165
#endif

static InputPlugin tone_ip;

static gboolean going;
static gboolean audio_error;
static pthread_t play_thread;

static void tone_about(void)
{
	static GtkWidget *box;
	box = xmms_show_message(
		_("About Tone Generator"),
		_("Sinus tone generator by Hvard Kvlen <havardk@xmms.org>\n"
                  "Modified by Daniel J. Peng <danielpeng@bigfoot.com>\n\n"
                  
		  "To use it, add a URL: tone://frequency1;frequency2;frequency3;...\n"
		  "e.g. tone://2000;2005 to play a 2000Hz tone and a 2005Hz tone"), _("Ok"),
		FALSE, NULL, NULL);
	gtk_signal_connect(GTK_OBJECT(box), "destroy",
			   gtk_widget_destroyed, &box);
}

static int tone_is_our_file(char *filename)
{
	if (!strncmp(filename, "tone://", 7))
		return TRUE;
	return FALSE;
}

#define BUF_SAMPLES 512
#define BUF_BYTES BUF_SAMPLES * 2

static void* play_loop(void *arg)
{
	GArray* frequencies = (GArray*)arg;
	gint16 data[BUF_SAMPLES];
	gint i, j;
	gdouble sum_sines;
	gint num_frequencies;
	gint* periods;
	gint* t;
	gdouble* wd;
	
	num_frequencies = 0;
	while(g_array_index(frequencies, gdouble, num_frequencies) != 0) {
		num_frequencies++;
	}

	periods = g_new(gint, num_frequencies);
        t = g_new(gint, num_frequencies);
        wd = g_new(gdouble, num_frequencies);

	for(j = 0; j < num_frequencies; j++) {
		t[j] = 0;
		wd[j] = 2 * PI * g_array_index(frequencies, gdouble, j) / 44100;
		periods[j] = 44100 / g_array_index(frequencies, gdouble, j);
	}

	g_array_free(frequencies,TRUE);

	while (going)
	{
		for (i = 0; i < BUF_SAMPLES; i++) {
			sum_sines = 0;
			for(j = 0; j < num_frequencies; j++) {
				sum_sines += sin(wd[j] * t[j]);
				if (t[j] > periods[j]) {
					t[j] -= periods[j];
				}
				t[j]++;
			}
			data[i] = GINT16_TO_LE(rint(((1 << 15) - 1) * (sum_sines / num_frequencies)));
		}
		tone_ip.add_vis_pcm(tone_ip.output->written_time(),
				    FMT_S16_LE, 1, BUF_BYTES, data);
		while(tone_ip.output->buffer_free() < BUF_BYTES && going)
			xmms_usleep(30000);
		if (going)
			tone_ip.output->write_audio(data, BUF_BYTES);
	}

	g_free((gpointer)periods);
	g_free((gpointer)t);
	g_free((gpointer)wd);

	/* Make sure the output plugin stops prebuffering */

	tone_ip.output->buffer_free();
	tone_ip.output->buffer_free();
	pthread_exit(NULL);
}

gint tone_chars_in_string(char* string, const gchar c)
{
	gint to_return = 0;

	string = strchr(string,c);
	while(string != NULL) {
		string++;
		to_return++;
		string = strchr(string,c);
	}
	return to_return;
}

GArray* tone_filename_parse(const gchar* filename)
{
	GArray* frequencies = g_array_new(TRUE, FALSE,sizeof(gdouble));
	gchar** frequencies_strings;
	const gchar* delim = ";";
	gint i;
	gdouble current_frequency;

	if(strncmp(filename,"tone://",strlen("tone://"))) {
		return NULL;
	}
	filename += strlen("tone://");

/*	printf("tone_filename_parse: string passed to g_strsplit is %s, delim is %s\n",filename,delim);*/
	frequencies_strings = g_strsplit(filename, delim, 500);
	
	i = 0;
	while(frequencies_strings[i] != NULL) {
		current_frequency = strtod(frequencies_strings[i],NULL);
/*		printf("tone_filename_parse: got %f\n",current_frequency);*/
		if (current_frequency >= MIN_FREQ && current_frequency <= MAX_FREQ) {
			g_array_append_val(frequencies, current_frequency);
		}
		i++;
	}

	g_strfreev(frequencies_strings);
	if (g_array_index(frequencies, gdouble, 0) == 0) {
		g_array_free(frequencies,TRUE);
		return NULL;
	} else {
		return frequencies;
	}
}

gchar* tone_title(char* filename)
{
	GArray* frequencies;
	gint num_frequencies;
	gchar** title_parts;
	gint i;
	gchar* title;

	frequencies = tone_filename_parse(filename);
	if (frequencies == NULL) {
		return NULL;
	}

	num_frequencies = 0;
	while(g_array_index(frequencies, gdouble, num_frequencies) != 0) {
		num_frequencies++;
	}

      	title_parts = g_new(char* , 2 + num_frequencies);
      	title_parts[0] = g_strdup(_("Tone Generator: "));
	title_parts[1] = g_strdup_printf("%.2f Hz", g_array_index(frequencies,gdouble,0));
	for(i = 2; i <= num_frequencies; i++) {
		title_parts[i] = g_strdup_printf(";%.2f Hz", g_array_index(frequencies,gdouble,i-1));
	}
	title_parts[num_frequencies + 1] = NULL;
	title = g_strjoinv(NULL,title_parts);
	g_strfreev(title_parts);
	return title;
}
	

static void tone_play(char *filename)
{
	GArray* frequencies;
	gchar *name;
	gint num_frequencies;

	frequencies = tone_filename_parse(filename);
	if (frequencies == NULL) {
		return;
	}

	num_frequencies = 0;
	while(g_array_index(frequencies, gdouble, num_frequencies) != 0) {
		num_frequencies++;
	}

	going = TRUE;
	audio_error = FALSE;
	if (tone_ip.output->open_audio(FMT_S16_LE, 44100, 1) == 0)
	{
		audio_error = TRUE;
		going = FALSE;
		return;
	}

	name = tone_title(filename);
	tone_ip.set_info(name, -1, 16 * 44100, 44100, 1);
	g_free(name);
	pthread_create(&play_thread, NULL, play_loop, frequencies);
}

static void tone_stop(void)
{
	if (going)
	{
		going = FALSE;
		pthread_join(play_thread, NULL);
		tone_ip.output->close_audio();
	}
}

static void tone_pause(short paused)
{
	tone_ip.output->pause(paused);
}

static int tone_get_time(void)
{
	if (audio_error)
		return -2;
	if (!going && !tone_ip.output->buffer_playing())
		return -1;
	return tone_ip.output->output_time();
}

static void tone_song_info(char *filename, char **title, int *length)
{
	*length = -1;
	*title = tone_title(filename);
}
		
	
	
static InputPlugin tone_ip = 
{
	NULL,
	NULL,
	NULL, /* Description */
	NULL,
	tone_about,
	NULL,
	tone_is_our_file,
	NULL,
	tone_play,
	tone_stop,
	tone_pause,
	NULL,
	NULL,
	tone_get_time,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tone_song_info,
	NULL,
	NULL
};

InputPlugin *get_iplugin_info(void)
{
	tone_ip.description =
		g_strdup_printf(_("Tone Generator %s"), VERSION);
	return &tone_ip;
}
