/*
 * Copyright (c) 1980, 1990 Regents of the University of California. All
 * rights reserved.
 * 
 * This code is derived from software contributed to Berkeley by Robert Elz at
 * The University of Melbourne.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution. 3. All advertising
 * materials mentioning features or use of this software must display the
 * following acknowledgement: This product includes software developed by the
 * University of California, Berkeley and its contributors. 4. Neither the
 * name of the University nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1980, 1990 Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char rcsid[] = "$Id: quotaio.c,v 1.1 1999/07/25 15:05:14 mvw Exp $";
#endif /* not lint */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/quota.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>

#include "dqblk.h"

static char *quotagroup = QUOTAGROUP;
static int __initialized_quotaio = 0;
static int need_dquot_convert;

static __inline__ void _initialize_quotaio()
{
   if (__initialized_quotaio)
      return;

   DQUOT_CONVERT_INIT();

   __initialized_quotaio = 1;
}

/*
 * Read a dqblk struct from the quotafile.
 */
int read_quota_from_file(const char *qfpathname, long id, struct dqblk *dqblk)
{
   int fd;
   struct group *grp;
   struct ondisk_dqblk od_dqblk;

   _initialize_quotaio();

   if ((fd = open(qfpathname, O_RDONLY)) < 0) {
      fd = open(qfpathname, O_RDWR | O_CREAT, 0640);
      if (fd < 0)
         return(1);

      sleep(3);

      if ((grp = getgrnam(quotagroup)) != (struct group *)NULL)
         (void) fchown(fd, getuid(), grp->gr_gid);
      else
         (void) fchown(fd, getuid(), 0);

      (void) fchmod(fd, 0640);
   }

   lseek(fd, (long) dqoff(id), L_SET);
   switch (read(fd, &od_dqblk, ondisk_dqblk_size)) {
      case 0:/* EOF */
         /*
          * Convert implicit 0 quota (EOF) into an
          * explicit one (zero'ed dqblk)
          */
         memset((caddr_t)dqblk, 0, sizeof(struct dqblk));
         break;
      case sizeof(struct ondisk_dqblk):   /* OK */
         CONVERT_TO_USER_DQUOT(dqblk, &od_dqblk);
         break;
      default: /* ERROR */
         close(fd);
         return(1);
   }

   close(fd);
   return(0);
}

/*
 * Write a dqblk struct to the quotafile.
 */
int write_quota_to_file(const char *qfpathname, long id, struct dqblk *dqblk)
{
   int fd;
   struct ondisk_dqblk od_dqblk;

   _initialize_quotaio();

   if ((fd = open(qfpathname, O_WRONLY)) < 0) {
      return(1);
   } else {
      CONVERT_TO_KERNEL_DQUOT(&od_dqblk, dqblk);
      lseek(fd, (long) dqoff(id), 0);
      if (write(fd, &od_dqblk, ondisk_dqblk_size) != ondisk_dqblk_size) {
         close(fd);
         return(1);
      }
      close(fd);
   }

   return(0);
}
