/* probe.c -- BIOS probes */
/*
Copyright 1999-2000 John Coffman.
All rights reserved.

Licensed under the terms contained in the file 'COPYING' in the 
source directory.

*/


#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <linux/unistd.h>
#include "common.h"
#include "partition.h"
#include "probe.h"
#include "lrmi.h"

static void do_ebda(void);
static void do_cr_pr(void);
static void do_help(void);
static void do_geom(char *bios);
static void do_geom_all(void);
static void do_table(char *part);

static char dev[] = "<device>";

extern CHANGE_RULE *change_rules;	/* defined in partition.c */


#if 1 /* def LCF_ALL_PARTITIONS */

#define SECTORSIZE 512LL

       _syscall5(int,  _llseek,  uint,  fd, ulong, hi, ulong, lo,
       loff_t *, res, uint, wh);

       int _llseek(unsigned int fd,  unsigned  long  offset_high,
       unsigned  long  offset_low,  loff_t * result, unsigned int
       whence);
       
       loff_t llseek(unsigned int fd, loff_t offs, unsigned int whence)
       { loff_t res;
       	return _llseek(fd, offs>>32, offs, &res, whence) < 0  ?
       			 (loff_t)(-1) : res;
       }
#endif


static
struct Probes {
	char *cmd;
	void (*prc)();
	char *str;
	char *help;
	}
	list[] = {
{ "help",  do_help,  NULL,  "Print list of -T(ell) options"	},
{ "ChRul", do_cr_pr, NULL,  "List partition change-rules"  },
{ "EBDA",  do_ebda,  NULL,  "Extended BIOS Data Area information" },
{ "geom=", do_geom,  "<bios>", "Geometry CHS data for BIOS code 0x80, etc." },
{ "geom" , do_geom_all, NULL, "Geometry for all BIOS drives" },
{ "table=", do_table, dev,   "Partition table information for /dev/hda, etc." },
{ NULL,    NULL,     NULL,   NULL}
	};


static struct partitions {
	char *name;
	unsigned char type;
	unsigned char hide;
	} ptab [] = {		/* Not complete, by any means */

    { "DOS12", PART_DOS12, HIDDEN_OFF },
    { "DOS16_small", PART_DOS16_SMALL, HIDDEN_OFF },
    { "DOS16_big", PART_DOS16_BIG, HIDDEN_OFF },
    { "NTFS or OS2_HPFS", PART_NTFS, HIDDEN_OFF },	/* same as HPFS; keep these two together */
/*  { "HPFS", PART_HPFS, HIDDEN_OFF },	*/	/* same as NTFS */
    { "FAT32", PART_FAT32, HIDDEN_OFF },
    { "FAT32_lba", PART_FAT32_LBA, HIDDEN_OFF },
    { "FAT16_lba", PART_FAT16_LBA, HIDDEN_OFF },
    { "OS/2 BootMgr", PART_OS2_BOOTMGR, 0 },
    { "DOS extended", PART_DOS_EXTD, 0 },
    { "WIN extended", PART_WIN_EXTD_LBA, 0 },
    { "Linux ext'd", PART_LINUX_EXTD, 0 },
    { "Linux Swap", PART_LINUX_SWAP, 0 },
    { "Linux Native", PART_LINUX_NATIVE, 0 },
    { "Minix", PART_LINUX_MINIX, 0 },
    { "Linux RAID", 0xfd, 0 },
    { NULL, 0, 0 }   };

static char phead[] = "\t\t Type  Boot     Start           End      Sector    #sectors";


/* convert octal, decimal, or hex character string to integer */
static int my_atoi(char *str)
{
    int res = 0;
    int base = 10;
    unsigned char c;

    if (!str || !*str) return -1;
    if (*str=='0') { base=8; str++; }
    if (*str && tolower(*str)=='x') {
	if (base==8) { base+=base; str++; }
	else return -1;
    }
    while ((c=*str++)) {
    	c = tolower(c);
    	if (c>'9') c -= 'a'-'0'-10;
    	c -= '0';
    	if (c>=base) return -1;
    	res = res*base + c;
    }
    
    return res;
}


/* print out the help page for -T flag */
static void do_help(void)
{
    struct Probes *pr;
    
    printf("usage:");
    for (pr=list; pr->cmd; pr++) {
    	printf("\tlilo -T %s%s\t%s\n", 
    			pr->cmd, 
    			pr->str ? pr->str : "        ",
    			pr->help);
    }
#if 0
    printf("    In some cases, the '-v' flag will produce more output.\n");
#endif
}


/* get the old BIOS disk geometry */
static int get_geom(int drive, struct disk_geom *geom)
{
   struct LRMI_regs regs;
   struct disk_param *dp;

   memset(&regs, 0, sizeof(regs));
   memset(geom, 0, sizeof(*geom));

   regs.eax = 0x1500;        /* check drive type */
   regs.edx = drive;

   LRMI_int(0x13, &regs);
   
   geom->type = (regs.eax>>8)&0xFF;
   if (geom->type == 0) return 1;
   if (geom->type == 3)
     geom->n_total_blocks = (regs.ecx << 16) | (regs.edx & 0xFFFF);
   
   regs.eax = 0x0800;
   regs.edx = drive;
   LRMI_int(0x13, &regs);
   
   if (regs.flags&1 || regs.eax&0xFF00 || regs.ecx==0)
     return 1 + (regs.eax & 0xFF00);
   
   geom->n_sect = regs.ecx & 0x3F;
   geom->n_head = ((regs.edx>>8)&0xFF)+1;
   geom->n_cyl  = (((regs.ecx>>8)&0xFF)|((regs.ecx&0xC0)<<2))+1;

   regs.eax = 0x4100;   /* check EDD extensions present */
   regs.edx = drive;
   regs.ebx = 0x55AA;
   LRMI_int(0x13, &regs);
   if ((regs.ebx&0xFFFF)==0xAA55) {
      geom->EDD_flags = regs.ecx;
      geom->EDD_rev = regs.eax >> 8;
   }
   if ((geom->EDD_flags) & (EDD_SUBSET | EDD_PACKET)) {
      dp = LRMI_alloc_real(sizeof(struct disk_param));
      dp->size = sizeof(struct disk_param);
      regs.eax = 0x4800;
      regs.edx = drive;
      regs.ds = (int)dp >> 4;
      regs.edi = (int)dp & 15;
      LRMI_int(0x13, &regs);
      
      if ((dp->flags) & EDD_PARAM_GEOM_VALID) {
         if (geom->n_sect != dp->n_sect ||
             geom->n_head != dp->n_head)
                printf("**Warning** Int 0x13 function 8 and function 0x48 return different\n"
                               "head/sector geometries for the following:\n");
         geom->n_cyl  = dp->n_cyl;
         geom->n_head = dp->n_head;
         geom->n_sect = dp->n_sect;
      }

      LRMI_free_real(dp);
   }
   
   return 0;
}


/* get the conventional memory size in Kb */
static int get_conv_mem(void)
{
	struct LRMI_regs regs;

	memset(&regs, 0, sizeof(regs));

	LRMI_init();
	/* Do it. */
	LRMI_int(0x12, &regs);

	/* Return. */
	return regs.eax;
}


/* print the conventional memory size */
static void do_ebda(void)
{
    int m, n;
    static char EBDA[]="Extended BIOS Data Area (EBDA)";
    
    m = get_conv_mem();
    if (m==640) printf("    no %s\n", EBDA);
    else printf("    %s = %dK\n", EBDA, 640-m);
    printf("    Conventional Memory = %dK    0x%06X\n", m, m<<10);
#ifndef LCF_LARGE_EBDA
    m <<= 10;
    m -= 0x200;
    n = m - (MAX_SECONDARY+6)*SECTOR_SIZE;
#else
#error("may not compile with LARGE_EBDA defined")
#endif
#if 0
    if (!verbose) return;
#else
    printf("\n");
#endif
    printf("    The First stage loader boots at:  0x%08X  (%04X:0000)\n",
    			FIRSTSEG<<4, FIRSTSEG);
    printf("    The Second stage loader runs at:  0x%08X  (%04X:%04X)\n",
    			n, n>>4, n&15);
    printf("    The kernel cmdline is passed at:  0x%08X  (%1X000:%04X)\n",
    			m, m>>16, m&0xFFFF);

}


/* print the CHS geometry information for the specified disk */
static void print_geom(int dr, struct disk_geom geom)
{
	 printf("    bios=0x%02x, cylinders=%d, heads=%d, sectors=%d\n", 
		dr, geom.n_cyl, geom.n_head, geom.n_sect);
	 if (geom.EDD_flags & EDD_PACKET)
	    printf("\tEDD packet calls allowed\n");
}


/* print disk drive geometry for all drives */
static void do_geom_all(void)
{
   int d, hd, dr;
   struct disk_geom geom;
   
   for (hd=0; hd<0x81; hd+=0x80)
   for (d=0; d<16; d++) {
      dr = d+hd;
      if (get_geom(dr, &geom)==0) {
      	 print_geom(dr, geom);
      }
   }
}


/* print disk drive geometry information for a particular drive */
static void do_geom(char *bios)
{
    int dr;
    struct disk_geom geom;
    
    dr = my_atoi(bios);
    if (get_geom(dr, &geom)==0) print_geom(dr, geom);
    else printf("Unrecognized BIOS device code 0x%02x\n", dr);
    
}


/* print an individual partition table entry */
static void print_pt(int index, struct partition pt)
{
    char bt[4], *ty, start[32], end[32], type[8];
    char x;
    int i;
    
    for (x=i=0; i<sizeof(pt); i++) x |= ((char*)&pt)[i];
    if (!x) {
    	printf("%4d\t\t\t     ** empty **\n", index);
    	return;
    }
    strcpy(bt,"   ");
    sprintf(type, "0x%02x", (int)pt.sys_ind);
    sprintf(start, "%d:%d:%d",
    	(int)pt.cyl+((pt.sector&0xC0)<<2),
    	(int)pt.head,
    	(int)pt.sector & 0x3f );
    sprintf(end, "%d:%d:%d",
    	(int)pt.end_cyl+((pt.end_sector&0xC0)<<2),
    	(int)pt.end_head,
    	(int)pt.end_sector & 0x3f );
    ty = type;
    if (pt.boot_ind==0x80) bt[1]='*';
    else if (pt.boot_ind==0) ; /* do nothing */
    else {
    	sprintf(bt+1,"%02x", (int)pt.boot_ind);
    }
    for (i=0; ptab[i].name; i++) {
	if (ptab[i].type == pt.sys_ind) {
	    ty = ptab[i].name;
	    break;
	} else if ((ptab[i].type|ptab[i].hide) == pt.sys_ind) {
	    bt[0] = 'H';
	    ty = ptab[i].name;
	    break;
	}
    }
    printf("%4d%18s%5s%10s%14s%12u%12u\n", index, ty, bt,
    	start, end, pt.start_sect, pt.nr_sects);
}

#define is_extd_part(x) ((x)==PART_DOS_EXTD||(x)==PART_WIN_EXTD_LBA||(x)==PART_LINUX_EXTD)

/* partition table display */
static void do_table(char *part)
{
    int fd, i;
    struct partition pt [PART_MAX];
    unsigned int second, base;
    unsigned short boot_sig;
    
    if (!strncmp(part, "/dev/", 5)) {
	if (strncmp(part+5, "md", 2) && isdigit(part[strlen(part)-1]) )
	    die("Not a device: '%s'", part);
	fd = open(part, O_RDONLY);
    } else
	fd = -1;
    if (fd<0) die("Unable to open '%s'", part);
    if (lseek(fd, PART_TABLE_OFFSET, SEEK_SET)<0) die("lseek failed");
    if (read(fd, pt, sizeof(pt)) != sizeof(pt)) die("read pt failed");
    if ( read(fd, &boot_sig, sizeof(boot_sig)) != sizeof(boot_sig)  ||
	boot_sig != BOOT_SIGNATURE ) die("read boot signature failed");
    printf("%s\n", phead);
    second=base=0;
    for (i=0; i<PART_MAX; i++) {
	print_pt(i+1, pt[i]);
	if (is_extd_part(pt[i].sys_ind)) {
	    if (!base) base = pt[i].start_sect;
	    else die("invalid partition table: second extended partition found");
	}
    }
    i=5;
    while (verbose && base) {
        if (llseek(fd, SECTORSIZE*(base+second) + PART_TABLE_OFFSET, SEEK_SET) < 0)
            die("secondary llseek failed");
	if (read(fd, pt, sizeof(pt)) != sizeof(pt)) die("secondary read pt failed");
	if ( read(fd, &boot_sig, sizeof(boot_sig)) != sizeof(boot_sig)  ||
	    boot_sig != BOOT_SIGNATURE ) die("read second boot signature failed");
        print_pt(i++, pt[0]);
        if (is_extd_part(pt[1].sys_ind)) second=pt[1].start_sect;
        else base = 0;
    }
        
    close(fd);
}


/* list partition change-rules */
static void do_cr_pr(void)
{
    CHANGE_RULE *cr;
    
    cr = change_rules;
    printf("\t\tType Normal Hidden\n");
    if (!cr) printf("\t **** no change-rules defined ****\n");
    while (cr) {
	printf ("%20s  0x%02x  0x%02x\n", cr->type, (int)cr->normal, (int)cr->hidden);
	cr = cr->next;
    }
}


/* entry from lilo.c for the '-T' (tell) switch */
void probe_tell (char *cmd)
{
    struct Probes *pr = list;
    int n;
    char *arg;
    
    if (!verbose) printf("\n");
    for (; pr->cmd; pr++) {
	n = strlen(pr->cmd);
	arg = NULL;
	if (pr->cmd[n-1] == '=') arg = cmd+n;
	if (!strncasecmp(cmd, pr->cmd, n)) {
	    pr->prc(arg);
	    printf("\n");
	    exit(0);
	}
    }
    printf("Unrecognized option to '-T' flag\n");
    do_help();
    
    exit(1);
}

