/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <string.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

static PedFileSystemType*	fs_types = NULL;

void
ped_file_system_type_register (PedFileSystemType* fs_type)
{
	PED_ASSERT (fs_type != NULL, return);
	PED_ASSERT (fs_type->ops != NULL, return);
	PED_ASSERT (fs_type->name != NULL, return);
	
	/* pretend that "next" isn't part of the struct :-) */
	((struct _PedFileSystemType*) fs_type)->next = fs_types;
	fs_types = (struct _PedFileSystemType*) fs_type;
}

void
ped_file_system_type_unregister (PedFileSystemType* fs_type)
{
	PedFileSystemType*	walk;
	PedFileSystemType*	last = NULL;

	PED_ASSERT (fs_type != NULL, return);

	for (walk = fs_types; walk != fs_type; walk = walk->next) {
		if (!walk) return;
		last = walk;
	}

	if (last) {
		((struct _PedFileSystemType*) last)->next = fs_type->next;
	} else {
		fs_types = fs_types->next;
	}
}

PedFileSystemType*
ped_file_system_type_get (const char* name)
{
	PedFileSystemType*	walk;

	PED_ASSERT (name != NULL, return NULL);

	for (walk = fs_types; walk != NULL; walk = walk->next) {
		if (!strcasecmp (walk->name, name))
			break;
	}
	return walk;
}

PedFileSystemType*
ped_file_system_type_get_next (PedFileSystemType* fs_type)
{
	if (fs_type)
		return fs_type->next;
	else
		return fs_types;
}

PedFileSystemType*
ped_file_system_probe (const PedGeometry* geom)
{
	PedFileSystemType*	walk = NULL;

	PED_ASSERT (geom != NULL, return NULL);

	ped_exception_fetch_all ();
	while (1) {
		walk = ped_file_system_type_get_next (walk);
		if (!walk) break;
		if (walk->ops->probe (geom)) break;
		ped_exception_catch ();
	}
	ped_exception_leave_all ();
	return walk;
}

PedFileSystem*
ped_file_system_open (PedGeometry* geom)
{
	PedFileSystemType*	type;

	PED_ASSERT (geom != NULL, return NULL);

	type = ped_file_system_probe (geom);
	if (!type) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
				     _("Could not detect file system."));
		return NULL;
	}
	return type->ops->open (geom);
}

PedFileSystem*
ped_file_system_create (PedGeometry* geom, PedFileSystemType* type)
{
	PED_ASSERT (geom != NULL, return NULL);
	PED_ASSERT (type != NULL, return NULL);

	if (!type->ops->create) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for formatting filesystem is "
				       "not yet implemented for %s."),
				     type->name);
		return NULL;
	}
	return type->ops->create (geom);
}

int
ped_file_system_close (PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);
	return fs->type->ops->close (fs);
}

int
ped_file_system_check (PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);

	if (!fs->type->ops->check) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for checking filesystem is "
				       "not yet implemented for %s."),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->check (fs);
}

int
ped_file_system_copy (PedFileSystem* fs, PedGeometry* geom)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);

	if (!fs->type->ops->copy) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for copying filesystem is "
				       "not yet implemented for %s."),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->copy (fs, geom);
}

int
ped_file_system_resize (PedFileSystem* fs, PedGeometry* geom)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);

	if (!fs->type->ops->resize) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("Support for resizing filesystem is "
				       "not yet implemented for %s."),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->resize (fs, geom);
}

PedSector
ped_file_system_get_min_size (const PedFileSystem* fs)
{
	PED_ASSERT (fs != NULL, return 0);

	if (!fs->type->ops->get_min_size) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				     PED_EXCEPTION_CANCEL,
				     _("No get_min_size() for %s!"),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->get_min_size (fs);
}

char
ped_file_system_get_system (const PedFileSystem* fs, const PedPartition* part,
			    PedDiskType* disk_type)
{
	PED_ASSERT (fs != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (disk_type != NULL, return 0);

	if (!fs->type->ops->get_system) {
		ped_exception_throw (PED_EXCEPTION_BUG, PED_EXCEPTION_CANCEL,
				     _("No get_system() for %s!"),
				     fs->type->name);
		return 0;
	}
	return fs->type->ops->get_system (fs, part, disk_type);
}

