/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <string.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

typedef struct _DosRawPartition		DosRawPartition;
typedef struct _DosRawTable		DosRawTable;

/* this MBR boot code is loaded into 0000:7c00 by the BIOS.  See mbr.s for
 * the source, and how to build it
 */

static char MBR_BOOT_CODE[] = {
	0xfa, 0xb8, 0x00, 0x10, 0x8e, 0xd0, 0xbc, 0x00,
	0xb0, 0xb8, 0x00, 0x00, 0x8e, 0xd8, 0x8e, 0xc0,
	0xfb, 0xbe, 0x00, 0x7c, 0xbf, 0x00, 0x06, 0xb9,
	0x00, 0x02, 0xf3, 0xa4, 0xea, 0x21, 0x06, 0x00,
	0x00, 0xbe, 0xbe, 0x07, 0x38, 0x04, 0x75, 0x0b,
	0x83, 0xc6, 0x10, 0x81, 0xfe, 0xfe, 0x07, 0x75,
	0xf3, 0xeb, 0x16, 0xb4, 0x02, 0xb0, 0x01, 0xbb,
	0x00, 0x7c, 0xb2, 0x80, 0x8a, 0x74, 0x01, 0x8b,
	0x4c, 0x02, 0xcd, 0x13, 0xea, 0x00, 0x7c, 0x00,
	0x00, 0xeb, 0xfe
};

#define MSDOS_MAGIC		0xAA55
#define PARTITION_MAGIC_MAGIC	0xf6f6

/* ripped from Linux source */
struct _DosRawPartition {
        __u8	boot_ind;	/* 0x80 - active */
	__u8	head;		/* starting head */
	__u8	sector;		/* starting sector */
	__u8	cyl;		/* starting cylinder */
	__u8	type;		/* partition type */
	__u8	end_head;	/* end head */
	__u8	end_sector;	/* end sector */
	__u8	end_cyl;	/* end cylinder */
	__u32	start;		/* starting sector counting from 0 */
	__u32	length;		/* nr of sectors in partition */
} __attribute__((packed));

struct _DosRawTable {
	char			boot_code [446];
	DosRawPartition		partitions [4];
	__u16			magic;
} __attribute__((packed));

void ped_disk_msdos_init ();
void ped_disk_msdos_done ();

static int msdos_probe (const PedDevice *dev);
static PedDisk* msdos_open (PedDevice* dev);
static PedDisk* msdos_create (PedDevice* dev);
static int msdos_close (PedDisk* disk);
static int msdos_read (PedDisk* disk);
static int msdos_write (PedDisk* disk);
static int msdos_align_partition (PedDisk* disk, PedPartition* part);
static int msdos_enumerate_partition (PedDisk* disk, PedPartition* part);
static int msdos_alloc_metadata (PedDisk* disk);
static int msdos_get_extended_system ();

static PedDiskOps msdos_disk_ops = {
	probe:			msdos_probe,
	open:			msdos_open,
	create:			msdos_create,
	close:			msdos_close,
	read:			msdos_read,
	write:			msdos_write,
	
	align_partition:	msdos_align_partition,
	enumerate_partition:	msdos_enumerate_partition,
	get_extended_system:	msdos_get_extended_system,
	alloc_metadata:		msdos_alloc_metadata
};

static PedDiskType msdos_disk_type = {
	next:	NULL,
	name:	"msdos",
	ops:	&msdos_disk_ops
};

void
ped_disk_msdos_init ()
{
	ped_register_disk_type (&msdos_disk_type);
}

void
ped_disk_msdos_done ()
{
	ped_unregister_disk_type (&msdos_disk_type);
}

static int
msdos_probe (const PedDevice *dev)
{
	DosRawTable	part_table;

	PED_ASSERT (dev != NULL, return 0);

	if (!ped_device_open ((PedDevice*) dev))
		return 0;
	if (!ped_device_read (dev, &part_table, 0, 1)) {
		ped_device_close ((PedDevice*) dev);
		return 0;
	}
	ped_device_close ((PedDevice*) dev);

	if (PED_LE16_TO_CPU (part_table.magic) != MSDOS_MAGIC)
		return 0;

	return 1;
}

static PedDisk*
msdos_open (PedDevice* dev)
{
	PedDisk*	disk;

	PED_ASSERT (dev != NULL, return 0);

	if (!msdos_probe (dev))
		goto error;

	ped_device_open ((PedDevice*) dev);
	disk = ped_disk_alloc (dev, &msdos_disk_type);
	if (!disk)
		goto error;

	disk->part_list = ped_partition_new (
		disk, PED_PARTITION_FREESPACE, NULL,
		0, dev->length);
	if (!disk->part_list)
		goto error_free_disk;

	if (!msdos_read (disk))
		goto error_free_part_list;

	return disk;

error_free_part_list:
	ped_free (disk->part_list);
error_free_disk:
	ped_free (disk);
error:
	return NULL;
}

/* Asks the user (via exceptions) if they want to use GNU Parted's bootloader.
 * If so, writes the boot code into the table.
 */
static void
query_write_boot_code (PedDevice* dev, DosRawTable* table)
{
	PedExceptionOption	ex_status;

	ex_status = ped_exception_throw (PED_EXCEPTION_INFORMATION,
		PED_EXCEPTION_YES_NO,
		_("There is no boot code on this disk.  Would you like "
		  "to use GNU parted's boot loader?"));

	switch (ex_status) {
		case PED_EXCEPTION_NO:
			return;

		case PED_EXCEPTION_UNHANDLED:
			ped_exception_catch ();
		case PED_EXCEPTION_YES:
			memset (table->boot_code, 0, 512);
			memcpy (table->boot_code, MBR_BOOT_CODE,
				sizeof (MBR_BOOT_CODE));
/*
			table->boot_code [MBR_BOOT_CODE_DISK_OFFSET]
				= 0x80 + dev->host * 2 + dev->did;
*/
	}
}

static PedDisk*
msdos_create (PedDevice* dev)
{
	DosRawTable		table;

	PED_ASSERT (dev != NULL, return 0);

	ped_device_read (dev, &table, 0, 1);

	if (!table.boot_code[0])
		query_write_boot_code (dev, &table);

	memset (table.partitions, 0, sizeof (DosRawPartition) * 4);
	table.magic = PED_CPU_TO_LE16 (MSDOS_MAGIC);

	if (!ped_device_write (dev, (void*) &table, 0, 1))
		return 0;
	if (!ped_device_sync (dev))
		return 0;

	return msdos_open (dev);
}

static int
msdos_close (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	ped_device_close (disk->dev);
	ped_disk_delete_all (disk);
	ped_free (disk);
	return 1;
}

static int
raw_part_is_extended (DosRawPartition* raw_part)
{
	PED_ASSERT (raw_part != NULL, return 0);

	switch (raw_part->type) {
	case 0x05:
	case 0x0f:
	case 0x85:
		return 1;

	default:
		return 0;
	}
}

static int
raw_part_is_hidden (DosRawPartition* raw_part)
{
	PED_ASSERT (raw_part != NULL, return 0);

	switch (raw_part->type) {
	case 0x11:
	case 0x14:
	case 0x16:
	case 0x1b:
	case 0x1c:
	case 0x1e:
	return 1;

	default:
		return 0;
	}
}

static PedSector
chs_to_sector (PedDevice* dev, int c, int h, int s)
{
	PedSector	real_c;		/* not measured in sectors, but need */
	PedSector	real_h;		/* lots of bits */
	PedSector	real_s;

	PED_ASSERT (dev != NULL, return 0);

	real_s = s % 0x40 - 1;
	real_h = h;
	real_c = c + ((s >> 6) << 8);

	if (real_c >= 1022)		/* MAGIC: C/H/S is irrelevant */
		return 0;
	return ((real_c * dev->heads + real_h) * dev->sectors + real_s)
		* (dev->sector_size / 512);
}

static void
sector_to_chs (const PedDevice* dev, PedSector sector, unsigned char * c,
	       unsigned char * h, unsigned char * s)
{
	PedSector	real_c, real_h, real_s;

	PED_ASSERT (dev != NULL, return);
	PED_ASSERT (c != NULL, return);
	PED_ASSERT (h != NULL, return);
	PED_ASSERT (s != NULL, return);

	sector /= (dev->sector_size / 512);

	real_c = sector / (dev->heads * dev->sectors);
	real_h = (sector / dev->sectors) % dev->heads;
	real_s = sector % dev->sectors;

	if (real_c > 1023) {
		real_c = 1023;
		real_h = dev->heads - 1;
		real_s = dev->sectors - 1;
	}

	*c = real_c % 0x100;
	*h = real_h;
	*s = real_s + 1 + (real_c >> 8 << 6);
}

static PedSector
legacy_start (PedDisk* disk, const DosRawPartition* raw_part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	return chs_to_sector (disk->dev, raw_part->cyl, raw_part->head,
			      raw_part->sector);
}

static PedSector
legacy_end (PedDisk* disk, const DosRawPartition* raw_part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	return chs_to_sector (disk->dev, raw_part->end_cyl,
			      raw_part->end_head, raw_part->end_sector);
}

static PedSector
linear_start (const PedDisk* disk, const DosRawPartition* raw_part,
	      PedSector offset)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	return offset
	       + PED_LE32_TO_CPU (raw_part->start)
	       	 	* (disk->dev->sector_size / 512);
}

static PedSector
linear_end (const PedDisk* disk, const DosRawPartition* raw_part,
	    PedSector offset)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	return linear_start (disk, raw_part, offset)
	       + (PED_LE32_TO_CPU (raw_part->length) - 1)
	       	 	* (disk->dev->sector_size / 512);
}

static int
probe_real_geom (PedDisk* disk, const DosRawPartition* raw_part,
		 PedSector offset)
{
	int		sector_list[] = {63, 61, 48, 32, 16, 0};
	int		head_list[] = {255, 192, 128, 96, 64, 61, 32, 17, 16, 0};
	int		s_index;
	int		h_index;
	PedSector	l_start, l_end;
	int		old_c, old_h, old_s;
	PedSector	start, end;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	old_c = disk->dev->cylinders;
	old_h = disk->dev->heads;
	old_s = disk->dev->sectors;
	start = linear_start (disk, raw_part, offset);
	end = linear_end (disk, raw_part, offset);

	for (s_index=0; sector_list [s_index]; s_index++) {
		disk->dev->sectors = sector_list [s_index];
		for (h_index=0; head_list [h_index]; h_index++) {
			disk->dev->heads = head_list [h_index];
			disk->dev->cylinders
			    = disk->dev->length /
					(disk->dev->sectors * disk->dev->heads);

			l_start = legacy_start (disk, raw_part);
			l_end = legacy_end (disk, raw_part);

			if (!l_start && !l_end)
				continue;
			if ((l_start
			    && start > disk->dev->sectors * disk->dev->heads)
			    && l_start != start)
				continue;
			if (l_end && l_end == end)
				return 1;
		}
	}

	disk->dev->cylinders = old_c;
	disk->dev->heads = old_h;
	disk->dev->sectors = old_s;
	return 0;
}

/*  Euclid's algorithm for finding the greatest common divisor.
 */
static PedSector
gcd (PedSector a, PedSector b)
{
	PED_ASSERT (a > b, return 0);

	if (b)
		return gcd (b, a % b);
	else
		return a;
}

/* returns the next largest factor of "n", after "factor", or 0 if "factor"
 * is the largest factor.
 */
static PedSector
get_next_factor (PedSector n, PedSector factor)
{
	for (factor++; n % factor; factor++) {
		if (factor > n / 2)
			return 0;
	}
	return factor;
}

/* Finds a dev->sectors, dev->heads that satisfies:
 * 	-  start % dev->sectors == 0
 * 	-  (end + 1) % (dev->heads * dev->sectors) == 0
 * FIXME: this code isn't exactly elegant...
 */
static int
probe_acceptable_geom (PedDisk* disk, const DosRawPartition* raw_part,
		       PedSector offset)
{
	PedSector	start, end;
	PedSector	heads, sectors;
	PedSector	tmp_heads, tmp_sectors;
	PedSector	h_factor, s_factor;	/* factor of h&s to remove */

	start = linear_start (disk, raw_part, offset);
	end = linear_end (disk, raw_part, offset);

	sectors = gcd (end + 1, start);
	if (sectors < 16)
		return 0;

	for (s_factor = 1; s_factor;
	     s_factor = get_next_factor (sectors, s_factor)) {

		tmp_sectors = sectors / s_factor;
		if (tmp_sectors > 63)
			continue;

		for (h_factor = 1; h_factor;
		     h_factor = get_next_factor ((end + 1) / tmp_sectors,
			     			 h_factor)) {

			tmp_heads = (end + 1) / tmp_sectors / h_factor;
			if (tmp_heads <= 255)
				break;
		}
		if (h_factor)
			break;
	}

	if (!s_factor)
		return 0;

	disk->dev->cylinders = disk->dev->length / (tmp_heads * tmp_sectors);
	disk->dev->heads = tmp_heads;
	disk->dev->sectors = tmp_sectors;
	disk->dev->geom_known = 0;

	return 1;
}

static int
process_inconsistent_table (PedDisk* disk, const DosRawPartition* raw_part,
			    PedSector offset)
{
	PedExceptionOption	ex_status;
	int			old_c, old_h, old_s;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	old_c = disk->dev->cylinders;
	old_h = disk->dev->heads;
	old_s = disk->dev->sectors;

	if (probe_real_geom (disk, raw_part, offset)) {
		ex_status = ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("The partition table on %s is inconsistent.  "
			"There are many reasons why this might be the case.  "
			"However, the most likely reason is that Linux "
			"detected the BIOS geometry for %s incorrectly.  "
			"GNU Parted suspects the real geometry should be "
			"%d/%d/%d (not %d/%d/%d).  You should check with your "
			"BIOS first, as this may not be correct.  You can "
			"inform Linux by adding the parameter %s=%d,%d,%d "
			"to the command line.  See the LILO documentation for "
			"more information.  If you think Parted's suggested "
			"geometry is correct, you may select Ignore to "
			"continue (and fix Linux later).  Otherwise, select "
			"Cancel (and fix Linux and/or the BIOS now)."),
			disk->dev->path,
			disk->dev->path,
			(int) disk->dev->cylinders,
			(int) disk->dev->heads,
			(int) disk->dev->sectors,
			old_c,
			old_h,
			old_s,
			disk->dev->path + strlen ("/dev/"),
			(int) disk->dev->cylinders,
			(int) disk->dev->heads,
			(int) disk->dev->sectors);

		switch (ex_status) {
			case PED_EXCEPTION_CANCEL:
				disk->dev->cylinders = old_c;
				disk->dev->heads = old_h;
				disk->dev->sectors = old_s;
				return 0;

			case PED_EXCEPTION_UNHANDLED:
				ped_exception_catch ();
			case PED_EXCEPTION_IGNORE:
				return 1;
		}
	} else {
		ex_status = ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("The partition table on %s is inconsistent.  "
			"There are many reasons why this might be the case.  "
			"Often, the reason is that Linux detected the BIOS "
			"geometry incorrectly.  However, this does not "
			"appear to be the case here."),
			disk->dev->path);

		switch (ex_status) {
			case PED_EXCEPTION_CANCEL:
				return 0;

			case PED_EXCEPTION_UNHANDLED:
				ped_exception_catch ();
			case PED_EXCEPTION_IGNORE:
				return probe_acceptable_geom (disk, raw_part,
							      offset);
		}
	}
}

static int
check_consistancy (PedDisk* disk, DosRawPartition* raw_part, PedSector offset)
{
	PedSector	leg_start, leg_end;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	if (!disk->dev->geom_known)
		return 1;

	leg_start = legacy_start (disk, raw_part);
	leg_end = legacy_end (disk, raw_part);

	if (leg_start && leg_start != linear_start (disk, raw_part, offset))
		return process_inconsistent_table (disk, raw_part, offset);
	if (leg_end && leg_end != linear_end (disk, raw_part, offset))
		return process_inconsistent_table (disk, raw_part, offset);
	return 1;
}

static int
read_table (PedDisk* disk, PedSector sector, int is_extended_table)
{
	int			i;
	DosRawTable		table;
	DosRawPartition*	raw_part;
	PedPartition*		part;
	PedPartitionType	type;
	PedSector		lba_offset;
	PedSector		part_start;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	if (!ped_device_read (disk->dev, (void*) &table, sector, 1))
		goto error;

	/* weird: empty extended partitions are filled with 0xf6 by PM */
	if (is_extended_table
	    && PED_LE16_TO_CPU (table.magic) == PARTITION_MAGIC_MAGIC)
		return 1;

	if (PED_LE16_TO_CPU (table.magic) != MSDOS_MAGIC) {
		if (ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_IGNORE_CANCEL,
			_("Invalid partition table on %s - wrong signature %x"),
			disk->dev->path,
			PED_LE16_TO_CPU (table.magic))
				!= PED_EXCEPTION_IGNORE)
			goto error;
	}

	for (i = 0; i < 4; i++) {
		raw_part = &table.partitions [i];
		if (raw_part->type == 0x0 || !raw_part->length)
			continue;

		if (is_extended_table) {
			if (raw_part_is_extended (raw_part))
				lba_offset = ped_disk_extended_partition
						(disk)->geom.start;
			else
				lba_offset = sector;
		} else {
			lba_offset = 0;
		}

		if (!check_consistancy (disk, raw_part, lba_offset))
			return 0;
		part_start = linear_start (disk, raw_part, lba_offset);

		if (part_start == sector) {
			if (ped_exception_throw (
				PED_EXCEPTION_ERROR,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("Invalid partition table - recursive "
				"partition on %s."),
				disk->dev->path)
					!= PED_EXCEPTION_IGNORE)
				return 0;
			continue;	/* avoid infinite recursion */
		}

		if (is_extended_table) {
			if (raw_part_is_extended (raw_part)) {
				/* hide M$ brain-damage ala fdisk, by not
				   adding nested extended partitions in a tree.
				   NOTE: sector offset is in terms of the
				   first extended partition */
				if (!read_table (disk, part_start, 1))
					return 0;
				continue;
			}
			type = PED_PARTITION_LOGICAL;
		} else {
			type = (raw_part_is_extended (raw_part))
				? PED_PARTITION_EXTENDED
				: PED_PARTITION_PRIMARY;
		}

		part = ped_partition_new (
			disk, type, NULL,
			part_start,
			linear_end (disk, raw_part, lba_offset));
		if (!part)
			goto error;
		part->system = raw_part->type;			/* hack */
		part->bootable = raw_part->boot_ind != 0;	/* hack */
		part->hidden = raw_part_is_hidden (raw_part);
		if (!is_extended_table)
			part->num = i + 1;
		if (!ped_disk_add_partition (disk, part))
			goto error;

		if (part->geom.start != part_start
		    || part->geom.end
				!= linear_end (disk, raw_part, lba_offset)) {
			ped_exception_throw (
				PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				"Partition %d isn't aligned to cylinder "
				"boundaries.  Need to add support for this.",
				part->num);
			goto error;
		}

		if (type == PED_PARTITION_EXTENDED) {
			if (!read_table (disk, part_start, 1))
				return 0;
		} else {
			part->fs_type = ped_file_system_probe (&part->geom);
		}
	}

	return 1;

error:
	ped_disk_delete_all (disk);
	return 0;
}

static int
msdos_read (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	ped_disk_delete_all (disk);
	return read_table (disk, 0, 0);
}

static int
fill_raw_part (DosRawPartition* raw_part, PedPartition* part, PedSector offset)
{
	PED_ASSERT (raw_part != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	raw_part->boot_ind = 0x80 * part->bootable;
	raw_part->type = part->system;
	raw_part->start = PED_CPU_TO_LE32 ((part->geom.start - offset)
				/ (part->geom.disk->dev->sector_size / 512));
	raw_part->length = PED_CPU_TO_LE32 (part->geom.length
				/ (part->geom.disk->dev->sector_size / 512));

	sector_to_chs (part->geom.disk->dev, part->geom.start,
		       &raw_part->cyl, &raw_part->head, &raw_part->sector);
	sector_to_chs (part->geom.disk->dev, part->geom.end,
		       &raw_part->end_cyl, &raw_part->end_head,
		       &raw_part->end_sector);

	return 1;
}

static int
fill_ext_raw_part_geom (DosRawPartition* raw_part, PedGeometry* geom,
			PedSector offset)
{
	PED_ASSERT (raw_part != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);
	PED_ASSERT (geom->disk != NULL, return 0);
	PED_ASSERT (geom->disk->dev != NULL, return 0);

	raw_part->boot_ind = 0;
	raw_part->type = 0x5;
	raw_part->start = PED_CPU_TO_LE32 ((geom->start - offset)
				/ (geom->disk->dev->sector_size / 512));
	raw_part->length = PED_CPU_TO_LE32 (geom->length
				/ (geom->disk->dev->sector_size / 512));

	sector_to_chs (geom->disk->dev, geom->start,
		       &raw_part->cyl, &raw_part->head, &raw_part->sector);
	sector_to_chs (geom->disk->dev, geom->start + geom->length - 1,
		       &raw_part->end_cyl, &raw_part->end_head,
		       &raw_part->end_sector);

	return 1;
}

static int
write_ext_table (PedDisk* disk, PedSector sector, PedPartition* logical)
{
	DosRawTable		table;
	PedGeometry*		geom;
	PedPartition*		part;
	PedSector		lba_offset;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (ped_disk_extended_partition (disk) != NULL, return 0);
	PED_ASSERT (logical != NULL, return 0);

	lba_offset = ped_disk_extended_partition (disk)->geom.start;

	memset (&table, 0, sizeof (DosRawTable));
	table.magic = PED_CPU_TO_LE16 (MSDOS_MAGIC);

	if (!fill_raw_part (&table.partitions[0], logical, sector))
		return 0;

	part = ped_disk_get_partition (disk, logical->num + 1);
	if (part) {
		geom = ped_geometry_new (disk, part->prev->geom.start,
				ped_disk_extended_partition (disk)->geom.end
				- part->prev->geom.start + 1);
		if (!geom)
			return 0;
		fill_ext_raw_part_geom (&table.partitions[1], geom, lba_offset);
		ped_geometry_destroy (geom);

		if (!write_ext_table (disk, part->prev->geom.start, part))
			return 0;
	}

	return ped_device_write (disk->dev, (void*) &table, sector, 1);
}

static int
write_empty_table (PedDisk* disk, PedSector sector)
{
	DosRawTable		table;

	PED_ASSERT (disk != NULL, return 0);

	memset (&table, 0, sizeof (DosRawTable));
	table.magic = PED_CPU_TO_LE16 (MSDOS_MAGIC);

	return ped_device_write (disk->dev, (void*) &table, sector, 1);
}

/* Find the first logical partition, and write the partition table for it.
 */
static int
write_extended_partitions (PedDisk* disk)
{
	PedPartition*		ext_part;
	PedPartition*		part;

	PED_ASSERT (disk != NULL, return 0);

	ext_part = ped_disk_extended_partition (disk);
	part = ped_disk_get_partition (disk, 5);
	if (part)
		return write_ext_table (disk, ext_part->geom.start, part);
	else
		return write_empty_table (disk, ext_part->geom.start);
}

static int
msdos_write (PedDisk* disk)
{
	DosRawTable		table;
	PedPartition*		part;
	int			i;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	ped_device_read (disk->dev, &table, 0, 1);

	if (!table.boot_code[0])
		query_write_boot_code (disk->dev, &table);

	memset (table.partitions, 0, sizeof (DosRawPartition) * 4);
	table.magic = PED_CPU_TO_LE16 (MSDOS_MAGIC);

	for (i=1; i<=4; i++) {
		part = ped_disk_get_partition (disk, i);
		if (!part)
			continue;

		if (!fill_raw_part (&table.partitions [i - 1], part, 0))
			return 0;

		if (part->type == PED_PARTITION_EXTENDED) {
			if (!write_extended_partitions (disk))
				return 0;
		}
	}

	if (!ped_device_write (disk->dev, (void*) &table, 0, 1))
		return 0;
	if (!ped_device_sync (disk->dev))
		return 0;
	return 1;
}

static PedSector
round_down_to (PedSector sector, PedSector grain_size)
{
	return sector - sector % grain_size;
}

static PedSector
round_up_to (PedSector sector, PedSector grain_size)
{
	if (sector % grain_size)
		return round_down_to (sector, grain_size) + grain_size;
	else
		return sector;
}

static PedSector
round_to_nearest (PedSector sector, PedSector grain_size)
{
	if (sector % grain_size > grain_size/2)
		return round_up_to (sector, grain_size);
	else
		return round_down_to (sector, grain_size);
}

static int
msdos_align_partition (PedDisk* disk, PedPartition* part)
{
	PedPartition*		ext_part;
	PedDevice*		dev;
	PedSector		cylinder_size;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	ext_part = ped_disk_extended_partition (disk);
	dev = disk->dev;
	cylinder_size = dev->sectors * dev->heads;

	part->geom.start = round_to_nearest (part->geom.start, cylinder_size);
	if (part->type & PED_PARTITION_LOGICAL) {
		PED_ASSERT (ext_part != NULL, return 0);

		/* if it's a logical partition outside an extended partition,
		 * back out immediately!  Let the support code deal with it...
		 */
		if (part->geom.start <= ext_part->geom.start - cylinder_size)
			return 1;

		if (part->geom.start <= ext_part->geom.start + dev->sectors) {
			part->geom.start = ext_part->geom.start + dev->sectors;
			if (part->num > 5)
				part->geom.start += dev->sectors;
		} else {
			part->geom.start += dev->sectors;
		}
	} else if (part->geom.start == 0) {
		part->geom.start += dev->sectors;
	}
	part->geom.end = round_to_nearest (part->geom.end, cylinder_size) - 1;
	part->geom.length = part->geom.end - part->geom.start + 1;
	return 1;
}

static int
add_metadata_part (PedDisk* disk, PedPartitionType type, PedSector start,
		   PedSector length)
{
	PedPartition*		new_part;

	PED_ASSERT (disk != NULL, return 0);

	new_part = ped_partition_new (disk, type | PED_PARTITION_METADATA, NULL,
				      start, start + length);
	if (!new_part)
		goto error;
	if (!ped_disk_add_partition (disk, new_part))
		goto error_destroy_new_part;

	return 1;

error_destroy_new_part:
	ped_partition_destroy (new_part);
error:
	return 0;
}

static int
msdos_alloc_metadata (PedDisk* disk)
{
	PedPartition*		ext_part;
	PedPartition*		log_part;
	int			i;
	PedDevice*		dev;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	dev = disk->dev;

	if (!add_metadata_part (disk, PED_PARTITION_PRIMARY,
				0, dev->sectors - 1))
		return 0;

	ext_part = ped_disk_extended_partition (disk);
	if (ext_part) {
		if (!add_metadata_part (disk, PED_PARTITION_LOGICAL,
					ext_part->geom.start, dev->sectors - 1))
			return 0;
		for (i=6; 1; i++) {
			log_part = ped_disk_get_partition (disk, i);
			if (!log_part)
				break;
			if (log_part->prev
			    && log_part->prev->type == PED_PARTITION_METADATA)
				continue;
			if (!add_metadata_part (disk, PED_PARTITION_LOGICAL,
					log_part->geom.start - dev->sectors,
					dev->sectors - 1))
				return 0;
		}
	}
	return 1;
}

static int
next_primary (PedDisk* disk)
{
	int	i;
	for (i=1; i<=4; i++) {
		if (!ped_disk_get_partition (disk, i))
			return i;
	}
	return 0;
}

static int
next_logical (PedDisk* disk)
{
	int	i;
	for (i=5; 1; i++) {
		if (!ped_disk_get_partition (disk, i))
			return i;
	}
}

static int
msdos_enumerate_partition (PedDisk* disk, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	/* don't re-number a primary partition */
	if (part->num != -1 && part->num <= 4)
		return 1;
	part->num = -1;

	switch (part->type) {
	case PED_PARTITION_PRIMARY:
	case PED_PARTITION_EXTENDED:
		part->num = next_primary (disk);
		if (!part->num) {
			part->num = -1;
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Can't add another primary partition - there "
				  "is a limit of 4 primary partitions."));
			return 0;
		}
		break;

	case PED_PARTITION_LOGICAL:
		part->num = next_logical (disk);
		break;

	default:
		return 0;
	}
	return 1;
}

static int
msdos_get_extended_system ()
{
	return 0x5;
}

