/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "bird.h"
#include "global.h"
#include "mathutil.h"
#include "window_state.h"

#define BIRD_A 1.6
#define BIRD_B 10

/* This is much simpiler than tux.c since the bird has only one
 * state.  Though the bird pathing is more dynamic and requires
 * some differential geometry.  (Well maybe requires is the
 * wrong word but thats how I'm doing it.)
 */

void
init_bird (Demo_State *state)
{
  Matrix rotate;
  
  state->bird_time = 0;
  state->bird_window = NULL;
  UniformScaleMat (state->bird_scale, 1.f / 10);
  IdentityMat (state->bird_orientation);
  RotateMat (state->bird_orientation, DEG_TO_RAD (90), 0, 0, 1);  
  RotateMat (rotate, DEG_TO_RAD (-90), 0, 1, 0);
  MatMultMat4x4 (state->bird_orientation, state->bird_orientation, 
  		 rotate);
  IdentityMat (state->bird_align);
}

void
put_bird_on_window (Demo_State *state)
{
  state->bird_time = 0;
  float width, height;

  if (!state->bird_window)
    return;
  
  width = state->bird_window->world_lower_right[0] -
    state->bird_window->world_upper_left[0];
  height = state->bird_window->world_upper_left[1] -
    state->bird_window->world_lower_right[1];

  /* Put the bird right in front of the window */
  state->bird_position[0] = 0.5 * width;
  state->bird_position[1] = 0.5 * -height;
  state->bird_position[2] = 5;
}

void
update_bird (Demo_State *state)
{
  Vector X, T, N, B;
  float length, oolength, sint, cost, a, b, a2, b2, width, height, K;

  if (!state->bird_window)
    return;

  /* The bird flys around a single window until it decides to
   * switch to another window.  It follows this curve
   *   (a * cos(t), 0, b * sin(t))
   * where a is proportional to the width of the window and 
   * b is some constant that keeps the bird's flight within
   * the space between windows.
   * I also  compute the TNB frame for the bird at each animation 
   * frame to track the bird's orientation.
   */

  /* Update the time */
  state->bird_time += state->dt;

  /* Compute the common stuff */
  sint = sin (-state->bird_time);
  cost = cos (-state->bird_time);

  width = state->bird_window->world_lower_right[0] -
    state->bird_window->world_upper_left[0];
  height = state->bird_window->world_upper_left[1] -
    state->bird_window->world_lower_right[1];
    
  a = BIRD_A * 0.5 * width;
  a2 = a * a;
  b = BIRD_B;
  b2 = b * b;
  

  /* The first terms are relative to the center of the window.  Then
   * add the position of the window center so that X is relative
   * to the upper left corner of the window. */   
  state->bird_position[0] = X[0] = a * cost + 0.5 * width;
  state->bird_position[1] = X[1] = 0.5 * -height;
  state->bird_position[2] = X[2] = b * sint;

  /* This is the velocity.  It will be T after normalization. */
  T[0] = -a * sint;
  T[1] = 0;
  T[2] = b * cost;

  /* I do the normalization of T myself since the length is useful
   * later. */
  length = Magnitude (T);
  oolength = 1.f / length;
  T[0] *= oolength;
  T[1] *= oolength;
  T[2] *= oolength;

  /* This is dT/dt which will be N after normalization */
  K = 2 * sint * cost * (a2 - b2) * oolength * oolength * oolength;
  N[0] =  -a * cost * oolength - K * a * sint;
  N[1] =  0;
  N[2] = -b * sint * oolength - K * b * cost;
  Normalize (N);
  
  /* B is the normalized cross product of T with N */
  CrossProduct (B, T, N);
  Normalize (B);   

  /* Upate the frame */
  state->bird_frame++;
  if (state->bird_frame > BIRD_FLY_MAX_FRAME)
    state->bird_frame = BIRD_FLY_MIN_FRAME;
 
  /* Build the alignment matrix.  This works like the spider matrix
   * but I align with the TNB frame 
   * (1, 0, 0) --> N
   * (0, 1, 0) --> T
   * (0, 0, 1) --> B
   */
  state->bird_align[0][0] = T[0];
  state->bird_align[1][0] = T[1];
  state->bird_align[2][0] = T[2];
  state->bird_align[3][0] = 0;

  state->bird_align[0][1] = B[0];
  state->bird_align[1][1] = B[1];
  state->bird_align[2][1] = B[2];
  state->bird_align[3][1] = 0;

  state->bird_align[0][2] = -N[0];
  state->bird_align[1][2] = -N[1];
  state->bird_align[2][2] = -N[2];
  state->bird_align[3][2] = 0;

  state->bird_align[0][3] = 0;
  state->bird_align[1][3] = 0;
  state->bird_align[2][3] = 0;
  state->bird_align[3][3] = 1;
}
