/* Copyright 1993-95 by Carl Harris, Jr.
 * All rights reserved
 *
 * Distribute freely, except: don't remove my name from the source or
 * documentation (don't take credit for my work), mark your changes (don't
 * get me blamed for your possible bugs), don't alter or remove this
 * notice.  May be sold if buildable source is provided to buyer.  No
 * warrantee of any kind, express or implied, is included with this
 * software; use at your own risk, responsibility for damages (if any) to
 * anyone resulting from the use of this software rests entirely with the
 * user.
 *
 * Send bug reports, bug fixes, enhancements, requests, flames, etc., and
 * I'll try to keep a version up to date.  I can be reached as follows:
 * Carl Harris <ceharris@mal.com>
 */


/***********************************************************************
  module:       poprc.c
  project:      popclient
  programmer:   Carl Harris, ceharris@mal.com
  description:  .poprc file functionality

  $Log: poprc.c,v $
  Revision 1.5  1995/09/07 22:37:37  ceharris
  Preparation for 3.0b4 release.

  Revision 1.4  1995/08/10 00:32:42  ceharris
  Preparation for 3.0b3 beta release:
  -	added code for --kill/--keep, --limit, --protocol, --flush
  	options; --pop2 and --pop3 options now obsoleted by --protocol.
  - 	added support for APOP authentication, including --with-APOP
  	argument for configure.
  -	provisional and broken support for RPOP
  -	added buffering to SockGets and SockRead functions.
  -	fixed problem of command-line options not being correctly
  	carried into the merged options record.

  Revision 1.3  1995/08/09 01:32:59  ceharris
  Version 3.0 beta 2 release.
  Added
  -	.poprc functionality
  -	GNU long options
  -	multiple servers on the command line.
  Fixed
  -	Passwords showing up in ps output.

  Revision 1.2  1995/08/08 01:01:32  ceharris
  Added GNU-style long options processing.
  Fixed password in 'ps' output problem.
  Fixed various RCS tag blunders.
  Integrated .poprc parser, lexer, etc into Makefile processing.

 ***********************************************************************/

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include "poprc_y.h"
#include "popclient.h"
#include "bzero.h"

extern int yydebug;
extern FILE *yyin;

char *prc_pathname;
int prc_errflag;


/* Node type for linked list of servers */
struct prc_node {
  struct prc_node *next;
  struct prc_server s;
};

/* Server parameters for current .poprc record */
static struct prc_server sp;

/* Head and tail of server list */
static struct prc_node *sp_head, *sp_tail;



/******************************************************************
  function:	prc_getpathname
  description:	figure out the pathname to poprc file based on
		command-line options and defaults.
  arguments:	
    cmd_opts	command-line options.
    def_opts	default options.

  ret. value:	pathname to .poprc
  globals:	none.
  calls:	none.
 *****************************************************************/

char *
prc_getpathname (cmd_opts, def_opts) 
struct optrec *cmd_opts, *def_opts;
{
  /* XXX -- this is a royal hack.  Because we don't merge the command-line
            options with the default options until we get around to querying
	    a particular server, we have to do this little merge here. */

  return (cmd_opts->poprcfile != (char *) 0 ?
          cmd_opts->poprcfile : def_opts->poprcfile);
}


/******************************************************************
  function:	prc_parse_file
  description:	Read the contents of the .poprc file, storing 
		each parsed record in a linked list.
  arguments:
    pathname	pathname for the .poprc file

  ret. value:	error code.
  globals:	writes sp_head, writes sp_tail, writes yyin,
		writes prc_pathname, writes prc_errflag.
  calls:	prc_reset, yyparse.
 *****************************************************************/

prc_parse_file (pathname)
char *pathname;
{
  struct stat statbuf;

  prc_pathname = pathname;
  prc_errflag = 0;
  sp_head = sp_tail = (struct prc_node *) 0;
  prc_reset();

  /* the .poprc file must have the same uid as the REAL uid of this process, 
     it must have permissions no greater than 600, and it must not be a 
     symbolic link.  We check these conditions here. */

  if (lstat(pathname, &statbuf) < 0) {
    if (errno == ENOENT) 
      return(0);
    else {
      perror(pathname);
      return(PS_IOERR);
    }
  }

  if ((statbuf.st_mode & S_IFLNK) == S_IFLNK) {
    fprintf(stderr, "File %s must not be a symbolic link.\n", pathname);
    return(PS_AUTHFAIL);
  }

  if (statbuf.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE)) {
    fprintf(stderr, "File %s must have no more than -rw------ permissions.\n", 
            pathname);
    return(PS_AUTHFAIL);
  }

  if (statbuf.st_uid != getuid()) {
    fprintf(stderr, "File %s must be owned by you.\n", pathname);
    return(PS_AUTHFAIL);
  }


  /* Open the poprc and feed it to the lexer. */
  if ((yyin = fopen(pathname,"r")) == (FILE *) 0) {
    perror(pathname);
    return(PS_IOERR);
  }

  yyparse();		/* parse entire file */

  fclose(yyin);

  if (prc_errflag) 
    return(PS_SYNTAX);
  else
    return(0);
}



/******************************************************************
  function:	prc_reset
  description:	clear the global sp record (server parameters)
		used by the parser.
  arguments:	none.
  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_reset()
{
  sp.server = (char *) 0;
  sp.protocol = DEF_PROTOCOL;
  sp.username = (char *) 0;
  sp.password = (char *) 0;
  sp.remotefolder = (char *) 0;
  sp.localfolder = (char *) 0;
}



/******************************************************************
  function:	prc_register
  description:	register the parsed server params by appending
		them to a linked list of server param records.
  arguments:	none.
  ret. value:	none.
  globals:	reads sp.
  calls:	none.
 *****************************************************************/

prc_register()
{
  struct prc_node *node;

  /* initialize new node */
  node = (struct prc_node *) xmalloc(sizeof(struct prc_node));
  node->next = (struct prc_node *) 0;
  bcopy(&sp, &node->s, sizeof(sp));
  
  /* append to end of list */
  if (sp_tail != (struct prc_node *) 0)
    /* list contains at least one element */
    sp_tail->next = node;
  else
    /* list is empty */
    sp_head = node;
  sp_tail = node;

}
 


/******************************************************************
  function:	prc_setserver
  description:	stub function to record parsed server name string
		into global sp record.
  arguments:	
    s		parsed server name

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setserver(s)
char *s;
{
  sp.server = s;
}


/******************************************************************
  function:	prc_setproto
  description:	stub function to record parsed protocol identifier
		into global sp record.
  arguments:	
    p		protocol identifier		

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setproto(p)
int p;
{
  sp.protocol = p;
}


/******************************************************************
  function:	prc_setusername
  description:	stub function to record parsed username string
		into global sp record.
  arguments:	
    s		username string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setusername(s)
char *s;
{
  sp.username = s;
}


/******************************************************************
  function:	prc_setpassword
  description:	stub function to record parsed password string
		into global sp record.
  arguments:	
    s		password string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setpassword(s)
char *s;
{
  sp.password = s;
}


/******************************************************************
  function:	prc_setremote
  description:	stub function to record parsed remote folder string
		into global sp record.
  arguments:	
    s		remote foldername string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setremote(s)
char *s;
{
  sp.remotefolder = s;

}


/******************************************************************
  function:	prc_setlocal
  description:	stub function to record parsed local folder string
		into global sp record.
  arguments:	
    s		local foldername string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setlocal(s)
char *s;
{
  sp.localfolder = s;
}



/******************************************************************
  function:	prc_mergeoptions
  description:	merge command-line options, default options, and
		.poprc options into a single options record.
  arguments:
    servername	name of the server to search in the .poprc
    cmd_opts 	command-line options.
    def_opts 	default options.
    mrg_opts	record to receive the merged options.

  ret. value:	none.
  globals:	reads sp_head.
  calls:	none.
 *****************************************************************/

int prc_mergeoptions (servername, cmd_opts, def_opts, mrg_opts)
char *servername;
struct optrec *cmd_opts, *def_opts, *mrg_opts;
{
  struct prc_node *sp_node;

  for (sp_node = sp_head;  
       sp_node != NULL && strcmp(sp_node->s.server, servername) != 0;
       sp_node = sp_node->next)
    ;  /* locate node for this server in list */
     
  /* copy those options which aren't subject to merging */
  bcopy(cmd_opts, mrg_opts, sizeof(*mrg_opts));

  strcpy(mrg_opts->loginid, def_opts->loginid);

  /* determine protocol to use */
  if (cmd_opts->whichpop != 0) 
    mrg_opts->whichpop = cmd_opts->whichpop; 
  else if (sp_node != NULL && sp_node->s.protocol != 0)
    mrg_opts->whichpop = sp_node->s.protocol;
  else
    mrg_opts->whichpop = DEF_PROTOCOL;

  /* Determine what the POP username will be */
  if (*(cmd_opts->userid) != '\0')
    strcpy(mrg_opts->userid, cmd_opts->userid);
  else if (sp_node != NULL && sp_node->s.username != (char *) 0)
    strncpy(mrg_opts->userid, sp_node->s.username, USERIDLEN - 1);
  else
    strcpy(mrg_opts->userid, def_opts->loginid);

  /* Deal with the password specification */
  if (*(cmd_opts->password) != '\0')
    strcpy(mrg_opts->password, cmd_opts->password);
  else if (sp_node != NULL && sp_node->s.password != (char *) 0)
    strncpy(mrg_opts->password, sp_node->s.password, PASSWORDLEN - 1);
  else 
    strncpy(mrg_opts->password,
         (char *) getpassword("Enter mailserver password: "), PASSWORDLEN - 1);
  
  /* Determine what the local folder will be */
  if (cmd_opts->foldertype != 0) {
    mrg_opts->foldertype = cmd_opts->foldertype;
    if (cmd_opts->foldertype == OF_USERMBOX) 
      strcpy(mrg_opts->userfolder, cmd_opts->userfolder);
    else
      ;
  }
  else if (sp_node != NULL && sp_node->s.localfolder != (char *) 0) 
    if (strcmp(sp_node->s.localfolder,"-") == 0)
      mrg_opts->foldertype = OF_STDOUT;
    else {
      mrg_opts->foldertype = OF_USERMBOX;
      strncpy(mrg_opts->userfolder, sp_node->s.localfolder, FOLDERLEN - 1);
    }
  else
    mrg_opts->foldertype = OF_SYSMBOX;
    
  /* Determine what the remote folder will be */
  if (*(cmd_opts->remotefolder) != '\0')
    strcpy(mrg_opts->remotefolder, cmd_opts->remotefolder);
  else if (sp_node != NULL && sp_node->s.remotefolder != (char *) 0)
    strncpy(mrg_opts->remotefolder, sp_node->s.remotefolder, FOLDERLEN - 1);
  else
    ;  /* mrg_opts->remotefolder is a null string */

}

