------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                 M A K E                                  --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.9 $                              --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with ALI;           use ALI;
with Csets;
with Debug;
with Fname;         use Fname;
with Namet;         use Namet;
with Opt;
with Osint;         use Osint;
with GNAT.OS_Lib;   use GNAT.OS_Lib;
with Gnatvsn;
with Output;        use Output;
with Table;
with Types;         use Types;

package body Make is

   use Ascii;
   --  Make control characters visible

   -------------------------------------
   -- Queue (Q) Manipulation Routines --
   -------------------------------------

   --  The Q is used in Compile_Sources below. Its implementation uses the
   --  GNAT generic package Table (basically an extensible array).  Q_Front
   --  points to the first valid element in the Q, whereas Q.First is the first
   --  element ever enqueued, while Q.Last - 1 is the last element in the Q.
   --
   --        +---+--------------+---+---+---+-----------+---+--------
   --    Q   |   |  ........    |   |   |   | .......   |   |
   --        +---+--------------+---+---+---+-----------+---+--------
   --          ^                  ^                       ^
   --       Q.First             Q_Front               Q.Last - 1
   --
   --  The elements comprised between Q.First and Q_Front - 1 are the
   --  elements that have been enqueued and then dequeued, while the
   --  elements between Q_Front and Q.Last - 1 are the elements currently
   --  in the Q. When the Q is intialized Q_Front = Q.First = Q.Last.
   --  After Compile_Sources has terminated its execution, Q_Front = Q.Last
   --  and the elements contained between Q.Front and Q.Last-1 are those that
   --  were explored and thus marked by Compile_Sources.  Whenever the Q is
   --  reinitialized, the elements between Q.First and Q.Last - 1 are unmarked.

   procedure Init_Q;
   --  Must be called to (re)initialize the Q.

   procedure Insert_Q (Source_File : File_Name_Type);
   --  Inserts Source_File at the end of Q.

   function Empty_Q return Boolean;
   --  Returns True if Q is empty.

   function Extract_From_Q return File_Name_Type;
   --  Extracts the first element from the Q.

   First_Q_Initialization : Boolean := True;
   --  Will be set to false after Init_Q has been called once.

   Q_Front : Natural;
   --  Points to the first valid element in the Q.

   package Q is new Table (
     Table_Component_Type => File_Name_Type,
     Table_Index_Type     => Natural,
     Table_Low_Bound      => 0,
     Table_Initial        => 4000,
     Table_Increment      => 100,
     Table_Name           => "Make.Q");
   --  This is the actual Q.

   ----------------------
   -- Marking Routines --
   ----------------------

   procedure Mark (Source_File : File_Name_Type);
   --  Mark Source_File. Marking is used to signal that Source_File has
   --  already been inserted in the Q.

   function Is_Marked (Source_File : File_Name_Type) return Boolean;
   --  Returns True if Source_File was previously marked.

   procedure Unmark (Source_File : File_Name_Type);
   --  Unmarks Source_File.

   ----------------------------------------------------
   -- Compiler, Binder & Linker Variables & Routines --
   ----------------------------------------------------

   Gcc      : constant String := "gcc";
   Gnatbind : constant String := "gnatbind";
   Gnatbl   : constant String := "gnatbl";

   Path      : constant String_Access := Getenv ("PATH");

   Comp_Flag : constant String_Access := new String'("-c");
   GNAT_Flag : constant String_Access := new String'("-gnatg");
   Link_Flag : constant String_Access := new String'("-linkonly");

   Display_Executed_Programs : Boolean := True;
   --  Set to True if name of commands should be output on stderr.

   function Execute (Program : String; Args : Argument_List) return Boolean;
   --  Executes Program.  Args contains the arguments to be passed to
   --  Program.  If the program is executed successfully True is returned.

   -------------------
   -- Misc Routines --
   -------------------

   function Full_Name (N : File_Name_Type) return Name_Id;
   --  Returns the full name of the file whose simple name is N. If the file
   --  cannot be located N is returned. The full name includes the
   --  appropriate directory information.

   ----------
   -- Bind --
   ----------

   procedure Bind (Ali_File : File_Name_Type; Args : Argument_List) is
      Bind_Args : Argument_List (Args'First .. Args'Last + 1);

   begin
      Bind_Args (Args'Range) := Args;

      Get_Name_String (Ali_File);
      Bind_Args (Args'Last + 1) := new String'(Name_Buffer (1 .. Name_Len));

      if not Execute (Gnatbind, Bind_Args) then
         raise Bind_Failed;
      end if;
   end Bind;

   -------------
   -- Compile --
   -------------

   procedure Compile (Source_File : File_Name_Type; Args : Argument_List) is
      Comp_Args : Argument_List (Args'First .. Args'Last + 3);
      Comp_Last : Integer := Args'Last + 1;

   begin
      Comp_Args (Args'First) := Comp_Flag;
      Comp_Args (Args'First + 1 .. Comp_Last) := Args;

      if Is_Predefined_File_Name (Source_File) then
         Comp_Last := Comp_Last + 1;
         Comp_Args (Comp_Last) := GNAT_Flag;
      end if;

      Comp_Last := Comp_Last + 1;
      Get_Name_String (Source_File);
      Comp_Args (Comp_Last) := new String'(Name_Buffer (1 .. Name_Len));

      if not Execute (Gcc, Comp_Args (Args'First .. Comp_Last)) then
         raise Compilation_Failed;
      end if;
   end Compile;

   ---------------------
   -- Compile_Sources --
   ---------------------

   procedure Compile_Sources
     (Main_Source          : File_Name_Type;
      Args                 : Argument_List;
      First_Compiled_File  : out Name_Id;
      Main_Unit            : out Boolean;
      Check_Internal_Files : Boolean := False;
      Dont_Execute         : Boolean := False;
      Force_Compilations   : Boolean := False;
      Verbose_Mode         : Boolean := False)
   is

      procedure Debug_Msg (S : String; N : Name_Id);
      --  If Debug.Debug_Flag_W is set outputs string S followed by name N.

      function First_New_Spec (A : ALI_Id) return File_Name_Type;
      --  Looks in the with table entries of A and returns the spec file name
      --  of the first withed unit (subprogram) for which no spec existed when
      --  A was generated but for which there exists one now, implying that A
      --  is now obsolete. If no such unit is found No_File is returned.
      --  Otherwise the spec file name of the unit is returned.
      --
      --  **WARNING** in the event of Uname format modifications, one *MUST*
      --  make sure this function is also updated.
      --
      --  This function should really be in ali.adb and use Uname services, but
      --  this causes the whole compiler to be dragged along from gnatbind and
      --  gnatmake.

      ---------------
      -- Debug_Msg --
      ---------------

      procedure Debug_Msg (S : String; N : Name_Id) is
      begin
         if Debug.Debug_Flag_W then
            Write_Str ("   ... ");
            Write_Str (S);
            Write_Name (N);
            Write_Eol;
         end if;
      end Debug_Msg;

      --------------------
      -- First_New_Spec --
      --------------------

      function First_New_Spec (A : ALI_Id) return File_Name_Type is

         Spec_File_Name : File_Name_Type := No_File;

         function New_Spec (Uname : Unit_Name_Type) return Boolean;
         --  Uname is the name of the spec or body of some ada unit.
         --  This function returns True if the Uname is the name of a body
         --  which has a spec not mentioned in ali file A. If True is returned
         --  Spec_File_Name above is set to the name of this spec file.

         function New_Spec (Uname : Unit_Name_Type) return Boolean is
            Spec_Name : Unit_Name_Type;
            File_Name : File_Name_Type;

         begin
            --  Test whether Uname is the name of a body unit (ie ends with %b)

            Get_Name_String (Uname);
            pragma
              Assert (Name_Len > 2 and then Name_Buffer (Name_Len - 1) = '%');

            if Name_Buffer (Name_Len) /= 'b' then
               return False;
            end if;

            --  Convert unit name into spec name

            Name_Buffer (Name_Len) := 's';
            Spec_Name := Name_Find;
            File_Name := Get_File_Name (Spec_Name);

            --  Look if File_Name is mentioned in A's sdep list.
            --  If not look if the file exists. If it does return True.

            for D in
              ALIs.Table (A).First_Sdep .. ALIs.Table (A).Last_Sdep
            loop
               if Sdep.Table (D).Sfile = File_Name then
                  return False;
               end if;
            end loop;

            if Full_Source_Name (File_Name) /= No_File then
               Spec_File_Name := File_Name;
               return True;
            end if;

            return False;
         end New_Spec;

      --  Start of processing for First_New_Spec

      begin
         U_Chk : for U in
           ALIs.Table (A).First_Unit .. ALIs.Table (A).Last_Unit
         loop
            exit U_Chk when New_Spec (Unit.Table (U).Uname);

            for W in Unit.Table (U).First_With .. Unit.Table (U).Last_With loop
               exit U_Chk when
                 Withs.Table (W).Afile /= No_File
                 and then New_Spec (Withs.Table (W).Uname);
            end loop;
         end loop U_Chk;

         return Spec_File_Name;
      end First_New_Spec;

      --  Compile_Sources Variables

      Source_File : File_Name_Type;
      --  Current source file

      Full_Source_File : File_Name_Type;
      --  Full name of the current source file

      Lib_File : File_Name_Type;
      --  Current library file

      Obj_File : File_Name_Type;
      --  Current object file

      Sfile : File_Name_Type;
      --  Contains, in turn, the source file of the units withed by Source_File

      Modified_Source : File_Name_Type;
      --  The first source in Lib_File whose current time stamp differs
      --  from that stored in Lib_File.

      New_Spec : File_Name_Type;
      --  If Lib_File contains in its W (with) section a body (for a
      --  subprogram) for which there exists a spec and the spec did not
      --  appear in the Sdep section of Lib_File, New_Spec contains the file
      --  name of this new spec.

      Lib_Stamp : Time_Stamp_Type;
      --  Time stamp of the current ada library file.

      Obj_Stamp : Time_Stamp_Type;
      --  Time stamp of the current object file.

      Processing_Main_Source : Boolean;
      Source_Not_Found       : Boolean;
      Need_To_Compile        : Boolean;

      Text : Text_Buffer_Ptr;
      Ali  : ALI_Id;

   --  Start of Processing for Compile_Sources

   begin
      --  Package and Queue initializations.

      Output.Set_Standard_Error;
      Initialize_ALI;
      Init_Q;

      --  The following two flags affects the behavior of Ali.Set_Source_Table.
      --  We set Opt.Check_Source_Files to True to ensure that source file
      --  time stamps are checked, and we set Opt.All_Sources to False to
      --  avoid checking the presence of the source files listed in the
      --  source dependency section of an ali file (which would be a mistake
      --  since the ali file may be obsolete).

      Opt.Check_Source_Files := True;
      Opt.All_Sources := False;

      Insert_Q (Main_Source);
      Mark (Main_Source);

      Processing_Main_Source := True;
      First_Compiled_File    := No_Name;
      Main_Unit              := False;

      Make_Loop : while not Empty_Q loop
         Need_To_Compile  := False;
         Source_Not_Found := False;

         Source_File      := Extract_From_Q;
         Full_Source_File := Osint.Full_Source_Name (Source_File);

         if Full_Source_File = No_Name then
            Source_Not_Found := True;
            Full_Source_File := Source_File;
         end if;

         if Verbose_Mode then
            Write_Str ("Checking --> ");
            Write_Name (Full_Source_File);
            Write_Eol;
         end if;

         Lib_File := Lib_File_Name (Source_File);

         if Source_Not_Found then
            if Belongs_To_Ada_Library (Lib_File) then
               Debug_Msg ("Skipping missing source because ali in Ada library",
                          Source_File);

               --  Skip remaining processing and execute next loop iteration

               goto Continue;

            else
               Osint.Write_Program_Name;
               Write_Str (": """);
               Write_Name (Source_File);
               Write_Str (""" file not found");
               Write_Eol;

               raise Compilation_Failed;
            end if;
         end if;

         Text := Read_Library_Info (Lib_File);

         if Text = null then
            Need_To_Compile := True;

            if Verbose_Mode then
               Lib_Stamp := Library_File_Stamp (Lib_File);

               if Lib_Stamp (Lib_Stamp'First) = ' ' then
                  Write_Str ("  """);
                  Write_Name (Lib_File);
                  Write_Str (""" missing.  **Recompile**");
                  Write_Eol;

               else
                  Obj_File  := Full_Object_File_Name;
                  Obj_Stamp := Library_File_Stamp (Obj_File);

                  if Obj_Stamp (Obj_Stamp'First) = ' ' then
                     Write_Str ("  """);
                     Write_Name (Obj_File);
                     Write_Str (""" missing.  **Recompile**");
                     Write_Eol;

                  else
                     Write_Str ("  """);
                     Write_Name (Lib_File);
                     Write_Str (""" (");
                     Write_Str (Lib_Stamp);
                     Write_Str (") is more recent than");
                     Write_Eol;
                     Write_Str ("  """);
                     Write_Name (Obj_File);
                     Write_Str (""" (");
                     Write_Str (Obj_Stamp);
                     Write_Str (").  **Recompile**");
                     Write_Eol;
                  end if;

               end if;
            end if;

         else
            Ali := Scan_ALI (Lib_File, Text);

            Set_Source_Table (Ali);
            --  get the source files and their time stamps. Note that some
            --  sources may be missing if Ali is out-of-date.

            Modified_Source := Time_Stamp_Mismatch (Ali);

            if Modified_Source /= No_File then
               Need_To_Compile := True;

               if Verbose_Mode then
                  Write_Str ("  """);
                  Write_Name (Full_Name (Modified_Source));
                  Write_Str (""" time stamp mismatch.  **Recompile**");
                  Write_Eol;
               end if;

            else
               New_Spec := First_New_Spec (Ali);

               if New_Spec /= No_File then
                  Need_To_Compile := True;

                  if Verbose_Mode then
                     Write_Str ("  """);
                     Write_Name (Full_Name (New_Spec));
                     Write_Str (""" new spec.  **Recompile**");
                     Write_Eol;
                  end if;
               end if;
            end if;
         end if;

         if Need_To_Compile or Force_Compilations then
            if First_Compiled_File = No_Name then
               First_Compiled_File := Full_Source_File;
            end if;

            if Dont_Execute then
               Main_Unit := False;
               return;
            end if;

            Compile (Full_Source_File, Args);

            --  If we get here the compilation succeded (or else the
            --  exception Compiltion_Failed would have been raised).

            --  Compile_Sources takes care not to put generics in the Q.
            --  The main source is the only case when we could compile
            --  a generic. Thus after compiling the main source we need
            --  to check whether the corresponding library file exists.
            --  If it doesn't the main source must be a generic.

            if Processing_Main_Source then
               Lib_Stamp := Library_File_Stamp (Lib_File);

               if Lib_Stamp (Lib_Stamp'First) = ' ' then
                  return;
               end if;
            end if;

            --  Otherwise re-read the updated library file.

            Text := Read_Library_Info (Lib_File, Fatal_Err => True);
            Ali  := Scan_ALI (Lib_File, Text);
         end if;

         if Processing_Main_Source then
            Processing_Main_Source := False;
            Main_Unit := ALIs.Table (Ali).Main_Program /= None;
         end if;

         --  Now insert in the Q the unmarked source files (i.e. those which
         --  have neever been inserted in the Q and hance never considered).

         for J in
           ALIs.Table (Ali).First_Unit .. ALIs.Table (Ali).Last_Unit
         loop
            for K in Unit.Table (J).First_With .. Unit.Table (J).Last_With loop
               Sfile := Withs.Table (K).Sfile;

               if Sfile = No_File then
                  Debug_Msg ("Skipping generic: ", Withs.Table (K).Uname);

               elsif Is_Marked (Sfile) then
                  Debug_Msg ("Skipping marked file: ", Sfile);

               elsif not Check_Internal_Files
                 and then Is_Predefined_File_Name (Sfile)
               then
                  Debug_Msg ("Skipping language defined file: ", Sfile);

               else
                  Insert_Q (Sfile);
                  Mark (Sfile);

               end if;
            end loop;
         end loop;

         <<Continue>>
         null;
      end loop Make_Loop;
   end Compile_Sources;

   ----------------------
   -- Display_Commands --
   ----------------------

   procedure Display_Commands (Display : Boolean := True) is
   begin
      Display_Executed_Programs := Display;
   end Display_Commands;

   -------------
   -- Empty_Q --
   -------------

   function Empty_Q return Boolean is
   begin
      if Debug.Debug_Flag_P then
         Write_Str ("   Q := [");

         for J in Q_Front .. Q.Last - 1 loop
            Write_Str (" ");
            Write_Name (Q.Table (J));
            Write_Eol;
            Write_Str ("         ");
         end loop;

         Write_Str ("]");
         Write_Eol;
      end if;

      return Q_Front >= Q.Last;
   end Empty_Q;

   --------------------
   -- Extract_From_Q --
   --------------------

   function Extract_From_Q return File_Name_Type is
      Elmt : constant File_Name_Type := Q.Table (Q_Front);

   begin
      if Debug.Debug_Flag_Q then
         Write_Str ("   Q := Q - [ ");
         Write_Name (Elmt);
         Write_Str (" ]");
         Write_Eol;
      end if;

      Q_Front := Q_Front + 1;
      return Elmt;
   end Extract_From_Q;

   -------------
   -- Execute --
   -------------

   function Execute (Program : String; Args : Argument_List) return Boolean is
      Success : Boolean;
      Execute : constant String_Access :=
        GNAT.OS_Lib.Locate_Regular_File (Program, Path.all);

   begin
      if Display_Executed_Programs then
         Write_Str (Program);

         for J in Args'Range loop
            Write_Str (" ");
            Write_Str (Args (J).all);
         end loop;

         Write_Eol;
      end if;

      GNAT.OS_Lib.Spawn (Execute.all, Args, Success);

      return Success;
   end Execute;

   ---------------
   -- Full_Name --
   ---------------

   function Full_Name (N : File_Name_Type) return Name_Id is
      Name : constant Name_Id := Osint.Full_Source_Name (N);

   begin
      if Name = No_Name then
         return N;
      else
         return Name;
      end if;
   end Full_Name;

   --------------
   -- Gnatmake --
   --------------

   procedure Gnatmake is

      procedure Makeusg;
      --  Outputs gnatmake usage information.

      function To_Lower (Name : Name_Id) return Name_Id;
      --  If Name does not have upper case characters, Name is returned,
      --  otherwise this routine creates and returns a new lower case
      --  version of Name.

      -------------
      -- Makeusg --
      -------------

      procedure Makeusg is
         procedure Write_Switch_Char;
         --  Write two spaces followed by appropriate switch character

         procedure Write_Switch_Char is
         begin
            Write_Str ("  ");
            Write_Char (Switch_Character);
         end Write_Switch_Char;

      begin
         --  Usage line

         Write_Str ("Usage: ");
         Osint.Write_Program_Name;
         Write_Char (' ');
         Write_Str ("switches unit[.adb] ");
         Write_Str ("{[-cargs opts] [-bargs opts] [-largs opts]}");
         Write_Eol;
         Write_Eol;

         --  Line for -a

         Write_Switch_Char;
         Write_Str ("a          Consider all files, even GNAT internal files");
         Write_Eol;

         --  Line for -c

         Write_Switch_Char;
         Write_Str ("c          Compile only, do not bind and link");
         Write_Eol;

         --  Line for -f

         Write_Switch_Char;
         Write_Str ("f          Force recompilations of non predefined units");
         Write_Eol;

         --  Line for -g

         Write_Switch_Char;
         Write_Str ("g          Compile with debugging information");
         Write_Eol;

         --  Line for -n

         Write_Switch_Char;
         Write_Str ("n          Just output the commands, don't execute them");
         Write_Eol;

         --  Line for -q

         Write_Switch_Char;
         Write_Str ("q          Be quiet, do not display executed commands");
         Write_Eol;

         --  Line for -s

         Write_Switch_Char;
         Write_Str ("s          Perform smart recompilations");
         Write_Eol;

         --  Line for -v

         Write_Switch_Char;
         Write_Str ("v          Motivate all (re)compilations");
         Write_Eol;
         Write_Eol;

         --  Line for -A

         Write_Switch_Char;
         Write_Str ("Adir       Skip missing library sources if ali in dir");
         Write_Eol;

         --  Line for -I

         Write_Switch_Char;
         Write_Str ("Idir       Look for source and ali files also in dir");
         Write_Eol;

         --  Line for -L

         Write_Switch_Char;
         Write_Str ("Ldir       Look for program libraries also in dir");
         Write_Eol;
         Write_Eol;

         --  Line for unit[.adb]

         Write_Str ("  unit[.adb]  Compilation unit name or source file");
         Write_Eol;
         Write_Eol;

         --  Line for -cargs

         Write_Switch_Char;
         Write_Str ("cargs opts Arguments to be passed to the compiler");
         Write_Eol;

         --  Line for -bargs

         Write_Switch_Char;
         Write_Str ("bargs opts Arguments to be passed to the binder");
         Write_Eol;

         --  Line for -largs

         Write_Switch_Char;
         Write_Str ("largs opts Arguments to be passed to the linker");
         Write_Eol;

         Write_Eol;
      end Makeusg;

      --------------
      -- To_Lower --
      --------------

      function To_Lower (Name : Name_Id) return Name_Id is
      begin
         Get_Name_String (Name);

         for I in 1 .. Name_Len loop
            if Csets.Is_Upper_Case_Letter (Name_Buffer (I)) then
               Name_Buffer (I) := Csets.Fold_Lower (Name_Buffer (I));
            end if;
         end loop;

         return Name_Enter;
      end To_Lower;

      ------------------------
      -- Gnatmake Variables --
      ------------------------

      Main_Name : Name_Id;
      --  The name of the main compilation unit or of the source containing it

      Main_Source_File : File_Name_Type;
      --  The actual source file corresponding to Main_Name

      Is_Main_Unit : Boolean;
      --  If True the Main_Source_File can be a main unit.

      Main_Ali_File : File_Name_Type;
      --  The ali file corresponding to Main_Source_File

   --  Start of Processing for Gnatmake

   begin
      --  Default initialization of the flags affecting gnatmake

      Opt.Check_Internal_Files     := False;
      Opt.Check_Object_Consistency := True;
      Opt.Compile_Only             := False;
      Opt.Dont_Execute             := False;
      Opt.Force_Compilations       := False;
      Opt.Quiet_Output             := False;
      Opt.Smart_Compilations       := False;
      Opt.Verbose_Mode             := False;

      --  Package initializations. The order of calls is important here.

      Output.Set_Standard_Error;
      Osint.Initialize (Osint.Make); --  Reads gnatmake switches
      Csets.Initialize;
      Namet.Initialize;

      if Opt.Verbose_Mode then
         Write_Eol;
         Write_Str ("GNAT Make Version ");
         Write_Str (Gnatvsn.Gnat_Version_String);
         Write_Str (" Copyright 1995 Free Software Foundation, Inc.");
         Write_Eol;
      end if;

      --  Output usage information if more than one file or compile unit

      if Osint.Number_Of_Files = 0 then
         Makeusg;
         Exit_Program (E_Fatal);

      elsif Osint.Number_Of_Files > 1 then
         Osint.Fail ("error, only one source or compilation unit allowed.");
      end if;

      --  ??? get rid of the following when smart compilation is implemented

      if Opt.Smart_Compilations then
         Osint.Write_Program_Name;
         Write_Str (": WARNING smart recompilation not yet implemented.");
         Write_Eol;
      end if;

      --  Now check if the user input a file or compilation unit name. If it
      --  is a compilation unit name first check the existence of the source
      --  file for the compilation unit body. If the file for the body of
      --  the compilation unit does not exist try the spec.

      Main_Name := Osint.Next_Main_Source;

      if Is_File_Name (Main_Name) then
         Main_Source_File := File_Name_Type (Main_Name);

      else
         Main_Source_File := File_Name_Of_Body (To_Lower (Main_Name));

         if Full_Source_Name (Main_Source_File) = No_Name then
            Main_Source_File := File_Name_Of_Spec (To_Lower (Main_Name));

            if Full_Source_Name (Main_Source_File) = No_Name then
               Osint.Write_Program_Name;
               Write_Str (": no file found for body or spec of """);
               Write_Name (Main_Name);
               Write_Char ('"');
               Write_Eol;
               Exit_Program (E_Fatal);
            end if;
         end if;
      end if;

      --  Consider GNAT predefined files only if -a switch is set.

      if Fname.Is_Predefined_File_Name (Main_Source_File)
        and then not Opt.Check_Internal_Files
      then
         Fail ("use the -a switch to compile GNAT predefined files");
      end if;

      Display_Commands (not Opt.Quiet_Output);

      --  Here is where the make process is started

      Recursive_Compilation_Step : declare
         Args : Argument_List (Gcc_Switches.First .. Gcc_Switches.Last);
         First_Compiled_File : Name_Id;

      begin
         for I in Gcc_Switches.First .. Gcc_Switches.Last loop
            Args (I) := Gcc_Switches.Table (I);
         end loop;

         Compile_Sources (Main_Source          => Main_Source_File,
                          Args                 => Args,
                          First_Compiled_File  => First_Compiled_File,
                          Main_Unit            => Is_Main_Unit,
                          Check_Internal_Files => Opt.Check_Internal_Files,
                          Dont_Execute         => Opt.Dont_Execute,
                          Force_Compilations   => Opt.Force_Compilations,
                          Verbose_Mode         => Opt.Verbose_Mode);

         if First_Compiled_File = No_Name and then not Opt.Quiet_Output then
            Osint.Write_Program_Name;
            Write_Str (": sources up to date. No recompilations needed.");
            Write_Eol;
         end if;

         if Opt.Dont_Execute and then First_Compiled_File /= No_Name then
            Write_Name (First_Compiled_File);
            Write_Eol;
         end if;

         if Opt.Dont_Execute
           or else not Is_Main_Unit
           or else Opt.Compile_Only
         then
            return;
         end if;
      end Recursive_Compilation_Step;

      Main_Ali_File := Osint.Lib_File_Name (Main_Source_File);

      Bind_Step : declare
         Args : Argument_List (Binder_Switches.First .. Binder_Switches.Last);

      begin
         for I in Binder_Switches.First .. Binder_Switches.Last loop
            Args (I) := Binder_Switches.Table (I);
         end loop;

         Bind (Main_Ali_File, Args);
      end Bind_Step;

      Link_Step : declare
         Args : Argument_List (Linker_Switches.First .. Linker_Switches.Last);

      begin
         for I in Linker_Switches.First .. Linker_Switches.Last loop
            Args (I) := Linker_Switches.Table (I);
         end loop;

         Link (Main_Ali_File, Args);
      end Link_Step;

      Exit_Program (E_Success);

   exception
      when Bind_Failed        => Osint.Fail ("*** bind failed.");
      when Compilation_Failed => Osint.Fail ("*** compilation failed.");
      when Link_Failed        => Osint.Fail ("*** link failed.");
      when others             => Osint.Fail ("INTERNAL ERROR. Please report.");

   end Gnatmake;

   ------------
   -- Init_Q --
   ------------

   procedure Init_Q is
   begin
      if not First_Q_Initialization then
         First_Q_Initialization := False;

         --  unmark source files which were previously marked & enqueued.

         for I in Q.First .. Q.Last - 1 loop
            Unmark (Source_File => Q.Table (I));
         end loop;
      end if;

      Q_Front := Q.First;
      Q.Set_Last (Q.First);
   end Init_Q;

   --------------
   -- Insert_Q --
   --------------

   procedure Insert_Q (Source_File : File_Name_Type) is
   begin
      if Debug.Debug_Flag_Q then
         Write_Str ("   Q := Q + [ ");
         Write_Name (Source_File);
         Write_Str (" ] ");
         Write_Eol;
      end if;

      Q.Table (Q.Last) := Source_File;
      Q.Increment_Last;
   end Insert_Q;

   ---------------
   -- Is_Marked --
   ---------------

   function Is_Marked (Source_File : File_Name_Type) return Boolean is
   begin
      return Get_Name_Table_Byte (Source_File) /= 0;
   end Is_Marked;

   ----------
   -- Link --
   ----------

   procedure Link (Ali_File : File_Name_Type; Args : Argument_List) is
      Link_Args : Argument_List (Args'First .. Args'Last + 2);

   begin
      Get_Name_String (Ali_File);
      Link_Args (Args'First) := new String'(Name_Buffer (1 .. Name_Len));

      Link_Args (Args'First + 1) := Link_Flag;
      Link_Args (Args'First + 2 .. Args'Last + 2) := Args;

      if not Execute (Gnatbl, Link_Args) then
         raise Link_Failed;
      end if;
   end Link;

   ----------
   -- Mark --
   ----------

   procedure Mark (Source_File : File_Name_Type) is
   begin
      Set_Name_Table_Byte (Source_File, 1);
   end Mark;

   ------------
   -- Unmark --
   ------------

   procedure Unmark (Source_File : File_Name_Type) is
   begin
      Set_Name_Table_Byte (Source_File, 0);
   end Unmark;

end Make;
