------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 6                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.75 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Intr; use Exp_Intr;
with Exp_Util; use Exp_Util;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Disp; use Sem_Disp;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;

package body Exp_Ch6 is

   -----------------
   -- Expand_Call --
   -----------------

   --  This procedure handles expansion of function calls and procedure call
   --  statements (i.e. it serves as the body for Expand_N_Function_Call and
   --  Expand_N_Procedure_Call_Statement. Processing for calls includes:

   --    Supply default expressions for missing arguments
   --    Replace "call" to enumeration literal function by literal itself
   --    Rewrite call to predefined operator as operator
   --    Expand dispatching call (see Expand_Dispatch_Call)

   procedure Expand_Call (N : Node_Id) is
      Loc           : constant Source_Ptr := Sloc (N);
      Subp          : Entity_Id;
      Parent_Subp   : Entity_Id;
      Parent_Formal : Entity_Id;
      Actual        : Node_Id;
      Formal        : Entity_Id;
      Prev          : Node_Id := Empty;
      Scop          : Entity_Id;

      procedure Insert_Default;
      --  Internal procedure to insert argument corresponding to Formal.
      --  The value is inserted immediately after Prev, or if Prev is Empty,
      --  (case of empty argument list), then into a new list. In both cases
      --  Prev is set to the inserted default for the next call.

      procedure Insert_Default is
         Default : Node_Id;
         Insert  : Node_Id;
         F_Name  : Node_Id;

      begin
         Insert := New_Node (N_Parameter_Association, Loc);
         F_Name := New_Node (N_Identifier, Loc);

         --  Copy the complete expression tree for each default parameter.
         --  This will ensure that a new Itype is generated (if applicable)
         --  for each such insertion of the expression in the subprogram call.

         Default := New_Copy_Tree (Default_Value (Formal));
         Set_Chars (F_Name, Chars (Formal));
         Set_Explicit_Actual_Parameter (Insert, Default);
         Set_Selector_Name (Insert, F_Name);

         --  Case of insertion is first named actual

         if No (Prev) or else
            Nkind (Parent (Prev)) /= N_Parameter_Association
         then
            Set_Next_Named_Actual (Insert, First_Named_Actual (N));
            Set_First_Named_Actual (N, Default);

            if No (Prev) then
               if not Present (Parameter_Associations (N)) then
                  Set_Parameter_Associations (N, New_List);
                  Append (Insert, Parameter_Associations (N));
               end if;
            else
               Insert_After (Prev, Insert);
            end if;

         --  Case of insertion is not first named actual

         else
            Set_Next_Named_Actual (Insert, Next_Named_Actual (Parent (Prev)));
            Set_Next_Named_Actual (Parent (Prev), Default);
            Append (Insert, Parameter_Associations (N));
         end if;

         Prev := Default;
      end Insert_Default;

   --  Start of processing for Expand_Call

   begin
      --  Case of access to subprogram, where the Name is an explicit
      --  dereference. The type of the name node is a subprogram type,
      --  from which we can retrieve the required signature.

      if Nkind (Name (N)) = N_Explicit_Dereference then
         Subp := Etype (Name (N));
         Parent_Subp := Empty;

      --  Case of call to simple entry, where the Name is a selected component
      --  whose prefix is the task, and whose selector name is the entry name

      elsif Nkind (Name (N)) = N_Selected_Component then
         Subp := Entity (Selector_Name (Name (N)));
         Parent_Subp := Empty;

      --  Case of call to member of entry family, where Name is an indexed
      --  component, with the prefix being a selected component giving the
      --  task and entry family name, and the index being the entry index.

      elsif Nkind (Name (N)) = N_Indexed_Component then
         Subp := Entity (Selector_Name (Prefix (Name (N))));
         Parent_Subp := Empty;

      --  Normal case

      else
         Subp := Entity (Name (N));
         Parent_Subp := Alias (Subp);

         if Ekind (Subp) = E_Entry then
            Parent_Subp := Empty;
         end if;
      end if;

      --  First step, insert default parameter values

      Formal := First_Formal (Subp);
      Actual := First_Actual (N);

      while Present (Formal) loop
         if Present (Actual) then

            --  Check for named and positional parameters in proper place

            if Nkind (Parent (Actual)) /= N_Parameter_Association
              or else Chars (Selector_Name (Parent (Actual))) = Chars (Formal)
            then
               Prev   := Actual;
               Actual := Next_Actual (Actual);
            else
               Insert_Default;
            end if;

         --  Trailing actuals are all defaults

         else
            Insert_Default;
         end if;

         Formal := Next_Formal (Formal);
      end loop;

      if Nkind (N) /= N_Entry_Call_Statement
        and then No (Controlling_Argument (N))
        and then Present (Parent_Subp)
      then
         while Present (Alias (Parent_Subp)) loop
            Parent_Subp := Alias (Parent_Subp);
         end loop;

         --  Expand an explicit conversion for parameter of the inherited type

         Formal := First_Formal (Subp);
         Parent_Formal := First_Formal (Parent_Subp);
         Actual := First_Actual (N);
         while Present (Formal) loop
            if (Etype (Formal) /= Etype (Parent_Formal)) then
               Replace_Substitute_Tree (Actual,
                 Make_Unchecked_Type_Conversion (Sloc (Actual),
                   Subtype_Mark =>
                     New_Occurrence_Of (Etype (Parent_Formal), Sloc (Actual)),
                   Expression   => Relocate_Node (Actual)));
               Set_Etype (Actual, Etype (Parent_Formal));
            end if;

            Formal := Next_Formal (Formal);
            Parent_Formal := Next_Formal (Parent_Formal);
            Actual := Next_Actual (Actual);
         end loop;

         Set_Entity (Name (N), Parent_Subp);
         Subp := Parent_Subp;
      end if;

      --  Some more special cases for cases other than explicit dereference

      if Nkind (Name (N)) /= N_Explicit_Dereference then

         --  Calls to an enumeration literal are replaced by the literal
         --  The only way that this case occurs is when we have a call to
         --  a function that is a renaming of an enumeration literal. The
         --  normal case of a direct reference to an enumeration literal
         --  has already been dealt with by Resolve_Call

         if Ekind (Subp) = E_Enumeration_Literal then
            Replace_Substitute_Tree (N, Name (N));
         end if;
      end if;

      --  Deals with Dispatch_Call if we still have a call

      if (Nkind (N) = N_Function_Call
           or else Nkind (N) =  N_Procedure_Call_Statement)
        and then Present (Controlling_Argument (N))
      then
         Expand_Dispatch_Call (N);
      end if;

      --  Create a transient scope if the result is uncontrained or tagged

      if Nkind (N) = N_Function_Call
        and then (Is_Unconstrained (N) or else Is_Tagged_Type (Etype (N)))
      then
         Establish_Transient_Scope (N);
         if Is_Class_Wide_Type (Etype (N)) then
            Error_Msg_N (
              "? calls returning class-wide objects not fully supported",
              N);
         end if;
      end if;

      --  If this is a call to an intrinsic subprogram, then perform the
      --  appropriate expansion to the corresponding tree node.

      if Is_Intrinsic_Subprogram (Subp) then
         Expand_Intrinsic_Call (N, Subp);
      end if;

      --  Check for a protected subprogram.  This should be an intra-object
      --  call; inter-object calls are not represented as subprogram calls.
      --  Expand the call into a call to the unprotected expansion of the
      --  subprogram.

      --  ??? This will probably not work when entry bodies or external
      --  function calls are implemented.

      Scop := Scope (Subp);

      if Nkind (N) /= N_Entry_Call_Statement and then
        Is_Protected_Type (Scop)
      then
         declare
            Param : Entity_Id;
            Rec   : Node_Id;

         begin
            Rec := Make_Identifier (Loc, Name_uObject);
            Set_Etype (Rec, Corresponding_Record_Type (Scop));

            Scop := Current_Scope;
            while not Is_Protected_Type (Scope (Scop)) loop
               Scop := Scope (Scop);
            end loop;

            Param := Defining_Identifier
              (First
                (Parameter_Specifications
                  (Parent
                    (Corresponding_Unprotected (Scop)))));
            Set_Entity (Rec, Param);
            Set_Analyzed (Rec);
            Rewrite_Substitute_Tree (N,
              Build_Protected_Subprogram_Call (N,
                Name => Name (N),
                Rec => Rec,
                External => False));
         end;

         Analyze (N);
      end if;

   end Expand_Call;

   --------------------------
   -- Expand_Dispatch_Call --
   --------------------------

   --  If Suppress_Tag_Checks are not on, some tag equality tests are
   --  performed before the call. For example, for the call:

   --    F (Tagged_Arg1, Arg2, Tagged_Arg3)

   --  where Tagged_Arg1 is the controlling argument, and Tagged_Arg3 is an
   --  argument that is required to have the same tag, we first generate:

   --    if Tagged_Type (Tagged_Arg1)._Tag /=
   --       Tagged_Type (Tagged_Arg3)._Tag
   --    then
   --       raise Constraint_Error;
   --    end if;

   --  where Tagged_Type is the root type. Then the actual call is:
   --      Acc_Dt (Tagged_Type (Ctrl_Arg)._Tag.all).F___n.all (parameters)

   procedure Expand_Dispatch_Call (Call_Node : Node_Id) is
      Arg           : constant Node_Id   := Controlling_Argument (Call_Node);
      Subp          : constant Entity_Id := Entity (Name (Call_Node));
      Param_List    : constant List_Id   := Parameter_Associations (Call_Node);
      Sloc_N        : constant Source_Ptr := Sloc (Call_Node);
      Prim_Ops      : Elist_Id;
      Prim_Op       : Elmt_Id;
      Actual        : Node_Id;
      Pos_In_Dt     : Natural;
      Tagged_Type   : Entity_Id;
      Ctrl_Value    : Node_Id;
      Tag_Ctrl      : Entity_Id;
      Tag_Actl      : Entity_Id;
      Prim_Ctrl     : Entity_Id;
      New_Actions   : List_Id := New_List;
      New_Call_Name : Node_Id;
      New_Call      : Node_Id;

   begin
      --  Expand_Dispatch is called directly from the semantics, so we need
      --  a check to see whether expansion is active before proceeding

      if not Expander_Active then
         return;
      end if;

      --  Definition of the Tagged Type

      if Is_Access_Type (Etype (Arg)) then
         Tagged_Type := Etype (Designated_Type (Etype (Arg)));
         Ctrl_Value  :=
           Make_Explicit_Dereference (Sloc_N, Prefix => New_Copy (Arg));

      else
         Tagged_Type := Etype (Etype (Arg));
         Ctrl_Value  := Arg;
      end if;

      Prim_Ops := Primitive_Operations (Tagged_Type);

      --  Compute the position in the dispatch table

      Prim_Op := First_Elmt (Prim_Ops);
      Pos_In_Dt := 1;

      while Node (Prim_Op) /= Subp loop
         Pos_In_Dt := Pos_In_Dt + 1;
         Prim_Op := Next_Elmt (Prim_Op);

         pragma Assert (Present (Prim_Op));
      end loop;

      --  Generate code for testing equality of tags among tagged actuals
      --  and convert class-wide actuals into the tagged type

      if Present (Param_List) then

         --  Get the tag component of the Tagged_Type

         Tag_Ctrl := Tag_Component (Tagged_Type);
         pragma Assert (Present (Tag_Ctrl));
         Actual := First_Actual (Call_Node);

         while Present (Actual) loop
            if not Tag_Checks_Suppressed (Etype (Actual))
              and then Present (Find_Controlling_Arg (Actual))
              and then Find_Controlling_Arg (Actual) /= Arg

            --  "=" is the only dispatching operation allowed to get
            --  operands with incompatible tags (it just returns false)

              and then Chars (Subp) /= Name_Op_Eq
            then

               --  Get the tag component of the Etype (Actual)

               Tag_Actl := Tag_Component (Etype (Actual));
               pragma Assert (Present (Tag_Actl));

               --  Generate code for tag equality check

               Append_To (New_Actions,
                 Make_If_Statement (Sloc_N,
                   Condition =>
                     Make_Op_Ne (Sloc_N,
                       Left_Opnd =>
                         Make_Selected_Component (Sloc_N,
                           Prefix =>
                             Make_Type_Conversion (Sloc_N,
                               Subtype_Mark =>
                                 New_Occurrence_Of (Tagged_Type, Sloc_N),
                               Expression => New_Copy (Ctrl_Value)),
                           Selector_Name =>
                             New_Reference_To (Tag_Ctrl, Sloc_N)),

                       Right_Opnd =>
                         Make_Selected_Component (Sloc_N,
                           Prefix =>
                             Make_Type_Conversion (Sloc_N,
                               Subtype_Mark =>
                                 New_Occurrence_Of (Tagged_Type, Sloc_N),
                               Expression => New_Copy (Actual)),
                           Selector_Name =>
                             New_Reference_To (Tag_Actl, Sloc_N))),

                   Then_Statements =>
                     New_List (New_Constraint_Error (Sloc_N))));
            end if;

            Actual := Next_Actual (Actual);
         end loop;
      end if;

      --  Generate the call itself by calling the right function in the
      --  dispatch table:

      --    Acc_Dt (Tagged_Type (Ctrl_Value)._Tag.all).Prim_{Pos_IN_Dt + 2}

      New_Call_Name :=
        Make_Selected_Component (Sloc_N,
          Prefix => Make_DT_Access (Sloc_N, Ctrl_Value, Tagged_Type),
          Selector_Name =>
            Make_DT_Component (Sloc_N, Tagged_Type, Pos_In_Dt + 2));

      if Is_Empty_List (New_Actions) then

         --  Replace subprogram call by the dispatching call

         if Nkind (Call_Node) = N_Function_Call then
            New_Call :=
              Make_Function_Call (Sloc_N,
                Name => New_Call_Name,
                Parameter_Associations => New_List_Copy (Param_List));

         else
            New_Call :=
              Make_Procedure_Call_Statement (Sloc_N,
                Name => New_Call_Name,
                Parameter_Associations => New_List_Copy (Param_List));

         end if;

      else
         --  If function call then generate expression_actions node whose
         --  actions are the if statements generated and expression is
         --  the dispatching call.

         if Nkind (Call_Node) = N_Function_Call then
            New_Call :=
              Make_Expression_Actions (Sloc_N,
                Actions => New_Actions,
                Expression =>
                  Make_Function_Call (Sloc_N,
                    Name => New_Call_Name,
                    Parameter_Associations => New_List_Copy (Param_List)));

         --  Else procedure call then analyze the if statements created,
         --  insert them before the subprogram call and create the
         --  dispatching call with a procedure call.

         else
            Insert_List_Before_And_Analyze (Call_Node, New_Actions);

            New_Call :=
              Make_Procedure_Call_Statement (Sloc_N,
                Name => New_Call_Name,
                Parameter_Associations => New_List_Copy (Param_List));
         end if;
      end if;

      Rewrite_Substitute_Tree (Call_Node, New_Call);
      Analyze (Call_Node);

      --  Resolution of Call_Node is not needed.
      --  It has already be done for the original node.

   end Expand_Dispatch_Call;

   ----------------------------
   -- Expand_N_Function_Call --
   ----------------------------

   procedure Expand_N_Function_Call (N : Node_Id) is
   begin
      Expand_Call (N);
   end Expand_N_Function_Call;

   ---------------------------------------
   -- Expand_N_Procedure_Call_Statement --
   ---------------------------------------

   procedure Expand_N_Procedure_Call_Statement (N : Node_Id) is
   begin
      Expand_Call (N);
   end Expand_N_Procedure_Call_Statement;

   ------------------------------
   -- Expand_N_Subprogram_Body --
   ------------------------------

   --  Add return statement if last statement in body is not a return
   --  statement (this makes things easier on Gigi which does not want
   --  to have to handle a missing return).

   --  Add call to Activate_Tasks if body is a task activator

   procedure Expand_N_Subprogram_Body (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      H        : constant Node_Id    := Handled_Statement_Sequence (N);
      Body_Ent : Entity_Id;
      Spec_Ent : Entity_Id;
      Except_H : Node_Id;
      Scop     : Entity_Id;
      Pdef     : Node_Id;
      Next_Sub : Node_Id;

      procedure Add_Termination (S : List_Id);
      --  Append to S a return statement in the procedure case or a Raise
      --  Program_Error in the function case if the last statement is not
      --  already a return or a goto statement.

      procedure Add_Termination (S : List_Id) is
         Last_S : constant Node_Id := Last (S);
         Loc_S  : constant Source_Ptr := Sloc (Last_S);
         Rtn    : Node_Id;

      begin
         if Nkind (Last_S) /= N_Return_Statement
           and then Nkind (Last_S) /= N_Goto_Statement
           and then Nkind (Last_S) /= N_Raise_Statement
         then
            if Ekind (Spec_Ent) = E_Procedure then
               Append_To (S, Make_Return_Statement (Loc_S));

            elsif Ekind (Spec_Ent) = E_Function then
               Append_To (S,
                 Make_Raise_Statement (Loc_S,
                   Name => New_Occurrence_Of (Standard_Program_Error, Loc_S)));
            end if;
         end if;
      end Add_Termination;

      procedure Set_New_Discriminals (Typ : Node_Id; Loc : Source_Ptr);
      --  Replace discriminals in a protected type for use by the
      --  next protected operation on the type.  Each operation needs a
      --  new set of discirminals, since it needs a unique renaming of
      --  the discriminant fields in the record used to implement the
      --  protected type.

      procedure Set_New_Discriminals (Typ : Node_Id; Loc : Source_Ptr) is
         D       : Entity_Id;
         D_Minal : Entity_Id;

      begin
         if Has_Discriminants (Typ) then
            D := First_Discriminant (Typ);

            while Present (D) loop

               D_Minal :=
                 Make_Defining_Identifier
                   (Loc, New_External_Name (Chars (D), 'D'));
               Set_Discriminal (D, D_Minal);

               D := Next_Discriminant (D);
            end loop;
         end if;
      end Set_New_Discriminals;

   --  Start of processing for Expand_N_Subprogram_Body

   begin
      --  Get entities for subprogram body and spec

      Body_Ent := Defining_Unit_Simple_Name (Specification (N));

      if Present (Corresponding_Spec (N)) then
         Spec_Ent := Corresponding_Spec (N);
      else
         Spec_Ent := Body_Ent;
      end if;

      --  Now, add a termination for all possible syntactic ends of the
      --  subprogram.  We don't bother to reanalyze the new body with the added
      --  return statement, since it would involve a lot of unnecessary work
      --  that would achieve precisely nothing.

      Add_Termination (Statements (H));

      if Present (Exception_Handlers (H)) then
         Except_H := First (Exception_Handlers (H));

         while Present (Except_H) loop
            Add_Termination (Statements (Except_H));
            Except_H := Next (Except_H);
         end loop;
      end if;

      Scop := Scope (Body_Ent);

      --  Add discriminal renamings to protected subprograms.
      --  Install new discriminals for expansion of the next
      --  subprogram of this protected type, if any.

      if Present (Scop) and then Is_Protected_Type (Scop) then
         Add_Discriminal_Declarations
           (Declarations (N), Scop, Name_uObject, Loc);
         Set_New_Discriminals (Base_Type (Scop), Loc);

         Pdef := Protected_Definition (Parent (Base_Type (Scop)));
         Add_Private_Declarations
           (Declarations (N), Pdef, Name_uObject, Loc);

         --  Associate privals with the next subprogram body to be expanded.
         --  These are used to expand references to private data objects.

         Next_Sub := Next (N);
         while Present (Next_Sub)
           and then Nkind (Next_Sub) /= N_Subprogram_Body
         loop
            Next_Sub := Next (Next_Sub);
         end loop;
         if Present (Next_Sub) then
            Set_Privals (Pdef,
              Defining_Unit_Name (Specification (Next_Sub)), Loc);
         end if;

      end if;

   end Expand_N_Subprogram_Body;

end Exp_Ch6;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.73
--  date: Tue Aug 16 20:14:30 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  revision 1.74
--  date: Wed Aug 24 23:25:59 1994;  author: giering
--  (Expand_Call): Expanded intraobject protected calls into calls to the
--   expanded unprotected version of the subprogram.
--  ----------------------------
--  revision 1.75
--  date: Thu Aug 25 01:04:34 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
