/*
 * man.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.  
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Some manpath, compression and locale related changes - aeb - 940320
 */

#define MAN_MAIN

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>
#include <signal.h>
#include "gripes.h"
#include "version.h"
#include "config.h"
#include "man.h"

#ifdef POSIX
#include <unistd.h>
#else
#ifndef R_OK
#define R_OK 4
#endif
#endif

#ifdef SECURE_MAN_UID
extern uid_t getuid ();
extern int setuid ();
#endif

#ifdef STDC_HEADERS
#include <stdlib.h>
#else
extern char *getenv ();
extern void free ();
extern int system ();
extern int strcmp ();
extern int strncmp ();
extern int exit ();
extern int fflush ();
extern int printf ();
extern FILE *fopen ();
extern int fclose ();
extern char *sprintf ();
#endif

extern char *my_malloc ();
extern char *get_expander ();
extern char *getval ();
extern char *convert_to_cat ();
extern char **glob_vector ();
extern char **glob_filename ();
extern int access ();
extern int unlink ();
extern int system ();
extern int chmod ();
extern int is_newer ();
extern int is_directory ();
extern int do_system_command ();

char *progname;
char *pager;
char *colon_sep_section_list;
char *roff_directive;
int apropos;
int whatis;
int findall;
int print_where;
int do_troff;
int debug;
struct dir srchlist;

static char *section;
static char **section_list;

static int have_roff = 1;

#ifdef DO_COMPRESS
int do_compress = 1;
#else
int do_compress = 0;
#endif

int alt_systems = 1;

int
main (argc, argv)
     int argc;
     char **argv;
{
  int status = 0;
  char *nextarg;
  char *tmp;
  extern int optind;
  extern char *mkprogname ();
  extern void prmanpath();
  char *is_section ();
  char **get_section_list ();
  void man_getopt ();
  void do_apropos ();
  void do_whatis ();
  int man ();

  progname = mkprogname (argv[0]);

  man_getopt (argc, argv);

  if (!strcmp (progname, "manpath") || (optind == argc && print_where)) {
      prmanpath();
      exit(0);
  }

  if (optind == argc)
    gripe(NO_NAME_NO_SECTION);

  section_list = get_section_list ();

  if (optind == argc - 1)
    {
      tmp = is_section (argv[optind]);

      if (tmp != NULL)
	gripe1 (NO_NAME_FROM_SECTION, tmp);
    }

  if (do_compress && !*getval("COMPRESS")) {
      if (debug)
	gripe (NO_COMPRESS);
      do_compress = 0;
  }

  while (optind < argc) {
      nextarg = argv[optind++];

      /*
       * See if this argument is a valid section name.  If not,
       * is_section returns NULL.
       */
      tmp = is_section (nextarg);

      if (tmp != NULL) {
	  section = tmp;

	  if (debug)
	    gripe1 (SECTION, section);

	  continue;
      }

      if (apropos)
	do_apropos (nextarg);
      else if (whatis)
	do_whatis (nextarg);
      else {
	  status = man (nextarg);

	  if (status == 0) {
	      if (section)
		gripe2 (NO_SUCH_ENTRY_IN_SECTION, nextarg, section);
	      else
		gripe1 (NO_SUCH_ENTRY, nextarg);
	  }
      }
  }
  return status;
}

void
usage ()
{
  gripe2 (VERSION, progname, version);
  gripe1 (USAGE1, progname);

  if (alt_systems)
    gripe (USAGE2);

  gripe (USAGE3);
  gripe (USAGE4);
  gripe (USAGE5);    /* maybe only if troff found? */
  gripe (USAGE6);

  if (alt_systems)
    gripe (USAGE7);

  gripe (USAGE8);
  exit(1);
}

/*
 * Check to see if the argument is a valid section number.  If the
 * first character of name is a numeral, or the name matches one of
 * the sections listed in section_list, we'll assume that it's a section.
 * The list of sections in config.h simply allows us to specify oddly
 * named directories like .../man3f.  Yuk. 
 */
char *
is_section (name)
     register char *name;
{
  register char **vs;

  for (vs = section_list; *vs != NULL; vs++)
    if ((strcmp (*vs, name) == 0)
	|| (isdigit (name[0]) && !isdigit (name[1])))
      return strdup (name);

  return NULL;
}

/*
 * Handle the apropos option.  Cheat by using another program.
 */
void
do_apropos (name)
     register char *name;
{
  register int len;
  register char *command;

  len = strlen (getval("APROPOS")) + strlen (name) + 2;

  command = my_malloc(len);

  sprintf (command, "%s %s", getval("APROPOS"), name);

  (void) do_system_command (command);

  free (command);
}

/*
 * Handle the whatis option.  Cheat by using another program.
 */
void
do_whatis (name)
     register char *name;
{
  register int len;
  register char *command;

  len = strlen (getval("WHATIS")) + strlen (name) + 2;

  command = my_malloc(len);

  sprintf (command, "%s %s", getval("WHATIS"), name);

  (void) do_system_command (command);

  free (command);
}

remove_file (file)
     char *file;
{
  int i;

  i = unlink (file);

  if (debug) {
      if (i)
	perror(file);
      else
	gripe1 (UNLINKED, file);
  }
}

remove_other_catfiles (catfile)
     char *catfile;
{
  char pathname[BUFSIZ];
  char *t;
  char **gf;
  int i, offset;

  strcpy (pathname, catfile);
  t = rindex(pathname, '.');
  if (t == NULL || strcmp(t, getval("COMPRESS_EXT")))
    return;
  offset = t - pathname;
  strcpy(t, "*");
  gf = glob_filename (pathname);

  if (gf != (char **) -1 && gf != NULL) {
      for ( ; *gf; gf++) {
	  /*
	   * Only remove files with a known extension, like .Z
	   * (otherwise we might kill a lot when called with
	   * catfile = ".gz" ...)
	   */
	  if (strlen (*gf) <= offset) {
	      if (strlen (*gf) == offset)  /* uncompressed version */
		  remove_file (*gf);
	      continue;
	  }

	  if (!strcmp (*gf + offset, getval("COMPRESS_EXT")))
	    continue;

	  if (get_expander (*gf) != NULL)
	    remove_file (*gf);
      }
  }
}

/*
 * Try to find the man page corresponding to the given name.  The
 * reason we do this with globbing is because some systems have man
 * page directories named man3 which contain files with names like
 * XtPopup.3Xt.  Rather than requiring that this program know about
 * all those possible names, we simply try to match things like
 * .../man[sect]/name[sect]*.  This is *much* easier.
 */
char **
glob_for_file (path, section, name, cat)
     register char *path;
     register char *section;
     register char *name;
     register int cat;
{
  char pathname[BUFSIZ];
  char **gf;

  if (cat)
    sprintf (pathname, "%s/cat%s/%s.%s*", path, section, name, section);
  else
    sprintf (pathname, "%s/man%s/%s.%s*", path, section, name, section);

  if (debug)
    gripe1 (GLOBBING, pathname);

  gf = glob_filename (pathname);

  if ((gf == (char **) -1 || gf == NULL || *gf == NULL) && isdigit (*section))
    {
      if (cat)
	sprintf (pathname, "%s/cat%s/%s.%c*", path, section, name, *section);
      else
	sprintf (pathname, "%s/man%s/%s.%c*", path, section, name, *section);

      gf = glob_filename (pathname);
    }

  if (gf == (char **) -1 || gf == NULL || *gf == NULL)
    {
      if (cat)
	sprintf (pathname, "%s/cat%s/%s.man", path, section, name);
      else
	sprintf (pathname, "%s/man%s/%s.man", path, section, name);

      gf = glob_filename (pathname);
    }

  if (gf == (char **) -1)
    gf = NULL;
  return gf;
}

/*
 * Simply display the preformatted page.
 */
int
display_cat_file (file)
     register char *file;
{
  register int found;
  char command[BUFSIZ];

  found = 0;

  if (access (file, R_OK) == 0)
    {
      char *expander = get_expander (file);

      if (expander != NULL && expander[0] != 0)
	sprintf (command, "%s %s | %s", expander, file, pager);
      else
	sprintf (command, "%s %s", pager, file);

      found = do_system_command (command);
    }
  return found;
}

/*
 * Try to find the ultimate source file.  If the first line of the
 * current file is not of the form
 *
 *      .so man3/printf.3s
 *
 * the input file name is returned.
 */
char *
ultimate_source (name0)
     char *name0;
{
  FILE *fp;
  char *name;
  char *expander;
  int expfl = 0;
  char *fgr;
  char *beg;
  char *end;
  char *cp;
  char buf[BUFSIZ];

/* TAMU FIX  -- this has to be static */
  static char ultname[BUFSIZ];

  name = name0;

 again:
  expander = get_expander (name);
  if (expander && *expander) {
      char command[BUFSIZ];

      sprintf (command, "%s %s", expander, name);
      fp = popen (command, "r");
      if (fp == NULL) {
	  perror("popen");
	  gripe1 (EXPANSION_FAILED, command);
	  return (NULL);
      }
      fgr = fgets (buf, BUFSIZ, fp);
      pclose (fp);
      expfl = 1;
  } else {
      fp = fopen (name, "r");
      if (fp == NULL && expfl) {
	  char *extp = rindex (name0, '.');
	  if (extp && *extp) {
	      strcat(name, extp);
	      fp = fopen (name, "r");
	  }
      }
      if (fp == NULL) {
	  perror("fopen");
	  gripe1 (OPEN_ERROR, name);
	  return (NULL);
      }
      fgr = fgets (buf, BUFSIZ, fp);
      fclose (fp);
  }

  if (fgr == NULL) {
      perror("fgets");
      gripe1 (READ_ERROR, name);
      return (NULL);
  }

  if (strncmp(buf, ".so", 3))
    return (name);

  beg = buf+3;
  while (*beg == ' ' || *beg == '\t')
    beg++;

  end = beg;
  while (*end != ' ' && *end != '\t' && *end != '\n' && *end != '\0')
    end++;

  *end = '\0';

  if (name != ultname) {
      strcpy(ultname, name);
      name = ultname;
  }

  /* If name ends in path/manx/foo.9x then use path, otherwise
     try same directory. */
  if ((cp = rindex(name, '/')) == NULL) /* very strange ... */
    return 0;
  *cp = 0;
  if((cp = rindex(name, '/')) != NULL && !strncmp(cp+1, "man", 3))
    strcpy(cp+1, beg);
  else if((cp = rindex(beg, '/')) != NULL)
    strcat(name, cp);
  else {
      strcat(name, "/");
      strcat(name, beg);
  }

  goto again;
}

static void
add_directive (d, file, buf)
     char *d;
     char *file;
     char *buf;
{
  if ((d = getval(d)) != 0 && *d)
    {
      if (*buf == 0)
	{
	  strcpy (buf, d);
	  strcat (buf, " ");
	  strcat (buf, file);
	}
      else
	{
	  strcat (buf, " | ");
	  strcat (buf, d);
	}
    }
}

static int
parse_roff_directive (cp, file, buf)
  char *cp;
  char *file;
  char *buf;
{
  char c;
  int tbl_found = 0;
  char *expander;

  while ((c = *cp++) != '\0')
    {
      switch (c)
	{
	case 'e':

	  if (debug)
	    gripe (FOUND_EQN);
	  add_directive ((do_troff ? "EQN" : "NEQN"), file, buf);
	  break;

	case 'g':

	  if (debug)
	    gripe (FOUND_GRAP);
	  add_directive ("GRAP", file, buf);
	  break;

	case 'p':

	  if (debug)
	    gripe (FOUND_PIC);
	  add_directive ("PIC", file, buf);
	  break;

	case 't':

	  if (debug)
	    gripe (FOUND_TBL);
	  tbl_found++;
	  add_directive ("TBL", file, buf);
	  break;

	case 'v':

	  if (debug)
	    gripe (FOUND_VGRIND);
	  add_directive ("VGRIND", file, buf);
	  break;

	case 'r':

	  if (debug)
	    gripe (FOUND_REFER);
	  add_directive ("REFER", file, buf);
	  break;

	case ' ':
	case '\t':
	case '\n':

	  goto done;

	default:

	  return -1;
	}
    }

 done:

  if (*buf == 0)
    return 1;

  strcat (buf, " | ");
  strcat (buf, getval(do_troff ? "TROFF" : "NROFF"));

  if (tbl_found && !do_troff && *getval("COL")) {
      strcat (buf, " | ");
      strcat (buf, getval("COL"));
  }

  return 0;
}

char *
make_roff_command (file)
     char *file;
{
  FILE *fp;
  char line [BUFSIZ];
  static char buf [BUFSIZ];
  int status;
  char *cp, *tblpath, *colpath, *roffpath, *expander, *fgr;

  expander = get_expander (file);
  if (expander && *expander)
    sprintf (buf, "%s %s", expander, file);
  else
    buf[0] = 0;

  if (roff_directive != NULL) {
      if (debug)
	gripe (ROFF_FROM_COMMAND_LINE);

      status = parse_roff_directive (roff_directive, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe (ROFF_CMD_FROM_COMMANDLINE_ERROR);
  }

  if (expander && *expander) {
      char command[BUFSIZ];

      sprintf (command, "%s %s", expander, file);
      fp = popen (command, "r");
      if (fp == NULL) {
	  perror("popen");
	  gripe1 (EXPANSION_FAILED, command);
	  return (NULL);
      }
      fgr = fgets (line, BUFSIZ, fp);
      pclose (fp);
  } else {
      fp = fopen (file, "r");
      if (fp == NULL) {
	  perror("fopen");
	  gripe1 (OPEN_ERROR, file);
	  return (NULL);
      }
      fgr = fgets (line, BUFSIZ, fp);
      fclose (fp);
  }

  if (fgr == NULL) {
      perror("fgets");
      gripe1 (READ_ERROR, file);
      return (NULL);
  }

  cp = &line[0];
  if (*cp++ == '\'' && *cp++ == '\\' && *cp++ == '"' && *cp++ == ' ') {
      if (debug)
	gripe1 (ROFF_FROM_FILE, file);

      status = parse_roff_directive (cp, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe1 (ROFF_CMD_FROM_FILE_ERROR, file);
  }

  if ((cp = getenv ("MANROFFSEQ")) != NULL) {
      if (debug)
	gripe (ROFF_FROM_ENV);

      status = parse_roff_directive (cp, file, buf);

      if (status == 0)
	return buf;

      if (status == -1)
	gripe (MANROFFSEQ_ERROR);
  }

  if (debug)
    gripe (USING_DEFAULT);

  (void) parse_roff_directive ("t", file, buf);

  return buf;
}

/*
 * Try to format the man page and create a new formatted file.  Return
 * 1 for success and 0 for failure.
 */
static int
make_cat_file (path, man_file, cat_file)
     register char *path;
     register char *man_file;
     register char *cat_file;
{
  int status;
  int mode;
  FILE *fp;
  char *roff_command;
  char command[BUFSIZ];

  if ((fp = fopen (cat_file, "w")) != NULL)
    {
      fclose (fp);
      unlink (cat_file);

      roff_command = make_roff_command (man_file, 0);
      if (roff_command == NULL)
	return 0;
      else if (do_compress)
	  /* The cd is necessary, because of .so commands,
	     like .so man1/bash.1 in bash_builtins.1.
	     But it changes the meaning of man_file and cat_file,
	     if these are not absolute. */
	
	sprintf (command, "(cd %s ; %s | %s > %s)", path,
		 roff_command, getval("COMPRESS"), cat_file);
      else
        sprintf (command, "(cd %s ; %s > %s)", path,
		 roff_command, cat_file);

      /*
       * Don't let the user interrupt the system () call and screw up
       * the formmatted man page if we're not done yet.
       */
      signal (SIGINT, SIG_IGN);

      gripe (PLEASE_WAIT);

      status = do_system_command (command);

      if (status == 1) {
	  mode = CATMODE;
	  chmod (cat_file, mode);

	  if (debug)
	    gripe2i (CHANGED_MODE, cat_file, mode);
      }

      signal (SIGINT, SIG_DFL);

      return 1;
    }
  else
    {
      if (debug)
	gripe1 (CAT_OPEN_ERROR, cat_file);

      return 0;
    }
}

static int
display_man_file(path, man_file)
     char *path, *man_file;
{
  char *roff_command;
  char command[BUFSIZ];

  roff_command = make_roff_command (man_file, 0);
  if (roff_command == NULL)
    return 0;
  else
    sprintf (command, "(cd %s ; %s | %s)", path,
	     roff_command, pager);

  return do_system_command (command);
}

/*
 * Try to format the man page source and save it, then display it.  If
 * that's not possible, try to format the man page source and display
 * it directly.
 */
static int
format_and_display (path, man_file)
     register char *path;
     register char *man_file;
{
  int status;
  register int found;
  char *roff_command;
  char *cat_file;
  char command[BUFSIZ];

  found = 0;

  if (access (man_file, R_OK) != 0)
    return 0;

  /* first test for contents  .so man1/xyzzy.1  */
  /* (in that case we do not want to make a cat file identical
     to cat1/xyzzy.1) */
  man_file = ultimate_source (man_file);
  if (man_file == NULL)
    return 0;

  if (do_troff) {
      roff_command = make_roff_command (man_file, 1);
      if (roff_command == NULL)
	return 0;
      else
	sprintf (command, "(cd %s ; %s)", path, roff_command);

      return do_system_command (command);
  } else if ((cat_file = convert_to_cat (man_file)) != NULL) {

      if (debug)
	gripe1 (PROPOSED_CATFILE, cat_file);

      /*
       * If cat_file exists, check whether it is more recent.
       * Otherwise, check for other cat files (maybe there are
       * old .Z files that should be removed).
       */

      status = is_newer (man_file, cat_file);
      if (debug)
	gripe1i (IS_NEWER_RESULT, status);
      if (status == -1 || status == -3) {
	  /* what? man_file does not exist anymore? */
	  gripe1 (CANNOT_STAT, man_file);
	  return(0);
      }

      if (status != 0) {
	  /*
	   * Cat file is out of date (status = 1) or does not exist
	   * (status = -2).  Try to format and save it.
	   */
	  if (print_where) {
	      printf ("%s\n", man_file);
	      found++;
	  } else {
	      found = make_cat_file (path, man_file, cat_file);

#ifdef SECURE_MAN_UID
	      if (!found) {
		  /*
		   * Try again as real user.  Note that for private
		   * man pages, we won't even get this far unless the
		   * effective user can read the real user's man page
		   * source.  Also, if we are trying to find all the
		   * man pages, this will probably make it impossible
		   * to make cat files in the system directories if
		   * the real user's man directories are searched
		   * first, because there's no way to undo this (is
		   * there?).  Yikes, am I missing something obvious?
		   */
		  setuid (getuid ());

		  found = make_cat_file (path, man_file, cat_file);
	      }
#endif
	      if (found) {
		  /*
		   * Creating the cat file worked.  Now just display it.
		   */
		  (void) display_cat_file (cat_file);

		  /*
		   * If we just created this cat file, unlink any others.
		   */
		  if (status == -2 && do_compress)
		    remove_other_catfiles(cat_file);
	      } else {
		  /*
		   * Couldn't create cat file.  Just format it and
		   * display it through the pager. 
		   */
		  found = display_man_file (path, man_file);
	      }
	  }
      }
      else if (access (cat_file, R_OK) == 0) {
	  /*
	   * Formatting not necessary.  Cat file is newer than source
	   * file, or source file is not present but cat file is.
	   */
	  if (print_where) {
	      printf ("%s (<-- %s)\n", cat_file, man_file);
	      found++;
	  } else {
	      found = display_cat_file (cat_file);
	  }
      }
  } else {
      /* could not construct name of cat_file */
      if (print_where) {
	  printf ("%s\n", man_file);
	  found++;
      } else {
	  found = display_man_file (path, man_file);
      }
  }
  return found;
}

/*
 * See if the preformatted man page or the source exists in the given
 * section.
 */
int
try_section (path, section, name)
     register char *path;
     register char *section;
     register char *name;
{
  register int found = 0;
  register int to_cat;
  register int cat;
  register char **names;
  register char **np;

  if (debug)
    gripe1 (TRYING_SECTION, section);

  names = NULL;

  if (*getval("NROFF")) {
    /*
     * Look for man page source files.
     */
      cat = 0;
      names = glob_for_file (path, section, name, cat);
  }

  if (names == NULL || *names == NULL) {
      /*
       * No files match.  See if there's a preformatted page around
       * that we can display. 
       */

      if (!do_troff) {
	  cat = 1;
	  names = glob_for_file (path, section, name, cat);

	  if (names != NULL) {
	      for (np = names; *np != NULL; np++) {
		  if (print_where) {
		      printf ("%s\n", *np);
		      found++;
		  } else {
		      found += display_cat_file (*np);
		  }
	      }
	  }
      }
  } else {
      for (np = names; *np != NULL; np++) {
	  found += format_and_display (path, *np);
      }
  }

  return found;
}

/*
 * Search for manual pages.
 *
 * If preformatted manual pages are supported, look for the formatted
 * file first, then the man page source file.  If they both exist and
 * the man page source file is newer, or only the source file exists,
 * try to reformat it and write the results in the cat directory.  If
 * it is not possible to write the cat file, simply format and display
 * the man file.
 *
 * If preformatted pages are not supported, or the troff option is
 * being used, only look for the man page source file.
 *
 * Note that globbing is necessary also if the section is given,
 * since a preformatted man page might be compressed.
 *
 */
int
man (name)
     char *name;
{
  register int found;
  register char **sp;
  FILE *fp;

  found = 0;

  /* allow  man ./manpage  for formatting explicitly given man pages */
  if (index(name, '/') && (fp = fopen(name, "r"))) {
      char fullname[BUFSIZ];
      char fullpath[BUFSIZ];
      char *path;
      char *cp;

      fclose (fp);
      if (*name != '/' && getcwd(fullname, sizeof(fullname))) {
	  strcat (fullname, "/");
	  strcat (fullname, name);
      } else
	strcpy (fullname, name);

      strcpy (fullpath, fullname);
      if ((cp = rindex(fullpath, '/')) != NULL) {
	  strcpy(cp+1, "..");
	  path = fullpath;
      } else
	path = ".";

      name = ultimate_source (fullname);
      
      return display_man_file (path, name);
  }

  fflush (stdout);
  if (section != NULL)
    found = man_from_section (section, name);
  else {
      for (sp = section_list; *sp != NULL; sp++) {
	  found += man_from_section (*sp, name);
	  if (found && !findall)
	    break;
      }
  }
  return found;
}

int
man_from_section (sec, name)
     char *sec, *name;
{
  register int found;
  register struct dir *dp;

  found = 0;

  for (dp = srchlist.nxt; dp; dp = dp->nxt) {
      if (debug)
	gripe1 (SEARCHING, dp->dir);

      found += try_section (dp->dir, sec, name);

      if (found && !findall)   /* i.e. only do this section... */
	return found;
  }
  return found;
}

char **
get_section_list ()
{
  int i;
  char *p;
  char *end;
  static char *tmp_section_list[100];

  if (colon_sep_section_list == NULL)
    {
      if ((p = getenv ("MANSECT")) == NULL)
	{
	  return std_sections;
	}
      else
	{
	  colon_sep_section_list = strdup (p);
	}
    }

  i = 0;
  for (p = colon_sep_section_list; ; p = end+1)
    {
      if ((end = strchr (p, ':')) != NULL)
	*end = '\0';

      tmp_section_list[i++] = strdup (p);

      if (end == NULL || i+1 == sizeof(tmp_section_list))
	break;
    }

  tmp_section_list [i] = NULL;
  return tmp_section_list;
}
