#include <OI/oi.H>

/*
 *	translations - using C++ member functions as callbacks
 *
 *	This program demonstrates the use of the translation mechanism
 *	to add callbacks to Mouse events.  This program should be
 *	executed as:
 *		./transM -config trans.ad
 *	Otherwise the appropriate mouse actions will not be set.
 *
 *	The reader should refer back to the README file for
 *	other translations examples.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *			OI_add_actions()
 *		- OI_d_tech
 *			layout_associated_object()
 *			set_associated_object()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_static_text
 *			oi_create_static_text()
 *			disallow_cut_paste()
 */


/*
 *	My_static_text subclass
 *
 *	contains data items deltaX and deltaY,
 *	contains the press, drag, and release member functions.
 */
class MY_static_text : public OI_static_text {
	protected:
		int	deltaX;
		int	deltaY;
	public:
			MY_static_text(char *nam, char *txt);
			~MY_static_text();
		void	press(OI_d_tech*, const XEvent*, const char **, unsigned int*);
		void	drag(OI_d_tech*, const XEvent*, const char **, unsigned int*);
		void	release(OI_d_tech*, const XEvent*, const char **, unsigned int*);
};
MY_static_text::~MY_static_text() { }


/*
 *	The OI_actions_rec structure allows the user to specify
 *	either C-functions, or C++ member functions, to be called
 *	when the specified mouse event occurs.
 */
static OI_actions_rec myactions[] = {
	{"press",	NULL,	NULL, (OI_translation_memfnp)&MY_static_text::press	},
	{"drag",	NULL,	NULL, (OI_translation_memfnp)&MY_static_text::drag	},
	{"release",	NULL,	NULL, (OI_translation_memfnp)&MY_static_text::release	},
};


/*
 *	My_static text constructor.
 *	simply adds the specified translations.
 */
MY_static_text::MY_static_text(char *nam, char *txt) : OI_static_text(nam, txt)
{
	push_actions(myactions, OI_count(myactions));
}


void
main (int argc, char **argv)
{
	OI_connection	*conp;				/* the connection to the server */
	OI_app_window	*appWindow;			/* the enclosing app window */
	MY_static_text	*staticText;			/* the text label */

	
	/*
	 *	Open a connection.
	 */
	if ((conp = OI_init(&argc, argv, "actions", "Actions"))) {

		/*
		 *	create application window.
		 */
		appWindow = oi_create_app_window("myapp", 200, 200, "C++ Translations");

		/*
		 *	create static text object.
		 *	disallow cut and paste, to allow the user to select the object with
		 *		the mouse.
		 *	set the static text object into the application window.
		 */
		staticText = new MY_static_text("label", " Drag Me ");
		staticText->disallow_cut_paste();
		staticText->set_associated_object(appWindow, 5, 5, OI_ACTIVE);

		/*
		 *	Display application window.
		 *	Begin interaction.
		 */
		appWindow->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}


/*
 *	on ButtonPress
 *		- initialize deltaX and deltaY
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as deltaX and deltaY.
 *		- refresh the object color.
 */
void
MY_static_text::press(OI_d_tech *, const XEvent *eventp, const char **, unsigned int *)
{
	deltaX = deltaY = 0;
	if (eventp->type == ButtonPress) {
		deltaX = eventp->xbutton.x;
		deltaY = eventp->xbutton.y;
	}
	set_pixels(fg_pixel(), bkg_pixel(), bdr_pixel());
}


/*
 *	on MotionNotify (drag event)
 *		- capture the event that caused this procedure to be called.
 *		- process all MotionNotify events in the queue (less choppy movement)
 *		- change the object's location from the old location to the new location
 */
void
MY_static_text::drag(OI_d_tech *, const XEvent *eventp, const char **, unsigned int *)
{
	XEvent event;
	if (eventp->type == MotionNotify) {
		event = *eventp;
		while (XCheckTypedEvent(display(), MotionNotify, &event));
		set_loc(event.xmotion.x+loc_x()-deltaX, event.xmotion.y+loc_y()-deltaY);
	}
}


/*
 *	on ButtonRelease
 *	refresh the object color.
 */
void
MY_static_text::release(OI_d_tech *, const XEvent *, const char **, unsigned int *)
{
	set_pixels(fg_pixel(), bkg_pixel(), bdr_pixel());
}
