/* Directory routines
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "global.h"
#include "dir.h"
#include "util.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/dir.c,v 1.1.1.1 1994/05/06 19:14:19 miguel Exp $";

/* If true show files starting with a dot */
int show_dot_files = 1;

/* If true show files ending in ~ */
int show_backups = 0;

int unsorted (const void *a, const void *b)
{
    return 0;
}

int sort_name (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd)
	return strcmp (a->fname, b->fname);
    else
	return bd-ad;
}

int sort_ext (const file_entry *a, const file_entry *b)
{
    char *exta, *extb;
    int r;
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd){
	exta = extension (a->fname);
	extb = extension (b->fname);
	r = strcmp (exta, extb);
	if (r)
	    return r;
	else
	    return sort_name (a, b);
    } else
	return bd-ad;
}

int sort_time (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd)
	return a->buf.st_mtime - b->buf.st_mtime;
    else
	return bd-ad;
}

int sort_size (const file_entry *a, const file_entry *b)
{
    int ad = S_ISDIR (a->buf.st_mode) ? 1 : 0;
    int bd = S_ISDIR (b->buf.st_mode) ? 1 : 0;

    if (ad == bd)
	return b->buf.st_size - a->buf.st_size;
    else
	return bd-ad;
}

void do_sort (dir_list *list, sortfn *sort, int top)
{
    qsort (&(*list) [1], top, sizeof (file_entry), sort);
}

void clean_dir (dir_list *list, int count)
{
    int i;
    
    for (i = 0; i < count; i++)
	if ((*list) [i].fname)
	    free ((*list) [i].fname);
}

int do_load_dir(dir_list *list, sortfn *sort)
{
    DIR           *dirp;
    struct dirent *dp;
    file_entry    tmp_entry;
    int           next_free = 0;

    dirp = opendir (".");
    
    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	if (!show_dot_files){
	    if (dp->d_name [0] == '.'){
		if (!(dp->d_name [1] == 0))
		    if (!(dp->d_name [1] == '.' && dp->d_namlen == 2))
			continue;
	    }
	}
	if (!show_backups && dp->d_name [dp->d_namlen-1] == '~')
	    continue;
	(*list) [next_free].fnamelen = dp->d_namlen;
	(*list) [next_free].fname = strdup (dp->d_name);
	(*list) [next_free].f.marked = 0;
	lstat (dp->d_name, &((*list) [next_free].buf));
	if (next_free == MAXFILES){
	    printf ("Too many files!\n");
	    break;
	}
	next_free++;
    }
    /* swap the . and .. entries */
    tmp_entry = (*list) [1];
    (*list) [1] = (*list) [0];
    (*list) [0] = tmp_entry;
    do_sort (list, sort, next_free-1);
    closedir (dirp);
    return next_free;
}

int link_isdir (file_entry *file)
{
    struct stat b;
    
    if (S_ISLNK (file->buf.st_mode)){
	stat (file->fname, &b);
	if (S_ISDIR (b.st_mode))
	    return 1;
    }
    return 0;
}

static dir_list dir_copy;

int do_reload_dir (dir_list *list, sortfn *sort, int count)
{
    DIR           *dirp;
    struct dirent *dp;
    file_entry    tmp_entry;
    int           next_free = 0;
    int           i, found;

    dirp = opendir (".");

    for (i = 0; i < count; i++){
	dir_copy [i].fnamelen = (*list) [i].fnamelen;
	dir_copy [i].fname =    (*list) [i].fname;
	dir_copy [i].f.marked = (*list) [i].f.marked;
    }

    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	if (!show_dot_files){
	    if (dp->d_name [0] == '.')
		if (!(dp->d_name [1] == 0))
		    if (!(dp->d_name [1] == '.' && dp->d_namlen == 2))
			continue;
	}
	if (!show_backups && dp->d_name [dp->d_namlen-1] == '~')
	    continue;

	for (found = i = 0; i < count; i++)
	    if (dp->d_namlen == dir_copy [i].fnamelen
		&& !strcmp (dp->d_name, dir_copy [i].fname)){
		(*list) [next_free].f.marked = dir_copy [i].f.marked;
		found = 1;
	    }
	
	if (!found)
	    (*list) [next_free].f.marked = 0;
	
	(*list) [next_free].fnamelen = dp->d_namlen;
	(*list) [next_free].fname = strdup (dp->d_name);
	lstat (dp->d_name, &((*list) [next_free].buf));
	if (next_free == MAXFILES-1){
	    printf ("Too many files!\n");
	    break;
	}
	next_free++;
    }
    /* swap the . and .. entries */
    tmp_entry = (*list) [1];
    (*list) [1] = (*list) [0];
    (*list) [0] = tmp_entry;
    do_sort (list, sort, next_free-1);
    clean_dir (&dir_copy, count);
    return next_free;
}
