/*
 *  linux/kernel/vm86.c
 *
 *  Copyright (C) 1994  Linus Torvalds
 */
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/string.h>
#include <linux/ptrace.h>

#include <asm/segment.h>
#include <asm/io.h>


/* Kernel MODULES interface - begin */
#include <linux/module.h>
#include "release.h"
extern void (*handle_vm86_fault_func)(struct vm86_regs *, long);


int
init_module(void)
{
   MOD_INC_USE_COUNT;
   handle_vm86_fault_func = handle_vm86_fault;
   return 0;
}

void
cleanup_module( void)
{
    int i;
    handle_vm86_fault_func = NULL;

    /*
     * wait arround to be sure no process is still in the module
     */
    for (i = 0; i < 100; i++) { 
      schedule();
    }
    /* now the memory can be freed */
    MOD_DEC_USE_COUNT;
}

/*
 * end of kernel modules interface.
 */
 
/*
 * Known problems:
 *
 * Interrupt handling is not guaranteed:
 * - a real x86 will disable all interrupts for one instruction
 *   after a "mov ss,xx" to make stack handling atomic even without
 *   the 'lss' instruction. We can't guarantee this in v86 mode,
 *   as the next instruction might result in a page fault or similar.
 * - a real x86 will have interrupts disabled for one instruction
 *   past the 'sti' that enables them. We don't bother with all the
 *   details yet..
 *
 * Hopefully these problems do not actually matter for anything.
 */

/*
 * 8- and 16-bit register defines..
 */
#define AL(regs)	(((unsigned char *)&((regs)->eax))[0])
#define AH(regs)	(((unsigned char *)&((regs)->eax))[1])
#define IP(regs)	(*(unsigned short *)&((regs)->eip))
#define SP(regs)	(*(unsigned short *)&((regs)->esp))

/*
 * virtual flags (16 and 32-bit versions)
 */
#define VFLAGS	(*(unsigned short *)&(current->v86flags))
#define VEFLAGS	(current->v86flags)

#define set_flags(X,new,mask) \
((X) = ((X) & ~(mask)) | ((new) & (mask)))

#define SAFE_MASK	(0xDD5)
#define RETURN_MASK	(0xDFF)

asmlinkage struct pt_regs * save_v86_state(struct vm86_regs * regs)
{
	unsigned long tmp;

	if (!current->vm86_info) {
		printk("no vm86_info: BAD\n");
		do_exit(SIGSEGV);
	}
	set_flags(regs->eflags, VEFLAGS, VIF_MASK | current->v86mask);
	memcpy_tofs(&current->vm86_info->regs,regs,sizeof(*regs));
	put_fs_long(current->screen_bitmap,&current->vm86_info->screen_bitmap);
	tmp = current->tss.esp0;
	current->tss.esp0 = current->saved_kernel_stack;
	current->saved_kernel_stack = 0;
	return (struct pt_regs *) tmp;
}

static inline void return_to_32bit(struct vm86_regs * regs16, int retval)
{
	struct pt_regs * regs32;

	regs32 = save_v86_state(regs16);
	regs32->eax = retval;
	__asm__ __volatile__("movl %0,%%esp\n\t"
		"jmp ret_from_sys_call"
		: : "r" (regs32));
}

static inline void set_IF(struct vm86_regs * regs)
{
	VEFLAGS |= VIF_MASK;
	if (VEFLAGS & VIP_MASK)
		return_to_32bit(regs, VM86_STI);
}

static inline void clear_IF(struct vm86_regs * regs)
{
	VEFLAGS &= ~VIF_MASK;
}

static inline void clear_TF(struct vm86_regs * regs)
{
	regs->eflags &= ~TF_MASK;
	VEFLAGS &= ~TF_MASK;
}

static inline void set_vflags_long(unsigned long eflags, struct vm86_regs * regs)
{
	set_flags(VEFLAGS, eflags, current->v86mask);
	set_flags(regs->eflags, eflags, SAFE_MASK);
	if (eflags & IF_MASK)
		set_IF(regs);
}

static inline void set_vflags_short(unsigned short flags, struct vm86_regs * regs)
{
	set_flags(VFLAGS, flags, current->v86mask);
	set_flags(regs->eflags, flags, SAFE_MASK);
	if (flags & IF_MASK)
		set_IF(regs);
}

static inline unsigned long get_vflags(struct vm86_regs * regs)
{
	unsigned long flags = regs->eflags & RETURN_MASK;

	if (VEFLAGS & VIF_MASK)
		flags |= IF_MASK;
	return flags | (VEFLAGS & current->v86mask);
}

static inline int is_revectored(int nr, struct revectored_struct * bitmap)
{
	__asm__ __volatile__("btl %2,%%fs:%1\n\tsbbl %0,%0"
		:"=r" (nr)
		:"m" (*bitmap),"r" (nr));
	return nr;
}

/*
 * Boy are these ugly, but we need to do the correct 16-bit arithmetic.
 * Gcc makes a mess of it, so we do it inline and use non-obvious calling
 * conventions..
 */
#define pushb(base, ptr, val) \
__asm__ __volatile__( \
	"decw %w0\n\t" \
	"movb %2,%%fs:0(%1,%0)" \
	: "=r" (ptr) \
	: "r" (base), "q" (val), "0" (ptr))

#define pushw(base, ptr, val) \
__asm__ __volatile__( \
	"decw %w0\n\t" \
	"movb %h2,%%fs:0(%1,%0)\n\t" \
	"decw %w0\n\t" \
	"movb %b2,%%fs:0(%1,%0)" \
	: "=r" (ptr) \
	: "r" (base), "q" (val), "0" (ptr))

#define pushl(base, ptr, val) \
__asm__ __volatile__( \
	"decw %w0\n\t" \
	"rorl $16,%2\n\t" \
	"movb %h2,%%fs:0(%1,%0)\n\t" \
	"decw %w0\n\t" \
	"movb %b2,%%fs:0(%1,%0)\n\t" \
	"decw %w0\n\t" \
	"rorl $16,%2\n\t" \
	"movb %h2,%%fs:0(%1,%0)\n\t" \
	"decw %w0\n\t" \
	"movb %b2,%%fs:0(%1,%0)" \
	: "=r" (ptr) \
	: "r" (base), "q" (val), "0" (ptr))

#define popb(base, ptr) \
({ unsigned long __res; \
__asm__ __volatile__( \
	"movb %%fs:0(%1,%0),%b2\n\t" \
	"incw %w0" \
	: "=r" (ptr), "=r" (base), "=r" (__res) \
	: "0" (ptr), "1" (base), "2" (0)); \
__res; })

#define popw(base, ptr) \
({ unsigned long __res; \
__asm__ __volatile__( \
	"movb %%fs:0(%1,%0),%b2\n\t" \
	"incw %w0\n\t" \
	"movb %%fs:0(%1,%0),%h2\n\t" \
	"incw %w0" \
	: "=r" (ptr), "=r" (base), "=r" (__res) \
	: "0" (ptr), "1" (base), "2" (0)); \
__res; })

#define popl(base, ptr) \
({ unsigned long __res; \
__asm__ __volatile__( \
	"movb %%fs:0(%1,%0),%b2\n\t" \
	"incw %w0\n\t" \
	"movb %%fs:0(%1,%0),%h2\n\t" \
	"incw %w0\n\t" \
	"rorl $16,%2\n\t" \
	"movb %%fs:0(%1,%0),%b2\n\t" \
	"incw %w0\n\t" \
	"movb %%fs:0(%1,%0),%h2\n\t" \
	"incw %w0\n\t" \
	"rorl $16,%2" \
	: "=r" (ptr), "=r" (base), "=r" (__res) \
	: "0" (ptr), "1" (base)); \
__res; })

static void do_int(struct vm86_regs *regs, int i, unsigned char * ssp, unsigned long sp)
{
	unsigned short seg = get_fs_word((void *) ((i<<2)+2));

	if (seg == BIOSSEG || regs->cs == BIOSSEG ||
	    is_revectored(i, &current->vm86_info->int_revectored))
		return_to_32bit(regs, VM86_INTx + (i << 8));
	if (i==0x21 && is_revectored(AH(regs), &current->vm86_info->int21_revectored))
		return_to_32bit(regs, VM86_INTx + (0x21 << 8));
	pushw(ssp, sp, get_vflags(regs));
	pushw(ssp, sp, regs->cs);
	pushw(ssp, sp, IP(regs));
	regs->cs = seg;
	SP(regs) -= 6;
	IP(regs) = get_fs_word((void *) (i<<2));
	clear_TF(regs);
	clear_IF(regs);
	return;
}

#ifdef VM86_DEBUG

void handle_vm86_fault(struct vm86_regs * regs, long error_code)
{
#if 0
	do_int(regs, 1, (unsigned char *) (regs->ss << 4), SP(regs));
#else
        if (current->flags & PF_PTRACED)
		current->blocked &= ~(1 << (SIGTRAP-1));
	send_sig(SIGTRAP, current, 1);
	current->tss.trap_no = 1;
	current->tss.error_code = error_code;
#endif
}

#else

void handle_vm86_fault(struct vm86_regs * regs, long error_code)
{
	unsigned char *csp, *ssp;
	unsigned long ip, sp;

	csp = (unsigned char *) (regs->cs << 4);
	ssp = (unsigned char *) (regs->ss << 4);
	sp = SP(regs);
	ip = IP(regs);

	switch (popb(csp, ip)) {

	/* operand size override */
	case 0x66:
		switch (popb(csp, ip)) {
		/* pushfd */
		case 0x9c:
			SP(regs) -= 4;
			IP(regs) += 2;
			pushl(ssp, sp, get_vflags(regs));
			return;

		/* popfd */
		case 0x9d:
			SP(regs) += 4;
			IP(regs) += 2;
			set_vflags_long(popl(ssp, sp), regs);
			return;
		}

	/* pushf */
	case 0x9c:
		SP(regs) -= 2;
		IP(regs)++;
		pushw(ssp, sp, get_vflags(regs));
		return;

	/* popf */
	case 0x9d:
		SP(regs) += 2;
		IP(regs)++;
		set_vflags_short(popw(ssp, sp), regs);
		return;

	/* int 3 */
	case 0xcc:
		IP(regs)++;
		do_int(regs, 3, ssp, sp);
		return;

	/* int xx */
	case 0xcd:
		IP(regs) += 2;
		do_int(regs, popb(csp, ip), ssp, sp);
		return;

	/* iret */
	case 0xcf:
		SP(regs) += 6;
		IP(regs) = popw(ssp, sp);
		regs->cs = popw(ssp, sp);
		set_vflags_short(popw(ssp, sp), regs);
		return;

	/* cli */
	case 0xfa:
		IP(regs)++;
		clear_IF(regs);
		return;

	/* sti */
	/*
	 * Damn. This is incorrect: the 'sti' instruction should actually
	 * enable interrupts after the /next/ instruction. Not good.
	 *
	 * Probably needs some horsing around with the TF flag. Aiee..
	 */
	case 0xfb:
		IP(regs)++;
		set_IF(regs);
		return;

	default:
		return_to_32bit(regs, VM86_UNKNOWN);
	}
}

#endif /* VM86_DEBUG */
