------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               T B U I L D                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.48 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Types; use Types;
package Tbuild is

--  This package contains various utility procedures to assist in
--  building specific types of tree nodes.

   procedure Create_Raise_Expression (N : Node_Id; Excep_Id : Entity_Id);
   --  This procedure converts an expression node into the special
   --  N_Expression_Actions node whose statments part is a single
   --  statement consisting of raise the predefined exception Excep_Id.

   procedure Initialize;
   --  Initialize data structures for Tbuild, to be called at the start of
   --  compiling each new main source file (i.e. called from Frontend). No
   --  ordering dependencies exist for this initialization call.

   function Make_DT_Component
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      I    : Positive)
      return Node_Id;
   --  Gives a reference to the Ith component of the Dispatch Table of
   --  a given Tagged Type.
   --
   --  I = 1    --> Inheritance_Depth
   --  I = 2    --> Tags (array of ancestors)
   --  I = 3, 4 --> predefined primitive
   --            function _Size (X : Typ) return Long_Long_Integer;
   --            function _Equality (X : Typ; Y : Typ'Class) return Boolean;
   --  I >= 5   --> User-Defined Primitive Operations
   --

   function Make_DT_Access
     (Loc : Source_Ptr; Rec : Node_Id; Typ : Entity_Id) return Node_Id;
   --  Create an access to the Dispatch Table by using the Tag field
   --  of a tagged record : Acc_Dt (Rec.tag).all

   function New_Constraint_Error (Loc : Source_Ptr) return Node_Id;
   --  This function builds a tree corresponding to the Ada statement
   --  "raise Constraint_Error" and returns the root of this tree,
   --  the N_Raise_Statement node.

   function New_External_Name
     (Related_Id   : Name_Id;
      Suffix       : Character := ' ';
      Suffix_Index : Nat := 0;
      Prefix       : Character := ' ')
      return         Name_Id;
   --  Builds a new entry in the names table of the form
   --    [Prefix  &] Related_Id [& Suffix] [& Suffix_Index'Image]
   --  where Prefix is prepended only if Prefix is non-blank (in which case
   --  it must be an upper case letter other than O,Q,U,W (which are used for
   --  identifier encoding, see Namet), and T is reserved for use by implicit
   --  types. Suffix_Index'Image is appended only if the value is non-zero.
   --  Suffix is also a single upper case letter other than O,Q,U,W and is a
   --  required parameter (T is permitted). The constructed name is stored
   --  using Find_Name so that it can be located using a subsequent Find_Name
   --  operation (i.e. it is properly hashed into the names table). The upper
   --  case letter given as the Suffix argument ensures that the name does
   --  not clash with any Ada identifier name. These generated names are
   --  permitted, but not required, to be made public by setting the flag
   --  Is_Public in the associated entity. Note: the reason that Prefix is
   --  Prefix is last is that it is almost always omitted. The notable case
   --  of Prefix being non-null is when it is 'T' for an implicit type.

   function New_External_Name
     (Suffix       : Character;
      Suffix_Index : Nat)
      return         Name_Id;
   --  Builds a new entry in the names table of the form
   --    Suffix & Suffix_Index'Image
   --  where Suffix is a single upper case letter other than O,Q,U,W and is a
   --  required parameter (T is permitted). The constructed name is stored
   --  using Find_Name so that it can be located using a subsequent Find_Name
   --  operation (i.e. it is properly hashed into the names table). The upper
   --  case letter given as the Suffix argument ensures that the name does
   --  not clash with any Ada identifier name. These generated names are
   --  permitted, but not required, to be made public by setting the flag
   --  Is_Public in the associated entity.

   function New_Internal_Name (Id_Char : Character) return Name_Id;
   --  Id_Char is an upper case letter other than O,Q,U,W (which are reserved
   --  for identifier encoding (see Namet package for details). The letter
   --  T is permitted, but is reserved by convention for the case of internal
   --  generated types. The result of the call is a new generated unique name
   --  of the form Xyyy where X is Id_Char, and yyy is a unique serial number.
   --  The name is entered into the names table using Name_Enter rather than
   --  Name_Find, because there can never be a need to locate the entry using
   --  the Name_Find procedure later on. Names created by New_Internal_Name
   --  must never have the Is_Public flag set. This is because there is no
   --  guarantee that the same serial number will be used when the same unit
   --  is included in different compilations. If a public name is required,
   --  New_External_Name must be used instead.

   function New_Occurrence_Of
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id;
   --  New_Occurrence_Of creates an N_Identifier node which is an
   --  occurrence of the defining identifier which is passed as its
   --  argument. The Entity and Etype of the result are set from
   --  the given defining identifier.

   function New_Reference_To
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id;
   --  This is like New_Occurrence_Of, but it does not set the Etype field.
   --  It is used from the expander, where Etype fields are generally not set,
   --  since they are set when the expanded tree is reanalyzed.

   procedure Raise_Warning
     (N        : Node_Id;
      Excep_Id : Entity_Id;
      Reason   : String);
   --  This procedure posts a warning message on node N stating that
   --  exception Excep_Id will be raised at run-time. The Reason string,
   --  (which should end with the characters ?!, since these are warning
   --  messages that should appear unconditionally) is used to provide an
   --  additional message which precedes the main message.

end Tbuild;
