------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 2                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.12 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;  use Atree;
with Einfo;  use Einfo;
with Nmake;  use Nmake;
with Sem;    use Sem;
with Sinfo;  use Sinfo;
with Tbuild; use Tbuild;

package body Exp_Ch2 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Expand_Discriminant (N : Node_Id);
   --  An occurence of a discriminant within a discriminated type is replaced
   --  with the corresponding discriminal, that is to say the formal parameter
   --  of the initialization procedure for the type that is associated with
   --  that particular discriminant. This is replacement is not performed for
   --  discriminants of records that appear in constraints of component of the
   --  record,  because Gigi uses the discriminant name to retrieve its value.
   --  See Einfo for more details, and Exp_Ch3, Exp_Ch9 for examples of use.

   -------------------------
   -- Expand_Discriminant --
   -------------------------

   procedure Expand_Discriminant (N : Node_Id) is
   begin
      if Ekind (Scope (Entity (N))) = E_Record_Type
        and then (Nkind (Parent (N)) = N_Range
          or else Nkind (Parent (N)) = N_Index_Or_Discriminant_Constraint)
      then
         null;
      else
         Set_Entity (N, Discriminal (Entity (N)));
      end if;
   end Expand_Discriminant;

   ----------------------------
   -- Expand_Entry_Parameter --
   ----------------------------

   --  A reference to an entry parameter is modified to be a reference to
   --  the corresponding component of the entry parameter record that is
   --  passed by the runtime to the accept body procedure

   procedure Expand_Entry_Parameter (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Ent_Formal : constant Entity_Id  := Entity (N);
      Ent_Spec   : constant Entity_Id  := Scope (Ent_Formal);
      Parm_Type  : constant Entity_Id  := Entry_Parameters_Type (Ent_Spec);
      Addr_Ent   : constant Entity_Id  := Accept_Address (Ent_Spec);
      P_Comp_Ref : Entity_Id;

   begin
      --  What we need is a reference to the corresponding component of the
      --  parameter record object. The Accept_Address field of the entry
      --  entity references the address variable that contains the address
      --  of the accept parameters record. We first have to do an unchecked
      --  conversion to turn this into a pointer to the parameter record and
      --  then we select the required parameter field.

      P_Comp_Ref :=
        Make_Selected_Component (Loc,
          Prefix =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Parm_Type, Loc),
              Expression   => New_Reference_To (Addr_Ent,  Loc)),
          Selector_Name =>
            New_Reference_To (Entry_Component (Ent_Formal), Loc));

      --  That's exactly what we want for the case of an elementary type,
      --  but for a composite type, we have a pointer, since we must pass
      --  such objects by reference. Thus we must dereference them to
      --  access them (this will often be redundant, since the needed
      --  deference is implicit, but no harm is done by making it explicit).

      if Is_Elementary_Type (Etype (Ent_Formal)) then
         Rewrite_Substitute_Tree (N, P_Comp_Ref);
      else
         Rewrite_Substitute_Tree (N,
           Make_Explicit_Dereference (Loc, P_Comp_Ref));
      end if;

      Analyze (N);
   end Expand_Entry_Parameter;

   ----------------------------
   -- Expand_N_Expanded_Name --
   ----------------------------

   --  Performs discriminal and entry parameter replacement as described above

   procedure Expand_N_Expanded_Name (N : Node_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      if Ekind (E) = E_Discriminant then
         Expand_Discriminant (N);
      elsif Is_Entry_Formal (E) then
         Expand_Entry_Parameter (N);
      end if;
   end Expand_N_Expanded_Name;

   -------------------------
   -- Expand_N_Identifier --
   -------------------------

   --  Performs discriminal and entry parameter replacement as described above

   procedure Expand_N_Identifier (N : Node_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      if Ekind (E) = E_Discriminant then
         Expand_Discriminant (N);
      elsif Is_Entry_Formal (E) then
         Expand_Entry_Parameter (N);
      end if;
   end Expand_N_Identifier;

end Exp_Ch2;
