/*
 *  Copyright (C) 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* Galeon includes */
#include "gul-gui.h"
#include "gul-string.h"
#include "eel-gconf-extensions.h"

/* system includes */
#include <ctype.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtkmain.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkimagemenuitem.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkmessagedialog.h>

static void
make_selection_list (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                     GList **list);

/**
 * gul_gui_menu_position_under_widget:
 */
void
gul_gui_menu_position_under_widget (GtkMenu   *menu,
				     gint      *x,
				     gint      *y,
				     gboolean  *push_in,
				     gpointer	user_data)
{
	GtkWidget *w = GTK_WIDGET (user_data);
	const GtkAllocation *allocation = &w->allocation;
	gint screen_width, screen_height;
	GtkRequisition requisition;

	gdk_window_get_origin (w->window, x, y);
	*x += allocation->x;
	*y += allocation->y + allocation->height;

	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
      
	screen_width = gdk_screen_width ();
	screen_height = gdk_screen_height ();
	  
	*x = CLAMP (*x, 0, MAX (0, screen_width - requisition.width));
	*y = CLAMP (*y, 0, MAX (0, screen_height - requisition.height));
}

/**
 * gul_gui_gtk_radio_button_get: get the active member of a radiobutton
 * group from one of the buttons in the group. This should be in GTK+!
 */
gint
gul_gui_gtk_radio_button_get (GtkRadioButton *radio_button)
{
	GtkToggleButton *toggle_button;
	gint i, length;
        GSList *list;
	
	/* get group list */
        list = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio_button));
        length = g_slist_length (list);

	/* iterate over list to find active button */
	for (i = 0; list != NULL; i++, list = g_slist_next (list))
	{
		/* get button and text */
		toggle_button = GTK_TOGGLE_BUTTON (list->data);
		if (gtk_toggle_button_get_active (toggle_button))
		{
			break;
		}
	}

	/* check we didn't run off end */
	g_assert (list != NULL);

	/* return index (reverse order!) */
	return (length - 1) - i;
}

/**
 * gul_gui_gtk_radio_button_set: set the active member of a radiobutton
 * group from one of the buttons in the group. This should be in GTK+!
 */
void
gul_gui_gtk_radio_button_set (GtkRadioButton *radio_button, gint index)
{
	GtkToggleButton *button;
	GSList *list;
	gint length;

	/* get the list */
        list = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio_button));

	/* check out the length */
        length = g_slist_length (list);

        /* new buttons are *preppended* to the list, so button added as first 
         * has last position in the list */
        index = (length - 1) - index;

	/* find the right button */
        button = GTK_TOGGLE_BUTTON (g_slist_nth_data (list, index));

	/* set it... this will de-activate the others in the group */
	if (gtk_toggle_button_get_active (button) == FALSE)
	{
		gtk_toggle_button_set_active (button, TRUE);
	}
}

/**
 * treeview_get_selection: get a list of selected iters
 */
GList *
gul_gui_treeview_get_selection_refs (GtkTreeView *treeview)
{
        GtkTreeSelection *selection = gtk_tree_view_get_selection (treeview);
        GList *list = NULL;

        gtk_tree_selection_selected_foreach (selection,
                                             (GtkTreeSelectionForeachFunc) make_selection_list,
                                             &list);

        return list;
}

/**
 * gul_gui_treeview_free_selection_refs: free the list returned from
 * gul_gui_treeview_get_selection_refs
 */
void
gul_gui_treeview_free_selection_refs (GList *l)
{
	g_list_foreach (l, (GFunc)gtk_tree_row_reference_free, NULL);
	g_list_free (l);
}

/**
 * make_selection_list: utl func to retreive the selection of a treeview
 * supporting multiple selection
 */
static void
make_selection_list (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                     GList **list)
{
	GtkTreeRowReference *node;

	node = gtk_tree_row_reference_new (model, path);
        *list = g_list_append (*list, node);
}

GtkWidget * 
gul_gui_append_new_menuitem (GtkWidget  *menu,
			     const char *mnemonic,
			     GCallback   callback,
			     gpointer    data)
{
	GtkWidget *menu_item;
	
	menu_item = gtk_menu_item_new_with_mnemonic (mnemonic);
	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu),
			       menu_item);
	
	if (callback)
	{
		g_signal_connect (G_OBJECT (menu_item),
				  "activate",
				  callback, data);
	}

	return menu_item;
}

GtkWidget * 
gul_gui_append_new_menuitem_stock (GtkWidget  *menu,
				   const char *stock_id,
				   GCallback   callback,
				   gpointer    data)
{
	GtkWidget *menu_item;
	
	menu_item = gtk_image_menu_item_new_from_stock (stock_id, NULL);
	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu),
			       menu_item);
	
	if (callback)
	{
		g_signal_connect (G_OBJECT (menu_item),
				  "activate",
				  callback, data);
	}
	
	return menu_item;
}

GtkWidget *
gul_gui_append_new_menuitem_stock_icon (GtkWidget *menu, 
					const char *stock_id,
					const char *mnemonic,
					GCallback callback,
					gpointer data)
{
	GtkWidget *menu_item;
	GtkWidget *image;

	menu_item = gtk_image_menu_item_new_with_mnemonic (mnemonic);
	image = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_MENU);
	gtk_widget_show (image);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menu_item), image);

	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu),
			       menu_item);
	
	if (callback)
	{
		g_signal_connect (G_OBJECT (menu_item),
				  "activate",
				  callback, data);
	}
	
	return menu_item;
}

GtkWidget * 
gul_gui_append_new_check_menuitem (GtkWidget  *menu,
				   const char *mnemonic,
				   gboolean value,
				   GCallback callback,
				   gpointer data)
{
	GtkWidget *menu_item;
	
	menu_item = gtk_check_menu_item_new_with_mnemonic (mnemonic);
	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu),
			       menu_item);
	
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item), value);
	
	if (callback)
	{
		g_signal_connect (G_OBJECT (menu_item),
				  "activate",
				  callback, data);
	}
	
	return menu_item;
}

GtkWidget *
gul_gui_append_separator (GtkWidget *menu)
{
	GtkWidget *menu_item;
	
	menu_item = gtk_menu_item_new ();
	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu),
			       menu_item);

	return menu_item;
}

gboolean 
gul_gui_confirm_overwrite_file (GtkWidget *parent, const char *filename)
{
	char *question;
	GtkWidget *dialog;
	gboolean res;
	
	if (!g_file_test (filename, G_FILE_TEST_EXISTS))
	{
		return TRUE;
	}

	question = g_strdup_printf (_("File %s will be overwritten.\n"
				    "If you choose yes, the contents will be lost.\n\n"
				    "Do you want to continue?"), filename);
	dialog = gtk_message_dialog_new (parent ? GTK_WINDOW(parent) : NULL, 
			                 GTK_DIALOG_MODAL,
				         GTK_MESSAGE_QUESTION, 
				         GTK_BUTTONS_YES_NO,
				         question);
	res = (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES);
	gtk_widget_destroy (dialog);		
	g_free (question);

	return res;
}

/** gul_gui_widget_hide_now:
 *
 * Same as gtk_widget_hide() but also flushes pending events so that the widget
 * really gets hidden, provides quicker response when closing windows.
 * Conveniently returns %FALSE so that it can be used as a signal handler for
 * "delete-event"
 *
 * Return value: %FALSE
 */
gboolean
gul_gui_widget_hide_now (GtkWidget *widget)
{
	gtk_widget_hide (widget);
	while (gtk_events_pending ()) gtk_main_iteration ();
	return FALSE;
}
