/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * utils.c - assorted utilities
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/shm.h>
#include <gtk/gtk.h>
#include "log.h"
#include "types.h"
#include "utils.h"
#include "preferences.h"
#include "support.h"
#include "history.h"

#if WANT_HAMLIB
#	include <hamlib/rig.h>
#endif

gint statusbartimer;
extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;
extern remotetype remote;
extern gchar **bandsplit;
extern gchar **modesplit;
extern gchar **qso;
extern gchar **dxcc;
extern gchar *xlogdir;
extern gint remotetimer;
extern gint clocktimer, savetimer;
extern glong msgid;
extern GHashTable *prefixes;
extern GList *riglist;
extern void *shareCall;
extern GList *logwindowlist;
extern GdkColormap *colormap;

#if WANT_HAMLIB
	extern gint hamlibtimer;
	extern	RIG *myrig;
#endif

/* saving of the log */
void savelog(GtkWidget *clist, gchar *logfile) {
	LOGDB *lp;
	gint i, j;
	gchar **item;

	item = g_new0(gchar *, 11);
	gtk_clist_freeze(GTK_CLIST(clist));
	lp = log_file_create(logfile, TYPE_FLOG, qso_field_nr, qso_fields, qso_widths);
	if (lp) {
		for (i = GTK_CLIST(clist)->rows - 1 ; i >= 0 ; i--) {
			for (j = 0; j < 11; j++)
				gtk_clist_get_text(GTK_CLIST(clist), i, j, &item[j]);
		log_file_qso_append(lp, item);
		}
		log_file_close(lp);
	}
	else {
		update_statusbar("Creation of logfile failed");
	}
	gtk_clist_thaw(GTK_CLIST(clist));
	g_free(item);
}


/* calculate location string from latitude and longitude information,
 * returned value has to be freed */
gchar *setlocation(gdouble lat, gint NS, gdouble lon, gint EW) {
	gchar *location, *ns, *ew;

	if (NS == 1) ns = g_strdup("N"); else ns = g_strdup("S");
	if (EW == 1) ew = g_strdup("E"); else ew = g_strdup("W");
	location = g_strdup_printf("%3.2f%s%3.2f%s", lat, ns, lon, ew);
	g_free(ns);
	g_free(ew);

	return(g_strdup(location));
}

/* update the menu items of an optionmenu */
void updateoptionmenu(GtkWidget *optionmenu, gchar **split) {
	GtkWidget *menu, *menuitem;
	gint index = 0;
	
	menu = gtk_menu_new();
	for (;;) {
		if (split[index] == NULL) break;
		menuitem = gtk_menu_item_new_with_label(split[index]);
		gtk_widget_show(menuitem);
		gtk_menu_append(GTK_MENU(menu), menuitem);
		index++;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), menu);
}

/* removing leading and trailing whitespaces from an array of strings */
void deletespaces(gchar **split) {
	gint index = 0;

	for (;;) {
		if (split[index] == NULL) break;
		g_strstrip(split[index]);
		index++;
	}
}

/* clear statusbar */
static gint statusbar_timeout(gpointer data) {
	GtkWidget *statusbar;

	statusbar = lookup_widget(mainwindow, "statusbar");
	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 1);
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, _("Ready."));
	gtk_timeout_remove(statusbartimer);
	state.statustimer = FALSE;
	return FALSE;
}

/* change the statusbar */
void update_statusbar(gchar *string) {
	GtkWidget *statusbar;

	statusbar = lookup_widget(mainwindow, "statusbar");
	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 1);
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, string);
	if (state.statustimer) gtk_timeout_remove(statusbartimer);
	statusbartimer = gtk_timeout_add(30000, statusbar_timeout, NULL);
	state.statustimer = TRUE;
}

/* Needed by g_hash_table_foreach_remove, to free every key in the hash table */
gboolean remove_prefixes(gchar *key, gpointer value, gpointer user_data) {
	g_free(key);
	return(TRUE);
}

/* check if latex is available */
gboolean latexavailable(void) {
	gint status;

	status = system("latex --version");
	if (status == 0) return(TRUE);
	return(FALSE);
}

/* free mem in state struct */
static void freestate(void)
{
	g_free(state.mylocation);
	g_free(state.rigfrequency);
	g_free(state.rigmode);
	g_free(state.rigrst);
}

/* free mem in prefs struct */
static void freeprefs(void)
{
	g_free(preferences.version);
	g_free(preferences.bands);
	g_free(preferences.modes);
	g_free(preferences.radio);
	g_free(preferences.device);
	g_free(preferences.themecolor);
	g_free(preferences.savedir);
	g_free(preferences.logstoload);
}

/* free mem in remote struct */
static void freeremote(void)
{
	g_free(remote.program);
}

/* cleanup of variables, used at exit */
void save_windowsize_and_cleanup(void) {
	gint i;

	/* free the dxcc array */
	for (i = 0; i < 400; i++)
		g_free(dxcc[i]);
	g_free(dxcc);

	/* free the qso array */
	for (i = 0; i < 20; i++)
		g_free(qso[i]);
	g_free(qso);

	g_list_free(logwindowlist);

	gdk_window_get_geometry(mainwindow->window, NULL, NULL, &preferences.width, &preferences.height, NULL);
	gdk_window_get_position(mainwindow->window, &preferences.x, &preferences.y);
/*	savecolumnsizes();*/
	savepreferences();
	savehistory();

#if WANT_HAMLIB
	if (riglist) g_list_free(riglist);
	if (hamlibtimer != -1) gdk_input_remove(hamlibtimer);
	if (preferences.hamlib > 0) {
		rig_close((RIG *)myrig);
		rig_cleanup((RIG *)myrig);
	}
#endif

	if (prefixes) {
		g_hash_table_foreach_remove(prefixes, (GHRFunc)remove_prefixes, NULL);
		g_hash_table_destroy(prefixes);
	}
	g_strfreev(bandsplit);
	g_strfreev(modesplit);
	g_free(xlogdir);
	if (remotetimer != -1) gdk_input_remove(remotetimer);
	if (savetimer != -1) gdk_input_remove(savetimer);
	if (clocktimer != -1) gdk_input_remove(clocktimer);
	
	/* remove message queue */
	if (msgid != -1) msgctl(msgid, IPC_RMID, 0);

	/* detach shared mem and destroy it */
	if (state.shmid != -1) {
		shmdt(shareCall);
		shmctl(state.shmid, IPC_RMID, NULL);
	}

	freestate();
	freeprefs();
	freeremote();
}

/* find dot in frequency and return correct band */
gchar *finddot(gchar *qsoband) {
	gchar *end, *j, *found = NULL;
	
	end = qsoband + strlen(qsoband);
	for (j = qsoband; j < end; ++j) {
		switch (*j) {
			case '.':
				*j = '\0';
			break;
		}
	}
	if (g_strcasecmp(qsoband, "1") == 0) found = g_strdup("1.8");
	else if (g_strcasecmp(qsoband, "3") == 0) found = g_strdup("3.5");
	else if (g_strcasecmp(qsoband, "7") == 0) found = g_strdup("7");
	else if (g_strcasecmp(qsoband, "29") == 0) found = g_strdup("28");
	else found = g_strdup(qsoband);
	return(g_strdup(found));
}

/* get the current date, returned value has to be freed */
gchar *getdate(void) {
	time_t current;
	struct tm *timestruct = NULL;
	gchar sdatenow[20];

	time(&current);
	timestruct = localtime(&current);
	strftime (sdatenow, 20, "%d %b %Y", timestruct);
	return(g_strdup(sdatenow));
}

/* get the current time, returned value has to be freed */
gchar *gettime(void) {
	time_t current;
	struct tm *timestruct = NULL;
	gchar stimenow[20];

	time(&current);
	timestruct = localtime (&current);
	strftime (stimenow, 20, "%H%M", timestruct);
	return(g_strdup(stimenow));
}

/* look up mode in a list of modes */
gchar *lookup_mode(gchar *selectedmode) {
	gint modeindex = 0;

	for (;;) {
		if (g_strcasecmp(selectedmode, modesplit[modeindex]) == 0) {
			break;
		}
		modeindex++;
	}
	return(g_strdup(modesplit[modeindex]));
}

/* look up band in a list of bands */
gchar *lookup_band(gchar *selectedband) {
	gint bandindex = 0;

	for (;;) {
		if (g_strcasecmp(selectedband, bandsplit[bandindex]) == 0) {
			break;
		}
		bandindex++;
	}
	return(g_strdup(bandsplit[bandindex]));
}

/* clock which updates a label in the statusbar */
gint updateclock(void) {
	GtkWidget *clocklabel;
	GString *timestr = g_string_new("");
	gchar *nowdate, *nowtime;

	nowdate = getdate();
	nowtime = gettime();
	clocklabel = lookup_widget(mainwindow, "clocklabel");
	g_string_sprintf(timestr, "%s", nowdate);
	timestr = g_string_append_c(timestr, ' ');
	timestr = g_string_append(timestr, nowtime);
	timestr = g_string_append(timestr, " GMT");
	gtk_label_set_text(GTK_LABEL(clocklabel), timestr->str);
	g_string_free(timestr, TRUE);
	g_free(nowdate);
	g_free(nowtime);
	return 1;
}

/* Put color in the right format for XParseColor, so it can be processed by gdk_color_parse */
gchar *color_parse(gchar *value){
	gchar **valuesplit = NULL;
	gchar *rgbsyntax;
        
	valuesplit = g_strsplit(value, ",", 3);
	rgbsyntax = g_strconcat("rgb:", valuesplit[0], "/", valuesplit[1], "/", valuesplit[2], NULL);
	g_strfreev(valuesplit);
	return(rgbsyntax);
}

/* set style of a new row which has been added to the log */
void setcallstyle(GtkWidget *clist, gint row, gchar *color) {
	GdkColor color2, color3, color4, themecolor;
	GtkStyle *callstyle;
	gchar *newcolor;

	newcolor = color_parse(color);
	gdk_color_parse(newcolor, &themecolor);
	gdk_colormap_alloc_color(colormap, &themecolor, FALSE, TRUE);
	
	/* colors other than the theme */
	color2.red = 0xffff;	/* white */
	color2.green = 0xffff;
	color2.blue	= 0xffff;
	color3.red = 0xcccc;	/* light gray */
	color3.green = 0xcccc;
	color3.blue	= 0xcccc;
	color4.red = 0x4924;	/* dark slate blue */
	color4.green = 0x3cf3;
	color4.blue	= 0x8e38;

	/* set style of the new row */
	callstyle = gtk_style_copy(clist->style);
	callstyle->base[GTK_STATE_NORMAL] = themecolor;
	callstyle->fg[GTK_STATE_NORMAL] = color2;
	callstyle->bg[GTK_STATE_SELECTED] = color3;
	callstyle->fg[GTK_STATE_SELECTED] = color4;
	gdk_font_unref(callstyle->font);
	callstyle->font = gdk_font_load("-adobe-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*");
	gtk_clist_set_cell_style(GTK_CLIST(clist), row, 3, callstyle);
	g_free(newcolor);
}

/* get color style of a label */
gchar *getlabelcolor(GtkWidget *label) {
	GtkStyle *labelstyle;
	GdkColor labelcolor;
	gchar *colorstr;
	
	labelstyle = gtk_style_copy(label->style);
	labelcolor = labelstyle->fg[GTK_STATE_NORMAL];
	colorstr = g_strdup_printf("%04X,%04X,%04X", labelcolor.red,
			labelcolor.green, labelcolor.blue);
	return(colorstr);
}

/* set style of a label */
void setlabelstyle(GtkWidget *label, gchar *color) {
	GtkStyle *labelstyle;
	GdkColor themecolor;
	gchar *newcolor;

	newcolor = color_parse(color);
	gdk_color_parse(newcolor, &themecolor);
	if(!gdk_colormap_alloc_color(colormap, &themecolor, FALSE, TRUE))
		g_print("Allocation of color failed in setlabelstyle\n");
	labelstyle = gtk_style_copy(label->style);
	labelstyle->fg[GTK_STATE_NORMAL] = themecolor;
	gdk_font_unref(labelstyle->font);
	labelstyle->font = gdk_font_load("-adobe-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*");
	gtk_widget_set_style(GTK_WIDGET(label), labelstyle);
	g_free(newcolor);
}

/* set style of a frame */
void setframestyle(GtkWidget *frame, gchar *color) {
	GtkStyle *framestyle;
	GdkColor themecolor;
	gchar *newcolor;

	newcolor = color_parse(color);
	gdk_color_parse(newcolor, &themecolor);
	if (!gdk_colormap_alloc_color(colormap, &themecolor, FALSE, TRUE))
		g_print("Allocation of color failed in setframestyle\n");
	framestyle = gtk_style_copy(frame->style);
	framestyle->bg[GTK_STATE_NORMAL] = themecolor; /* frame */
	framestyle->fg[GTK_STATE_NORMAL] = themecolor; /* frame title */
	gdk_font_unref(framestyle->font);
	framestyle->font = gdk_font_load("-adobe-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*");
	gtk_widget_set_style(GTK_WIDGET(frame), framestyle);
	g_free(newcolor);
}

gint autosave(void) {
	gint i;
	logtype *logwindow;
	gchar *logfile, *temp;
	gboolean message = FALSE;

	for (i = 0; i < g_list_length(logwindowlist); i++)
	{
		logwindow = g_list_nth_data(logwindowlist, i);
		if (logwindow->logchanged)
		{
			logfile = g_strconcat(preferences.savedir, G_DIR_SEPARATOR_S, logwindow->logname, ".xlog", NULL);
			savelog(logwindow->clist, logfile);
			logwindow->logchanged = FALSE;
			gtk_label_set_text(GTK_LABEL(logwindow->label), logwindow->logname);
			g_free(logfile);
			message = TRUE;
		}
	}

	if (message)
	{
		temp = g_strdup_printf(_("Log(s) autosaved"));
		update_statusbar(temp);
		g_free(temp);
	}
	return 1;
}

