/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
*/

#include "kbg.h"
#include "kbg.moc"

#include <iostream.h>
#include <kapp.h>
#include <kiconloader.h>
#include <kstddirs.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qprinter.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <qlayout.h>
#include <qpixmap.h>
#include <qstring.h>
#include <klocale.h>
#include <kstdaction.h>
#include <qwhatsthis.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <qvaluelist.h> 
#include <kmessagebox.h>
#include <kcmenumngr.h>
#include <qiconset.h>
#include <kconfig.h>
#include <kcompletion.h>
#include <kcompletionbox.h>

#include "kbgoffline.h"
#include "kbgfibs.h"

#include "version.h"



// == setup ====================================================================

/*
 * Get things rolling, no tricks here
 */
KBg::KBg(QWidget *parent, const char *name)
	: KMainWindow(parent, name)
{
	/*
	 * initially there is no engine - user setting ?
	 */
	engine = 0;
	statusBar()->message(i18n("No Engine"));
	
	/*
	 * The main view is shared between the board and a small MLE
	 */
	panner = new QSplitter(Vertical, this, "panner");
	board  = new KBgBoardSetup(panner, "board");
	//status = new KEdit(panner);
	status = new KBgTextView(panner, "status");
	//status->setReadOnly(true);
	setCentralWidget(panner);

	/*
	 * create all necessary actions
	 */
	setupActions();

	/*
	 * toolbars
	 */
	commandLabel = new QLabel(i18n("Command: "), toolBar("cmdToolBar"));
	commandLine  = new KLineEdit(toolBar("cmdToolBar"), "commandline");

     	toolBar("cmdToolBar")->insertWidget(0, 200, commandLabel);
	toolBar("cmdToolBar")->insertWidget(1,  10, commandLine );
     	toolBar("cmdToolbar")->show();

	commandLine->completionObject()->setOrder(KCompletion::Weighted);
	connect(commandLine, SIGNAL(returnPressed(const QString &)), this, SLOT(handleCmd(const QString &)));
	commandLine->setFocus();

	/*
	 * set up configuration handling
	 */
	connect(this, SIGNAL(readSettings()), board, SLOT(readConfig()));
	connect(this, SIGNAL(saveSettings()), board, SLOT(saveConfig()));

	/*
	 * what is messages
	 */
	QWhatsThis::add(status, 
			i18n("This area contains the status messages for the game. "
			     "Most of these messages are sent to you from the current "
			     "engine."));
	QWhatsThis::add(toolBar("cmdToolBar"), 
			i18n("This is the command line. You can type special "
			     "commands related to the current engine in here. "
			     "Most relevant commands are also available "
			     "through the menus."));
	QWhatsThis::add(toolBar("mainToolBar"), 
			i18n("This is the button bar tool bar. It gives "
			     "you easy access to game related commands. "
			     "You can drag the bar to a different location "
			     "within the window."));
	QWhatsThis::add(statusBar(), 
			i18n("This is the status bar. It shows you the currently "
			     "selected engine in the left corner."));
}

/*
 * Empty destructor
 */
KBg::~KBg()
{}

/*
 * Create and connect all the actions
 */
void KBg::setupActions()
{
	// file actions
	KStdAction::print(this, SLOT(print()), actionCollection(), "print");
	KStdAction::quit(kapp,  SLOT(quit()),  actionCollection(), "quit");
	
	// help actions
	KStdAction::help(this, SLOT(help()), actionCollection(), "help");
	KStdAction::whatsThis(this, SLOT(whatsThis()), actionCollection(), "whatsthis");
	new KAction(i18n("www: "PROG_NAME), 0, this, SLOT(showWWWHome()), actionCollection(), "help_www_home");
	new KAction(i18n("www: FIBS"),      0, this, SLOT(showWWWFIBS()), actionCollection(), "help_www_fibs");

	// settings actions
	KStdAction::showMenubar(this, SLOT(toggleMenubar()), actionCollection());
	KStdAction::showToolbar(this, SLOT(toggleMainToolbar()), actionCollection());
	KStdAction::showStatusbar(this, SLOT(toggleStatusbar()), actionCollection());
	KStdAction::keyBindings(this, SLOT(configureKeys()), actionCollection());
	KStdAction::configureToolbars(this,SLOT(configureToolbars()), actionCollection());
	KStdAction::saveOptions(this, SLOT(saveConfig()), actionCollection());
	KStdAction::preferences(this, SLOT(setupDlg()), actionCollection());
        (new KToggleAction(i18n("Show Command Line"), 0, this, SLOT(toggleCmdline()), 
			   actionCollection(), "show_cmd_line"))->setChecked(true);
	new KAction(i18n("Info Font..."),  0, status, SLOT(selectFont()), actionCollection(), "info_font");
	new KAction(i18n("Board Font..."), 0, board,  SLOT(selectFont()), actionCollection(), "board_font");

	// edit actions
	KStdAction::undo(this, SLOT(undo()), actionCollection(), "undo")->setEnabled(false);
	KStdAction::redo(this, SLOT(redo()), actionCollection(), "redo")->setEnabled(false);
	
	// game actions - initially disabled
	KStdAction::redisplay(this, SLOT(load()), actionCollection(), "load")->setEnabled(false);
	(new KAction(i18n("Roll Dice"),   
		     QIconSet(kapp->iconLoader()->loadIcon(PROG_NAME "-roll.xpm", KIcon::Toolbar)),
		     0, this, SLOT(roll()), 
		     actionCollection(), "roll"))->setEnabled(false);
	(new KAction(i18n("Done Moving"), 
		     QIconSet(kapp->iconLoader()->loadIcon(PROG_NAME "-send.xpm", KIcon::Toolbar)),
		     0, this, SLOT(done()), 
		     actionCollection(), "done"))->setEnabled(false);
	(new KAction(i18n("Double Cube"), 
		     QIconSet(kapp->iconLoader()->loadIcon(PROG_NAME "-double.xpm", KIcon::Toolbar)),
		     0, this, SLOT(cube()), 
		     actionCollection(), "cube"))->setEnabled(false);

	// engine menu
	(new KToggleAction(i18n("Open Board"),        0, this, SLOT(setupEngineOffline()), 
			   actionCollection(), "engine_offline"))->setEnabled(true);
	(new KToggleAction(i18n("Online on FIBS"),    0, this, SLOT(setupEngineFIBS()), 
			   actionCollection(), "engine_fibs"   ))->setEnabled(true);
	(new KToggleAction(i18n("Offline vs. GNUbg"), 0, this, SLOT(setupEngineGNUbg()), 
			   actionCollection(), "engine_gnubg"  ))->setEnabled(false);

	// over and out
	createGUI();
}


// == configuration handing ====================================================

/*
 * Save all settings that should persist
 */
void KBg::saveConfig()
{	
	KConfig* config = kapp->config();

	config->setGroup("main window");

	/*
	 * save main window visual settings
	 */
        config->writeEntry("origin", pos());
        config->writeEntry("height", height());
        config->writeEntry("width",  width());

	config->writeEntry("font",   status->font());
	config->writeEntry("panner", (double)board->height()/(double)panner->height());

	saveMainWindowSettings(config, "main window");

	/*
	 * save the history
	 */
	config->writeEntry("history", commandLine->completionObject()->items());

	/*
	 * store current engine
	 */
	config->setGroup("state");
	config->writeEntry("engine", currentEngine);
	
	/*
	 * Tell other objects to save their config
	 */
	emit saveSettings();

	config->sync();
}

/*
 * Read the stored configuration and apply it
 */
void KBg::readConfig()
{
	KConfig* config = kapp->config();
	QPoint pos, defpos(10, 10);
	QFont kappFont = kapp->font();

	config->setGroup("main window");

	/*
	 * restore main window visual settings
	 */
        pos = config->readPointEntry("origin", &defpos);
        setGeometry(pos.x(), pos.y(), config->readNumEntry("width",520), config->readNumEntry("height",473));

	status->setFont(config->readFontEntry("font", &kappFont));

	QValueList<int> l;
	l.append(   config->readDoubleNumEntry("panner", 0.75) *panner->height());
	l.append((1-config->readDoubleNumEntry("panner", 0.75))*panner->height());
	panner->setSizes(l);

	/*
	 * restore the toolbars
	 */
	applyMainWindowSettings(config, "main window");

	/*
	 * restore the history
	 */
	commandLine->completionObject()->setItems(config->readListEntry("history"));

	/*
	 * tell other objects to read their configurations
	 */
	emit readSettings();

	/*
	 * restore last engine
	 */
	config->setGroup("state");		
	currentEngine = (Engines)config->readNumEntry("engine", None);
	engine = 0;
	if (currentEngine != None)
		setupEngine(currentEngine);
}

/*
 * Connected to the setup dialog applyButtonPressed signal
 */
void KBg::setupOk()
{
	saveConfig();
	delete nb;
}

/*
 * Connected to the setup dialog cancelButtonPressed signal
 */
void KBg::setupCancel()
{
	board->undoSetupChanges();
	delete nb;
}

/*
 * Initialize and display the setup dialog
 */
void KBg::setupDlg()
{
	nb = new QTabDialog(this, "setup", true);
	
	nb->setOkButton();
  	nb->setCancelButton();

	board->getSetupPages(nb);
	if (engine) engine->getSetupPages(nb);
	
	connect(nb, SIGNAL(applyButtonPressed()),  this, SLOT(setupOk()));
	connect(nb, SIGNAL(cancelButtonPressed()), this, SLOT(setupCancel()));	
	
	nb->resize(nb->minimumSize());
	nb->show();
}


// == engine handling ==========================================================

/*
 * Generic function that sets up the engine type.
 */
void KBg::setupEngine(int type)
{
	/*
	 * Engine doesn't need to be changed?
	 */
	if (engine && (currentEngine == type)) return;		

	/*
	 * Remove the  old engine
	 */
	if (engine) {
		engine->removeMenuItems((QPopupMenu *)factory()->container("play_menu", this));
		delete engine;
	}

	/*
	 * Create new engine and hook up menu and slots/signals
	 */
	QString s = PROG_NAME;
	switch (currentEngine = type) {
	case Offline:
		engine = new KBgEngineOffline(this, &s);
		statusBar()->message(i18n("Open board"));
		break;
	case FIBS:
		engine = new KBgEngineFIBS(this, &s);
		statusBar()->message(i18n("FIBS"));
		break;
	case GNUbg:
		// doesn't exist yet
		statusBar()->message(i18n("GNU Backgammon"));
		break;
	}
	((KToggleAction *)actionCollection()->action("engine_offline"))->setChecked(currentEngine == Offline);
	((KToggleAction *)actionCollection()->action("engine_gnubg"  ))->setChecked(currentEngine == GNUbg  );
	((KToggleAction *)actionCollection()->action("engine_fibs"   ))->setChecked(currentEngine == FIBS   );
	
	// get the engine and menus hooked
	board->setContextMenu(engine->getMenu());
	engine->addMenuItems((QPopupMenu *)factory()->container("play_menu", this));
	KContextMenuManager::insert(status->viewport(), engine->getMenu());

	// engine -> this
	connect(engine, SIGNAL(statText(const QString &)), this, SLOT(updateCaption(const QString &)));
	connect(engine, SIGNAL(infoText(const QString &)), status, SLOT(write(const QString &)));
	connect(engine, SIGNAL(allowCommand(int, bool)),   this, SLOT(allowCommand(int, bool)));
	
	// this -> engine 
	connect(this, SIGNAL(readSettings()), engine, SLOT(readConfig()));
	connect(this, SIGNAL(saveSettings()), engine, SLOT(saveConfig()));

	// board -> engine
	connect(board, SIGNAL(rollDice(const int)),    engine, SLOT(rollDice(const int)));
	connect(board, SIGNAL(doubleCube(const int)),  engine, SLOT(doubleCube(const int)));
	connect(board, SIGNAL(currentMove(QString *)), engine, SLOT(handleMove(QString *)));
	
	// engine -> board
	connect(engine, SIGNAL(undoMove()), board, SLOT(undoMove()));
	connect(engine, SIGNAL(redoMove()), board, SLOT(redoMove()));
	connect(engine, SIGNAL(setEditMode(const bool)), board, SLOT(setEditMode(const bool)));
	connect(engine, SIGNAL(allowMoving(const bool)), board, SLOT(allowMoving(const bool)));
	connect(engine, SIGNAL(getState(KBgBoardStatus *)), board, SLOT(getState(KBgBoardStatus *)));
	connect(engine, SIGNAL(newState(const KBgBoardStatus *)), board, SLOT(setState(const KBgBoardStatus *)));
}

// set offline engine
void KBg::setupEngineOffline()
{
	setupEngine(Offline);
}

// make GNUbg the new engine
void KBg::setupEngineGNUbg()
{
	setupEngine(GNUbg);
}

// make FIBS the new engine
void KBg::setupEngineFIBS()
{
	setupEngine(FIBS);
}


// == various slots for the actions ============================================

// Tell the board to print itself - restore and save user settings
void KBg::print()
{
	QPrinter *prt = new QPrinter();
	
	KConfig* config = kapp->config();
	config->setGroup("printing");

	prt->setNumCopies(config->readNumEntry("numcopies", 1));
	prt->setOutputFileName(config->readEntry("outputfile", ""));
	prt->setOutputToFile(config->readBoolEntry("tofile", false));
	prt->setPageSize((QPrinter::PageSize) config->readNumEntry("pagesize", QPrinter::A4));
	prt->setOrientation((QPrinter::Orientation)config->readNumEntry("orientation", QPrinter::Landscape));

	if (prt->setup() ) {
		QPainter p;
		p.begin(prt);
		board->print(&p);
		p.end();
		config->writeEntry("tofile",      prt->outputToFile());
		config->writeEntry("outputfile",  prt->outputFileName());
		config->writeEntry("pagesize",    (int)prt->pageSize());
		config->writeEntry("orientation", (int)prt->orientation());
		config->writeEntry("numcopies",   prt->numCopies());
	}
	delete prt;
}

// Invokes the KDE help browser
void KBg::help()
{
	kapp->invokeHelp();
}

// Open the homepage in the default browser
void KBg::showWWWHome() 
{ 
	kapp->invokeBrowser(PROG_HOME);
}

// Open the FIBS homepage in the default browser
void KBg::showWWWFIBS() 
{ 
	kapp->invokeBrowser("http://www.fibs.com/");
}

// Toggle visibility of menubar - careful, the menu is gone too
void KBg::toggleMenubar()
{
	if (menuBar()->isVisible())
		menuBar()->hide();
	else
		menuBar()->show();
}

// Toggle visibility of the bar with the XML name s
void KBg::toggleToolbar(const char *s)
{
	QToolBar *bar = toolBar(s);
	if(bar->isVisible())
		bar->hide();
	else
		bar->show();
}

// toggle the main toolbar - the one with pictures
void KBg::toggleMainToolbar()
{	
	toggleToolbar("mainToolBar");
}

// toggle the command line toolbar
void KBg::toggleCmdline()
{	
	toggleToolbar("cmdToolBar");
}

// toggle visibility of the statusbar
void KBg::toggleStatusbar()
{
	if (statusBar()->isVisible())
		statusBar()->hide();
	else
		statusBar()->show();
}

// get the standard dialog for key bindings
void KBg::configureKeys()
{
	KKeyDialog::configureKeys(actionCollection(), xmlFile(), true, this);
}

// configure the toolbars
void KBg::configureToolbars()
{
	KEditToolbar *dlg = new KEditToolbar(guiFactory(), this);
	if (dlg->exec())
		createGUI();
	delete dlg;
}

// undoes last move
void KBg::undo() 
{ 
	if (engine) engine->undo(); 
}

// redo the last move
void KBg::redo() 
{ 
	if (engine) engine->redo(); 
}

// roll the dice
void KBg::roll() 
{ 
	if (engine) engine->roll(); 
}

// double the cube
void KBg::cube() 
{ 
	if (engine) engine->cube(); 
}

// finish the move
void KBg::done() 
{ 
	if (engine) engine->done(); 
}

// reload the board
void KBg::load() 
{ 
	if (engine) engine->load(); 
}


// == various slots - not for actions ==========================================

/*
 * About to be closed. Let the engine know and clean up. If the engine returns
 * false - for whatever reason - we do not quit.
 */
bool KBg::queryExit()
{
	bool ret = true;
	if (engine)
		ret = engine->queryExit();
	saveConfig();	
	return ret;
}

/*
 * Set the caption of the main window. If the user has 
 * requested pip counts, they are appended, too.
 */
void KBg::updateCaption(const QString &s)
{
	QString msg = "";
	if (!s.isEmpty()) {
		msg = s;
		if (board->getPipCount(US) >= 0) {
			QString tmp;
			tmp.setNum(board->getPipCount(US  ));
			msg += " - " + tmp;
			tmp.setNum(board->getPipCount(THEM));
			msg += "-"  + tmp;
		}
	}
	setCaption(msg, false);
}

/*
 * Take the string from commandlin, give it to the engine,
 * append to the history and clear the buffer.
 */
void KBg::handleCmd(const QString &s)
{
	if (!s.stripWhiteSpace().isEmpty()) {
		if (engine) 
			engine->handleCommand(s);
		commandLine->completionObject()->addItem(s);
	}
	commandLine->clear();
	commandLine->completionBox()->close();
}

/*
 * Reflect the availability of commands in the button bar.
 */ 
void KBg::allowCommand(int cmd, bool f)
{
	switch (cmd) {
	case KBgEngine::Undo:
		actionCollection()->action("undo")->setEnabled(f);
		break;
	case KBgEngine::Redo:
		actionCollection()->action("redo")->setEnabled(f);
		break;
	case KBgEngine::Roll:
		actionCollection()->action("roll")->setEnabled(f);
		break;
	case KBgEngine::Cube:
		actionCollection()->action("cube")->setEnabled(f);
		break;
	case KBgEngine::Done:
		actionCollection()->action("done")->setEnabled(f);
		break;
	case KBgEngine::Load:
		actionCollection()->action("load")->setEnabled(f);
		break;
	}
}

// EOF
