/*
 * @(#)ThreadInfo.java	1.18 00/02/02
 *
 * Copyright 1998-2000 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the proprietary information of Sun Microsystems, Inc.  
 * Use is subject to license terms.
 * 
 */
/*
 * Copyright (c) 1997-1999 by Sun Microsystems, Inc. All Rights Reserved.
 * 
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 * 
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 * 
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package com.sun.tools.example.debug.tty;

import com.sun.jdi.ThreadReference;
import com.sun.jdi.ThreadGroupReference;
import com.sun.jdi.IncompatibleThreadStateException;
import com.sun.jdi.StackFrame;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.io.*;

class ThreadInfo {
    // This is a list of all known ThreadInfo objects. It survives 
    // ThreadInfo.invalidateAll, unlike the other static fields below. 
    private static List threads = Collections.synchronizedList(new ArrayList());
    private static boolean gotInitialThreads = false;

    static ThreadInfo current = null;
    private static ThreadGroupReference group = null;

    final ThreadReference thread;
    int currentFrameIndex = 0;

    private ThreadInfo(ThreadReference thread) {
        this.thread = thread;
        if (thread == null) {
            Env.fatalError("Internal error: null ThreadInfo created");
        }
    }

    private static void initThreads() {
        if (!gotInitialThreads) {
            Iterator iter = Env.vm().allThreads().iterator();
            while (iter.hasNext()) {
                ThreadReference thread = (ThreadReference)iter.next();
                threads.add(new ThreadInfo(thread));
            }
            gotInitialThreads = true;
        }
    }

    static void addThread(ThreadReference thread) {
        synchronized (threads) {
            initThreads();
            ThreadInfo ti = new ThreadInfo(thread);
            // Guard against duplicates. Duplicates can happen during 
            // initialization when a particular thread might be added both
            // by a thread start event and by the initial call to threads()
            if (getThread(thread) == null) {
                threads.add(ti);
            }
        }
    }

    static void removeThread(ThreadReference thread) {
        if (thread.equals(ThreadInfo.current)) {
            // Current thread has died.

            // Be careful getting the thread name. If its death happens
            // as part of VM termination, it may be too late to get the 
            // information, and an exception will be thrown.
            String currentThreadName;
            try {
               currentThreadName = " \"" + thread.name() + "\"";
            } catch (Exception e) {
               currentThreadName = "";
            }
                 
            setCurrentThread(null);

            Env.errorln("");
            Env.errorln("Current thread" + currentThreadName + 
                        " died. Execution continuing...");
        }
        threads.remove(getThread(thread));
    }

    static List threads() {
        synchronized(threads) {
            initThreads();
            // Make a copy to allow iteration without synchronization
            return new ArrayList(threads);
        }
    }

    static void invalidateAll() {
        current = null;
        group = null;
        synchronized (threads) {
            Iterator iter = threads().iterator();
            while (iter.hasNext()) {
                ThreadInfo ti = (ThreadInfo)iter.next();
                ti.invalidate();
            }
        }
    }

    static void setThreadGroup(ThreadGroupReference tg) {	
        group = tg;
    }
    
    static void setCurrentThread(ThreadReference tr) { 
        if (tr == null) {
            current = null;
        } else {
            ThreadInfo tinfo = getThread(tr);
            current = tinfo;
            current.invalidate();
        }
    }
    
    static ThreadGroupReference group() {
	if (group == null) {
            // Current thread group defaults to the first top level
            // thread group.
	    setThreadGroup((ThreadGroupReference)
                           Env.vm().topLevelThreadGroups().get(0));
	}
        return group;
    }
    
    static ThreadInfo getThread(long id) {
        ThreadInfo retInfo = null;

        synchronized (threads) {
            Iterator iter = threads().iterator();
            while (iter.hasNext()) {
                ThreadInfo ti  = (ThreadInfo)iter.next();
                if (ti.thread.uniqueID() == id) {
                   retInfo = ti;
                   break;
                }
            }
        }
        return retInfo;
    }

    static ThreadInfo getThread(ThreadReference tr) {
        return getThread(tr.uniqueID());
    }

    static ThreadInfo getThread(String idToken) {
        ThreadInfo tinfo = null;
        if (idToken.startsWith("t@")) {
            idToken = idToken.substring(2);
        }
        try {
            long threadId = Long.decode(idToken).longValue();
            tinfo = getThread(threadId);
        } catch (NumberFormatException e) {
            tinfo = null;
        }
        return tinfo;
    }

    List stack() throws IncompatibleThreadStateException {
        return thread.frames();
    }

    StackFrame getCurrentFrame() throws IncompatibleThreadStateException {
        if (thread.frameCount() == 0) {
            //Env.out.println("No stack");
            return null;
        }
        return thread.frame(currentFrameIndex);
    }

    void invalidate() {
        currentFrameIndex = 0;
    }

    /* Throw IncompatibleThreadStateException if not suspended */
    private void assureSuspended() throws IncompatibleThreadStateException {
        if (!thread.isSuspended()) {
            throw new IncompatibleThreadStateException();
        }
    }

    /**
     * Change the current stackframe to be one or more frames higher
     * (as in, away from the current program counter).
     *
     * @param nFrames	the number of stackframes
     * @exception IllegalAccessError when the thread isn't 
     * suspended or waiting at a breakpoint
     * @exception ArrayIndexOutOfBoundsException when the 
     * requested frame is beyond the stack boundary
     */
    void up(int nFrames) throws IncompatibleThreadStateException {
        assureSuspended();
        if ((currentFrameIndex + nFrames) >= thread.frameCount()) {
            throw new ArrayIndexOutOfBoundsException();
        }
        currentFrameIndex += nFrames;
    }

    /**
     * Change the current stackframe to be one or more frames lower
     * (as in, toward the current program counter).     *
     * @param nFrames	the number of stackframes
     * @exception IllegalAccessError when the thread isn't 
     * suspended or waiting at a breakpoint
     * @exception ArrayIndexOutOfBoundsException when the 
     * requested frame is beyond the stack boundary
     */
    void down(int nFrames) throws IncompatibleThreadStateException {
        assureSuspended();
        if ((currentFrameIndex - nFrames) < 0) {
            throw new ArrayIndexOutOfBoundsException();
        }
        currentFrameIndex -= nFrames;
    }

}
                            
