/*
 * up.c - version 0.3 - by: David Cantrell (david@burdell.org)
 * Displays the uptime of system, accounting for the following:
 *
 *   o Decades
 *   o Years
 *   o Weeks
 *   o Days
 *   o Hours
 *   o Minutes
 *
 * It can also display uptime in the standard for quick comparison
 * with other systems ( XX day(s), HH:MM )
 *
 * Portions of this program are taken from procps-1.2.7
 * This program is protected by the GNU General Public License.
 * Blah, blah, blah....go find and read the copyleft...too much text
 * for the comments section.
 *
 * Last modified 25-Aug-1999
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>

#define BAD_PROC                                                \
"Error: /proc must be mounted\n"                                \
"  To mount /proc at boot you need an /etc/fstab line like:\n"  \
"      /proc   /proc   proc      defaults\n"                    \
"  In the meantime, mount -t proc /proc /proc\n"
#define UPTIME_FILE  "/proc/uptime"
static char buf[1024];

/* This macro opens FILE only if necessary and seeks to 0 so that successive
   calls to the functions are more efficient.  It also reads the current
   contents of the file into the global buf.
*/
#define FILE_TO_BUF(FILE) {					\
    static int n, fd = -1;					\
    if (fd == -1 && (fd = open(FILE, O_RDONLY)) == -1) {	\
	fprintf(stderr, BAD_PROC);				\
	return 0;						\
    }								\
    lseek(fd, 0L, SEEK_SET);					\
    if ((n = read(fd, buf, sizeof buf - 1)) < 0) {		\
	perror(FILE);						\
	close(fd);						\
	fd = -1;						\
	return 0;						\
    }								\
    buf[n] = '\0';						\
}

#define SET_IF_DESIRED(x,y)  if (x) *(x) = (y)	/* evals 'x' twice */

static int uptime(double *uptime_secs, double *idle_secs) 
{
   double up=0, idle=0;
    
   FILE_TO_BUF(UPTIME_FILE)
   if (sscanf(buf, "%lf %lf", &up, &idle) < 2) {
      fprintf(stderr, "bad data in " UPTIME_FILE "\n");
      return 0;
   }
   SET_IF_DESIRED(uptime_secs, up);
   SET_IF_DESIRED(idle_secs, idle);
   return up;	/* assume never be zero seconds in practice */
}

static void print_uptime(const char *otype)
{
   /*
    * This routine calculates the uptime and displays it in the
    * desired format (the unique anal format of this program, or
    * the traditional format of 'uptime'). I wrote this routine
    * from scratch (most of the others are hacked from procps...
    * well, except main) and it's not the most pretty thing in
    * the world, but DAMMIT it works!
    */

   int uptimes[6];
   int idx;
   double upseconds, upidle;
   char *upstrs[6];
   const char * const words[6] = {"decade", "year", "week", "day", "hour", "minute"};

   uptime(&upseconds, &upidle);

   if (!strcmp(otype, "std")) {
      /* calculate uptime values */
      uptimes[3] = (int) upseconds / (60*60*24);
      uptimes[4] = ((int) upseconds / (60*60)) % 24;
      uptimes[5] = ((int) upseconds / 60) % 60;
   }
   else {
      /* calculate the uptime values */
      uptimes[0] = (int) upseconds / (60*60*24*365*10);
      uptimes[1] = ((int) upseconds / (60*60*24*365)) % 10;
      uptimes[2] = ((int) upseconds / (60*60*24*7)) % 52;
      uptimes[3] = ((int) upseconds / (60*60*24)) % 7;
      uptimes[4] = ((int) upseconds / (60*60)) % 24;
      uptimes[5] = ((int) upseconds / 60) % 60;
   }

   /* fixup the plurals */
   for(idx = 0; idx <= 5; idx++)
      upstrs[idx] = (uptimes[idx] != 1) ? "s" : "";

   /* print the uptime string, eliminating zero values */
   if (!strcmp(otype, "std")) {
      printf(" uptime:  up %d %s%s, ", uptimes[3], words[3], upstrs[3]);
      printf("%2d:%02d", uptimes[4], uptimes[5]);
   }
   else if (!strcmp(otype, "num")) {		/* output numbers only */
      for(idx = 0; idx <= 5; idx++)
        printf("%d ", uptimes[idx]);
   }
   else {
      if (strcmp(otype, "noup"))
        printf(" up");
      for(idx = 0; idx <= 5; idx++) {
        if(uptimes[idx] != 0)
           printf(" %d %s%s", uptimes[idx], words[idx], upstrs[idx]);
      }
   }
 
   printf("\n");
}

int main(int argc, char *argv[])
{
   /*
    * Main program routine, checks command line 
    * argument and does the appropriate action.
    */

   int ret = EXIT_SUCCESS;

   if (argc == 1)
      print_uptime("norm");
   else if (argc == 2) {
      if (!strcmp(argv[1], "--version") || !strcmp(argv[1], "-v"))
         printf("up v0.2 -- by: David Cantrell (david@burdell.org)\n");
      else if (!strcmp(argv[1], "--help") || !strcmp(argv[1], "-h")) {
         printf("up v0.2 -- by: David Cantrell (david@burdell.org)\n");
         printf("up is a robust uptime display utility for Linux systems.\n\n");
         printf("Usage: %s [option]\n", argv[0]);
         printf("       --compare | -c     Display uptime in standard uptime format, for quick\n");
         printf("                          comparison with other systems.\n");
         printf("       --numbers | -n     Output numbers only (useful for scripts?). They\n");
         printf("                          are in this order: decades, years, weeks, days,\n");
         printf("                          hours, minutes.  Zero values are not eliminated.\n");
         printf("       --noup    | -u     Displays the default uptime format without the\n");
         printf("                          preceding 'up'.\n");
         printf("       --version | -v     Return the version number and author information.\n");
         printf("       --help    | -h     Display this screen.\n");
      }
      else if (!strcmp(argv[1], "--compare") || !strcmp(argv[1], "-c"))
         print_uptime("std");
      else if (!strcmp(argv[1], "--numbers") || !strcmp(argv[1], "-n"))
         print_uptime("num");
      else if (!strcmp(argv[1], "--noup") || !strcmp(argv[1], "-u"))
         print_uptime("noup");
      else {
         printf("Invalid command line argument. Type %s --help for help.\n", argv[0]);
         ret = EXIT_FAILURE;
      }
   }

   return(ret);
}

