/*
    interface.c -- parted binding glue to libext2resize
    Copyright (C) 1998-2000 Free Software Foundation, Inc.
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* VERSION: libext2resize 1.1.6 (by Lennert)
 * merged 1.1.11 changes (by Andrew)
 */

static const char _interface_c[] = "$Id: interface.c,v 1.6 1999/10/02 20:56:47 aclausen Exp $";

#include "config.h"

#include <parted/parted.h>
#include <parted/disk_dos.h>
#include <parted/disk_loop.h>
#include <parted/disk_pc98.h>
#include <parted/disk_mac.h>
#include <parted/disk_bsd.h>
#include <parted/disk_sun.h>

#include <string.h>

#include "ext2.h"
#include "parted_io.h"

struct ext2_dev_handle* ext2_make_dev_handle_from_parted_geometry(PedGeometry* geom);



static int
_ext2_probe (const PedGeometry* geom);

static int
_ext2_clobber (PedGeometry* geom);

static PedFileSystem*
_ext2_open (const PedGeometry *geom);

static PedFileSystem*
_ext2_create (PedGeometry *geom);

static int
_ext2_close (PedFileSystem *fs);

static int
_ext2_check (PedFileSystem *fs);

static int
_ext2_resize (PedFileSystem* fs, const PedGeometry* geom);

static PedConstraint*
_ext2_get_resize_constraint (const PedFileSystem* fs);

static int
_ext2_set_system (const PedFileSystem* fs, PedPartition* part,
		  const PedDiskType* disk_type);




static PedFileSystemOps _ext2_ops = {
	probe:		_ext2_probe,
	clobber:	_ext2_clobber,
	open:		_ext2_open,
	create:         _ext2_create,
	close:		_ext2_close,
	check:          _ext2_check,
	resize:		_ext2_resize,
	copy:           NULL,
	get_resize_constraint:	_ext2_get_resize_constraint,
	set_system:	_ext2_set_system
};

static PedFileSystemType _ext2_type = {
       next:		 NULL,
       ops:		 &_ext2_ops,
       name:		 "ext2"
};

void ped_file_system_ext2_init ()
{
	ped_file_system_type_register (&_ext2_type);
}

void ped_file_system_ext2_done ()
{
	ped_file_system_type_unregister (&_ext2_type);
}



static int
_ext2_probe (const PedGeometry* geom)
{
	struct ext2_super_block sb;

	if (!ped_geometry_read(geom, &sb, 2, 2)
	    || EXT2_SUPER_MAGIC(sb) != EXT2_SUPER_MAGIC_CONST)
		return 0;

	return 1;
}

static int
_ext2_clobber (PedGeometry* geom)
{
	struct ext2_super_block sb;

	if (!ped_geometry_read(geom, &sb, 2, 2))
		return 0;
	if (EXT2_SUPER_MAGIC(sb) != EXT2_SUPER_MAGIC_CONST)
		return 1;

	sb.s_magic = 0;
	return ped_geometry_write(geom, &sb, 2, 2);
}

static PedFileSystem*
_ext2_open (const PedGeometry *geom)
{
	PedFileSystem*		fs;
	struct ext2_fs*		fs_info;
	struct ext2_dev_handle*	handle;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs) goto error;

	fs->type = &_ext2_type;
	fs->geom = ped_geometry_duplicate (geom);

	handle = ext2_make_dev_handle_from_parted_geometry(fs->geom);
	if (!handle) goto error_free_fs;

	fs_info = (struct ext2_fs*) ext2_open(handle, 0);
	if (!fs_info) goto error_free_handle;

	fs->type_specific = (void*) fs_info;
	fs_info->opt_verbose = 0;

	return fs;

error_free_handle:
	ext2_destroy_dev_handle(handle);
error_free_fs:
	ped_free(fs);
error:
	return NULL;
}

static PedFileSystem*
_ext2_create (PedGeometry *geom)
{
	PedFileSystem*		fs;
	struct ext2_fs*		fs_info;
	struct ext2_dev_handle*	handle;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs) goto error;

	fs->type = &_ext2_type;
	fs->geom = ped_geometry_duplicate (geom);

	handle = ext2_make_dev_handle_from_parted_geometry(fs->geom);
	if (!handle) goto error_free_fs;

	fs_info = ext2_mkfs (handle, 0, 0, 0, 0, -1, -1);
	if (!fs_info) goto error_free_handle;

	fs->type_specific = (void*) fs_info;
	fs_info->opt_verbose = 0;

	return fs;

error_free_handle:
	ext2_destroy_dev_handle(handle);
error_free_fs:
	ped_free(fs);
error:
	return NULL;
}

static int
_ext2_close (PedFileSystem *fs)
{
	struct ext2_dev_handle* handle;

	handle = ((struct ext2_fs*)fs->type_specific)->devhandle;
	ext2_close(fs->type_specific);
	ext2_destroy_dev_handle(handle);

	ped_free(fs);
	return 1;
}

static int
_ext2_check (PedFileSystem *fs)
{
	ped_exception_throw (PED_EXCEPTION_INFORMATION, PED_EXCEPTION_OK,
		_("The ext2 filesystem passed a basic check.  For a more "
		  "comprehensive check, use the e2fsck program."));
	return 1;
}

static int
_ext2_resize (PedFileSystem* fs, const PedGeometry* geom)
{
	struct ext2_fs* f;
	PedSector	old_length = fs->geom->length;

	if (fs->geom->disk->dev != geom->disk->dev)
	{
		fprintf(stderr, "huh? ungrokkable!\n");
		return 0;
	}

	if (fs->geom->start != geom->start)
	{
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
		      PED_EXCEPTION_CANCEL,
		      _("Sorry, can't move the start of ext2 partitions yet!"));
		return 0;
	}

	geom->disk->dev->boot_dirty = 1;

	f = (struct ext2_fs *) fs->type_specific;

/* ensure that the geometry contains the new and old geometry */
	if (old_length > geom->length) {
		if (!ext2_resize_fs(f, geom->length >> (f->logsize - 9)))
			goto error;

		fs->geom->length = geom->length;
		fs->geom->end = fs->geom->start + geom->length - 1;
	} else {
		fs->geom->length = geom->length;
		fs->geom->end = fs->geom->start + geom->length - 1;

		if (!ext2_resize_fs(f, geom->length >> (f->logsize - 9)))
			goto error;
	}
	return 1;

error:
	fs->geom->length = old_length;
	fs->geom->end = fs->geom->start + old_length - 1;
	return 0;
}

static PedConstraint*
_ext2_get_resize_constraint (const PedFileSystem* fs)
{
	struct ext2_fs* f = (struct ext2_fs *) fs->type_specific;
	PedDisk*	disk = fs->geom->disk;
	PedAlignment	start_align;
	PedGeometry	start_sector;
	PedGeometry	full_disk;
	PedSector	min_size;

	if (!ped_alignment_init (&start_align, fs->geom->start, 0))
		return NULL;
	if (!ped_geometry_init (&full_disk, disk, 0, disk->dev->length - 1))
		return NULL;
	if (!ped_geometry_init (&start_sector, disk, fs->geom->start, 1))
		return NULL;
	min_size = (EXT2_SUPER_BLOCKS_COUNT(f->sb)
		   	- EXT2_SUPER_FREE_BLOCKS_COUNT(f->sb))
		   * (f->blocksize / PED_SECTOR_SIZE);

	return ped_constraint_new (&start_align, ped_alignment_any,
				   &start_sector, &full_disk, min_size);
}

static int
_ext2_set_system (const PedFileSystem* fs, PedPartition* part,
		  const PedDiskType* disk_type)
{
	if (strcmp (disk_type->name, LOOP_NAME) == 0)
		return 1;

	if (strcmp (disk_type->name, DOS_NAME) == 0) {
		DosPartitionData*	dos_data = part->disk_specific;

		PED_ASSERT (dos_data != NULL, return 0);

		if (dos_data->raid) {
			dos_data->system = 0xfd;
			return 1;
		}
		if (dos_data->lvm) {
			dos_data->system = 0x8e;
			return 1;
		}

		if (dos_data->hidden) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
			       	PED_EXCEPTION_FIX,
				_("Ext2 partitions can't be hidden on msdos "
				  "disk labels."));
			dos_data->hidden = 0;
		}

		dos_data->system = 0x83;
		return 1;
	}

	if (strcmp (disk_type->name, PC98_NAME) == 0) {
		PC98PartitionData*	pc98_data = part->disk_specific;

		PED_ASSERT (pc98_data != NULL, return 0);

		pc98_data->boot = 1;
		pc98_data->system = 0xa0e2;
		return 1;
	}

	if (strcmp (disk_type->name, MAC_NAME) == 0) {
		MacPartitionData*	mac_data = part->disk_specific;

		PED_ASSERT (mac_data != NULL, return 0);

		if (mac_data->is_boot)
			strcpy (mac_data->system_name, "Apple_Bootstrap");
		else
			strcpy (mac_data->system_name, "Apple_UNIX_SVR2");
		mac_data->status = 0x33;
		return 1;
	}

	if (strcmp (disk_type->name, BSD_NAME) == 0) {
		BSDPartitionData* bsd_data = part->disk_specific;
		bsd_data->type = 0x8;
		return 1;
	}

	if (strcmp (disk_type->name, SUN_NAME) == 0) {
		SunPartitionData* sun_data = part->disk_specific;

		if (sun_data->is_lvm) {
			sun_data->type = 0x8e;
			return 1;
		}

		sun_data->type = 0x83;
		return 1;
	}

	return 0;
}
