/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-2001 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "didadit.h"
#include "didadit.moc"

#include "toolbox.h"
#include "channel.h"
#include "global.h"
#include "main.h"
#include "sendqueue.h"
#include "crc.h"
#include "checksums.h"

#include <kwin.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include <errno.h>
#include <zlib.h>


extern TopLevel *toplevel;


//#define DIDADIT_DEBUG
#undef DIDADIT_DEBUG


#define LASTBLOCK_INFO 1
#define LASTBLOCK_START 2
#define LASTBLOCK_REQ 3
#define LASTBLOCK_FIN 4
#define LASTBLOCK_ECHOREQUEST 5


// Fehlercodes, die in LinKT verwendet werden.
// fatale Fehler
char error_wrongtype[] = "100 Unexpected block type\r";
char error_wrongreq[] = "101 REQ block with data out of range\r";
char error_wrongmd5[] = "102 MD5 validation error\r";
char error_wrongoffset[] = "104 Invalid offset/blocksize\r";
char error_wronginfoline[] = "200 Illegal line in INFO block\r";
char error_nomd5hash[] = "202 INFO block without MD5 info\r";
char error_nosize[] = "203 Empty file or missing size info in INFO block\r";
char error_noblocksize[] = "204 INFO block without BLOCKSIZE info\r";
char error_nofilename[] = "205 INFO block without FILENAME info\r";
char error_cannotopen[] = "300 Could not create file\r";
// non-fatale Fehler
char error_notimpl[] = "150 Unknown block type\r";


//   DIDADIT::DIDADIT()
//
// Konstruktor der Klasse DIDADIT.
DIDADIT::DIDADIT( QWidget *_chan )
				: QObject()
{
   chan = _chan;
   blocksize = conf->getRBlocksize()*1024;
   fd_data = -1;
   fd_r = -1;
   rxlistroot = NULL;
   transInfo = NULL;
   reqCount = 0;
   reqLength = 0;
   lastBlockSent = -1;
   gzipfilesize = -1;
   fileSent = false;
	size = -1;
	modtime = -1;
	offset = 0;
   rxData = NULL;
   rxDataLen = 0;
   block.data = NULL;
   starttime = -1;

	status = 100;
	firstblock = true;
	waitForOkAbort = false;
	toplevel->chanListTable->setMode( chan, MODE_DIDADITRX );

   timer = new QTimer( this );
   connect(timer, SIGNAL(timeout()), this, SLOT(sendBlock()));
}


DIDADIT::~DIDADIT()
{
	s_rxliste *tmp;


   if (fd_data != -1)
   {
     ::close(fd_data);
     fd_data = -1;
   }

   if (!allReceived())
      writeRFile();
   else
      deleteRFile();

   if (rxData != NULL) free(rxData);

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   delete timer;

   while (rxlistroot != NULL)
   {
      tmp = rxlistroot;
      rxlistroot = rxlistroot->next;
      free( tmp );
   }

   toplevel->chanListTable->setMode( chan, MODE_INFO );
}


void EscapeIt (char *dest, int & dlen, char ch)
{
  if (ch == FEND)
  {
     dest[dlen] = FESC;
     dest[dlen+1] = TFEND;
     dlen += 2;
  }
  else if (ch == FESC)
       {
          dest[dlen] = FESC;
          dest[dlen+1] = TFESC;
          dlen += 2;
       }
       else
       {
          dest[dlen] = ch;
          dlen++;
       }
}



void DIDADIT::SendBlock( const char *_data, int datalen, unsigned short blockid, int when )
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;
   char *data;

   data = (char *) malloc(datalen*2+100);

   if ((blockid == RBLOCK_ERR) ||
       (blockid == RBLOCK_DATA) ||
       (blockid == RBLOCK_FIN) ||
       (blockid == RBLOCK_REQ) ||
       (blockid == RBLOCK_FINACK))
   {
      // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
      memcpy(data, md5, 16);
      datalen += 16;
      memcpy(data+16, _data, datalen);
   }
   else memcpy(data, _data, datalen);

   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = CheckSums::calcCrcthpStr( data, datalen, CRC );

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
      EscapeIt(sSend, slen, data[i]);

   sSend[slen] = FEND;
   slen++;

   free(data);

   switch (when)
   {
   	case WHEN_NOW:
			SendBytesNow( sSend, slen );
         break;
   	default:
		   SendBytes( sSend, slen);
	}
}


//   void DIDADIT::SendBytes(char *data, int len)
// Eine bestimmte Anzahl Bytes soll ausgesendet werden (nicht anzeigen).
void DIDADIT::SendBytes(char *data, int len)
{
   ((Channel *)chan)->sendString( len, data, false );
}


//   void DIDADIT::SendBytesNow( char *data, int len )
// Eine bestimmte Anzahl Bytes soll JETZT ausgesendet werden (nicht
// anzeigen).
void DIDADIT::SendBytesNow( char *data, int len )
{
   ((Channel *)chan)->sendStringNow( len, data, false );
}


/*
 * void didadit::DecodeBlock(char *data, int len, t_RBlock *block)
 *
 * Den Block in data dekodieren und in die t_RBlock-Struktur
 * packen.
 */
void DIDADIT::DecodeBlock(char *data, int len, t_RBlock *block)
{
   int i;
   unsigned short crc=0;

   block->error = false;


   block->data = (char *) malloc(len);

   block->len = 0;
   i = 1;
   while (i < len)
   {
      if (data[i] == FESC)
      {
         i++;
         switch (data[i])
         {
            case TFEND: block->data[block->len] = FEND; break;
            case TFESC: block->data[block->len] = FESC; break;
            default: block->error = true;
         }
      }
      else block->data[block->len] = data[i];

      i++;
      block->len++;
   }

   // Block-ID
   memcpy(&block->id,block->data,sizeof(block->id));

   // CRC
   memcpy(&block->crc, block->data+block->len-2, sizeof(block->crc));

   // CRC der Daten und der Block-ID errechnen (ohne CRC)
   crc = CheckSums::calcCrcthpStr( block->data, block->len-2, crc );

   // Die Laenge der CRC und die der ID wieder abziehen
   block->len -= 4;
   if (block->len < 0)
   {
   	block->error = true;
      free( block->data );
      block->data = NULL;
   	return;
   }
   memmove(block->data, block->data+2, block->len);


   if (crc != block->crc) block->error = true;

   if (block->error) return;

   // Wenn in diesem Block ein MD5-Hash drin ist, wird er aus dem
   // daten-Feld in das Hash-Feld gepackt
   if ((block->id == RBLOCK_ERR) ||
       (block->id == RBLOCK_DATA) ||
       (block->id == RBLOCK_FIN) ||
       (block->id == RBLOCK_REQ) ||
       (block->id == RBLOCK_FINACK))
   {
      // Ja, MD5-Hash
      memcpy(block->md5, block->data, 16);
      memmove(block->data, block->data+16, block->len-16);
      block->len -= 16;
	}
	block->data[block->len] = '\0';
}


//   void DIDIADIT::sendMessage(char *str)
// Eine Nachricht (Statusmeldung) auf dem Bildschirm anzeigen und, wenn
// moeglich (Gegenstation keine automatische Station) auch aussenden.
void DIDADIT::sendMessage(char *str)
{
   if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( str );
   else
      ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );
}


// void DIDADIT::checkBlock( const char *data, int & len )
//
// Diese Funktion wird immer dann aufgerufen, wenn Daten angekommen
// sind.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist,
//            1, wenn der Block fehlerhaft ist
//            2, wenn der Block OK ist und verarbeitet werden soll
//            3, wenn der Block noch nicht vollstaendig empfangen wurde
//				  4, wenn der Block uebersprungen wurde und die diese Routine
//					  nochmal aufgerufen werden will
int DIDADIT::checkBlock( const char *data, int & len )
{
   int i;

   // Erstmal warten, bis dieser Block komplett empfangen wurde
   if (rxData == NULL)
   {
      // Speicher reservieren
      rxData = (char *) malloc(MAXROHBLOCKSIZE*2);
      rxDataLen = 0;
   }

   // Die jetzt gerade empfangenen Daten an rxData anhaengen
   memcpy(rxData+rxDataLen,data,len);
   rxDataLen += len;
   len = 0;

	// Alles vor dem ersten Auftreten von FEND loeschen
	if ((i = nPOS(FEND, rxData, rxDataLen)) == -1)
   {
      // FEND ist garnicht vorhanden. rxData loeschen
      free(rxData);
      rxData = NULL;
      rxDataLen = 0;
      return 3;
   }
   else
   {
      memmove(rxData,rxData+i,rxDataLen-i);
		rxDataLen -= i;
		rxData[rxDataLen] = '\0';
	}

	// Gucken, ob FEND nochmal vorhanden ist. Wenn ja, ist der ganze
	// Block empfangen worden. In diesem Fall wird der Block dekodiert
	// und in rxData geloescht. Ansonsten wars das erstmal.
	if ((i = nPOS(FEND, rxData+1, rxDataLen-1)) == -1)
	{
		return 3;
	}

	if (i == 0)
	{
		// Erstes FEND entfernen und wieder nach oben in der Schleife
		rxDataLen--;
		memcpy(rxData, rxData+1, rxDataLen);
		return 4;
	}

	// Das uebersprunge erste FEND dazu
	i++;

	// Den Block dekodieren
	DecodeBlock(rxData,i,&block);

	memcpy(rxData,rxData+i,rxDataLen-i);
	rxDataLen -= i;

	// Wenn der Block ungueltig ist, wird abgebrochen.
	if (block.error)
	{
printf("block.error\n");
		return 1;
   }

   return 2;
}


//   bool DIDADIT::readInfoBlock()
//
// "Packt" den Info-Block "aus" und speichert alle Infos ab.
//
// Rueckgabe: true, wenn alles OK ist, false wenn ein Fehler aufgetreten
//            ist. Bei false wird die DIDADIT-Uebertragung abgebrochen.
bool DIDADIT::readInfoBlock()
{
   char tmp[500], *data;
   int i,len;
   char id[600];
   bool md5da=false;

   blocksize = -1;
	size = -1;
   filename = "";


   ((Channel *)chan)->sendString( 5, "#OK#\r", true );

   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }
      else
      {
		  	strcpy(tmp, error_wronginfoline);
			SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
			showError( error_wronginfoline );
         free(data);
      	return false;
      }

      if (!strcmp(id, "FILENAME"))
      {
         filename = tmp;

         // Eventuell fuehrenden Pfad entfernen
         if ((i = filename.findRev('\\')) != -1)
         	filename.remove( 0, i+1 );
         if ((i = filename.findRev('/')) == -1)
         	filename.remove( 0, i+1 );
      }
      if (!strcmp(id, "SIZE"))
      {
         i = atoi(tmp);
         if (i != 0) size = i;
      }
      if (!strcmp(id, "MD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, md5);
         md5da = true;
      }
      if (!strcmp(id, "TIME"))
      {
         i = atoi(tmp);
         if (i != 0) modtime = i;
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
/*      if (!strcmp(id, "COMPRESS"))
      {
      	if ((i = POS(',', tmp)) != -1)
         {
            memcpy(tmp2, tmp, i);
            tmp2[i] = '\0';
            if (!strcmp(tmp2, "gzip"))
            {
            	len = strlen(tmp)-i-1;
	            memmove(tmp, tmp+i+1, len);
               tmp[len] = '\0';
               if ((gzipfilesize = atoi(tmp)) == 0) gzipfilesize = -1;
				}
         }
      }*/
   }

   free(data);


   // Fehlermeldungen, wenn irgendwelche Felder fehlen
   if (!md5da)
   {
   	strcpy(tmp, error_nomd5hash);
		SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		showError( error_nomd5hash );
      return false;
   }

   if (blocksize == -1)
   {
   	strcpy(tmp, error_noblocksize);
		SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		showError( error_noblocksize );
      return false;
   }

   if (filename.isEmpty())
   {
   	strcpy(tmp, error_nofilename);
		SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		showError( error_nofilename );
      return false;
   }

   if (size == -1)
   {
   	strcpy(tmp, error_nosize);
		SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		showError( error_nosize );
      return false;
   }


   // Gucken, ob ein .r - File existiert. Wenn ja wird da der zu benutzende
   // Offset und die Liste der schon empfangenen Bytes eingelesen, wenn der
   // MD5-Hash stimmt.
   offset = lookForRFile();

   // Wenn die beantragte Blocksize groesser ist als die maximale, die wir
   // unterstuetzen, verlangen wir eine kleinere.
   sendBlocksize = false;
   if (blocksize > MAXBLOCKSIZE)
   {
		sendBlocksize = true;
   	blocksize = MAXBLOCKSIZE;
	}

   return true;
}


//   void DIDADIT::sendStartBlock()
//
// Sendet den zu dieser Datei gehoerenden Startblock aus.
void DIDADIT::sendStartBlock()
{
	char tmp[1000];


   if (sendBlocksize)
		sprintf(tmp,"OFFSET=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
						offset, blocksize, DIDADIT_VERSION);
	else
		sprintf(tmp,"OFFSET=%li\rVERSION=%s\r",
						offset, DIDADIT_VERSION);

	SendBlock (tmp, strlen(tmp), RBLOCK_START );

	// DIDADIT-Empfangs-Fenster erstellen
	if (transInfo == NULL)
	{
		// Entsprechendes Fenster oeffnen
		transInfo = new TransferInfo( ((Channel *)chan)->centralWidget, TRANSART_DIDADITRX, filename, size, offset );
		((Channel *)chan)->withTransInfo( transInfo );
	}

   starttime = time(NULL);

	sprintf(tmp, "<LinKT>: DIDADIT-Download started at offset %li.\r" \
                "         Filename: %s, %li bytes\r", offset, filename.latin1(), size);
	((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
}


//   int DIDADIT::proceed(char *data, int len)
//
// Diese Funktion wird vom Terminalprogramm mit allen Didadit-Daten
// aufgerufen und wickelt eigentlich alles ab.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist;
//            1, wenn dies der erste Block ist, der fehlerhaft ist.
//            2, wenn alles OK ist und nix gemacht werden muss
//
// Ist der allererste Block fehlerhaft (im Empfangsfall), scheint die
// Uebertragung nicht mit Didadit zu erfolgen -> fallback auf AutoBIN.
int DIDADIT::proceed(char *data, int len)
{
   char tmp[500];
   int errcode, i;

   if (waitForOkAbort)
   {
      if (!strncmp(data, "#ABORT#", 7))
      {
         strcpy(tmp, "<LinKT>: DIDADIT-Transmission aborted by receiver.\r");
         ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
         return 0;
      }
      if (strncmp(data, "#OK#", 4)) return 2;
      waitForOkAbort = false;
      ((Channel *)chan)->flags &= ~CH_LINEMODE;
   }

   for (;;)
   {
   	if (block.data != NULL)
      {
      	free( block.data );
         block.data = NULL;
		}

	   switch (checkBlock(data,len))
	   {
	      case 0: return 0;
	      case 1: if (firstblock)
	                 return 1;
	              else
	                 continue;
	      case 3: return 2;
         case 4: continue;
	   }
	   firstblock = false;


      // Soll die Uebertragung abgebrochen werden?
      if (block.id == RBLOCK_ABORT)
      {
		   // Ergebnis der Uebertragung ausgeben
         strcpy(tmp, "<LinKT>: Transfer aborted by peer\r");
         ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

		   if (transInfo != NULL)
		   {
		      delete transInfo;
		      transInfo = NULL;
		      ((Channel *)chan)->withoutTransInfo();
		   }

         return 0;
      }

      if (block.id == RBLOCK_ERR)
		{
      	// Ein Fehler ist aufgetreten. Fehlerinfo ausgeben und eventuell
         // abbrechen.
         if ((i = POS(' ', block.data)) != -1)
         {
            memcpy(tmp, block.data, i);
            tmp[i] = '\0';
            block.len -= (i+1);
            memmove( block.data, block.data+i+1, block.len);
            block.data[block.len] = '\0';
            errcode = atoi(tmp);

            if (errcode > 0)
            {
					sprintf(tmp, "<LinKT>: DIDADIT-Error:\r         %s\r", block.data);
					((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

					if (block.data[1] >= '0' && block.data[1] < '5')
						return 0;
               else
                  continue;
				}
         }
      }


      if (block.id == RBLOCK_ECHOREQ)
      {
			SendBlock( block.data, block.len, RBLOCK_ECHOREP );
         continue;
		}

	   switch (status)
	   {
	      case 100: // Erwarte INFO-Block - alles andere wird ignoriert
	                if (block.id == RBLOCK_INFO)
	                {
	                   if (!readInfoBlock()) return 0;
	                   if (!createDataFiles()) return 0;
	                   sendStartBlock();
	                   status = 101;
	                }
	                break;
	      case 101: // 102 = 101, RBLOCK_FIN wurde schon empfangen
	      case 102: switch (block.id)
	                {
	                   case RBLOCK_DATA:
                      		if (!checkMD5()) return 0;
	                        if (readDataBlock())
	                           return 0;
	                        break;
	                   case RBLOCK_FIN:
	                        if (readFinBlock())
	                           return 0;
	                        status = 102;
	                        break;
	                }
	                break;
	   }
   }
}


//   void DIDADIT::readDataBlock()
//
// Ein Datenblock wird empfangen. Abspeichern und merken.
//
// Rueckgabe: true, wenn die komplette Uebertragung fertig ist,
//            false wenn nicht.
bool DIDADIT::readDataBlock()
{
   unsigned long offset;
   unsigned short blocklen;
   char tmp[50];

   // Die ersten 4 Bytes ist der Offset, danach kommt die Blocklaenge
   // (2 Bytes)
   memcpy(&offset, block.data, 4);
   memcpy(&blocklen, block.data+4, 2);
   memmove(block.data, block.data+6, block.len-6);
   block.len -= 6;

   if (offset+blocklen > (unsigned long)size)
   {
		SendBlock( error_wrongoffset, strlen(error_wrongoffset), RBLOCK_ERR );
		showError( error_wrongoffset );
      return true;
   }

   lseek(fd_data, offset, SEEK_SET);
   write(fd_data, block.data, block.len);

   // addRXListe gibt die Anzahl der Bytes zurueck, die bislang empfangen
   // wurden.
   transInfo->setReceivedBytes( addRXListe( offset, offset+blocklen-1 ) );


   if (status == 102)
      if (allReceived())
      {
         SendBlock( tmp, 0, RBLOCK_FINACK );

         // Ergebnis der Uebertragung ausgeben
         showEndText();

         // .r - File loeschen
         deleteRFile();

         if (transInfo != NULL)
         {
            delete transInfo;
            transInfo = NULL;
            ((Channel *)chan)->withoutTransInfo();
         }
         return true;
      }

   return false;
}


//   bool DIDADIT::createDataFiles()
//
// Erzeugt ein File mit dem Namen <filename>.r im abin-Verzeichnis, in dem
// alle moeglichen Informationen abgespeichert werden sowie das eigentliche
// Datenfile (<filename>).
//
// Rueckgabe: true, wenn ok, false wenn ein Fehler aufgetreten ist.
bool DIDADIT::createDataFiles()
{
	char tmp[500];

	sprintf(tmp, "%s/%s", conf->getDirABin().latin1(), filename.latin1());

   if ((fd_data = open(tmp, O_WRONLY|O_CREAT)) == -1)
   {
   	strcpy(tmp, error_cannotopen);
		SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		showError( error_cannotopen );
      return false;
   }
   // Zugriffsrechte einstellen
   fchmod(fd_data, S_IRUSR|S_IWUSR);

   return true;
}


//   long DIDADIT::addRXListe( long start, long ende )
//
// Ein neuer Block wurde empfangen - Eintragen in die Liste im Speicher
//
// Rueckgabe: Anzahl der bislang empfangenen Bytes
long DIDADIT::addRXListe( unsigned long start, unsigned long ende )
{
   s_rxliste *tmp, *neu, *next, *last;
   bool ready;
   long len=0;


   // Allererster Eintrag: Eintragen und 'raus
   if (rxlistroot == NULL)
   {
      rxlistroot = (s_rxliste *) malloc(sizeof(s_rxliste));
      rxlistroot->next = NULL;
      rxlistroot->start = start;
      rxlistroot->ende = ende;
      len = ende-start+1;
      return len;
   }

   // Gucken, ob das im Augenblick empfangene Frame das naechste in
   // der Folge ist
   tmp = rxlistroot;
   ready = false;
   while (tmp != NULL && !ready)
   {
      if (tmp->ende == start-1)
      {
         tmp->ende = ende;
         ready = true;
      }
      else
	      if (ende == tmp->start-1)
         {
            tmp->start = start;
         	ready = true;
         }
      tmp = tmp->next;
   }

   if (!ready)
   {
      // Irgendwo war ein Loch. Gucken, an welcher Stelle dieser Eintrag
      // 'rein muss und einen neuen Eintrag erzeugen
		tmp = rxlistroot;
		ready = false;
		last = NULL;
		while (tmp != NULL && !ready)
      {
			if (start < tmp->start)
         {
				// Neuer Eintrag
				neu = (s_rxliste *) malloc(sizeof(s_rxliste));
				if (last == NULL)
				{
					rxlistroot = neu;
					rxlistroot->next = tmp;
				}
				else
				{
					last->next = neu;
					neu->next = tmp;
				}
				neu->start = start;
				neu->ende = ende;
				ready = true;
         }

         last = tmp;
			if (!ready)
         	tmp = tmp->next;
      }
   }


   if (!ready)
   {
		neu = (s_rxliste *) malloc(sizeof(s_rxliste));
		last->next = neu;
      neu->next = NULL;
      neu->start = start;
      neu->ende = ende;
   }


   // Gucken, ob irgendwelche Bereiche doppelt abgedeckt sind, Eintraege
   // also ueberlappen
   tmp = rxlistroot;
   while (tmp != NULL)
   {
      next = tmp->next;
      if (next != NULL)
         while (tmp->ende >= next->start-1)
         {
            // next reicht in den Bereich von tmp 'rein
            tmp->ende = next->ende;
            // next loeschen
            tmp->next = next->next;
            free(next);
            next = tmp->next;
            if (next == NULL) break;
         }
		len += tmp->ende - tmp->start + 1;
      tmp = tmp->next;
   }



   // DEBUG
#ifdef DIDADIT_DEBUG
   for (tmp=rxlistroot; tmp; tmp=tmp->next)
		printf("start: %li, ende: %li\n", tmp->start, tmp->ende);
	printf("-----------\n");
	printf("len: %li\n", len);
#endif
	return len;
}


//   void DIDADIT::showEndText()
//
// Gibt Statusinformationen am Ende der Uebertragungen zurueck.
void DIDADIT::showEndText()
{
   time_t timediff;
   char *timeptr;
   char tmp2[500];
   char text[500];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   strcpy(text,"<LinKT>: DIDADIT-RX OK. (time: %s, %li bit/s)\xD" \
    				"         filename: %s, %i bytes\xD" \
               "         Requested Blocks: %li (%li bytes)\xD");
   sprintf(tmp2, text,
                 timeptr,((size-offset)*8)/timediff, filename.latin1(), size, reqCount, reqLength);


   if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp2 );
   else
      ((Channel *)chan)->outText( tmp2, strlen(tmp2), OUTCOLOR_TXTEXT );

   free(timeptr);
}



void DIDADIT::sendReqBlock( char *data, int & len, unsigned long start, unsigned long stop )
{
   unsigned long rlen=stop-start+1;
   char str[100];


   memcpy(data+len, &start, 4);
   memcpy(data+len+4, &rlen, 4);
   len += 8;

   reqCount++;
   reqLength += rlen;

	sprintf(str, "<LinKT>: REQ-Block %i sent. Offset=%li, len=%li.\xD", len/8, start, rlen);
	((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );
}



//   void DIDADIT::readFinBlock()
//
// Ein FIN-Block gucken, ob alles da ist. Wenn nicht werden die fehlenden
// Teile neu angefordert.
bool DIDADIT::readFinBlock()
{
	char tmp[MAXBLOCKSIZE];
   int len=0;
   s_rxliste *ltmp;


   if (rxlistroot == NULL)
   {
      // Alles neu, es wurde garnichts empfangen
      sendReqBlock( tmp, len, 0, size );
   	SendBlock( tmp, len, RBLOCK_REQ );
      status = 101;
      return false;
   }


   if ((rxlistroot->start == 0) && (rxlistroot->ende == (unsigned long)size-1))
   {
      // Alles wurde empfangen. FINACK-Block senden; Feierabend

      SendBlock( "", 0, RBLOCK_FINACK );

      // Ergebnis der Uebertragung ausgeben
      showEndText();

      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }

      return true;
   }


   // Teile wurden nicht empfangen
   ltmp = rxlistroot;

   if (rxlistroot != NULL)
      if (rxlistroot->start != 0)
	      sendReqBlock( tmp, len, 0, ltmp->start-1 );

   while (ltmp != NULL && len+8 <= blocksize)
   {
      if (ltmp->next != NULL)
	      sendReqBlock( tmp, len, ltmp->ende+1, ltmp->next->start-1 );
      else
         if (ltmp->ende+1 < (unsigned long)size)
		      sendReqBlock( tmp, len, ltmp->ende+1, size-ltmp->ende );

      ltmp = ltmp->next;
   }

  	SendBlock( tmp, len, RBLOCK_REQ );
   status = 101;

   return false;
}


bool DIDADIT::allReceived()
{
   if (rxlistroot == NULL)
      return false;
   return (rxlistroot->start == 0 && rxlistroot->ende == (unsigned long)size-1);
}


//   void DIDADIT::writeRFile()
//
// Die Infos zu diesem Transfer werden unter <filename>.r abgespeichert
// (Fuer den Resume-Mode)
void DIDADIT::writeRFile()
{
   char tmp[500];
   s_rxliste *rxlist;
   unsigned char hash[16];


   sprintf(tmp, "%s/%s.r", conf->getDirABin().latin1(), filename.latin1());

   if ((fd_r = open(tmp, O_RDWR|O_CREAT)) == -1)
      return;

   // Zugriffsrechte einstellen
   fchmod(fd_r,S_IRUSR|S_IWUSR);

   sprintf(tmp, "%s/%s", conf->getDirABin().latin1(), filename.latin1());
   if (!CheckSums::calcMD5wholefile( tmp, hash ))
   {
   }

   // Die ersten 16 Bytes ist der MD5-Hash des fertigen Files
   write(fd_r, md5, 16);
   // Danach kommt der Hash des schon empfangenen Files
   write(fd_r, hash, 16);

   // Dann kommt die Liste der empfangenen Bytes
   rxlist = rxlistroot;
   while (rxlist != NULL)
   {
      sprintf(tmp,"%li %li\xA", rxlist->start, rxlist->ende);
      write(fd_r, tmp, strlen(tmp));

      rxlist = rxlist->next;
   }

   ::close(fd_r);
}


//   void DIDADIT::deleteRFile()
//
// Wenn das R-File (<filename>.r) existiert, wird es geloescht
void DIDADIT::deleteRFile()
{
   char tmp[500];

   sprintf(tmp, "%s/%s.r", conf->getDirABin().latin1(), filename.latin1());

   if (file_exist(tmp))
      unlink(tmp);
}


//   unsigned long DIDADIT::lookForRFile()
//
// Guckt, ob ein .r-File fuer diesen Dateinamen vorhanden ist. Wenn
// ja wird es eingelesen und der zu verwendende Offset zurueckgegeben.
unsigned long DIDADIT::lookForRFile()
{
   int fd;
   char tmp[500];
   int i;
   unsigned char hash[16];
   FILE *f;
   unsigned long start, ende;
   s_rxliste *rxliste, *last=NULL;


   sprintf(tmp, "%s/%s.r", conf->getDirABin().latin1(), filename.latin1());

   if ((fd = open(tmp, O_RDONLY)) == -1)
      return 0;

   sprintf(tmp, "%s/%s", conf->getDirABin().latin1(), filename.latin1());
   if (!CheckSums::calcMD5wholefile( tmp, hash ))
   {
   	return 0;
   }

   // Der Hash des schon gespeicherten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, md5, 16) != 0)
      return 0;

   // Der Hash des kompletten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, hash, 16) != 0)
      return 0;

   if ((f = fdopen(fd, "r")) == NULL)
      return 0;

   while (fgets(tmp, 499, f) != NULL)
   {
      if ((i = POS('\xA', tmp)) != -1) tmp[i] = '\0';

      if ((i = POS(' ',tmp)) != -1)
      {
         if (sscanf(tmp, "%li %li", &start, &ende) == 2)
         {
            // Neuer Eintrag
            if (rxlistroot == NULL)
            {
               // Erster Eintrag
               rxlistroot = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxlistroot;
            }
            else
            {
               rxliste = rxlistroot;
               while (rxliste->next != NULL) rxliste = rxliste->next;

               rxliste->next = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxliste->next;
            }

            rxliste->next = NULL;
            rxliste->start = start;
            rxliste->ende = ende;
            last = rxliste;
         }
      }
   }

   if (last == NULL) return 0;

   return (last->ende+1);

   ::close(fd);
}


void DIDADIT::getHexMD5( unsigned char *md5, char *hexmd5 )
{
   sprintf(hexmd5, "%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X",
                   md5[0],md5[1],md5[2],md5[3],md5[4],md5[5],md5[6],md5[7],
                   md5[8],md5[9],md5[10],md5[11],md5[12],md5[13],md5[14],md5[15]);
}


void DIDADIT::md5HexToBin( char *hex, unsigned char *bin )
{
   int i;
   char tmp[3];

   tmp[3] = '\0';

   for (i=0; i<16; i++)
   {
      tmp[0] = hex[i*2];
      tmp[1] = hex[(i*2)+1];
      bin[i] = get_hex (tmp);
   }
}


//   void DIDADIT::abortIt()
//
// Sendet einen ABORT-Block an die Gegenseite
void DIDADIT::abortIt()
{
	char tmp[2];
   SendBlock( tmp, 0, RBLOCK_ABORT, WHEN_NOW );
}


void DIDADIT::sendBlock()
{
}


int DIDADIT::getStatus()
{
   return status;
}


bool DIDADIT::checkMD5()
{
	char tmp[100];


	if (!memcmp(md5, block.md5, 16)) return true;

  	strcpy(tmp, error_wrongmd5);
	SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
	showError( error_wrongmd5 );
   return false;
}


void DIDADIT::showError( const char *text )
{
	char tmp[500], tmp2[500];
   int i, len;


   if ((i = POS(' ', text)) == -1)
   	return;

	len = strlen(text)-i-1;
	memcpy(tmp, text+i+1, len);
   tmp[len] = '\0';

	sprintf(tmp2, "<LinKT>: Sent DIDADIT-Error:\r         %s\r", tmp);
	((Channel *)chan)->outText( tmp2, strlen(tmp2), OUTCOLOR_TXTEXT );
}


//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------



DIDADIT_TX::DIDADIT_TX( Channel *chan, const QString & filename, int blocksize )
					: FileTransfer( chan, filename )
{
   struct stat stat_data;


	this->chan = chan;
   if (blocksize > MAXBLOCKSIZE)
   	blocksize = MAXBLOCKSIZE;
   this->blocksize = blocksize;

   fd_data = -1;
   reqlistroot = NULL;
   rxbytes = 0;
   txbytes = 0;
   reqCount = 0;
   reqLength = 0;
   lastBlockSent = -1;
   rxData = NULL;
   rxDataLen = 0;
   starttime = -1;
   waitForOkAbort = true;
   fileSent = false;
	txready = SQTRSTAT_NOSTOP;
	block.data = NULL;


	stat(longname.latin1(), &stat_data);
	size = stat_data.st_size;
	modtime = stat_data.st_mtime;
	if (!CheckSums::calcMD5wholefile( longname, md5 ))
	{
   	txready = SQTRSTAT_EMERGSTOP;
		return;
	}

	status = 1;
}


DIDADIT_TX::~DIDADIT_TX()
{
   s_reqliste *tmp;
	if (rxData != NULL) free( rxData );

   while (reqlistroot != NULL)
   {
   	tmp = reqlistroot;
      reqlistroot = reqlistroot->next;
      free(tmp);
   }

   emit unshowStatus();
}


void DIDADIT_TX::startSending()
{
	sendText( "#DIDADIT#\r" );
	sendInfoBlock();
}


void DIDADIT_TX::sendText( const char *data )
{
	emit sendString( data, strlen(data), true );
}


void DIDADIT_TX::SendBlock( const char *_data, int datalen, unsigned short blockid )
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;
   char *data;

   data = (char *) malloc(datalen*2+100);

   if ((blockid == RBLOCK_ERR) ||
       (blockid == RBLOCK_DATA) ||
       (blockid == RBLOCK_FIN) ||
       (blockid == RBLOCK_REQ) ||
       (blockid == RBLOCK_FINACK))
	{
      // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
      memcpy(data, md5, 16);
      datalen += 16;
      memcpy(data+16, _data, datalen);
   }
   else memcpy(data, _data, datalen);

   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = CheckSums::calcCrcthpStr( data, datalen, CRC );

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
      EscapeIt(sSend, slen, data[i]);

   sSend[slen] = FEND;
   slen++;

   free(data);


	emit sendString( sSend, slen, false );
}


/*
 * void didadit::DecodeBlock(char *data, int len, t_RBlock *block)
 *
 * Den Block in data dekodieren und in die t_RBlock-Struktur
 * packen.
 */
void DIDADIT_TX::DecodeBlock(char *data, int len, t_RBlock *block)
{
   int i;
   unsigned short crc=0;

   block->error = false;


   block->data = (char *) malloc(len);

   block->len = 0;
   i = 1;
   while (i < len)
   {
      if (data[i] == FESC)
      {
         i++;
         switch (data[i])
         {
            case TFEND: block->data[block->len] = FEND; break;
            case TFESC: block->data[block->len] = FESC; break;
            default: block->error = true;
         }
      }
      else block->data[block->len] = data[i];

      i++;
      block->len++;
   }

   // Block-ID
   memcpy(&block->id,block->data,sizeof(block->id));

   // CRC
   memcpy(&block->crc, block->data+block->len-2, sizeof(block->crc));

   // CRC der Daten und der Block-ID errechnen (ohne CRC)
   crc = CheckSums::calcCrcthpStr( block->data, block->len-2, crc);

   // Die Laenge der CRC und die der ID wieder abziehen
   block->len -= 4;
   if (block->len < 0)
   {
   	block->error = true;
      free( block->data );
      block->data = NULL;
   	return;
   }
   memmove(block->data, block->data+2, block->len);


   if (crc != block->crc) block->error = true;

   if (block->error) return;

   // Wenn in diesem Block ein MD5-Hash drin ist, wird er aus dem
   // daten-Feld in das Hash-Feld gepackt
   if ((block->id == RBLOCK_ERR) ||
       (block->id == RBLOCK_DATA) ||
       (block->id == RBLOCK_FIN) ||
       (block->id == RBLOCK_REQ) ||
       (block->id == RBLOCK_FINACK))
   {
      // Ja, MD5-Hash
      memcpy(block->md5, block->data, 16);
      memmove(block->data, block->data+16, block->len-16);
      block->len -= 16;
	}
	block->data[block->len] = '\0';
}


// void DIDADIT_TX::checkBlock( const char *data, int & len )
//
// Diese Funktion wird immer dann aufgerufen, wenn Daten angekommen
// sind.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist,
//            1, wenn der Block fehlerhaft ist
//            2, wenn der Block OK ist und verarbeitet werden soll
//            3, wenn der Block noch nicht vollstaendig empfangen wurde
//				  4, wenn der Block uebersprungen wurde und die diese Routine
//					  nochmal aufgerufen werden will
int DIDADIT_TX::checkBlock( const char *data, int & len )
{
   int i;

   // Erstmal warten, bis dieser Block komplett empfangen wurde
   if (rxData == NULL)
   {
      // Speicher reservieren
      rxData = (char *) malloc(MAXROHBLOCKSIZE*2);
      rxDataLen = 0;
   }

   // Die jetzt gerade empfangenen Daten an rxData anhaengen
   memcpy(rxData+rxDataLen,data,len);
   rxDataLen += len;
   len = 0;

	// Alles vor dem ersten Auftreten von FEND loeschen
	if ((i = nPOS(FEND, rxData, rxDataLen)) == -1)
   {
      // FEND ist garnicht vorhanden. rxData loeschen
      free(rxData);
      rxData = NULL;
      rxDataLen = 0;
      return 3;
   }
   else
   {
      memmove(rxData,rxData+i,rxDataLen-i);
		rxDataLen -= i;
		rxData[rxDataLen] = '\0';
	}

	// Gucken, ob FEND nochmal vorhanden ist. Wenn ja, ist der ganze
	// Block empfangen worden. In diesem Fall wird der Block dekodiert
	// und in rxData geloescht. Ansonsten wars das erstmal.
	if ((i = nPOS(FEND, rxData+1, rxDataLen-1)) == -1)
	{
		return 3;
	}

	if (i == 0)
	{
		// Erstes FEND entfernen und wieder nach oben in der Schleife
		rxDataLen--;
		memcpy(rxData, rxData+1, rxDataLen);
		return 4;
	}

	// Das uebersprunge erste FEND dazu
	i++;

	// Den Block dekodieren
	DecodeBlock(rxData,i,&block);

	memcpy(rxData,rxData+i,rxDataLen-i);
	rxDataLen -= i;

	// Wenn der Block ungueltig ist, wird abgebrochen.
	if (block.error)
	{
printf("block.error\n");
		return 1;
   }

   return 2;
}


//   void DIDADIT_TX::sendInfoBlock()
//
// Sendet ein Info-Frame mit Informationen zu dieser Datei aus
void DIDADIT_TX::sendInfoBlock()
{
   char tmp[1000];
   char hexmd5[33];


   getHexMD5(md5, hexmd5);

   sprintf(tmp,"FILENAME=%s\rSIZE=%li\rMD5=%s\rTIME=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
               shortname.latin1(), size, hexmd5, modtime, blocksize, DIDADIT_VERSION);

/*   if (gzipfilesize < size)
   	sprintf(tmp+strlen(tmp), "COMPRESS=gzip,%i\r",gzipfilesize);*/

   SendBlock (tmp, strlen(tmp), RBLOCK_INFO );

/*   lastBlockSent = LASTBLOCK_INFO;
   timer->start(*/
}


//   void DIDADIT_TX::readStartBlock()
//
// "Packt" den Start-Blocks "aus" und speichert alle Infos ab.
void DIDADIT_TX::readStartBlock()
{
   char tmp[600],*data;
   int i,len,partoffset=-1;
   char id[600];
   unsigned char partmd5[16], calcpartmd5[16];

   partmd5[0] = '\0';
//   blocksize = -1;

   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }

      if (!strcmp(id, "OFFSET"))
      {
         offset = atoi(tmp);
      }
      if (!strcmp(id, "PARTMD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, partmd5);
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
   }

   // Gucken, ob partmd5 korrekt ist, wenn er uebergeben wurde
   if (partmd5[0] != '\0' && offset != 0)
   {
   	partoffset = offset;
      if (!CheckSums::calcMD5partfile( longname, calcpartmd5, offset ))
      	partoffset = 0;
      else
	      if (!memcmp(partmd5, calcpartmd5, 16))
	      	partoffset = 0;
   }

   if (partoffset != -1) offset = partoffset;


	sprintf(tmp, "<LinKT>: DIDADIT-Upload started at offset %li.\r" \
                "         filename: %s, %li bytes\r", offset, shortname.latin1(), size);
	chan->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
   emit showStatus( this, shortname.latin1(), size, offset, TRANSART_DIDADITTX );

   // Auch hier nochmal gucken, ob die Gegenseite zu grosse Bloecke haben
   // will...
   if (blocksize > MAXBLOCKSIZE)
   	blocksize = MAXBLOCKSIZE;

   starttime = time(NULL);
   txbytes = offset;

   free(data);
}


//   void DIDADIT_TX::slotReceivedString( const char *data, int len )
//
// Diese Funktion wird vom Terminalprogramm mit allen Didadit-Daten
// aufgerufen und wickelt eigentlich alles ab.
//
// Ist der allererste Block fehlerhaft (im Empfangsfall), scheint die
// Uebertragung nicht mit Didadit zu erfolgen -> fallback auf AutoBIN.
void DIDADIT_TX::slotReceivedString( const char *data, int len )
{
   char tmp[500];
   int errcode, i;


   if (waitForOkAbort)
   {
      if (!strncmp(data, "#ABORT#", 7))
      {
         strcpy(tmp, "<LinKT>: DIDADIT-Transmission aborted by receiver.\r");
         chan->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
         txready = SQTRSTAT_EMERGSTOP;
         return;
      }
      if (strncmp(data, "#OK#", 4)) return;
      waitForOkAbort = false;
      emit setLinemode( false );
   }

   for (;;)
   {
   	if (block.data != NULL)
      {
      	free( block.data );
         block.data = NULL;
		}
	   switch (checkBlock(data,len))
	   {
	      case 0:
         	txready = SQTRSTAT_STDSTOP;
	      	return;
	      case 1: continue;
	      case 3: return;
         case 4: continue;
	   }


      // Soll die Uebertragung abgebrochen werden?
      if (block.id == RBLOCK_ABORT)
      {
		   // Ergebnis der Uebertragung ausgeben
         strcpy(tmp, "<LinKT>: Transfer aborted by peer\r");
         chan->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

         emit unshowStatus();

         txready = SQTRSTAT_EMERGSTOP;
         return;
      }

      if (block.id == RBLOCK_ERR)
		{
      	// Ein Fehler ist aufgetreten. Fehlerinfo ausgeben und eventuell
         // abbrechen.
         if ((i = POS(' ', block.data)) != -1)
         {
            memcpy(tmp, block.data, i);
            tmp[i] = '\0';
            block.len -= (i+1);
            memmove( block.data, block.data+i+1, block.len);
            block.data[block.len] = '\0';
            errcode = atoi(tmp);

            if (errcode > 0)
            {
					sprintf(tmp, "<LinKT>: DIDADIT-Error:\r         %s\r", block.data);
					chan->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

					if (block.data[1] >= '0' && block.data[1] < '5')
               {
               	txready = SQTRSTAT_EMERGSTOP;
						return;
               }
               else
                  continue;
				}
         }
      }


      if (block.id == RBLOCK_ECHOREQ)
      {
			SendBlock( block.data, block.len, RBLOCK_ECHOREP );
         continue;
		}

	   switch (status)
	   {
	      case 1: // Erwarte START-Block
	              readStartBlock();
	              status = 2;
	              if (offset == (unsigned long)size)
					  {
		              SendBlock( "", 0, RBLOCK_FIN );
                    fileSent = true;
					  }
					  else
                    emit sendData();
	              break;
	      case 2: switch (block.id)
	              {
	                 case RBLOCK_FINACK:
	                      readFinAck();
                         txready = SQTRSTAT_STDSTOP;
	                      return;
	                 case RBLOCK_REQ:
	                      if (!readReqBlock())
	                      {
                         	 txready = SQTRSTAT_EMERGSTOP;
	                      	 return;
								 }
	                      break;
                    default:
   	                   strcpy(tmp, error_notimpl);
		                   SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
		                   showError( error_notimpl );
                         break;
	              }
                 break;
	   }
   }
}


void DIDADIT_TX::getFileData()
{
   static char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;
   unsigned short blockid=RBLOCK_DATA;
   long datalen;
   unsigned short dlen;
   bool request=false;
   char data[MAXROHBLOCKSIZE];
   unsigned long loffset;
   s_reqliste *tmp;
   char str[100];


   if (status != 2)
   	return;

   // Weitere Daten aus dem Sourcefile holen
   if (fd_data == -1)
   {
      if ((fd_data = open(longname.latin1(), O_RDONLY)) == -1)
      {
printf("cannot open sourcefile: %s\n", strerror(errno));
	   	strcpy(str, error_cannotopen);
			SendBlock( str, strlen(str), RBLOCK_ERR );
			showError( error_cannotopen );
         return;
      }
   }

   if (reqlistroot != NULL)
   {
   	request = true;
      loffset = lseek( fd_data, reqlistroot->offset, SEEK_SET );
      datalen = read( fd_data, data, reqlistroot->len );

		if (datalen > blocksize)
		{
      	datalen = blocksize;
         reqlistroot->len -= blocksize;
         reqlistroot->offset += blocksize;
		}
		else
		{
			tmp = reqlistroot;
			reqlistroot = reqlistroot->next;
			free(tmp);
		}
   }
   else
   {
	  	if (fileSent)
      	return;

      loffset = lseek(fd_data, offset, SEEK_SET);

      if ((datalen = read(fd_data, data, blocksize)) != blocksize)
      {
         if (datalen == -1)
         {
            printf("read()-error: %s\n", strerror(errno));
            return;
         }
         // Das File ist komplett ausgesendet worden. FIN-Block senden
         fileSent = true;
      }
      offset += datalen;
   }

   // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
   dlen = (unsigned short) datalen;
   memmove(data+22, data, datalen);
   datalen += 22;
   // Offset und Blocklaenge dazu
   memcpy(data, md5, 16);
   memcpy(data+16, &loffset, 4);
   memcpy(data+20, &dlen, 2);


   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = CheckSums::calcCrcthpStr( data, datalen, CRC );

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
      EscapeIt(sSend, slen, data[i]);

   sSend[slen] = FEND;
   slen++;


/////////////////////////// DEBUGGING //////////////////////////////////
#ifdef DIDADIT_DEBUG
   if ((1+(int) (100.0*rand()/(RAND_MAX+1.0))) < 20)
   {
      sSend[100] = 'A';
      sSend[101] = 'B';
      sSend[102] = 'C';

      sprintf(str, "<LinKT>: Error sent - offset: %li, len: %i\r", loffset, dlen);
		chan->outText( str, strlen(str), OUTCOLOR_TXTEXT );
   }
#endif
/////////////////////////// DEBUGGING //////////////////////////////////

   if (!request)
   {
	   txbytes += dlen;

      emit statusRxBytes( txbytes );
	}

	// Block aussenden
   emit sendString( sSend, slen, false );

	if (fileSent && reqlistroot == NULL)
	{
		SendBlock( "", 0, RBLOCK_FIN );
		sprintf(str, "<LinKT>: FIN sent\r");
		chan->outText( str, strlen(str), OUTCOLOR_TXTEXT );
	}
}



//   void DIDADIT_TX::showEndText()
//
// Gibt Statusinformationen am Ende der Uebertragungen zurueck.
// Parameter rx ist true, wenn das File empfangen wurde, sonst false.
void DIDADIT_TX::showEndText()
{
   time_t timediff;
   char *timeptr;
   char tmp2[500];
   char text[500];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   strcpy(text,"<LinKT>: DIDADIT-TX OK. (time: %s, %li bit/s)\xD" \
					"         filename: %s, %i bytes\xD" \
					"         Retransmitted Blocks: %li (%li bytes)\xD");
   sprintf(tmp2, text,
                 timeptr,((size-offset)*8)/timediff, shortname.latin1(), size, reqCount, reqLength);

	sendMessage(tmp2);

   free(timeptr);
}



void DIDADIT_TX::readFinAck()
{
   // Ergebnis der Uebertragung ausgeben
   showEndText();

   emit unshowStatus();
   txready = SQTRSTAT_STDSTOP;
}


void DIDADIT_TX::getHexMD5( unsigned char *md5, char *hexmd5 )
{
   sprintf(hexmd5, "%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X",
                   md5[0],md5[1],md5[2],md5[3],md5[4],md5[5],md5[6],md5[7],
                   md5[8],md5[9],md5[10],md5[11],md5[12],md5[13],md5[14],md5[15]);
}


void DIDADIT_TX::md5HexToBin( char *hex, unsigned char *bin )
{
   int i;
   char tmp[3];

   tmp[3] = '\0';

   for (i=0; i<16; i++)
   {
      tmp[0] = hex[i*2];
      tmp[1] = hex[(i*2)+1];
      bin[i] = get_hex (tmp);
   }
}


bool DIDADIT_TX::checkMD5()
{
	char tmp[100];


	if (!memcmp(md5, block.md5, 16)) return true;

  	strcpy(tmp, error_wrongmd5);
	SendBlock( tmp, strlen(tmp), RBLOCK_ERR );
	showError( error_wrongmd5 );
   return false;
}


void DIDADIT_TX::showError( const char *text )
{
	char tmp[500], tmp2[500];
   int i, len;


   if ((i = POS(' ', text)) == -1)
   	return;

	len = strlen(text)-i-1;
	memcpy(tmp, text+i+1, len);
   tmp[len] = '\0';

	sprintf(tmp2, "<LinKT>: Sent DIDADIT-Error:\r         %s\r", tmp);
	chan->outText( tmp2, strlen(tmp2), OUTCOLOR_TXTEXT );
}


//   bool DIDADIT_TX::readReqBlock()
//
// Ein REQ-Block wird empfangen. Daten in die entsprechende verkettete
// Liste eintragen und das Aussenden anfordern, wenn gerade keine REQ-
// Bloecke empfangen werden.
bool DIDADIT_TX::readReqBlock()
{
	unsigned long offset, len, dataoff=0;
	s_reqliste *tmp;
	char str[100];


   // Da immer nur ein REQ-Block zu einer Zeit ausstehen kann, wird bei jedem
   // neuen REQ-Block die alte Request-Liste geloescht.
   while (reqlistroot != NULL)
   {
   	tmp = reqlistroot;
      reqlistroot = reqlistroot->next;
      free(tmp);
   }


   if (block.len % 8 != 0)
   {
		SendBlock( error_wrongreq, strlen(error_wrongreq), RBLOCK_ERR );
		showError( error_wrongreq );
		return false;
   }

   while (block.len >= (int)dataoff+8)
   {
		memcpy( &offset, block.data+dataoff, 4 );
		memcpy( &len, block.data+dataoff+4, 4 );
      dataoff += 8;

	   if ((int)(offset+len) > size)
	   {
			SendBlock( error_wrongreq, strlen(error_wrongreq), RBLOCK_ERR );
			showError( error_wrongreq );
	   	return false;
	   }

		reqCount++;
		reqLength += len;

		if (reqlistroot == NULL)
		{
			// Erster Eintrag
			reqlistroot = (s_reqliste *)malloc(sizeof(s_reqliste));
			tmp = reqlistroot;
		}
		else
		{
			tmp = reqlistroot;
			while (tmp->next != NULL) tmp = tmp->next;

			tmp->next = (s_reqliste *)malloc(sizeof(s_reqliste));
			tmp = tmp->next;
		}

		tmp->next = NULL;
		tmp->offset = offset;
		tmp->len = len;

		sprintf(str, "<LinKT>: Received REQ-Block %li: Offset=%li, len=%li.\xD", dataoff/8, offset, len);
		chan->outText( str, strlen(str), OUTCOLOR_TXTEXT );
	}

	emit newTXData();
   return true;
}


int DIDADIT_TX::sentReady()
{
   return txready;
}


void DIDADIT_TX::abortTransmission()
{
	char tmp[100];


   strcpy(tmp,"<LinKT>: DIDADIT-TX aborted.\xD");
	sendMessage(tmp);

   emit deleteFTData();
	txready = SQTRSTAT_ABORT;
	SendBlock( "", 0, RBLOCK_ABORT );
   emit setLinemode( true );
}



