#include "system.h"

#include "rpmlib.h"

#include "rpmio.h"

/* =============================================================== */

/*@null@*/ FD_t fdNew(void) {
    FD_t fd = (FD_t)malloc(sizeof(struct _FD));
    if (fd == NULL)
	return NULL;
    fd->fd_fd = -1;
    fd->fd_bzd = NULL;
    fd->fd_gzd = NULL;
    fd->fd_url = NULL;
    return fd;
}

int fdValid(FD_t fd) {
    return (fd != NULL && fd->fd_fd >= 0);
}

int fdFileno(FD_t fd) {
    return (fd != NULL ? fd->fd_fd : -1);
}

/*@null@*/ FD_t fdOpen(const char *pathname, int flags, mode_t mode) {
    FD_t fd;
    int fdno;
    if ((fdno = open(pathname, flags, mode)) < 0)
	return NULL;
    fd = fdNew();
    fd->fd_fd = fdno;
    return fd;
}

/*@null@*/ FD_t fdDup(int fdno) {
    FD_t fd;
    int nfdno;
    if ((nfdno = dup(fdno)) < 0)
	return NULL;
    fd = fdNew();
    fd->fd_fd = nfdno;
    return fd;
}

off_t fdLseek(FD_t fd, off_t offset, int whence) {
    int rc = lseek(fdFileno(fd), offset, whence);
    return rc;
}

ssize_t fdRead(FD_t fd, void * buf, size_t count) {
    int rc = read(fdFileno(fd), buf, count);
    return rc;
}

ssize_t fdWrite(FD_t fd, const void * buf, size_t count) {
    int rc = write(fdFileno(fd), buf, count);
    return rc;
}

int fdClose(/*@only@*/ FD_t fd) {
    int fdno;

    if (fd != NULL && (fdno = fdFileno(fd)) >= 0) {
	fd->fd_fd = -1;
	fd->fd_bzd = NULL;
	fd->fd_gzd = NULL;
	free(fd);
    	return close(fdno);
    }
    return -2;
}

/*@-mustfree*/
/*@shared@*/ FILE *fdFdopen(/*@owned@*/ FD_t fd, const char *mode) {
    FILE *f = fdopen(fdFileno(fd), mode);
    if (f != NULL) {
	fd->fd_fd = -1;
	free(fd);
	return f;
    }
    return NULL;
}
/*@=mustfree*/

/* =============================================================== */
/* Support for GZIP library.
 */
#ifdef	HAVE_ZLIB_H

#include <zlib.h>

gzFile * gzdFileno(FD_t fd) {
    return (fd != NULL ? ((gzFile *)fd->fd_gzd) : NULL);
}

/*@null@*/ FD_t gzdOpen(const char *pathname, const char *mode) {
    FD_t fd;
    gzFile *gzfile;;
    if ((gzfile = gzopen(pathname, mode)) == NULL)
	return NULL;
    fd = fdNew();
    fd->fd_gzd = gzfile;
    return fd;
}

/*@shared@*/ FD_t gzdFdopen(FD_t fd, const char *mode) {
    gzFile *gzfile  = gzdopen(fdFileno(fd), mode);
    if (gzfile != NULL) {
	fd->fd_fd = -1;
	fd->fd_gzd = gzfile;
	return fd;
    }
    return NULL;
}

ssize_t gzdRead(FD_t fd, void * buf, size_t count) {
    return gzread(gzdFileno(fd), buf, count);
}

ssize_t gzdWrite(FD_t fd, const void * buf, size_t count) {
    return gzwrite(gzdFileno(fd), (void *)buf, count);
}

#if 0	/* XXX zlib-1.0.4 has not */
off_t gzdLseek(FD_t fd, off_t offset, int whence) {
    return gzseek(gzdFileno(fd), offset, whence);
}
#endif

int gzdFlush(FD_t fd) {
    return gzflush(gzdFileno(fd), Z_SYNC_FLUSH);	/* XXX W2DO? */
}

char * gzdStrerror(FD_t fd) {
    static char *zlib_err [] = {
	"OK"
	"Errno",
	"Stream", 
	"Data", 
	"Memory",
	"Buffer",
	"Version"
    };  

    int zerror;

    gzerror(gzdFileno(fd), &zerror);
    switch (zerror) {
    case Z_ERRNO:
	return strerror(errno);
	break;
    default:
	break;
    }
    return zlib_err[-zerror];
}

int gzdClose(/*@only@*/ FD_t fd) {
    gzFile *gzfile;
    int zerror;

    if (fd != NULL && (gzfile = gzdFileno(fd)) != NULL) {
	fd->fd_fd = -1;
	fd->fd_bzd = NULL;
	fd->fd_gzd = NULL;
	free(fd);
    	zerror = gzclose(gzfile);
	return 0;
    }
    return -2;
}

#endif	/* HAVE_ZLIB_H */
