/*
 **********************************************************************
 *     cardwo.c - PCM output HAL for emu10k1 driver
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

#include <linux/poll.h>
#include "hwaccess.h"
#include "8010.h"
#include "voicemgr.h"
#include "cardwo.h"
#include "audio.h"

static u32 samplerate_to_linearpitch(u32 samplingrate)
{
	samplingrate = (samplingrate << 8) / 375;
	return (samplingrate >> 1) + (samplingrate & 1);
}

static void query_format(struct wave_format *wave_fmt)
{
	if ((wave_fmt->channels != 1) && (wave_fmt->channels != 2))
		wave_fmt->channels = 2;

	if (wave_fmt->samplingrate >= 0x2ee00)
		wave_fmt->samplingrate = 0x2ee00;

	if ((wave_fmt->bitsperchannel != 8) && (wave_fmt->bitsperchannel != 16))
		wave_fmt->bitsperchannel = 16;

	wave_fmt->bytesperchannel = wave_fmt->bitsperchannel >> 3;
	wave_fmt->bytespersample = wave_fmt->channels * wave_fmt->bytesperchannel;
	wave_fmt->bytespersec = wave_fmt->bytespersample * wave_fmt->samplingrate;

	return;
}

static int alloc_buffer(struct emu10k1_card *card, struct waveout_buffer *buffer)
{
	u32 pageindex, pagecount;
	unsigned long busaddx;
	int i;

	buffer->pos = 0;
	buffer->silence_pos = 0;
	buffer->hw_pos = 0;

	DPD(2, "requested pages is: %d\n", buffer->pages);

	if ((buffer->emupageindex = emu10k1_addxmgr_alloc(buffer->pages * PAGE_SIZE, card)) < 0)
		return CTSTATUS_ERROR;

	/* Fill in virtual memory table */
	for (pagecount = 0; pagecount < buffer->pages; pagecount++) {
		if ((buffer->addr[pagecount] = pci_alloc_consistent(card->pci_dev, PAGE_SIZE, &buffer->dma_handle[pagecount])) == NULL) {
			buffer->pages = pagecount;
			return CTSTATUS_ERROR;
		}

		DPD(2, "Virtual Addx: %p\n", buffer->addr[pagecount]);

		for (i = 0; i < PAGE_SIZE / EMUPAGESIZE; i++) {
			busaddx = buffer->dma_handle[pagecount] + i * EMUPAGESIZE;

			DPD(3, "Bus Addx: %lx\n", busaddx);

			pageindex = buffer->emupageindex + pagecount * PAGE_SIZE / EMUPAGESIZE + i;

			((u32 *) card->virtualpagetable.addr)[pageindex] = (busaddx * 2) | pageindex;
		}
	}

	return CTSTATUS_SUCCESS;
}

static void free_buffer(struct emu10k1_card *card, struct waveout_buffer *buffer)
{
	u32 pagecount, pageindex;
	int i;

	if (buffer->emupageindex < 0)
		return;

	for (pagecount = 0; pagecount < buffer->pages; pagecount++) {
		pci_free_consistent(card->pci_dev, PAGE_SIZE, buffer->addr[pagecount], buffer->dma_handle[pagecount]);

		for (i = 0; i < PAGE_SIZE / EMUPAGESIZE; i++) {
			pageindex = buffer->emupageindex + pagecount * PAGE_SIZE / EMUPAGESIZE + i;
			((u32 *) card->virtualpagetable.addr)[pageindex] = (card->silentpage.dma_handle * 2) | pageindex;
		}
	}

	emu10k1_addxmgr_free(card, buffer->emupageindex);
	buffer->emupageindex = -1;

	return;
}

static int get_voice(struct emu10k1_card *card, struct woinst *woinst)
{
	struct voice_allocdesc voice_allocdesc;

	/* Allocate voices here, if no voices available, return error.
	 * Init voice_allocdesc first.*/

	voice_allocdesc.usage = VOICE_USAGE_PLAYBACK;

	voice_allocdesc.flags = 0;

	if (woinst->format.channels == 2)
		voice_allocdesc.flags |= VOICE_FLAGS_STEREO;

	if (woinst->format.bitsperchannel == 16)
		voice_allocdesc.flags |= VOICE_FLAGS_16BIT;

	if (emu10k1_voice_alloc(card, &woinst->voice, &voice_allocdesc) != CTSTATUS_SUCCESS)
		return CTSTATUS_ERROR;

	/* Calculate pitch */
	woinst->voice.initial_pitch = (u16) (srToPitch(woinst->format.samplingrate) >> 8);
	woinst->voice.pitch_target = samplerate_to_linearpitch(woinst->format.samplingrate);

	DPD(2, "Initial pitch --> %x\n", woinst->voice.initial_pitch);

	woinst->voice.start = 2 * (woinst->buffer.emupageindex << 11) / woinst->format.bytespersample;
	woinst->voice.end = woinst->voice.start + woinst->buffer.size / woinst->format.bytespersample;
	woinst->voice.startloop = woinst->voice.start;
	woinst->voice.endloop = woinst->voice.end;

	woinst->voice.params[0].send_a=0x00;
	woinst->voice.params[0].send_b=0x00;
	woinst->voice.params[0].send_c=0xff;
	woinst->voice.params[0].send_d=0x00;

	woinst->voice.params[0].bus_routing= (woinst->device == 1) ? 0xd23c : 0xd01c;
	woinst->voice.params[0].volume_target=0xffff;
	woinst->voice.params[0].initial_fc=0xff;
	woinst->voice.params[0].initial_attn=0x00;
	woinst->voice.params[0].byampl_env_sustain = 0x7f;
	woinst->voice.params[0].byampl_env_decay = 0x7f;

	if (woinst->voice.flags & VOICE_FLAGS_STEREO) {
		woinst->voice.params[1].send_a=0x00;
		woinst->voice.params[1].send_b=0xff;		
		woinst->voice.params[1].send_c=0x00;
		woinst->voice.params[1].send_d=0x00;

		woinst->voice.params[1].bus_routing= (woinst->device == 1) ? 0xd23c : 0xd01c;
		woinst->voice.params[1].volume_target=0xffff;
		woinst->voice.params[1].initial_fc=0xff;
		woinst->voice.params[1].initial_attn=0x00;
		woinst->voice.params[1].byampl_env_sustain = 0x7f;
		woinst->voice.params[1].byampl_env_decay = 0x7f;
	} else {
		woinst->voice.params[0].send_b=0xff;
	}

	DPD(2, "voice: start=%x, end=%x, startloop=%x, endloop=%x\n", woinst->voice.start, woinst->voice.end,woinst->voice.startloop, woinst->voice.endloop);

	emu10k1_voice_playback_setup(&woinst->voice);

	return CTSTATUS_SUCCESS;
}

int emu10k1_waveout_open(struct emu10k1_wavedevice *wave_dev)
{
	struct emu10k1_card *card = wave_dev->card;
	struct woinst *woinst = wave_dev->woinst;
	u32 delay;

	DPF(2, "emu10k1_waveout_open()\n");

	if (alloc_buffer(card, &woinst->buffer) != CTSTATUS_SUCCESS) {
		ERROR();
		emu10k1_waveout_close(wave_dev);
		return CTSTATUS_ERROR;
	}

	woinst->buffer.fill_silence = 0;
	woinst->buffer.silence_bytes = 0;

	if (get_voice(card, woinst) != CTSTATUS_SUCCESS) {
		ERROR();
		emu10k1_waveout_close(wave_dev);
		return CTSTATUS_ERROR;
	}

	delay = (48000 * woinst->buffer.fragment_size) / woinst->format.bytespersec;

	emu10k1_timer_install(card, &woinst->timer, emu10k1_waveout_bh, (unsigned long) wave_dev, delay / 2);

	woinst->state = WAVE_STATE_OPEN;

	return CTSTATUS_SUCCESS;
}

void emu10k1_waveout_close(struct emu10k1_wavedevice *wave_dev)
{
	struct emu10k1_card *card = wave_dev->card;
	struct woinst *woinst = wave_dev->woinst;

	DPF(2, "emu10k1_waveout_close()\n");

	emu10k1_waveout_stop(wave_dev);

	emu10k1_timer_uninstall(card, &woinst->timer);

	emu10k1_voice_free(&woinst->voice);

	free_buffer(card, &woinst->buffer);

	woinst->state = WAVE_STATE_CLOSED;

	return;
}

void emu10k1_waveout_start(struct emu10k1_wavedevice *wave_dev)
{
	struct emu10k1_card *card = wave_dev->card;
	struct woinst *woinst = wave_dev->woinst;

	DPF(2, "emu10k1_waveout_start()\n");
	/* Actual start */
	emu10k1_voice_start(&woinst->voice);

	emu10k1_timer_enable(card, &woinst->timer);

	woinst->state |= WAVE_STATE_STARTED;

	return;
}

int emu10k1_waveout_setformat(struct emu10k1_wavedevice *wave_dev, struct wave_format *format)
{
	struct emu10k1_card *card = wave_dev->card;
	struct woinst *woinst = wave_dev->woinst;
	u32 delay;

	DPF(2, "emu10k1_waveout_setformat()\n");

	if (woinst->state & WAVE_STATE_STARTED)
		return CTSTATUS_ERROR;

	query_format(format);

	if (woinst->format.samplingrate != format->samplingrate ||
	    woinst->format.channels != format->channels ||
	    woinst->format.bitsperchannel != format->bitsperchannel) {

		woinst->format = *format;

		if (woinst->state == WAVE_STATE_CLOSED)
			return CTSTATUS_SUCCESS;

		emu10k1_timer_uninstall(card, &woinst->timer);
		emu10k1_voice_free(&woinst->voice);

		if (get_voice(card, woinst) != CTSTATUS_SUCCESS) {
			ERROR();
			emu10k1_waveout_close(wave_dev);
			return CTSTATUS_ERROR;
		}

		delay = (48000 * woinst->buffer.fragment_size) / woinst->format.bytespersec;

		emu10k1_timer_install(card, &woinst->timer, emu10k1_waveout_bh, (unsigned long) wave_dev, delay / 2);
	}

	return CTSTATUS_SUCCESS;
}

void emu10k1_waveout_stop(struct emu10k1_wavedevice *wave_dev)
{
	struct emu10k1_card *card = wave_dev->card;
	struct woinst *woinst = wave_dev->woinst;

	DPF(2, "emu10k1_waveout_stop()\n");

	if (!(woinst->state & WAVE_STATE_STARTED))
		return;

	emu10k1_timer_disable(card, &woinst->timer);

	/* Stop actual voice */
	emu10k1_voice_stop(&woinst->voice);

	emu10k1_waveout_update(woinst);

	DPD(2, "position is %x\n", woinst->buffer.hw_pos / woinst->format.bytespersample);

	woinst->state &= ~WAVE_STATE_STARTED;

	return;
}

void emu10k1_waveout_getxfersize(struct woinst *woinst, u32 * size)
{
	struct waveout_buffer *buffer = &woinst->buffer;
	u32 pending;

	if (buffer->hw_pos > buffer->silence_pos ||
	   (buffer->hw_pos == buffer->silence_pos && woinst->state & WAVE_STATE_STARTED)) {
		*size = buffer->hw_pos - buffer->silence_pos;
	} else {
		*size = buffer->size + buffer->hw_pos - buffer->silence_pos;
	}

	pending = buffer->size - *size;

	buffer->fill_silence = (pending < buffer->fragment_size) ? 1 : 0;

	if (pending > buffer->silence_bytes) {
		*size += buffer->silence_bytes;
	} else {
		*size = buffer->size;
		buffer->pos = buffer->hw_pos;
	}

	return;
}

static void copy_block(void **dst, u32 str, u8 *src, u32 len)
{
	int i, j, k;

	i = str / PAGE_SIZE;
	j = str % PAGE_SIZE;
	k = (len > PAGE_SIZE - j) ? PAGE_SIZE - j : len;
	copy_from_user(dst[i] + j, src, k);
	len -= k;
	while (len >= PAGE_SIZE) {
		copy_from_user(dst[++i], src + k, PAGE_SIZE);
		k += PAGE_SIZE;
		len -= PAGE_SIZE;
	}
	copy_from_user(dst[++i], src + k, len);

	return;
}

static void fill_block(void **dst, u32 str, u8 src, u32 len)
{
	int i, j, k;

	i = str / PAGE_SIZE;
	j = str % PAGE_SIZE;
	k = (len > PAGE_SIZE - j) ? PAGE_SIZE - j : len;
	memset(dst[i] + j, src, k);
	len -= k;
	while (len >= PAGE_SIZE) {
		memset(dst[++i], src, PAGE_SIZE);
		len -= PAGE_SIZE;
	}
	memset(dst[++i], src, len);

	return;
}

void emu10k1_waveout_xferdata(struct woinst *woinst, u8 *data, u32 *size)
{
	struct waveout_buffer *buffer = &woinst->buffer;
	u32 sizetocopy, sizetocopy_now, start;
	unsigned long flags;

	sizetocopy = min(buffer->size, *size);
	*size = sizetocopy;

	if (!sizetocopy)
		return;

	spin_lock_irqsave(&woinst->lock, flags);
	buffer->silence_bytes = 0;
	sizetocopy_now = buffer->size - buffer->pos;

	start = buffer->pos;

	if (sizetocopy > sizetocopy_now) {
		sizetocopy -= sizetocopy_now;
		buffer->pos = sizetocopy;
		buffer->silence_pos = buffer->pos;
		spin_unlock_irqrestore(&woinst->lock, flags);

		copy_block(buffer->addr, start, data, sizetocopy_now);
		copy_block(buffer->addr, 0, data + sizetocopy_now, sizetocopy);
	} else {
		if (sizetocopy == sizetocopy_now)
			buffer->pos = 0;
		else
			buffer->pos += sizetocopy;

		buffer->silence_pos = buffer->pos;
		spin_unlock_irqrestore(&woinst->lock, flags);

		copy_block(buffer->addr, start, data, sizetocopy);
	}

	return;
}

void emu10k1_waveout_fillsilence(struct woinst *woinst)
{
	struct waveout_buffer *buffer = &woinst->buffer;
	u16 filldata;
	u32 sizetocopy, sizetocopy_now, start;
	unsigned long flags;

	sizetocopy = woinst->buffer.fragment_size;

	if (woinst->format.bitsperchannel == 16)
		filldata = 0x0000;
	else
		filldata = 0x8080;

	spin_lock_irqsave(&woinst->lock, flags);
	buffer->silence_bytes += sizetocopy;
	start = buffer->silence_pos;
	sizetocopy_now = buffer->size - start;

	if (sizetocopy > sizetocopy_now) {
		sizetocopy -= sizetocopy_now;
		buffer->silence_pos = sizetocopy;
		spin_unlock_irqrestore(&woinst->lock, flags);
		fill_block(buffer->addr, start, filldata, sizetocopy_now);
		fill_block(buffer->addr, 0, filldata, sizetocopy);
	} else {
		if (sizetocopy == sizetocopy_now)
			buffer->silence_pos = 0;
		else
			buffer->silence_pos += sizetocopy;

		spin_unlock_irqrestore(&woinst->lock, flags);

		fill_block(buffer->addr, start, filldata, sizetocopy);
	}

	return;
}

void emu10k1_waveout_update(struct woinst *woinst)
{
	u32 hw_pos;

	/* There is no actual start yet */
	if (!(woinst->state & WAVE_STATE_STARTED)) {
		hw_pos = woinst->buffer.hw_pos;
	} else {
		u32 samples;
		/* hw_pos in sample units */
		emu10k1_voice_getcontrol(&woinst->voice, CCCA_CURRADDR, &hw_pos);

		hw_pos -= woinst->voice.start;

		hw_pos *= woinst->format.bytespersample;

		/* Refer to voicemgr.c, CA is not started at zero.
		 * We need to take this into account. */
		samples = 64 * woinst->format.channels - 4 * woinst->format.bytesperchannel;

		if (hw_pos >= samples)
			hw_pos -= samples;
		else
			hw_pos += woinst->buffer.size - samples;
	}

	if (hw_pos >= woinst->buffer.hw_pos)
		woinst->total_played += hw_pos - woinst->buffer.hw_pos;
	else
		woinst->total_played += woinst->buffer.size + hw_pos - woinst->buffer.hw_pos;

	woinst->buffer.hw_pos = hw_pos;

	return;
}
