/*
 *  MonKT - a monitor for LinKT
 *  Copyright (C) 1998-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "dxwin.h"
#include "global.h"
#include "toolbox.h"
#include "main.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <kapp.h>
#include <kmsgbox.h>

#include <qlayout.h>
#include <qwidget.h>
#include <qlabel.h>


#define BUTTONWIDTH 65
#define BUTTONHEIGHT 25



extern TopLevel *toplevel;



void DXWinTbl::mousePressEvent( QMouseEvent *e )
{

   if (e->button() == RightButton)
   {
      emit showKontextMenu();
      return;
   }

   KTabListBox::mousePressEvent( e );
}


DXWinTbl::DXWinTbl( QWidget *parent ) : KTabListBox( parent )
{
}



DXWin::DXWin( QWidget *parent, QWidget *tl )
                    : QWidget(parent)
{
   tabbar = new QTabBar( this );
   connect(tabbar, SIGNAL(selected(int)), this, SLOT(changeTab(int)));

   dxviewroot = NULL;
   akttab = -1;
   aktview = NULL;


   kontextmenu = new QPopupMenu();
   kontextmenu->insertItem( klocale->translate("&New view"), (TopLevel *)tl, SLOT(slotNewView()));
   kontextmenu->insertSeparator( -1 );
   kontextmenu->insertItem( klocale->translate("&Delete view"), (TopLevel *)tl, SLOT(slotDeleteView()));


   loadViewsList();
}


void DXWin::resizeEvent( QResizeEvent *e )
{
   s_dxviews *views;

   QWidget::resizeEvent(e);

   views = dxviewroot;

   while (views != NULL)
   {
      views->tbl->setGeometry(0, tabbar->height(), width(), height()-tabbar->height());
      views = views->next;
   }

   QWidget::resizeEvent( e );
}


s_dxviews * DXWin::addDXView(char *label, double lowerqrg, double upperqrg)
{
   s_dxviews *tmp;
   bool found=false;
   struct QTab *tab;


   // Gucken, ob noch ein View "von frueher" existiert, also einer, bei
   // dem label auf NULL steht.
   tmp = dxviewroot;
   while (tmp != NULL && !found)
   {
      if (tmp->label == NULL)
         found = true;
      else
         tmp = tmp->next;
   }


   if (!found)
   {
      if (dxviewroot == NULL)
      {
         // Erster Eintrag
         dxviewroot = (s_dxviews *)malloc(sizeof(s_dxviews));
         tmp = dxviewroot;
      }
      else
      {
         tmp = dxviewroot;
         while (tmp->next != NULL) tmp = tmp->next;

         tmp->next = (s_dxviews *)malloc(sizeof(s_dxviews));
         tmp = tmp->next;
      }

      tmp->next = NULL;
   }
   tmp->lowerqrg = lowerqrg;
   tmp->upperqrg = upperqrg;
   tmp->dataroot = NULL;
   tmp->datacount = 0;
   tmp->label = (char *) strdup(label);

   if (!found)
   {
      tab = new QTab();
      tab->label = label;
      tmp->tabid = tabbar->addTab( tab );
   }
   else
   {
      tab = tabbar->tab(tmp->tabid);
      tab->label = label;
      tabbar->setTabEnabled( tmp->tabid, true );
   }
   tabbar->resize(tabbar->sizeHint());

   if (!found)
   {
      tmp->tbl = new DXWinTbl( this );
      tmp->tbl->setGeometry(0, tabbar->height(), width(), height()-tabbar->height());
      tmp->tbl->setNumCols(5);
      tmp->tbl->setColumn(0, klocale->translate("DX-Call"), 80, KTabListBox::TextColumn);
      tmp->tbl->setColumn(1, klocale->translate("QRG"), 100, KTabListBox::TextColumn);
      tmp->tbl->setColumn(2, klocale->translate("Time"), 50, KTabListBox::TextColumn);
      tmp->tbl->setColumn(3, klocale->translate("Comment"), 200, KTabListBox::TextColumn);
      tmp->tbl->setColumn(4, klocale->translate("Sender"), 80, KTabListBox::TextColumn);

      connect(tmp->tbl, SIGNAL(showKontextMenu()), this, SLOT(slotShowKontextMenu()));
      connect(tmp->tbl, SIGNAL(popupMenu(int, int)), this, SLOT(slotShowKontextMenu(int, int)));
      connect(tmp->tbl, SIGNAL(highlighted(int, int)), this, SLOT(slotSelectLine(int, int)));
   }
   tmp->tbl->show();


   if (akttab != -1)
   {
      tmp->tbl->hide();
      akttab = tmp->tabid;
      aktview = tmp;
   }

   return tmp;
}


void DXWin::addDXMessage(char *dx_call, char *absender, float qrg, int zeit, char *bemerkung)
{
   s_dxdata *tmp;
   s_dxviews *views;
   bool ausgeben, unten_bleiben=false;
   char str[1000];
   char sep;


   views = dxviewroot;

   while (views != NULL)
   {
      ausgeben = true;

      if (qrg < views->lowerqrg) ausgeben = false;
      if (qrg > views->upperqrg) ausgeben = false;

      if (ausgeben)
      {
         if (views->dataroot == NULL)
         {
            // Erster Eintrag
            views->dataroot = (s_dxdata *)malloc(sizeof(s_dxdata));
            tmp = views->dataroot;
         }
         else
         {
            tmp = views->dataroot;
            while (tmp->next != NULL) tmp = tmp->next;

            tmp->next = (s_dxdata *)malloc(sizeof(s_dxdata));
            tmp = tmp->next;
         }

         tmp->next = NULL;
         tmp->dx_call = (char *) strdup(dx_call);
         tmp->absender = (char *) strdup(absender);
         tmp->qrg = qrg;
         tmp->zeit = zeit;
         tmp->bemerkung = (char *) strdup(bemerkung);

         views->datacount++;

         if (views->datacount > config->tblsize)
         {
            tmp = views->dataroot;
            views->dataroot = views->dataroot->next;
            free(tmp->dx_call);
            free(tmp->absender);
            free(tmp->bemerkung);
            free(tmp);
         }

         // Eintrag' in die entsprechende Tabelle
         if (views->tbl->lastRowVisible() == (int)views->tbl->count()-1)
            unten_bleiben = true;

         sep = views->tbl->separator();
         if (bemerkung[0] == '\0')
            sprintf(str, "%s%c%.1f%c%.4i%c %c%s", dx_call, sep, qrg, sep, zeit, sep, sep, absender);
         else
            sprintf(str, "%s%c%.1f%c%.4i%c%s%c%s", dx_call, sep, qrg, sep, zeit, sep, bemerkung, sep, absender);
         views->tbl->insertItem(str);
         if (unten_bleiben)
         {
            views->tbl->setTopItem( views->tbl->count() );
/*            int cell = views->tbl->count()-1 - views->tbl->lastRowVisible()-views->tbl->topItem();
            if (cell < 0) cell=0;
            views->tbl->setTopItem( cell );*/
         }
         while ((int)views->tbl->count() > config->tblsize)
            views->tbl->removeItem(0);
      }

      views = views->next;
   }
}


void DXWin::changeTab(int id)
{
   s_dxviews *views;
   bool found=false;


   views = dxviewroot;

   while (views != NULL && !found)
   {
      if (views->tabid == akttab)
      {
         found = true;
         views->tbl->hide();
      }
      else views = views->next;
   }

   views = dxviewroot;

   while (views != NULL)
   {
      if (views->tabid == id)
      {
         views->tbl->show();
         akttab = id;
         aktview = views;
         return;
      }
      views = views->next;
   }
}


void DXWin::slotShowKontextMenu(int, int)
{
   kontextmenu->popup(QCursor::pos());
}


void DXWin::slotShowKontextMenu()
{
   slotShowKontextMenu(0,0);
}


void DXWin::mousePressEvent( QMouseEvent *e )
{
   if (e->button() == RightButton)
   {
      kontextmenu->popup(QCursor::pos());
      return;
   }

   QWidget::mousePressEvent(e);
}


void DXWin::delDXView( const char  *label )
{
   s_dxviews *tmp;
   s_dxdata *dxtmp, *dxtmpn;

   tmp = dxviewroot;
   while (tmp != NULL)
   {
      if (!strcmp(tmp->label, label))
      {
         // Ok, dieser View soll versteckt werden...
         tmp->tbl->clear();
         tmp->tbl->hide();
         dxtmp = tmp->dataroot;
         while (dxtmp != NULL)
         {
            dxtmpn = dxtmp;
            dxtmp = dxtmp->next;

            if (dxtmpn->dx_call != NULL)
               free(dxtmpn->dx_call);
            if (dxtmpn->bemerkung != NULL)
               free(dxtmpn->bemerkung);
            if (dxtmpn->absender != NULL)
               free(dxtmpn->absender);
            free(dxtmpn);
         }
         tmp->dataroot = NULL;

         // Tab verstecken
         tabbar->setTabEnabled( tmp->tabid, false );

         free(tmp->label);
         tmp->label = NULL;

         if (akttab == tmp->tabid)
         {
            akttab = -1;
            aktview = NULL;
         }

         return;
      }
      tmp = tmp->next;
   }
}


void DXWin::slotSaveViewList()
{
   s_dxviews *tmp;
   char str[1000];
   int fd;


   sprintf(str, "%s/dxviews.list", config->maindir);
   if ((fd = open(str, O_CREAT|O_WRONLY|O_TRUNC)) == -1) return;

   fchmod(fd,S_IRUSR|S_IWUSR);

   tmp = dxviewroot;
   while (tmp != NULL)
   {
      if (tmp->label != NULL)
      {
         sprintf(str, "%.1f;%.1f;%s\n",
                      tmp->lowerqrg, tmp->upperqrg, tmp->label);

         write(fd, str, strlen(str));
      }

      tmp = tmp->next;
   }

   ::close(fd);
}


void DXWin::loadViewsList()
{
   FILE *f;
   char str[1000];
   int i;
   double low,up;
   char label[500];


   sprintf(str, "%s/dxviews.list", config->maindir);
   if ((f = fopen(str, "r")) == NULL) return;

   while (fgets(str, 999, f) != NULL)
   {
      if ((i = POS('\n', str)) != -1) str[i] = '\0';

      if (sscanf(str, "%lf;%lf;%s", &low, &up, label) == 3)
      {
         strcpy(label, str+lPOS(';', str)+1);
         addDXView( label, low, up );
      }
   }

   fclose(f);
}


void DXWin::slotSelectLine(int index, int)
{
   if (aktview == NULL)
      toplevel->updateStatusbar( "" );
   else
      toplevel->updateStatusbar( aktview->tbl->text(index, 0) );
}


void DXWin::resizeTabBar()
{
   tabbar->resize(tabbar->sizeHint());
}


s_dxviews * DXWin::getViewsRoot()
{
   return dxviewroot;
}



//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////



DlgAddView::DlgAddView() : QTabDialog( 0, "", true )
{
   setCaption(klocale->translate("MonKT: Add new view..."));

   setFixedSize(350, 250);

   setupWindowGeneral();
   addTab(winGeneral, klocale->translate("&General"));

   setupWindowQRG();
   addTab(winQRG, klocale->translate("&QRG"));

   readData();

   setCancelButton();
   connect( this, SIGNAL(applyButtonPressed()), SLOT(saveData()));
}


DlgAddView::~DlgAddView()
{
}


void DlgAddView::setupWindowQRG()
{
   winQRG = new QWidget( this );

   QLabel *l = new QLabel( winQRG );
   l->setGeometry( 10, 10, width()-20, 20 );
   l->setText( klocale->translate("You can set the minimum- and the maximum-QRG") );

   l = new QLabel( winQRG );
   l->setGeometry( 10, 30, width()-20, 20 );
   l->setText( klocale->translate("for this view here:") );

   l = new QLabel( winQRG );
   l->setGeometry( 10, 60, width()-20, 20 );
   l->setText("minimum");

   minQRG = new QLineEdit( winQRG );
   minQRG->setGeometry( 10, 80, 80, 20);

   l = new QLabel( winQRG );
   l->setGeometry( 100, 80, 40, 20 );
   l->setText("khz");

   l = new QLabel( winQRG );
   l->setGeometry( 160, 60, width()-20, 20 );
   l->setText("maximum");

   maxQRG = new QLineEdit( winQRG );
   maxQRG->setGeometry( 160, 80, 80, 20);

   l = new QLabel( winQRG );
   l->setGeometry( 250, 80, 40, 20 );
   l->setText("khz");
}


void DlgAddView::setupWindowGeneral()
{
   winGeneral = new QWidget( this );

   QLabel *l = new QLabel( winGeneral );
   l->setGeometry( 10, 40, width()-20, 20 );
   l->setText( klocale->translate("Short description:") );

   name = new QLineEdit( winGeneral );
   name->setGeometry( 10, 60, 100, 20);
   name->setMaxLength( 20 );
}


void DlgAddView::readData()
{
   // Voreinstellungen
   name->setText("new view");
   minQRG->setText("0.0");
   maxQRG->setText("99999999.9");
}


void DlgAddView::saveData()
{
   // Check, ob die Variablen korrekte Werte haben
   lowerqrg = atof(minQRG->text());
   upperqrg = atof(maxQRG->text());

   if (lowerqrg >= upperqrg)
   {
      KMsgBox::message( 0, klocale->translate("Error"), klocale->translate("The min-QRG must be lower than the max-QRG!") );
      return;
   }

   accept();
}


void DlgAddView::getData( double & min, double & max, char *desc )
{
   min = lowerqrg;
   max = upperqrg;
   strcpy(desc, name->text());
}


//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////


DlgDelView::DlgDelView( s_dxviews *views ) : QDialog( 0, "", true )
{
   s_dxviews *tmp;

   setCaption(klocale->translate("MonKT: Delete view..."));

   setFixedSize(250, 200);

   dxviews = views;


   QLabel *l = new QLabel( this );
   l->setGeometry( 10, 10, width()-20, 20 );
   l->setText( klocale->translate("Which view do you want to delete?") );

   liste = new QListBox( this );
   liste->setGeometry( 10, 30, width()-20, 100 );


   // OK- und Abbruch-Knopf
   int abstand = (width()-(2*BUTTONWIDTH))/3;
   ok = new QPushButton( this );
   ok->setText( klocale->translate ("&Ok") );
   ok->setGeometry(abstand, height()-BUTTONHEIGHT-10, BUTTONWIDTH, BUTTONHEIGHT);
   connect( ok, SIGNAL(clicked()), this, SLOT(slotOkClicked()) );

   abort = new QPushButton( this );
   abort->setText( klocale->translate ("&Abort") );
   abort->setGeometry(abstand*2+BUTTONWIDTH, height()-BUTTONHEIGHT-10, BUTTONWIDTH, BUTTONHEIGHT);
   connect( abort, SIGNAL(clicked()), this, SLOT(reject()) );


    tmp = views;
    while (tmp != NULL)
    {
       liste->insertItem( tmp->label );
       tmp = tmp->next;
    }
}


DlgDelView::~DlgDelView()
{
}


void DlgDelView::slotOkClicked()
{
   s_dxviews *tmp;

   if (liste->currentItem() == -1)
   {
      KMsgBox::message( 0, klocale->translate("Error"), klocale->translate("You must select a view.") );
      return;
   }

   tmp = dxviews;
   while (tmp != NULL)
   {
      if (!strcmp(tmp->label, liste->text(liste->currentItem())))
      {
         accept();
         return;
      }
      tmp = tmp->next;
   }

   KMsgBox::message( 0, klocale->translate("Error"), klocale->translate("You must select a view.") );
}


const char * DlgDelView::getListName()
{
   return liste->text(liste->currentItem());
}


#include "dxwin.moc"

