/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"

void fsincos(float angle, float *sin_angle, float *cos_angle) 
{
	*sin_angle = (float)sin(angle);

	*cos_angle = (float)cos(angle);
}

// Gary Tarolli's clever inverse square root technique
float fsqrt_inv(float f)
{
	long i;
	float x2, y;

	x2 = 0.5f*f;
	i = *(long *)&f;
	i = 0x5f3759df - (i>>1);
	y = *(float *)&i;

	// repeat this iteration for more accuracy
	y = 1.5f*y - (x2*y * y*y);

	return y;
}

/*
const float ONE_HALF = 0.5f;
const float THREE_HALVES = 1.5f;
float fsqrt_inv(float f)
{
	__asm // 18 cycles
	{
		fld			dword ptr [esp + 4]
		// f
		fmul		dword ptr [ONE_HALF]
		// x2 = 0.5f*f

		mov			eax, [esp + 4]
		mov			ecx, 0x5f3759df

		shr			eax, 1

		sub			ecx, eax

		mov			[esp + 4], ecx

		fmul		dword ptr [esp + 4]
		// x2*y
		fld			dword ptr [esp + 4]
		// y
		// x2*y
		fmul		dword ptr [esp + 4]
		// y*y
		// x2*y
		fld			dword ptr [THREE_HALVES]
		// 1.5f
		// y*y
		// x2*y
		fmul		dword ptr [esp + 4]
		// 1.5f*y
		// y*y
		// x2*y
		fxch		st(2)
		// x2*y
		// y*y
		// 1.5f*y
		// ******** stall 1 clock ********
		fmulp		st(1), st
		// x2*y * y*y
		// 1.5f*y
		// ******** stall 2 clocks ********
		fsubp		st(1), st
		// y = 1.5f*y - (x2*y * y*y)

		ret
	}
}
*/
float DotProduct(Vector a, Vector b)
{
	return ( a[X]*b[X] + a[Y]*b[Y] + a[Z]*b[Z] );
	/*
	__asm
	{
		mov			ecx, [esp + 4]
		mov			edx, [esp + 8]

		fld			dword ptr [ecx]
		// a[X]
		fmul		dword ptr [edx]
		// a[X]*b[X] (2)
		fld			dword ptr [ecx + 4]
		// a[Y]
		// a[X]*b[X] (1)
		fmul		dword ptr [edx + 4]
		// a[Y]*b[Y] (2)
		// a[X]*b[X] (1)
		fld			dword ptr [ecx + 8]
		// a[Z]
		// a[Y]*b[Y] (1)
		// a[X]*b[X] (0)
		fmul		dword ptr [edx + 8]
		// a[Z]*b[Z] (2)
		// a[Y]*b[Y] (0)
		// a[X]*b[X] (0)
		fxch		st(2)
		// a[X]*b[X] (0)
		// a[Y]*b[Y] (0)
		// a[Z]*b[Z] (2)
		faddp		st(1), st
		// a[X]*b[X] + a[Y]*b[Y] (2)
		// a[Z]*b[Z] (1)
		// ******** stall 2 cycles ********
		faddp		st(1), st

		ret
	}*/
}

void CrossProduct(Vector v, Vector v1, Vector v2)
{
	v[X] = v1[Y]*v2[Z] - v1[Z]*v2[Y]; // = A - B
	v[Y] = v1[Z]*v2[X] - v1[X]*v2[Z]; // = C - D
	v[Z] = v1[X]*v2[Y] - v1[Y]*v2[X]; // = E - F
	/*__asm
	{
		mov			eax, [esp + 4]
		mov			ecx, [esp + 8]

		mov			edx, [esp + 12]

		fld			dword ptr [ecx + 4]
		// v1[Y]
		fmul		dword ptr [edx + 8]
		// v1[Y]*v2[Z] (2)
		fld			dword ptr [ecx + 8]
		// v1[Z]
		// v1[Y]*v2[Z] (1)
		fmul		dword ptr [edx + 4]
		// v1[Z]*v2[Y] (2)
		// v1[Y]*v2[Z] (0)
		fld			dword ptr [ecx + 8]
		// v1[Z]
		// v1[Z]*v2[Y] (1)
		// v1[Y]*v2[Z] (0)
		fmul		dword ptr [edx]
		// v1[Z]*v2[X] (2)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fld			dword ptr [ecx]
		// v1[X]
		// v1[Z]*v2[X] (1)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fmul		dword ptr [edx + 8]
		// v1[X]*v2[Z] (2)
		// v1[Z]*v2[X] (0)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fld			dword ptr [ecx]
		// v1[X]
		// v1[X]*v2[Z] (1)
		// v1[Z]*v2[X] (0)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fmul		dword ptr [edx + 4]
		// v1[X]*v2[Y] (2)
		// v1[X]*v2[Z] (0)
		// v1[Z]*v2[X] (0)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fld			dword ptr [ecx + 4]
		// v1[Y]
		// v1[X]*v2[Y] (1)
		// v1[X]*v2[Z] (0)
		// v1[Z]*v2[X] (0)
		// v1[Z]*v2[Y] (0)
		// v1[Y]*v2[Z] (0)
		fmul		dword ptr [edx]
		// v1[Y]*v2[X] = F (2)
		// v1[X]*v2[Y] = E (0)
		// v1[X]*v2[Z] = D (0)
		// v1[Z]*v2[X] = C (0)
		// v1[Z]*v2[Y] = B (0)
		// v1[Y]*v2[Z] = A (0)
		fxch		st(5)
		// v1[Y]*v2[Z] = A (0)
		// v1[X]*v2[Y] = E (0)
		// v1[X]*v2[Z] = D (0)
		// v1[Z]*v2[X] = C (0)
		// v1[Z]*v2[Y] = B (0)
		// v1[Y]*v2[X] = F (2)
		fsubrp	st(4), st
		// v1[X]*v2[Y] = E (0)
		// v1[X]*v2[Z] = D (0)
		// v1[Z]*v2[X] = C (0)
		// A - B (2)
		// v1[Y]*v2[X] = F (1)
		fxch		st(2)
		// v1[Z]*v2[X] = C (0)
		// v1[X]*v2[Z] = D (0)
		// v1[X]*v2[Y] = E (0)
		// A - B (2)
		// v1[Y]*v2[X] = F (1)
		fsubrp	st(1), st
		// C - D (2)
		// v1[X]*v2[Y] = E (0)
		// A - B (1)
		// v1[Y]*v2[X] = F (0)
		fxch		st(1)
		// v1[X]*v2[Y] = E (0)
		// C - D (2)
		// A - B (1)
		// v1[Y]*v2[X] = F (0)
		fsubrp	st(3), st
		// C - D (1)
		// A - B (0)
		// E - F (2)
		fxch		st(1)
		// A - B (0)
		// C - D (1)
		// E - F (2)
		// ******** stall 1 cycle ********
		// (result must be ready one cycle in advance)
		fstp		dword ptr [eax]
		fstp		dword ptr [eax + 4]
		fstp		dword ptr [eax + 8]

		ret
	}*/
}

// v0, v1, and v2 must be structures with the x and y component
// as the first 2 32-bit members
// if the vertices v0, v1, and v2 are in counter clockwise order
// the return value will be positive, otherwise it'll be negative
// if they're collinear, the return value will be zero
float NormZ(float *v0, float *v1, float *v2)
{
	// (v2-v1) x (v0-v1)
	return ((v2[X] - v1[X]) * (v0[Y] - v1[Y])) - ((v2[Y] - v1[Y]) * (v0[X] - v1[X]));
	/*__asm
	{
		mov			ecx, [esp + 8] // ecx = v1
		mov			edx, [esp + 12] // edx = v2

		mov			eax, [esp + 4] // eax = v0

		fld			dword ptr [edx]
		// v2.x
		fsub		dword ptr [ecx]
		// v2.x - v1.x (2)
		fld			dword ptr [eax + 4]
		// v0.y
		// v2.x - v1.x (1)
		fsub		dword ptr [ecx + 4]
		// v0.y - v1.y (2)
		// v2.x - v1.x (0)
		fld			dword ptr [edx + 4]
		// v2.y
		// v0.y - v1.y (1)
		// v2.x - v1.x (0)
		fsub		dword ptr [ecx + 4]
		// v2.y - v1.y (2)
		// v0.y - v1.y (0)
		// v2.x - v1.x (0)
		fld			dword ptr [eax]
		// v0.x
		// v2.y - v1.y (1)
		// v0.y - v1.y (0)
		// v2.x - v1.x (0)
		fsub		dword ptr [ecx]
		// v0.x - v1.x (2)
		// v2.y - v1.y (0)
		// v0.y - v1.y (0)
		// v2.x - v1.x (0)
		fxch		st(2)
		// v0.y - v1.y (0)
		// v2.y - v1.y (0)
		// v0.x - v1.x (2)
		// v2.x - v1.x (0)
		fmulp		st(3), st
		// v2.y - v1.y (0)
		// v0.x - v1.x (1)
		// (v0.y-v1.y)*(v2.x-v1.x) (3)
		// ******** STALL 1 clock ********
		// v2.y - v1.y (0)
		// v0.x - v1.x (0)
		// (v0.y-v1.y)*(v2.x-v1.x) (2)
		fmulp		st(1), st
		// (v2.y-v1.y)*(v0.x-v1.x) (2)
		// (v0.y-v1.y)*(v2.x-v1.x) (1)
		// ******** STALL 2 clocks ********
		fsubp		st(1), st

		ret
	}
	*/
}

float Magnitude(Vector v)
{
	return fsqrt(SQR(v[X]) + SQR(v[Y]) + SQR(v[Z]));
}

float Distance(Vector v0, Vector v1)
{
	return fsqrt(SQR(v1[X]-v0[X]) + SQR(v1[Y]-v0[Y]) + SQR(v1[Z]-v0[Z]));
}

void Normalize(Vector v)
{
	float m = 1.0f/fsqrt(SQR(v[X]) + SQR(v[Y]) + SQR(v[Z]));
	v[X] *= m;
	v[Y] *= m;
	v[Z] *= m;
}

void FastApproxNormalize(Vector v)
{
	float m = fsqrt_inv(SQR(v[X]) + SQR(v[Y]) + SQR(v[Z]));
	v[X] *= m;
	v[Y] *= m;
	v[Z] *= m;
}

// rotates point about the x axis by angle (in radians)
void RotateByXAxis(Vector res, Vector point, float angle)
{
	float cos_angle, sin_angle;
	float y, z;

	fsincos(angle, &sin_angle, &cos_angle);

	y = (point[Y] * cos_angle) - (point[Z] * sin_angle);
	z = (point[Y] * sin_angle) + (point[Z] * cos_angle);

	res[X] = point[X];
	res[Y] = y;
	res[Z] = z;
	res[W] = point[W];
}

// rotates point about the y axis by angle (in radians)
void RotateByYAxis(Vector res, Vector point, float angle)
{
	float cos_angle, sin_angle;
	float x, z;

	fsincos(angle, &sin_angle, &cos_angle);

	x = (point[X] * cos_angle) - (point[Z] * sin_angle);
	z = (point[X] * sin_angle) + (point[Z] * cos_angle);

	res[X] = x;
	res[Y] = point[Y];
	res[Z] = z;
	res[W] = point[W];
}

// rotates point about the z axis by angle (in radians)
void RotateByZAxis(Vector res, Vector point, float angle)
{
	float cos_angle, sin_angle;
	float x, y;

	fsincos(angle, &sin_angle, &cos_angle);

	x = (point[X] * cos_angle) - (point[Y] * sin_angle);
	y = (point[X] * sin_angle) + (point[Y] * cos_angle);

	res[X] = x;
	res[Y] = y;
	res[Z] = point[Z];
	res[W] = point[W];
}

// rotates point about rotation_axis by angle (in radians)
// using a quaternian formula
void RotateByAxis(Vector res, Vector point, float angle, Vector rotation_axis)
{
	float cos_ang, sin_ang, cos_ang_sqr, dp;
	Vector v, vp, vvp;

	fsincos(0.5f*angle, &sin_ang, &cos_ang);
	cos_ang_sqr = cos_ang*cos_ang;

	v[X] = rotation_axis[X]*sin_ang;
	v[Y] = rotation_axis[Y]*sin_ang;
	v[Z] = rotation_axis[Z]*sin_ang;

	CrossProduct(vp, v, point);
	CrossProduct(vvp, v, vp);
	dp = DotProduct(point, v);

	res[X] = cos_ang_sqr*point[X] + v[X]*dp + 2.0f*cos_ang*vp[X] + vvp[X];
	res[Y] = cos_ang_sqr*point[Y] + v[Y]*dp + 2.0f*cos_ang*vp[Y] + vvp[Y];
	res[Z] = cos_ang_sqr*point[Z] + v[Z]*dp + 2.0f*cos_ang*vp[Z] + vvp[Z];
	res[W] = point[W];
}

void Transpose(Matrix res, Matrix m)
{
	Matrix tmp;
	tmp[0][0] = m[0][0]; tmp[0][1] = m[1][0]; tmp[0][2] = m[2][0]; tmp[0][3] = m[3][0];
	tmp[1][0] = m[0][1]; tmp[1][1] = m[1][1]; tmp[1][2] = m[2][1]; tmp[1][3] = m[3][1];
	tmp[2][0] = m[0][2]; tmp[2][1] = m[1][2]; tmp[2][2] = m[2][2]; tmp[2][3] = m[3][2];
	tmp[3][0] = m[0][3]; tmp[3][1] = m[1][3]; tmp[3][2] = m[2][3]; tmp[3][3] = m[3][3];
	MatrixCopy(res, tmp);
}

void MatMultMat4x4(Matrix res, Matrix m1, Matrix m2)
{
	Matrix tmp;
	tmp[0][0] = m1[0][0]*m2[0][0] + m1[0][1]*m2[1][0] + m1[0][2]*m2[2][0] + m1[0][3]*m2[3][0];
	tmp[0][1] = m1[0][0]*m2[0][1] + m1[0][1]*m2[1][1] + m1[0][2]*m2[2][1] + m1[0][3]*m2[3][1];
	tmp[0][2] = m1[0][0]*m2[0][2] + m1[0][1]*m2[1][2] + m1[0][2]*m2[2][2] + m1[0][3]*m2[3][2];
	tmp[0][3] = m1[0][0]*m2[0][3] + m1[0][1]*m2[1][3] + m1[0][2]*m2[2][3] + m1[0][3]*m2[3][3];
	tmp[1][0] = m1[1][0]*m2[0][0] + m1[1][1]*m2[1][0] + m1[1][2]*m2[2][0] + m1[1][3]*m2[3][0];
	tmp[1][1] = m1[1][0]*m2[0][1] + m1[1][1]*m2[1][1] + m1[1][2]*m2[2][1] + m1[1][3]*m2[3][1];
	tmp[1][2] = m1[1][0]*m2[0][2] + m1[1][1]*m2[1][2] + m1[1][2]*m2[2][2] + m1[1][3]*m2[3][2];
	tmp[1][3] = m1[1][0]*m2[0][3] + m1[1][1]*m2[1][3] + m1[1][2]*m2[2][3] + m1[1][3]*m2[3][3];
	tmp[2][0] = m1[2][0]*m2[0][0] + m1[2][1]*m2[1][0] + m1[2][2]*m2[2][0] + m1[2][3]*m2[3][0];
	tmp[2][1] = m1[2][0]*m2[0][1] + m1[2][1]*m2[1][1] + m1[2][2]*m2[2][1] + m1[2][3]*m2[3][1];
	tmp[2][2] = m1[2][0]*m2[0][2] + m1[2][1]*m2[1][2] + m1[2][2]*m2[2][2] + m1[2][3]*m2[3][2];
	tmp[2][3] = m1[2][0]*m2[0][3] + m1[2][1]*m2[1][3] + m1[2][2]*m2[2][3] + m1[2][3]*m2[3][3];
	tmp[3][0] = m1[3][0]*m2[0][0] + m1[3][1]*m2[1][0] + m1[3][2]*m2[2][0] + m1[3][3]*m2[3][0];
	tmp[3][1] = m1[3][0]*m2[0][1] + m1[3][1]*m2[1][1] + m1[3][2]*m2[2][1] + m1[3][3]*m2[3][1];
	tmp[3][2] = m1[3][0]*m2[0][2] + m1[3][1]*m2[1][2] + m1[3][2]*m2[2][2] + m1[3][3]*m2[3][2];
	tmp[3][3] = m1[3][0]*m2[0][3] + m1[3][1]*m2[1][3] + m1[3][2]*m2[2][3] + m1[3][3]*m2[3][3];
	MatrixCopy(res, tmp);
}

void MatMultMat3x4(Matrix res, Matrix m1, Matrix m2)
{
	Matrix tmp;
	tmp[0][0] = m1[0][0]*m2[0][0] + m1[0][1]*m2[1][0] + m1[0][2]*m2[2][0];
	tmp[0][1] = m1[0][0]*m2[0][1] + m1[0][1]*m2[1][1] + m1[0][2]*m2[2][1];
	tmp[0][2] = m1[0][0]*m2[0][2] + m1[0][1]*m2[1][2] + m1[0][2]*m2[2][2];
	tmp[0][3] = m1[0][0]*m2[0][3] + m1[0][1]*m2[1][3] + m1[0][2]*m2[2][3] + m1[0][3];
	tmp[1][0] = m1[1][0]*m2[0][0] + m1[1][1]*m2[1][0] + m1[1][2]*m2[2][0];
	tmp[1][1] = m1[1][0]*m2[0][1] + m1[1][1]*m2[1][1] + m1[1][2]*m2[2][1];
	tmp[1][2] = m1[1][0]*m2[0][2] + m1[1][1]*m2[1][2] + m1[1][2]*m2[2][2];
	tmp[1][3] = m1[1][0]*m2[0][3] + m1[1][1]*m2[1][3] + m1[1][2]*m2[2][3] + m1[1][3];
	tmp[2][0] = m1[2][0]*m2[0][0] + m1[2][1]*m2[1][0] + m1[2][2]*m2[2][0];
	tmp[2][1] = m1[2][0]*m2[0][1] + m1[2][1]*m2[1][1] + m1[2][2]*m2[2][1];
	tmp[2][2] = m1[2][0]*m2[0][2] + m1[2][1]*m2[1][2] + m1[2][2]*m2[2][2];
	tmp[2][3] = m1[2][0]*m2[0][3] + m1[2][1]*m2[1][3] + m1[2][2]*m2[2][3] + m1[2][3];
	tmp[3][0] = 0.0f;
	tmp[3][1] = 0.0f;
	tmp[3][2] = 0.0f;
	tmp[3][3] = 1.0f;
	MatrixCopy(res, tmp);
}

void MatMultVec3x3_2(Vector res, Matrix m, Vector v)
{
	res[0] = m[0][0]*v[0] + m[0][1]*v[1] + m[0][2]*v[2];
	res[1] = m[1][0]*v[0] + m[1][1]*v[1] + m[1][2]*v[2];
	/*__asm
	{
		mov			eax, [esp + 4] // eax = res
		mov			ecx, [esp + 8] // ecx = m

		mov			edx, [esp + 12] // edx = v

		fld			dword ptr [ecx]
		// m00
		fmul		dword ptr [edx]
		// m00*v0
		fld			dword ptr [ecx + 4]
		// m01
		// m00*v0
		fmul		dword ptr [edx + 4]
		// m01*v1
		// m00*v0
		fld			dword ptr [ecx + 8]
		// m02
		// m01*v1
		// m00*v0
		fmul		dword ptr [edx + 8]
		// m02*v2
		// m01*v1
		// m00*v0
		fld			dword ptr [ecx + 16]
		// m10
		// m02*v2
		// m01*v1
		// m00*v0
		fmul		dword ptr [edx]
		// m10*v0
		// m02*v2
		// m01*v1
		// m00*v0
		fld			dword ptr [ecx + 20]
		// m11
		// m10*v0
		// m02*v2
		// m01*v1
		// m00*v0
		fmul		dword ptr [edx + 4]
		// m11*v1
		// m10*v0
		// m02*v2
		// m01*v1
		// m00*v0
		fld			dword ptr [ecx + 24]
		// m12
		// m11*v1
		// m10*v0
		// m02*v2
		// m01*v1
		// m00*v0
		fmul		dword ptr [edx + 8]
		// m12*v2
		// m11*v1
		// m10*v0
		// m02*v2
		// m01*v1
		// m00*v0
		fxch		st(5)
		// m00*v0
		// m11*v1
		// m10*v0
		// m02*v2
		// m01*v1
		// m12*v2
		faddp		st(4), st
		// m11*v1
		// m10*v0
		// m02*v2
		// m00*v0 + m01*v1
		// m12*v2
		faddp		st(1), st
		// m10*v0 + m11*v1
		// m02*v2
		// m00*v0 + m01*v1
		// m12*v2
		fxch		st(1)
		// m02*v2
		// m10*v0 + m11*v1
		// m00*v0 + m01*v1
		// m12*v2
		faddp		st(2), st
		// m10*v0 + m11*v1
		// m00*v0 + m01*v1 + m02*v2
		// m12*v2
		faddp		st(2), st
		// m00*v0 + m01*v1 + m02*v2
		// m10*v0 + m11*v1 + m12*v2
		fstp		dword ptr [eax]
		fstp		dword ptr [eax + 4]
		ret
	}*/
}

// assumes v[W] == 1.0f
void MatMultVec3x4_3(Vector res, Matrix m, Vector v)
{
	res[0] = m[0][0]*v[0] + m[0][1]*v[1] + m[0][2]*v[2] + m[0][3];
	res[1] = m[1][0]*v[0] + m[1][1]*v[1] + m[1][2]*v[2] + m[1][3];
	res[2] = m[2][0]*v[0] + m[2][1]*v[1] + m[2][2]*v[2] + m[2][3];
	res[3] = 1.0f;
/*	__asm // 38 cycles
	{
		mov			eax, [esp + 4]
		mov			ecx, [esp + 8]

		mov			edx, [esp + 12]

		fld			dword ptr [ecx]
		// m00
		fmul		dword ptr [edx]
		// m00*v0 (2)
		fld			dword ptr [ecx + 4]
		// m01
		// m00*v0 (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		fxch		st(1)
		// m00*v0 (0)
		// m01*v1 (2)
		fadd		dword ptr [ecx + 12]
		// m00*v0 + m03 (2)
		// m01*v1 (1)
		fld			dword ptr [ecx + 8]
		// m02
		// m00*v0 + m03 (1)
		// m01*v1 (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m00*v0 + m03 (0)
		// m01*v1 (0)
		fxch		st(2)
		// m01*v1 (0)
		// m00*v0 + m03 (0)
		// m02*v2 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m03 (2)
		// m02*v2 (1)
		fld			dword ptr [ecx + 16]
		// m10
		// m00*v0 + m01*v1 + m03 (1)
		// m02*v2 (0)
		fmul		dword ptr [edx]
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m03 (0)
		// m02*v2 (0)
		fxch		st(2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 + m03 (0)
		// m10*v0 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 + m03 (2)
		// m10*v0 (1)
		fld			dword ptr [ecx + 16 + 4]
		// m11
		// m00*v0 + m01*v1 + m02*v2 + m03 (1)
		// m10*v0 (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m10*v0 (0)
		fxch		st(2)
		// m10*v0 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (2)
		fadd		dword ptr [ecx + 16 + 12]
		// m10*v0 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (1)
		fld			dword ptr [ecx + 16 + 8]
		// m12
		// m10*v0 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fxch		st(3)
		// m11*v1 (0)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (1)
		fld			dword ptr [ecx + 32]
		// m20
		// m10*v0 + m11*v1 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fmul		dword ptr [edx]
		// m20*v0 (2)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fxch		st(3)
		// m12*v2 (0)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (1)
		fld			dword ptr [ecx + 32 + 4]
		// m21
		// m10*v0 + m11*v1 + m12*v2 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 4]
		// m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fxch		st(3)
		// m20*v0 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (2)
		fadd		dword ptr [ecx + 32 + 12]
		// m20*v0 + m23 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (1)
		fld			dword ptr [ecx + 32 + 8]
		// m22
		// m20*v0 + m23 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fmul		dword ptr [edx + 8]
		// m22*v2 (2)
		// m20*v0 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fxch		st(4)
		// m21*v1 (0)
		// m20*v0 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m23 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (1)
		fxch		st(1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m20*v0 + m21*v1 + m23 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (1)
		fstp		dword ptr [eax + 4]
		// m20*v0 + m21*v1 + m23 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (0)
		faddp		st(2), st
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23 (2)
		fstp		dword ptr [eax]
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		mov			dword ptr [eax + 12], 0x3f800000
		fstp		dword ptr [eax + 8]

		ret
	}
*/
}

void MatMultVec3x4_4(Vector res, Matrix m, Vector v)
{
	res[0] = m[0][0]*v[0] + m[0][1]*v[1] + m[0][2]*v[2] + m[0][3]*v[3];
	res[1] = m[1][0]*v[0] + m[1][1]*v[1] + m[1][2]*v[2] + m[1][3]*v[3];
	res[2] = m[2][0]*v[0] + m[2][1]*v[1] + m[2][2]*v[2] + m[2][3]*v[3];
	res[3] = 1.0f;

/*	__asm // 44 cycles
	{
		mov			eax, [esp + 4]
		mov			ecx, [esp + 8]

		mov			edx, [esp + 12]

		fld			dword ptr [ecx]
		// m00
		fmul		dword ptr [edx]
		// m00*v0 (2)
		fld			dword ptr [ecx + 4]
		// m01
		// m00*v0 (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		fld			dword ptr [ecx + 8]
		// m02
		// m01*v1 (1)
		// m00*v0 (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m01*v1 (0)
		// m00*v0 (0)
		fxch		st(2)
		// m00*v0 (0)
		// m01*v1 (0)
		// m02*v2 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 (2)
		// m02*v2 (1)
		fld			dword ptr [ecx + 12]
		// m03
		// m00*v0 + m01*v1 (1)
		// m02*v2 (0)
		fmul		dword ptr [edx + 12]
		// m03*v3 (2)
		// m00*v0 + m01*v1 (0)
		// m02*v2 (0)
		fxch		st(2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 (0)
		// m03*v3 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 (2)
		// m03*v3 (1)
		fld			dword ptr [ecx + 16]
		// m10
		// m00*v0 + m01*v1 + m02*v2 (1)
		// m03*v3 (0)
		fmul		dword ptr [edx]
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m02*v2 (0)
		// m03*v3 (0)
		fxch		st(2)
		// m03*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 (0)
		// m10*v0 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (2)
		// m10*v0 (1)
		fld			dword ptr [ecx + 16 + 4]
		// m11
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (1)
		// m10*v0 (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fld			dword ptr [ecx + 16 + 8]
		// m12
		// m11*v1 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fxch		st(3)
		// m10*v0 (0)
		// m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (1)
		fld			dword ptr [ecx + 16 + 12]
		// m13
		// m10*v0 + m11*v1 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (0)
		fmul		dword ptr [edx + 12]
		// m13*v3 (2)
		// m10*v0 + m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (0)
		fxch		st(3)
		// m12*v2 (0)
		// m10*v0 + m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (1)
		fld			dword ptr [ecx + 32]
		// m20
		// m10*v0 + m11*v1 + m12*v2 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (0)
		fmul		dword ptr [edx]
		// m20*v0 (2)
		// m10*v0 + m11*v1 + m12*v2 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (0)
		fxch		st(3)
		// m13*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (1)
		fld			dword ptr [ecx + 32 + 4]
		// m21
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 4]
		// m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fld			dword ptr [ecx + 32 + 8]
		// m22
		// m21*v1 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 8]
		// m22*v2 (2)
		// m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fxch		st(4)
		// m20*v0 (0)
		// m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (1)
		fld			dword ptr [ecx + 32 + 12]
		// m23
		// m20*v0 + m21*v1 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (0)
		fmul		dword ptr [edx + 12]
		// m23*v3 (2)
		// m20*v0 + m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (0)
		fxch		st(4)
		// m22*v2 (0)
		// m20*v0 + m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m22*v2 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (1)
		fxch		st(1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m20*v0 + m21*v1 + m22*v2 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (1)
		fstp		dword ptr [eax + 4]
		// m20*v0 + m21*v1 + m22*v2 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (0)
		faddp		st(2), st
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (2)
		fstp		dword ptr [eax]
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		mov			dword ptr [eax + 12], 0x3f800000
		fstp		dword ptr [eax + 8]

		ret
	}
	*/
}

// assumes v[W] == 1.0f
void MatMultVec4x4_3(Vector res, Matrix m, Vector v)
{
	res[0] = m[0][0]*v[0] + m[0][1]*v[1] + m[0][2]*v[2] + m[0][3];
	res[1] = m[1][0]*v[0] + m[1][1]*v[1] + m[1][2]*v[2] + m[1][3];
	res[2] = m[2][0]*v[0] + m[2][1]*v[1] + m[2][2]*v[2] + m[2][3];
	res[3] = m[3][0]*v[0] + m[3][1]*v[1] + m[3][2]*v[2] + m[3][3];

/*	__asm // 48 cycles
	{
		mov			eax, [esp + 4]
		mov			ecx, [esp + 8]

		mov			edx, [esp + 12]

		fld			dword ptr [ecx]
		// m00
		fmul		dword ptr [edx]
		// m00*v0 (2)
		fld			dword ptr [ecx + 4]
		// m01
		// m00*v0 (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		fxch		st(1)
		// m00*v0 (0)
		// m01*v1 (2)
		fadd		dword ptr [ecx + 12]
		// m00*v0 + m03 (2)
		// m01*v1 (1)
		fld			dword ptr [ecx + 8]
		// m02
		// m00*v0 + m03 (1)
		// m01*v1 (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m00*v0 + m03 (0)
		// m01*v1 (0)
		fxch		st(2)
		// m01*v1 (0)
		// m00*v0 + m03 (0)
		// m02*v2 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m03 (2)
		// m02*v2 (1)
		fld			dword ptr [ecx + 16]
		// m10
		// m00*v0 + m01*v1 + m03 (1)
		// m02*v2 (0)
		fmul		dword ptr [edx]
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m03 (0)
		// m02*v2 (0)
		fxch		st(2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 + m03 (0)
		// m10*v0 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 + m03 (2)
		// m10*v0 (1)
		fld			dword ptr [ecx + 16 + 4]
		// m11
		// m00*v0 + m01*v1 + m02*v2 + m03 (1)
		// m10*v0 (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m10*v0 (0)
		fxch		st(2)
		// m10*v0 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (2)
		fadd		dword ptr [ecx + 16 + 12]
		// m10*v0 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (1)
		fld			dword ptr [ecx + 16 + 8]
		// m12
		// m10*v0 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fxch		st(3)
		// m11*v1 (0)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (1)
		fld			dword ptr [ecx + 32]
		// m20
		// m10*v0 + m11*v1 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fmul		dword ptr [edx]
		// m20*v0 (2)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fxch		st(3)
		// m12*v2 (0)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (1)
		fld			dword ptr [ecx + 32 + 4]
		// m21
		// m10*v0 + m11*v1 + m12*v2 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 4]
		// m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fxch		st(3)
		// m20*v0 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (2)
		fadd		dword ptr [ecx + 32 + 12]
		// m20*v0 + m23 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (1)
		fld			dword ptr [ecx + 32 + 8]
		// m22
		// m20*v0 + m23 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fmul		dword ptr [edx + 8]
		// m22*v2 (2)
		// m20*v0 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fxch		st(4)
		// m21*v1 (0)
		// m20*v0 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m23 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (1)
		fld			dword ptr [ecx + 48]
		// m30
		// m20*v0 + m21*v1 + m23 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (0)
		fmul		dword ptr [edx]
		// m30*v0 (2)
		// m20*v0 + m21*v1 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (0)
		fxch		st(4)
		// m22*v2 (0)
		// m20*v0 + m21*v1 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m22*v2 + m23 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (1)
		fld			dword ptr [ecx + 48 + 4]
		// m31
		// m20*v0 + m21*v1 + m22*v2 + m23 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (0)
		fmul		dword ptr [edx + 4]
		// m31*v1 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (0)
		fxch		st(4)
		// m30*v0 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (2)
		fadd		dword ptr [ecx + 48 + 12]
		// m30*v0 + m33 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (1)
		fld			dword ptr [ecx + 48 + 8]
		// m32
		// m30*v0 + m33 (1)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (0)
		fmul		dword ptr [edx + 8]
		// m32*v2 (2)
		// m30*v0 + m33 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (0)
		fxch		st(5)
		// m31*v1 (0)
		// m30*v0 + m33 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (2)
		faddp		st(1), st
		// m30*v0 + m31*v1 + m33 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (1)
		fxch		st(1)
		// m20*v0 + m21*v1 + m22*v2 + m23 (0)
		// m30*v0 + m31*v1 + m33 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (1)
		fstp		dword ptr [eax + 8]
		// m30*v0 + m31*v1 + m33 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (0)
		faddp		st(3), st
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 + m31*v1 + m32*v2 + m33 (2)
		fstp		dword ptr [eax + 4]
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 + m31*v1 + m32*v2 + m33 (0)
		fstp		dword ptr [eax]
		// m30*v0 + m31*v1 + m32*v2 + m33 (0)
		fstp		dword ptr [eax + 12]

		ret
	}
*/
}

void MatMultVec4x4_4(Vector res, Matrix m, Vector v)
{
	res[0] = m[0][0]*v[0] + m[0][1]*v[1] + m[0][2]*v[2] + m[0][3]*v[3];
	res[1] = m[1][0]*v[0] + m[1][1]*v[1] + m[1][2]*v[2] + m[1][3]*v[3];
	res[2] = m[2][0]*v[0] + m[2][1]*v[1] + m[2][2]*v[2] + m[2][3]*v[3];
	res[3] = m[3][0]*v[0] + m[3][1]*v[1] + m[3][2]*v[2] + m[3][3]*v[3];
/*
	__asm // 56 cycles
	{
		mov			eax, [esp + 4]  // eax = res
		mov			ecx, [esp + 8]  // ecx = m

		mov			edx, [esp + 12] // edx = v

		fld			dword ptr [ecx]
		// m00
		fmul		dword ptr [edx]
		// m00*v0 (2)
		fld			dword ptr [ecx + 4]
		// m01
		// m00*v0 (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		fld			dword ptr [ecx + 8]
		// m02
		// m01*v1 (1)
		// m00*v0 (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m01*v1 (0)
		// m00*v0 (0)
		fxch		st(2)
		// m00*v0 (0)
		// m01*v1 (0)
		// m02*v2 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 (2)
		// m02*v2 (1)
		fld			dword ptr [ecx + 12]
		// m03
		// m00*v0 + m01*v1 (1)
		// m02*v2 (0)
		fmul		dword ptr [edx + 12]
		// m03*v3 (2)
		// m00*v0 + m01*v1 (0)
		// m02*v2 (0)
		fxch		st(2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 (0)
		// m03*v3 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 (2)
		// m03*v3 (1)
		fld			dword ptr [ecx + 16]
		// m10
		// m00*v0 + m01*v1 + m02*v2 (1)
		// m03*v3 (0)
		fmul		dword ptr [edx]
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m02*v2 (0)
		// m03*v3 (0)
		fxch		st(2)
		// m03*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 (0)
		// m10*v0 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (2)
		// m10*v0 (1)
		fld			dword ptr [ecx + 16 + 4]
		// m11
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (1)
		// m10*v0 (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fld			dword ptr [ecx + 16 + 8]
		// m12
		// m11*v1 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m10*v0 (0)
		fxch		st(3)
		// m10*v0 (0)
		// m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (1)
		fld			dword ptr [ecx + 16 + 12]
		// m13
		// m10*v0 + m11*v1 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (0)
		fmul		dword ptr [edx + 12]
		// m13*v3 (2)
		// m10*v0 + m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m12*v2 (0)
		fxch		st(3)
		// m12*v2 (0)
		// m10*v0 + m11*v1 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (1)
		fld			dword ptr [ecx + 32]
		// m20
		// m10*v0 + m11*v1 + m12*v2 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (0)
		fmul		dword ptr [edx]
		// m20*v0 (2)
		// m10*v0 + m11*v1 + m12*v2 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m13*v3 (0)
		fxch		st(3)
		// m13*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (1)
		fld			dword ptr [ecx + 32 + 4]
		// m21
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 4]
		// m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fld			dword ptr [ecx + 32 + 8]
		// m22
		// m21*v1 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 8]
		// m22*v2 (2)
		// m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m20*v0 (0)
		fxch		st(4)
		// m20*v0 (0)
		// m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (1)
		fld			dword ptr [ecx + 32 + 12]
		// m23
		// m20*v0 + m21*v1 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (0)
		fmul		dword ptr [edx + 12]
		// m23*v3 (2)
		// m20*v0 + m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m22*v2 (0)
		fxch		st(4)
		// m22*v2 (0)
		// m20*v0 + m21*v1 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m22*v2 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (1)
		fld			dword ptr [ecx + 48]
		// m30
		// m20*v0 + m21*v1 + m22*v2 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (0)
		fmul		dword ptr [edx]
		// m30*v0 (2)
		// m20*v0 + m21*v1 + m22*v2 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m23*v3 (0)
		fxch		st(4)
		// m23*v3 (0)
		// m20*v0 + m21*v1 + m22*v2 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (1)
		fld			dword ptr [ecx + 48 + 4]
		// m31
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (0)
		fmul		dword ptr [edx + 4]
		// m31*v1 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (0)
		fld			dword ptr [ecx + 48 + 8]
		// m32
		// m31*v1 (1)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (0)
		fmul		dword ptr [edx + 8]
		// m32*v2 (2)
		// m31*v1 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 (0)
		fxch		st(5)
		// m30*v0 (0)
		// m31*v1 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m32*v2 (2)
		faddp		st(1), st
		// m30*v0 + m31*v1 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m32*v2 (1)
		fld			dword ptr [ecx + 48 + 12]
		// m33
		// m30*v0 + m31*v1 (1)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m32*v2 (0)
		fmul		dword ptr [edx + 12]
		// m33*v3 (2)
		// m30*v0 + m31*v1 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m32*v2 (0)
		fxch		st(5)
		// m32*v2 (0)
		// m30*v0 + m31*v1 (0)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m33*v3 (2)
		faddp		st(1), st
		// m30*v0 + m31*v1 + m32*v2 (2)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m33*v3 (1)
		fxch		st(1)
		// m20*v0 + m21*v1 + m22*v2 + m23*v3 (0)
		// m30*v0 + m31*v1 + m32*v2 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m33*v3 (1)
		fstp		dword ptr [eax + 8]
		// m30*v0 + m31*v1 + m32*v2 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m33*v3 (0)
		faddp		st(3), st
		// m10*v0 + m11*v1 + m12*v2 + m13*v3 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 + m31*v1 + m32*v2 + m33*v3 (2)
		fstp		dword ptr [eax + 4]
		// m00*v0 + m01*v1 + m02*v2 + m03*v3 (0)
		// m30*v0 + m31*v1 + m32*v2 + m33*v3 (0)
		fstp		dword ptr [eax]
		// m30*v0 + m31*v1 + m32*v2 + m33*v3 (0)
		fstp		dword ptr [eax + 12]

		ret
	}
*/
}

void IdentityMat(Matrix m)
{
	m[0][0] = 1.0f; m[0][1] = 0.0f; m[0][2] = 0.0f; m[0][3] = 0.0f;
	m[1][0] = 0.0f; m[1][1] = 1.0f; m[1][2] = 0.0f; m[1][3] = 0.0f;
	m[2][0] = 0.0f; m[2][1] = 0.0f; m[2][2] = 1.0f; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void ScaleMat(Matrix m, float sx, float sy, float sz)
{
	m[0][0] = sx; m[0][1] = 0.0f; m[0][2] = 0.0f; m[0][3] = 0.0f;
	m[1][0] = 0.0f; m[1][1] = sy; m[1][2] = 0.0f; m[1][3] = 0.0f;
	m[2][0] = 0.0f; m[2][1] = 0.0f; m[2][2] = sz; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void TranslateMat(Matrix m, float dx, float dy, float dz)
{
	m[0][0] = 1.0f; m[0][1] = 0.0f; m[0][2] = 0.0f; m[0][3] = dx;
	m[1][0] = 0.0f; m[1][1] = 1.0f; m[1][2] = 0.0f; m[1][3] = dy;
	m[2][0] = 0.0f; m[2][1] = 0.0f; m[2][2] = 1.0f; m[2][3] = dz;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void RotateXMat(Matrix m, float angle)
{
	float cos_angle, sin_angle;
	fsincos(angle, &sin_angle, &cos_angle);
	m[0][0] = 1.0f; m[0][1] = 0.0f; m[0][2] = 0.0f; m[0][3] = 0.0f;
	m[1][0] = 0.0f; m[1][1] = cos_angle; m[1][2] = -sin_angle; m[1][3] = 0.0f;
	m[2][0] = 0.0f; m[2][1] = sin_angle; m[2][2] = cos_angle; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void RotateYMat(Matrix m, float angle)
{
	float cos_angle, sin_angle;
	fsincos(angle, &sin_angle, &cos_angle);
	m[0][0] = cos_angle; m[0][1] = 0.0f; m[0][2] = sin_angle; m[0][3] = 0.0f;
	m[1][0] = 0.0f; m[1][1] = 1.0f; m[1][2] = 0.0f; m[1][3] = 0.0f;
	m[2][0] = -sin_angle; m[2][1] = 0.0f; m[2][2] = cos_angle; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void RotateZMat(Matrix m, float angle)
{
	float cos_angle, sin_angle;
	fsincos(angle, &sin_angle, &cos_angle);
	m[0][0] = cos_angle; m[0][1] = -sin_angle; m[0][2] = 0.0f; m[0][3] = 0.0f;
	m[1][0] = sin_angle; m[1][1] = cos_angle; m[1][2] = 0.0f; m[1][3] = 0.0f;
	m[2][0] = 0.0f; m[2][1] = 0.0f; m[2][2] = 1.0f; m[2][3] = 0.0f;
	m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

void ViewportMat(Matrix m, float min_x, float max_x, float min_y, float max_y, float min_z, float max_z)
{
	Matrix T, S_vp, T_vp;

	TranslateMat(T, 1.0f, 1.0f, 1.0f);
	ScaleMat(S_vp, 0.5f*(max_x - min_x), 0.5f*(max_y - min_y), 0.5f*(max_z - min_z));
	TranslateMat(T_vp, min_x, min_y, min_z);

	MatMultMat3x4(m, T_vp, S_vp);
	MatMultMat3x4(m, m, T);
}

void OrthoMat(Matrix m, float left_plane, float right_plane, float bottom_plane, float top_plane, float near_plane, float far_plane)
{
	m[0][0] = 2.0f/(right_plane - left_plane);
	m[0][1] = 0.0f;
	m[0][2] = 0.0f;
	m[0][3] = -(right_plane + left_plane)/(right_plane - left_plane);

	m[1][0] = 0.0f;
	m[1][1] = 2.0f/(top_plane - bottom_plane);
	m[1][2] = 0.0f;
	m[1][3] = -(top_plane + bottom_plane)/(top_plane - bottom_plane);

	m[2][0] = 0.0f;
	m[2][1] = 0.0f;
	m[2][2] = -2.0f/(far_plane - near_plane);
	m[2][3] = -(far_plane + near_plane)/(far_plane - near_plane);

	m[3][0] = 0.0f;
	m[3][1] = 0.0f;
	m[3][2] = 0.0f;
	m[3][3] = 1.0f;

	// flip y to compensate for upside down screen coords
	m[1][1] = -m[1][1];
}

void FrustumMat(Matrix m, float left_plane, float right_plane, float bottom_plane, float top_plane, float near_plane, float far_plane)
{
	m[0][0] = 2.0f*near_plane/(right_plane - left_plane);
	m[0][1] = 0.0f;
	m[0][2] = (right_plane + left_plane)/(right_plane - left_plane);
	m[0][3] = 0.0f;

	m[1][0] = 0.0f;
	m[1][1] = 2.0f*near_plane/(top_plane - bottom_plane);
	m[1][2] = (top_plane + bottom_plane)/(top_plane - bottom_plane);
	m[1][3] = 0.0f;

	m[2][0] = 0.0f;
	m[2][1] = 0.0f;
	m[2][2] = -(far_plane + near_plane)/(far_plane - near_plane);
	m[2][3] = -2.0f*far_plane*near_plane/(far_plane - near_plane);

	m[3][0] = 0.0f;
	m[3][1] = 0.0f;
	m[3][2] = -1.0f;
	m[3][3] = 0.0f;

	// flip y to compensate for upside down screen coords
	m[1][1] = -m[1][1];
}

void PerspectiveMat(Matrix m, float fovy, float aspect_ratio, float near_plane, float far_plane)
{
	float sin_angle, cos_angle;
	float left_plane, right_plane, bottom_plane, top_plane;

	fsincos(0.5f*fovy, &sin_angle, &cos_angle);

	top_plane = near_plane*sin_angle/cos_angle;
	bottom_plane = -top_plane;
	right_plane = top_plane*aspect_ratio;
	left_plane = -right_plane;

	FrustumMat(m, left_plane, right_plane, bottom_plane, top_plane, near_plane, far_plane);
}

void ParallelMat(Matrix m, float fovy, float aspect_ratio, float near_plane, float far_plane)
{
	float sin_angle, cos_angle;
	float left_plane, right_plane, bottom_plane, top_plane;

	fsincos(0.5f*fovy, &sin_angle, &cos_angle);

	top_plane = near_plane*sin_angle/cos_angle;
	bottom_plane = -top_plane;
	right_plane = top_plane*aspect_ratio;
	left_plane = -right_plane;

	OrthoMat(m, left_plane, right_plane, bottom_plane, top_plane, near_plane, far_plane);
}


void MatrixCopy(Matrix dst, Matrix src)
{
	dst[0][0] = src[0][0];
	dst[0][1] = src[0][1];
	dst[0][2] = src[0][2];
	dst[0][3] = src[0][3];
	dst[1][0] = src[1][0];
	dst[1][1] = src[1][1];
	dst[1][2] = src[1][2];
	dst[1][3] = src[1][3];
	dst[2][0] = src[2][0];
	dst[2][1] = src[2][1];
	dst[2][2] = src[2][2];
	dst[2][3] = src[2][3];
	dst[3][0] = src[3][0];
	dst[3][1] = src[3][1];
	dst[3][2] = src[3][2];
	dst[3][3] = src[3][3];
}
/*
void MatrixCopy(Matrix dst, Matrix src)
{
	__asm // 21 cycles
	{
		mov			edx, [esp + 4]
		mov			ecx, [esp + 8]

		// ******** 1 cycle delay ********
		// (waiting for ecx)
		mov			eax, [ecx]

		mov			[edx], eax
		mov			eax, [ecx + 4]

		mov			[edx + 4], eax
		mov			eax, [ecx + 8]

		mov			[edx + 8], eax
		mov			eax, [ecx + 12]

		mov			[edx + 12], eax
		mov			eax, [ecx + 16]

		mov			[edx + 16], eax
		mov			eax, [ecx + 20]

		mov			[edx + 20], eax
		mov			eax, [ecx + 24]

		mov			[edx + 24], eax
		mov			eax, [ecx + 28]

		mov			[edx + 28], eax
		mov			eax, [ecx + 32]

		mov			[edx + 32], eax
		mov			eax, [ecx + 36]

		mov			[edx + 36], eax
		mov			eax, [ecx + 40]

		mov			[edx + 40], eax
		mov			eax, [ecx + 44]

		mov			[edx + 44], eax
		mov			eax, [ecx + 48]

		mov			[edx + 48], eax
		mov			eax, [ecx + 52]

		mov			[edx + 52], eax
		mov			eax, [ecx + 56]

		mov			[edx + 56], eax
		mov			eax, [ecx + 60]

		mov			[edx + 60], eax

		ret
	}
}
*/