------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                U I N T P                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.42 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;  use Alloc;
with Debug;  use Debug;
with Namet;  use Namet;
with Output; use Output;

package body Uintp is

   --  Values outside the range that is represented direction are stored
   --  using two tables. The secondary table Udigits contains sequences of
   --  Int values consisting of the digits of the number in a radix Base
   --  system. The digits are stored from most significant to least significant
   --  with the first digit only carrying the sign.

   --  There is one entry in the primary Uints table for each distinct Uint
   --  value. This table entry contains the length (number of digits) and
   --  a starting offset of the value in the Udigits table.

   Uint_First_Entry : constant Uint := Uint (Uint_Direct_Last + 1);

   --  Some subprograms defined in this package manipulate the Udigits
   --  table directly, while for others it is more convenient to work with
   --  locally defined arrays of the digits of the the Universal Integers.
   --  The type UI_Vector is defined for this purpose and some internal
   --  subprograms used for converting from one to the other are defined.

   type UI_Vector is array (Pos range <>) of Int;
   --  Vector containing the integer values of a Uint value

   --  Note: An earlier version of this package used pointers of arrays
   --  of Ints (dynamically allocated) for the Uint type. The change
   --  leads to a few less natural idioms used throughout this code, but
   --  eliminates all uses of the heap except for the table package itself.
   --  For example, Uint parameters are often converted to UI_Vectors for
   --  internal manipulation. This is done by creating the local UI_Vector
   --  using the function N_Digits on the Uint to find the size needed for
   --  the vector, and then calling Init_Operand to copy the values out
   --  of the table into the vector.

   type Uint_Entry is record
      Length : Pos;
      --  Length of entry in Udigits table in digits (i.e. in words)

      Loc : Int;
      --  Starting location in Udigits table of this Uint value
   end record;

   package Uints is new Table (
     Table_Component_Type => Uint_Entry,
     Table_Index_Type     => Uint,
     Table_Low_Bound      => Uint_First_Entry,
     Table_Initial        => Alloc_Uints_Initial,
     Table_Increment      => Alloc_Uints_Increment,
     Table_Name           => "Uints");

   package Udigits is new Table (
     Table_Component_Type => Int,
     Table_Index_Type     => Int,
     Table_Low_Bound      => 0,
     Table_Initial        => Alloc_Udigits_Initial,
     Table_Increment      => Alloc_Udigits_Increment,
     Table_Name           => "Udigits");

   Uint_Int_First : Uint := Uint_0;
   --  Uint value containing Int'First value, set by Initialize. The initial
   --  value of Uint_0 is used for an assertion check that ensures that this
   --  value is not used before it is initialized. This value is used in the
   --  UI_Is_In_Int_Range predicate, and it is right that this is a host
   --  value, since the issue is host representation of integer values.

   Uint_Int_Last : Uint;
   --  Uint value containing Int'Last value set by Initialize.

   UI_Power_2 : array (Int range 0 .. 64) of Uint;
   --  This table is used to memoize exponentiations by powers of 2. The Nth
   --  entry, if set, contains the Uint value 2 ** N. Initially UI_Power_2_Set
   --  is zero and only the 0'th entry is set, the invariant being that all
   --  entries in the range 0 .. UI_Power_2_Set are initialized.

   UI_Power_2_Set : Nat;
   --  Number of entries set in UI_Power_2;

   UI_Power_10 : array (Int range 0 .. 64) of Uint;
   --  This table is used to memoize exponentiations by powers of 10 in the
   --  same manner as described above for UI_Power_2.

   UI_Power_10_Set : Nat;
   --  Number of entries set in UI_Power_10;

   Uints_Min   : Uint;
   Udigits_Min : Int;
   --  These values are used to make sure that the mark/release mechanism
   --  does not destroy values saved in the U_Power tables. Whenever an
   --  entry is made in the U_Power tables, Uints_Min and Udigits_Min are
   --  updated to protect the entry, and Release never cuts back beyond
   --  these minimum values.

   -----------------------
   -- Local Subprograms --
   -----------------------

   function N_Digits (Input : Uint) return Int;
   pragma Inline (N_Digits);
   --  Returns number of "digits" in a Uint

   procedure Image_Out (Input : Uint; To_Buffer : Boolean);
   --  Common processing for UI_Image and UI_Write, To_Buffer is set
   --  True for UI_Image, and false for UI_Write.

   procedure Init_Operand (UI : Uint; Vec : out UI_Vector);
   pragma Inline (Init_Operand);
   --  This procedure copies the digits from UI or the table into the
   --  vector parameter. The parameter should be of the correct size
   --  as determined by a previous call to N_Digits with UI.

   function Vector_To_Uint
     (In_Vec   : UI_Vector;
      Negative : Boolean)
      return     Uint;
   --  Functions that calculate values in UI_Vectors, call this function
   --  to create and return the Uint value.

   ---------------
   -- Image_Out --
   ---------------

   procedure Image_Out (Input : Uint; To_Buffer : Boolean) is
      Marks             : constant Uintp.Save_Mark := Uintp.Mark;
      Save_Uints_Last   : constant Uint := Uints.Last;
      Save_Udigits_Last : constant Int  := Udigits.Last;

      Exponent : Int;
      --  This is used only if the image buffer overflows, we reserve
      --  the last 6 digits for a possible exponent.

      procedure Image_Char (C : Character);
      --  Internal procedure to output one character

      procedure Image_Uint (U : Uint);
      --  Internal procedure to output characters of non-negative Uint

      procedure Image_Char (C : Character) is
      begin
         if To_Buffer then
            if UI_Image_Length = 32 then
               Exponent := Exponent + 1;

            else
               UI_Image_Length := UI_Image_Length + 1;
               UI_Image_Buffer (UI_Image_Length) := C;
            end if;

         else
            Write_Char (C);
         end if;
      end Image_Char;

      procedure Image_Uint (U : Uint) is
      begin
         if U >= Uint_10 then
            Image_Uint (U / Uint_10);
            Image_Uint (U rem Uint_10);
         else
            Image_Char (Character'Val
                         (UI_To_Int (U) + Int (Character'Pos ('0'))));
         end if;
      end Image_Uint;

   --  Start of processing for Image_Out

   begin
      UI_Image_Length := 0;
      Exponent := 0;

      if Input < Uint_0 then
         Image_Char ('-');
         Image_Uint (-Input);
      else
         Image_Uint (Input);
      end if;

      --  Output exponent if value overflowed buffer. Use last 8 character
      --  positions for an exponent.

      if Exponent > 0 then
         UI_Image_Buffer (25) := 'E';
         UI_Image_Length := 26;
         Image_Uint (UI_From_Int (Exponent + (32 - 24)));
      end if;

      --  Reset table pointers to remove temporary junk created by this
      --  processing, since none of these intermediate values are needed

      Uintp.Release (Marks);
   end Image_Out;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      Uints.Init;
      Udigits.Init;

      Uint_Int_First := UI_From_Int (Int'First);
      Uint_Int_Last  := UI_From_Int (Int'Last);

      UI_Power_2 (0) := Uint_1;
      UI_Power_2_Set := 0;

      UI_Power_10 (0) := Uint_1;
      UI_Power_10_Set := 0;

      Uints_Min := Uints.Last;
      Udigits_Min := Udigits.Last;

   end Initialize;

   -------------------
   -- Init_Operand --
   -------------------

   procedure Init_Operand (UI : Uint; Vec : out UI_Vector) is
      Loc : Int;

   begin
      if Int (UI) <= Int (Uint_Direct_Last) then
         Vec (1) := Int (UI) - Int (Uint_Direct_Bias);
      else
         Loc := Uints.Table (UI).Loc;

         for J in 1 .. Uints.Table (UI).Length loop
            Vec (J) := Udigits.Table (Loc + J - 1);
         end loop;
      end if;
   end Init_Operand;

   ----------
   -- Mark --
   ----------

   function Mark return Save_Mark is
   begin
      if Debug_Flag_D then
         w ("Mark: Save_Udigit = ", Int (Udigits.Last));
      end if;

      return (Save_Uint => Uints.Last, Save_Udigit => Udigits.Last);
   end Mark;

   ---------------
   -- N_Digits --
   ---------------

   function N_Digits (Input : Uint) return Int is
   begin
      if Int (Input) <= Int (Uint_Direct_Last) then
         return 1;
      else
         return Uints.Table (Input).Length;
      end if;
   end N_Digits;

   -------------
   -- Release --
   -------------

   procedure Release (M : Save_Mark) is
   begin
      Uints.Set_Last   (Uint'Max (M.Save_Uint,   Uints_Min));
      Udigits.Set_Last (Int'Max  (M.Save_Udigit, Udigits_Min));

      if Debug_Flag_D then
         w ("Udigits.Last = ", Udigits.Last);
         w ("        Save = ", Int (M.Save_Udigit));
         w ("        Min  = ", Int (Udigits_Min));
      end if;

   end Release;

   ---------------
   -- Tree_Read --
   ---------------

   procedure Tree_Read is
   begin
      Uints.Tree_Read;
      Udigits.Tree_Read;
   end Tree_Read;

   ----------------
   -- Tree_Write --
   ----------------

   procedure Tree_Write is
   begin
      Uints.Tree_Write;
      Udigits.Tree_Write;
   end Tree_Write;

   -------------
   -- UI_Abs --
   -------------

   function UI_Abs (Right : Uint) return Uint is
   begin
      if Right < Uint_0 then
         return -Right;
      else
         return Right;
      end if;
   end UI_Abs;

   -------------
   -- UI_Add --
   -------------

   function UI_Add (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Add (UI_From_Int (Left), Right);
   end UI_Add;

   function UI_Add (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Add (Left, UI_From_Int (Right));
   end UI_Add;

   function UI_Add (Left : Uint; Right : Uint) return Uint is
   begin
      --  First try simple case where Int "+" can be used;

      if Int (Left) <= Int (Uint_Direct_Last)
        and then Int (Right) <= Int (Uint_Direct_Last)
      then
         return
           UI_From_Int
             ((Int (Left)  - Int (Uint_Direct_Bias)) +
              (Int (Right) - Int (Uint_Direct_Bias)));

      --  Otherwise full circuit is needed

      else
         declare
            L_Length   : Int := N_Digits (Left);
            R_Length   : Int := N_Digits (Right);
            L_Vec      : UI_Vector (1 .. L_Length);
            R_Vec      : UI_Vector (1 .. R_Length);
            Sum_Length : Int;
            Tmp_Int    : Int;
            Carry      : Int;
            Borrow     : Int;
            X_Bigger   : Boolean := False;
            Y_Bigger   : Boolean := False;
            Result_Neg : Boolean := False;

         begin
            Init_Operand (Left, L_Vec);
            Init_Operand (Right, R_Vec);

            --  At least one more than 1 digit, so calculation is needed. First
            --  calculate the number of digits sufficient to hold result.

            if L_Length > R_Length then
               Sum_Length := L_Length + 1;
               X_Bigger := True;
            else
               Sum_Length := R_Length + 1;
               if R_Length > L_Length then Y_Bigger := True; end if;
            end if;

            --  Make copies of the absolute values of L_Vec and R_Vec into
            --  X and Y both with lengths equal to the maximum possibly
            --  needed. This makes looping over the digits much simpler.

            declare
               X      : UI_Vector (1 .. Sum_Length);
               Y      : UI_Vector (1 .. Sum_Length);
               Tmp_UI : UI_Vector (1 .. Sum_Length);

            begin
               for I in 1 .. Sum_Length - L_Length loop
                  X (I) := 0;
               end loop;

               X (Sum_Length - L_Length + 1) := abs L_Vec (1);

               for I in 2 .. L_Length loop
                  X (I + (Sum_Length - L_Length)) := L_Vec (I);
               end loop;

               for I in 1 .. Sum_Length - R_Length loop
                  Y (I) := 0;
               end loop;

               Y (Sum_Length - R_Length + 1) := abs R_Vec (1);

               for I in 2 .. R_Length loop
                  Y (I + (Sum_Length - R_Length)) := R_Vec (I);
               end loop;

               if (L_Vec (1) < 0) = (R_Vec (1) < 0) then

                  --  Same sign so just add

                  Carry := 0;
                  for I in reverse 1 .. Sum_Length loop
                     Tmp_Int := X (I) + Y (I) + Carry;
                     if Tmp_Int > Base then
                        Tmp_Int := Tmp_Int - Base;
                        Carry := 1;
                     else
                        Carry := 0;
                     end if;
                     X (I) := Tmp_Int;
                  end loop;

                  return Vector_To_Uint (X, L_Vec (1) < 0);

               else
                  --  Find which one has bigger magnitude

                  if not (X_Bigger or Y_Bigger) then
                     for I in L_Vec'Range loop
                        if abs L_Vec (I) > abs R_Vec (I) then
                           X_Bigger := True;
                           exit;
                        elsif abs R_Vec (I) > abs L_Vec (I) then
                           Y_Bigger := True;
                           exit;
                        end if;
                     end loop;
                  end if;

                  --  If they have identical magnitude, just return 0, else
                  --  swap if necessary so that X had the bigger magnitude.
                  --  Determine if result is negative at this time.

                  Result_Neg := False;

                  if not (X_Bigger or Y_Bigger) then
                     return Uint_0;

                  elsif Y_Bigger then
                     if R_Vec (1) < 0 then
                        Result_Neg := True;
                     end if;

                     Tmp_UI := X;
                     X := Y;
                     Y := Tmp_UI;

                  else
                     if L_Vec (1) < 0 then
                        Result_Neg := True;
                     end if;
                  end if;

                  --  Subtract Y from the bigger X

                  Borrow := 0;

                  for J in reverse 1 .. Sum_Length loop
                     Tmp_Int := X (J) - Y (J) + Borrow;

                     if Tmp_Int < 0 then
                        Tmp_Int := Tmp_Int + Base;
                        Borrow := -1;
                     else
                        Borrow := 0;
                     end if;

                     X (J) := Tmp_Int;
                  end loop;

                  return Vector_To_Uint (X, Result_Neg);

               end if;
            end;
         end;
      end if;
   end UI_Add;

   --------------------------
   -- UI_Decimal_Digits_Hi --
   --------------------------

   function UI_Decimal_Digits_Hi (U : Uint) return Nat is
   begin
      --  The maximum value of a "digit" is 32767, which is 5 decimal
      --  digits, so an N_Digit number could take up to 5 times this
      --  number of digits. This is certainly too high for large
      --  numbers but it is not worth worrying about.

      return 5 * N_Digits (U);
   end UI_Decimal_Digits_Hi;

   --------------------------
   -- UI_Decimal_Digits_Lo --
   --------------------------

   function UI_Decimal_Digits_Lo (U : Uint) return Nat is
   begin
      --  The maximum value of a "digit" is 32767, which is more than four
      --  decimal digits, but not a full five digits. The easily computed
      --  minimum number of decimal digits is thus 1 + 4 * the number of
      --  digits. This is certainly too low for large numbers but it is
      --  not worth worrying about.

      return 1 + 4 * (N_Digits (U) - 1);
   end UI_Decimal_Digits_Lo;

   ------------
   -- UI_Div --
   ------------

   function UI_Div (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Div (UI_From_Int (Left), Right);
   end UI_Div;

   function UI_Div (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Div (Left, UI_From_Int (Right));
   end UI_Div;

   function UI_Div (Left, Right : Uint) return Uint is
      L_Length    : constant Int := N_Digits (Left);
      R_Length    : constant Int := N_Digits (Right);
      Q_Length    : constant Int := L_Length - R_Length + 1;
      L_Vec       : UI_Vector (1 .. L_Length);
      R_Vec       : UI_Vector (1 .. R_Length);
      D           : Int;
      Remainder   : Int;
      Tmp_Divisor : Int;
      Carry       : Int;
      Tmp_Int     : Int;
      Tmp_Dig     : Int;

   begin
      pragma Assert (Right /= Uint_0);

      --  Some special cases that are simpler to compute than the general
      --  case are treated first.

      if L_Length = 1 and then R_Length = 1 then
         return UI_From_Int (UI_To_Int (Left) / UI_To_Int (Right));
      elsif  L_Length < R_Length then
         return Uint_0;
      end if;

      Init_Operand (Left, L_Vec);
      Init_Operand (Right, R_Vec);

      --  Case of right operand is single digit. Here we can simply divide
      --  each digit of the left operand by the divisor, from most to least
      --  significant, carrying the remainder to the next digit (just like
      --  ordinary long division by hand).

      if R_Length = 1 then
         Remainder := 0;
         Tmp_Divisor := abs R_Vec (1);

         declare
            Quotient : UI_Vector (1 .. L_Length);

         begin
            for J in L_Vec'Range loop
               Tmp_Int      := Remainder * Base + abs L_Vec (J);
               Quotient (J) := Tmp_Int / Tmp_Divisor;
               Remainder    := Tmp_Int rem Tmp_Divisor;
            end loop;

            return
              Vector_To_Uint (Quotient, (L_Vec (1) < 0 xor R_Vec (1) < 0));
         end;
      end if;

      --  The possible simple cases have been exhausted. Now turn to the
      --  algorithm D from the section of Knuth mentioned at the top of
      --  this package.

      Algorithm_D : declare
         Dividend     : UI_Vector (1 .. L_Length + 1);
         Divisor      : UI_Vector (1 .. R_Length);
         Quotient     : UI_Vector (1 .. Q_Length);
         Divisor_Dig1 : Int;
         Divisor_Dig2 : Int;
         Q_Guess      : Int;

      begin
         --  [ NORMALIZE ] (step D1 in the algorithm). First calculate the
         --  scale d, and then multiply Left and Right (u and v in the book)
         --  by d to get the dividend and divisor to work with.

         D := Base / (abs R_Vec (1) + 1);

         Dividend (1) := 0;
         Dividend (2) := abs L_Vec (1);

         for J in 3 .. L_Length + Int (1) loop
            Dividend (J) := L_Vec (J - 1);
         end loop;

         Divisor (1) := abs R_Vec (1);

         for J in Int (2) .. R_Length loop
            Divisor (J) := R_Vec (J);
         end loop;

         if D > 1 then

            --  Multiply Dividend by D

            Carry := 0;
            for J in reverse Dividend'Range loop
               Tmp_Int      := Dividend (J) * D + Carry;
               Dividend (J) := Tmp_Int rem Base;
               Carry        := Tmp_Int / Base;
            end loop;

            --  Multiply Divisor by d.

            Carry := 0;
            for J in reverse Divisor'Range loop
               Tmp_Int      := Divisor (J) * D + Carry;
               Divisor (J)  := Tmp_Int rem Base;
               Carry        := Tmp_Int / Base;
            end loop;

         end if;

         --  Main loop of long division algorithm.

         Divisor_Dig1 := Divisor (1);
         Divisor_Dig2 := Divisor (2);

         for J in Quotient'Range loop

            --  [ CALCULATE Q (hat) ] (step D3 in the algorithm).

            Tmp_Int := Dividend (J) * Base + Dividend (J + 1);

            --  Initial guess

            if Dividend (J) = Divisor_Dig1 then
               Q_Guess := Base - 1;
            else
               Q_Guess := Tmp_Int / Divisor_Dig1;
            end if;

            --  Refine the guess

            while Divisor_Dig2 * Q_Guess >
                  (Tmp_Int - Q_Guess * Divisor_Dig1) * Base + Dividend (J + 2)
            loop
               Q_Guess := Q_Guess - 1;
            end loop;

            --  [ MULTIPLY & SUBTRACT] (step D4). Q_Guess * Divisor is
            --  subtracted from the remaining dividend.

            Carry := 0;
            for K in reverse Divisor'Range loop
               Tmp_Int := Dividend (J + K) - Q_Guess * Divisor (K) + Carry;
               Tmp_Dig := Tmp_Int rem Base;
               Carry   := Tmp_Int / Base;

               if Tmp_Dig < 0 then
                  Tmp_Dig := Tmp_Dig + Base;
                  Carry   := Carry - 1;
               end if;

               Dividend (J + K) := Tmp_Dig;
            end loop;

            Dividend (J) := Dividend (J) + Carry;

            --  [ TEST REMAINDER ] & [ ADD BACK ] (steps D5 and D6)
            --  Here there is a slight difference from the book: the last
            --  carry is always added in above and below (cancelling each
            --  other). In fact the dividend going negative is used as
            --  the test.

            --  If the Dividend went negative, then Q_Guess was off by
            --  one, so it is decremented, and the divisor is added back
            --  into the relevant portion of the dividend.

            if Dividend (J) < 0 then
               Q_Guess := Q_Guess - 1;

               Carry := 0;
               for K in reverse Divisor'Range loop
                  Tmp_Int := Dividend (J + K) + Divisor (K) + Carry;

                  if Tmp_Int > Base then
                     Tmp_Int := Tmp_Int - Base;
                     Carry := 1;
                  else
                     Carry := 0;
                  end if;

                  Dividend (J + K) := Tmp_Int;
               end loop;

               Dividend (J) := Dividend (J) + Carry;
            end if;

            --  Finally we can get the next quotient digit

            Quotient (J) := Q_Guess;

         end loop;

         return Vector_To_Uint (Quotient, (L_Vec (1) < 0 xor R_Vec (1) < 0));

      end Algorithm_D;

   end UI_Div;

   ------------
   -- UI_Eq --
   ------------

   function UI_Eq (Left : Int; Right : Uint) return Boolean is
   begin
      return not UI_Ne (UI_From_Int (Left), Right);
   end UI_Eq;

   function UI_Eq (Left : Uint; Right : Int) return Boolean is
   begin
      return not UI_Ne (Left, UI_From_Int (Right));
   end UI_Eq;

   function UI_Eq (Left : Uint; Right : Uint) return Boolean is
   begin
      return not UI_Ne (Left, Right);
   end UI_Eq;

   --------------
   -- UI_Expon --
   --------------

   function UI_Expon (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Expon (UI_From_Int (Left), Right);
   end UI_Expon;

   function UI_Expon (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Expon (Left, UI_From_Int (Right));
   end UI_Expon;

   function UI_Expon (Left : Int; Right : Int) return Uint is
   begin
      return UI_Expon (UI_From_Int (Left), UI_From_Int (Right));
   end UI_Expon;

   function UI_Expon (Left : Uint; Right : Uint) return Uint is
   begin
      pragma Assert (Right >= Uint_0);

      --  Any value raised to power of 0 is 1

      if Right = Uint_0 then
         return Uint_1;

      --  0 to any positive power is 0.

      elsif Left = Uint_0 then
         return Uint_0;

      --  Any value raised to power of 1 is that value

      elsif Right = Uint_1 then
         return Left;

      --  Cases which can be done by table lookup

      elsif Right <= Uint_64 then

         --  2 ** N for N in 2 .. 64

         if Left = Uint_2 then
            declare
               Right_Int : constant Int :=
                             Int (Right) - Int (Uint_Direct_Bias);

            begin
               if Right_Int > UI_Power_2_Set then
                  for J in UI_Power_2_Set + Int (1) .. Right_Int loop
                     UI_Power_2 (J) := UI_Power_2 (J - Int (1)) * Int (2);
                     Uints_Min := Uints.Last;
                     Udigits_Min := Udigits.Last;
                  end loop;

                  UI_Power_2_Set := Right_Int;
               end if;

               return UI_Power_2 (Right_Int);
            end;

         --  10 ** N for N in 2 .. 64

         elsif Left = Uint_10 then
            declare
               Right_Int : constant Int :=
                             Int (Right) - Int (Uint_Direct_Bias);

            begin
               if Right_Int > UI_Power_10_Set then
                  for J in UI_Power_10_Set + Int (1) .. Right_Int loop
                     UI_Power_10 (J) := UI_Power_10 (J - Int (1)) * Int (10);
                     Uints_Min := Uints.Last;
                     Udigits_Min := Udigits.Last;
                  end loop;

                  UI_Power_10_Set := Right_Int;
               end if;

               return UI_Power_10 (Right_Int);
            end;
         end if;
      end if;

      --  If we fall through, then we have the general case (see Knuth 4.6.3)

      declare
         N       : Uint := Right;
         Squares : Uint := Left;
         Result  : Uint := Uint_1;

      begin
         loop
            if (N mod Uint_2) = Uint_1 then
               Result := Result * Squares;
            end if;

            N := N / Uint_2;
            exit when N = Uint_0;
            Squares := Squares *  Squares;
         end loop;

         return Result;
      end;
   end UI_Expon;

   ------------------
   -- UI_From_Int --
   ------------------

   function UI_From_Int (Input : Int) return Uint is
   begin
      --  The case -Base < Input < Base is the usual and simple case.

      if -Base < Input and then Input < Base then
         return Uint (Int (Uint_Direct_Bias) + Input);

      --  For values of larger magnitude, compute digits into a vector and
      --  call Vector_To_Uint.

      else
         declare
            Max_For_Int : constant := 4;
            --  Base is defined so that 4 Uint digits is sufficient
            --  to hold the largest possible Int value.

            V : UI_Vector (1 .. Max_For_Int);
            Temp_Integer : Int;
            Ret_Value : Uint;

         begin
            for I in V'Range loop
               V (I) := 0;
            end loop;

            Temp_Integer := Input;

            for I in reverse V'Range loop
               V (I) := abs (Temp_Integer rem Base);
               Temp_Integer := Temp_Integer / Base;
            end loop;

            return Vector_To_Uint (V, Input < 0);
         end;
      end if;
   end UI_From_Int;

   ------------
   -- UI_Ge --
   ------------

   function UI_Ge (Left : Int; Right : Uint) return Boolean is
   begin
      return not UI_Lt (UI_From_Int (Left), Right);
   end UI_Ge;

   function UI_Ge (Left : Uint; Right : Int) return Boolean is
   begin
      return not UI_Lt (Left, UI_From_Int (Right));
   end UI_Ge;

   function UI_Ge (Left : Uint; Right : Uint) return Boolean is
   begin
      return not UI_Lt (Left, Right);
   end UI_Ge;

   ------------
   -- UI_Gt --
   ------------

   function UI_Gt (Left : Int; Right : Uint) return Boolean is
   begin
      return UI_Lt (Right, UI_From_Int (Left));
   end UI_Gt;

   function UI_Gt (Left : Uint; Right : Int) return Boolean is
   begin
      return UI_Lt (UI_From_Int (Right), Left);
   end UI_Gt;

   function UI_Gt (Left : Uint; Right : Uint) return Boolean is
   begin
      return UI_Lt (Right, Left);
   end UI_Gt;

   ---------------
   -- UI_Image --
   ---------------

   procedure UI_Image (Input : Uint) is
   begin
      Image_Out (Input, True);
   end UI_Image;

   -------------------------
   -- UI_Is_In_Int_Range --
   -------------------------

   function UI_Is_In_Int_Range (Input : Uint) return Boolean is
   begin
      --  Make sure we don't get called before Initialize

      pragma Assert (Uint_Int_First /= Uint_0);

      return Input >= Uint_Int_First
        and then Input <= Uint_Int_Last;
   end UI_Is_In_Int_Range;

   ------------
   -- UI_Le --
   ------------

   function UI_Le (Left : Int; Right : Uint) return Boolean is
   begin
      return not UI_Lt (Right, UI_From_Int (Left));
   end UI_Le;

   function UI_Le (Left : Uint; Right : Int) return Boolean is
   begin
      return not UI_Lt (UI_From_Int (Right), Left);
   end UI_Le;

   function UI_Le (Left : Uint; Right : Uint) return Boolean is
   begin
      return not UI_Lt (Right, Left);
   end UI_Le;

   ------------
   -- UI_Lt --
   ------------

   function UI_Lt (Left : Int; Right : Uint) return Boolean is
   begin
      return UI_Lt (UI_From_Int (Left), Right);
   end UI_Lt;

   function UI_Lt (Left : Uint; Right : Int) return Boolean is
   begin
      return UI_Lt (Left, UI_From_Int (Right));
   end UI_Lt;

   function UI_Lt (Left : Uint; Right : Uint) return Boolean is
      L_Length : constant Int := N_Digits (Left);
      R_Length : constant Int := N_Digits (Right);

   begin
      --  Quick processing for identical arguments

      if Int (Left) = Int (Right) then
         return False;

      --  Quick processing for both arguments one digit long

      elsif L_Length = 1 and then R_Length = 1 then
         return Int (Left) < Int (Right);

      --  At least one argument is more than one digit long

      else
         declare
            L_Vec : UI_Vector (1 .. L_Length);
            R_Vec : UI_Vector (1 .. R_Length);

         begin
            Init_Operand (Left, L_Vec);
            Init_Operand (Right, R_Vec);

            if L_Vec (1) < 0 then

               --  First argument negative, second argument non-negative

               if R_Vec (1) >= 0 then
                  return True;

               --  Both arguments negative

               else
                  if L_Length /= R_Length then
                     return L_Length > R_Length;

                  elsif L_Vec (1) /= R_Vec (1) then
                     return L_Vec (1) < R_Vec (1);

                  else
                     for J in 2 .. L_Vec'Last loop
                        if L_Vec (J) /= R_Vec (J) then
                           return L_Vec (J) > R_Vec (J);
                        end if;
                     end loop;

                     return False;
                  end if;
               end if;

            else
               --  First argument non-negative, second argument negative

               if R_Vec (1) < 0 then
                  return False;

               --  Both arguments non-negative

               else
                  if L_Length /= R_Length then
                     return L_Length < R_Length;
                  else
                     for J in L_Vec'Range loop
                        if L_Vec (J) /= R_Vec (J) then
                           return L_Vec (J) < R_Vec (J);
                        end if;
                     end loop;

                     return False;
                  end if;
               end if;
            end if;
         end;
      end if;
   end UI_Lt;

   ------------
   -- UI_Max --
   ------------

   function UI_Max (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Max (UI_From_Int (Left), Right);
   end UI_Max;

   function UI_Max (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Max (Left, UI_From_Int (Right));
   end UI_Max;

   function UI_Max (Left : Uint; Right : Uint) return Uint is
   begin
      if Left >= Right then
         return Left;
      else
         return Right;
      end if;
   end UI_Max;

   ------------
   -- UI_Min --
   ------------

   function UI_Min (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Min (UI_From_Int (Left), Right);
   end UI_Min;

   function UI_Min (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Min (Left, UI_From_Int (Right));
   end UI_Min;

   function UI_Min (Left : Uint; Right : Uint) return Uint is
   begin
      if Left <= Right then
         return Left;
      else
         return Right;
      end if;
   end UI_Min;

   -------------
   -- UI_Mod --
   -------------

   function UI_Mod (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Mod (UI_From_Int (Left), Right);
   end UI_Mod;

   function UI_Mod (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Mod (Left, UI_From_Int (Right));
   end UI_Mod;

   function UI_Mod (Left : Uint; Right : Uint) return Uint is
      Urem : constant Uint := Left rem Right;

   begin
      if (Left < Uint_0) = (Right < Uint_0)
        or else Urem = Uint_0
      then
         return Urem;
      else
         return Right + Urem;
      end if;
   end UI_Mod;

   ------------
   -- UI_Mul --
   ------------

   function UI_Mul (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Mul (UI_From_Int (Left), Right);
   end UI_Mul;

   function UI_Mul (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Mul (Left, UI_From_Int (Right));
   end UI_Mul;

   function UI_Mul (Left : Uint; Right : Uint) return Uint is
      L_Length : constant Int := N_Digits (Left);
      R_Length : constant Int := N_Digits (Right);
      L_Vec    : UI_Vector (1 .. L_Length);
      R_Vec    : UI_Vector (1 .. R_Length);

   begin
      --  Simple case of single length operands. Note that we chose our base
      --  precisely to make this simple (the product always fits in Int range)

      if L_Length = 1 and then R_Length = 1 then
         return UI_From_Int
           ((Int (Left)  - Int (Uint_Direct_Bias)) *
            (Int (Right) - Int (Uint_Direct_Bias)));
      end if;

      --  Otherwise we have the general case (Algorithm M in Knuth)

      Init_Operand (Left, L_Vec);
      Init_Operand (Right, R_Vec);

      Algorithm_M : declare
         Product : UI_Vector (1 .. L_Length + R_Length);
         Tmp_Sum : Int;
         Carry   : Int;

      begin
         for J in Product'Range loop
            Product (J) := 0;
         end loop;

         for J in reverse R_Vec'Range loop
            Carry := 0;
            for K in reverse L_Vec'Range loop
               Tmp_Sum :=
                 abs (L_Vec (K) * R_Vec (J)) + Product (J + K) + Carry;
               Product (J + K) := Tmp_Sum rem Base;
               Carry := Tmp_Sum / Base;
            end loop;

            Product (J) := Carry;
         end loop;

         return Vector_To_Uint (Product, (L_Vec (1) < 0 xor R_Vec (1) < 0));
      end Algorithm_M;
   end UI_Mul;

   ------------
   -- UI_Ne --
   ------------

   function UI_Ne (Left : Int; Right : Uint) return Boolean is
   begin
      return UI_Ne (UI_From_Int (Left), Right);
   end UI_Ne;

   function UI_Ne (Left : Uint; Right : Int) return Boolean is
   begin
      return UI_Ne (Left, UI_From_Int (Right));
   end UI_Ne;

   function UI_Ne (Left : Uint; Right : Uint) return Boolean is
      Size      : constant Int := N_Digits (Left);
      Left_Loc  : Int;
      Right_Loc : Int;

   begin
      --  Quick processing for identical arguments

      if Int (Left) = Int (Right) then
         return False;

      --  Certainly not equal if sizes are different

      elsif Size /= N_Digits (Right) then
         return True;

      --  Quick processing for one digit case

      elsif Size = 1 then
         return Int (Left) /= Int (Right);

      --  Otherwise do comparison

      else
         Left_Loc  := Uints.Table (Left).Loc;
         Right_Loc := Uints.Table (Right).Loc;

         for J in Int (0) .. Size - Int (1) loop
            if Udigits.Table (Left_Loc + J) /=
               Udigits.Table (Right_Loc + J)
            then
               return True;
            end if;
         end loop;

         return False;
      end if;
   end UI_Ne;

   ----------------
   -- UI_Negate --
   ----------------

   function UI_Negate (Right : Uint) return Uint is
   begin
      --  Quick processing for single digit case. Note that the negative of
      --  a single digit value always fits in a single digit, because the
      --  range is symmetrical.

      if Int (Right) <= Int (Uint_Direct_Last) then
         return Uint (Int (Uint_Direct_Bias) -
                     (Int (Right) - Int (Uint_Direct_Bias)));

      --  Else copy the value to the end of the table, negating 1st digit

      else
         declare
            Length : Int := Uints.Table (Right).Length;
            Loc    : Int := Uints.Table (Right).Loc;

         begin
            Uints.Increment_Last;
            Uints.Table (Uints.Last).Length := Length;
            Uints.Table (Uints.Last).Loc := Udigits.Last + 1;

            Udigits.Increment_Last;
            Udigits.Table (Udigits.Last) := -Udigits.Table (Loc);

            for Idx in 2 .. Length loop
               Udigits.Increment_Last;
               Udigits.Table (Udigits.Last) := Udigits.Table (Loc + Idx - 1);
            end loop;

            if Debug_Flag_D then
               w ("In negate: Length = ", Int (Length));
               w ("      Digits.Last = ", Int (Udigits.Last));
            end if;

            return Uints.Last;
         end;
      end if;
   end UI_Negate;

   -------------
   -- UI_Rem --
   -------------

   function UI_Rem (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Rem (UI_From_Int (Left), Right);
   end UI_Rem;

   function UI_Rem (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Rem (Left, UI_From_Int (Right));
   end UI_Rem;

   function UI_Rem (Left, Right : Uint) return Uint is
   begin
      if N_Digits (Left) = 1 and then N_Digits (Right) = 1 then
         return UI_From_Int (UI_To_Int (Left) rem UI_To_Int (Right));
      else
         return Left - (Left / Right) * Right;
      end if;
   end UI_Rem;

   ------------
   -- UI_Sub --
   ------------

   function UI_Sub (Left : Int; Right : Uint) return Uint is
   begin
      return UI_Add (Left, -Right);
   end UI_Sub;

   function UI_Sub (Left : Uint; Right : Int) return Uint is
   begin
      return UI_Add (Left, -Right);
   end UI_Sub;

   function UI_Sub (Left : Uint; Right : Uint) return Uint is
   begin
      return UI_Add (Left, -Right);
   end UI_Sub;

   ----------------
   -- UI_To_Int --
   ----------------

   function UI_To_Int (Input : Uint) return Int is
   begin
      if Int (Input) <= Int (Uint_Direct_Last) then
         return Int (Input) - Int (Uint_Direct_Bias);

      --  Case of input is more than one digit

      else
         declare
            In_Length : constant Int := N_Digits (Input);
            In_Vec    : UI_Vector (1 .. In_Length);
            Ret_Int   : Int;

         begin
            --  Uints of more than one digit could be outside the range for
            --  Ints. Caller should have checked for this if not certain.
            --  Fatal error to attempt to convert from value outside Int'Range.

            pragma Assert (UI_Is_In_Int_Range (Input));

            --  Otherwise, proceed ahead, we are OK

            Init_Operand (Input, In_Vec);
            Ret_Int := 0;

            --  Calculate -|Input| and then negates if value is positive.
            --  This handles our current definition of Int (based on
            --  2s complement). Is it secure enough?

            for Idx in In_Vec'Range loop
               Ret_Int := Ret_Int * Base - abs In_Vec (Idx);
            end loop;

            if In_Vec (1) < 0 then
               return Ret_Int;
            else
               return -Ret_Int;
            end if;
         end;
      end if;
   end UI_To_Int;

   --------------
   -- UI_Write --
   --------------

   procedure UI_Write (Input : Uint) is
   begin
      Image_Out (Input, False);
   end UI_Write;

   ---------------------
   -- Vector_To_Uint --
   ---------------------

   function Vector_To_Uint
     (In_Vec   : UI_Vector;
      Negative : Boolean)
      return     Uint
   is
      Size : Int;

   begin
      --  The vector can contain leading zeros. These are not stored in the
      --  table, so loop through the vector looking for first non-zero digit

      for J in In_Vec'Range loop
         if In_Vec (J) /= 0 then

            --  The length of the value is the length of the rest of the vector

            Size := In_Vec'Last - J + 1;

            if Size = 1 then
               if Negative then
                  return Uint (Int (Uint_Direct_Bias) - In_Vec (J));
               else
                  return Uint (Int (Uint_Direct_Bias) + In_Vec (J));
               end if;
            end if;

            --  The value takes more than one digit, so it is stored in the
            --  table. Expand the table to contain the count and digits.
            --  the index of the first new location is the return value.

            Uints.Increment_Last;
            Uints.Table (Uints.Last).Length := Size;
            Uints.Table (Uints.Last).Loc    := Udigits.Last + 1;

            Udigits.Increment_Last;

            if Negative then
               Udigits.Table (Udigits.Last) := -In_Vec (J);
            else
               Udigits.Table (Udigits.Last) := +In_Vec (J);
            end if;

            for K in 2 .. Size loop
               Udigits.Increment_Last;
               Udigits.Table (Udigits.Last) := In_Vec (J + K - 1);
            end loop;

            if Debug_Flag_D then
               w ("In V_To_U: Size   = ", Int (Size));
               w ("      Digits.Last = ", Int (Udigits.Last));
            end if;

            return Uints.Last;
         end if;
      end loop;

      --  Dropped through loop only if vector contained all zeros

      return Uint_0;
   end Vector_To_Uint;

end Uintp;
