------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             P A R . P R A G                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.81 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Generally the parser checks the basic syntax of pragmas, but does not
--  do specialized syntax checks for individual pragmas, these are deferred
--  to semantic analysis time (see unit Sem_Prag). There are some pragmas
--  which require recognition and either partial or complete processing
--  during parsing, and this unit performs this required processing.

with Stringt; use Stringt;
with Uintp;   use Uintp;

separate (Par)

function Prag (Pragma_Node : Node_Id; Semi : Source_Ptr) return Node_Id is
   Pragma_Name : constant Name_Id    := Chars (Pragma_Node);
   Pragma_Sloc : constant Source_Ptr := Sloc (Pragma_Node);
   Arg_Count   : Nat;
   Arg_Node    : Node_Id;
   Expr_Node   : Node_Id;

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Arg1 return Node_Id;
   function Arg2 return Node_Id;
   function Arg3 return Node_Id;
   function Arg4 return Node_Id;
   --  Obtain specified Pragma_Argument_Association. It is allowable to call
   --  the routine for the argument one past the last present argument, but
   --  that is the only case in which a non-present argument can be referenced.

   procedure Check_Ada_83_Warning;
   --  Issues a warning message for the current pragma if operating in Ada 83
   --  mode (used for language pragmas that are not a standard part of Ada 83).
   --  This procedure does not raise Error_Resync. Also notes use of 95 pragma.

   procedure Check_Arg_Count (Required : Int);
   --  Check argument count for pragma = Required.
   --  If not give error and raise Error_Resync.

   procedure Check_Arg_Is_Convention (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that it
   --  is a valid convention name. If not give error and raise Error_Resync.
   --  This procedure also checks for the possible allowed presence of the
   --  identifier Convention for this argument.

   procedure Check_Arg_Is_Identifier (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that it
   --  is an identifier. If not give error and raise Error_Resync.

   procedure Check_Arg_Is_String_Literal (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that it
   --  is a string literal. If not give error and raise Error_Resync.

   procedure Check_Arg_Is_Library_Unit_Name (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that it
   --  is of the form of a library unit name, i.e. that it is an identifier
   --  or a selected component with a selector name that is itself an
   --  identifier. If not of this form, give error and raise Error_Resync.

   procedure Check_Arg_Is_Local_Name (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that
   --  it has the proper syntactic form for a local name.

   procedure Check_Arg_Is_On_Or_Off (Arg : Node_Id);
   --  Check the expression of the specified argument to make sure that it
   --  is an identifier which is either ON or OFF, and if not, then issue
   --  an error message and raise Error_Resync.

   procedure Check_At_Least_One_Argument;
   --  Check there is at least one argument.
   --  If not give error and raise Error_Resync.

   procedure Check_External_And_Or_Link_Name (A1 : Node_Id; A2 : Node_Id);
   --  Check last two arguments of pragma Import, Export or Interface_Name
   --  to check for appropriate optional identifiers. A1 is definitely
   --  present, but A2 may be missing if either External_Name or Link_Name
   --  is omitted.

   procedure Check_Library_Unit_Pragma;
   --  Library unit pragmas (10.1.5) have at most one argument, which must
   --  be the current compilation unit.

   procedure Check_No_Identifier (Arg : Node_Id);
   --  Checks that the given argument does not have an identifier. If an
   --  identifier is present, then an error message is issued, and
   --  Error_Resync is raised.

   procedure Check_No_Identifiers;
   --  Checks that none of the arguments to the pragma has an identifier.
   --  If any argument has an identifier, then an error message is issued,
   --  and Error_Resync is raised.

   procedure Check_Optional_Identifier (Arg : Node_Id; Id : Name_Id);
   --  Checks if the given argument has an identifier, and if so, requires
   --  it to match the given identifier name. If there is a non-matching
   --  identifier, then an error message is given and Error_Resync raised.

   ----------
   -- Arg1 --
   ----------

   function Arg1 return Node_Id is
   begin
      return First (Pragma_Argument_Associations (Pragma_Node));
   end Arg1;

   ----------
   -- Arg2 --
   ----------

   function Arg2 return Node_Id is
   begin
      return Next (Arg1);
   end Arg2;

   ----------
   -- Arg3 --
   ----------

   function Arg3 return Node_Id is
   begin
      return Next (Arg2);
   end Arg3;

   ----------
   -- Arg4 --
   ----------

   function Arg4 return Node_Id is
   begin
      return Next (Arg3);
   end Arg4;

   --------------------------
   -- Check_Ada_83_Warning --
   --------------------------

   procedure Check_Ada_83_Warning is
   begin
      Note_Feature (New_Pragmas, Pragma_Sloc);

      if Ada_83 then
         Error_Msg ("(Ada 83) pragma% is non-standard", Pragma_Sloc);
      end if;

      --  Put back the node for subsequent error messages, because this is a
      --  situation where we do not raise Error_Resync and get out immediately

      Error_Msg_Name_1 := Pragma_Name;
   end Check_Ada_83_Warning;

   ---------------------
   -- Check_Arg_Count --
   ---------------------

   procedure Check_Arg_Count (Required : Int) is
   begin
      if Arg_Count /= Required then
         Error_Msg ("wrong number of arguments for pragma%", Pragma_Sloc);
         raise Error_Resync;
      end if;
   end Check_Arg_Count;

   -----------------------------
   -- Check_Arg_Is_Convention --
   -----------------------------

   procedure Check_Arg_Is_Convention (Arg : Node_Id) is
   begin
      Check_Arg_Is_Identifier (Arg);
      Check_Optional_Identifier (Arg, Name_Convention);

      if not Is_Convention_Name (Chars (Expression (Arg))) then
         Error_Msg
           ("argument of pragma% is not valid convention name",
             Sloc (Expression (Arg)));
         raise Error_Resync;
      end if;
   end Check_Arg_Is_Convention;

   -----------------------------
   -- Check_Arg_Is_Identifier --
   -----------------------------

   procedure Check_Arg_Is_Identifier (Arg : Node_Id) is
   begin
      if Nkind (Expression (Arg)) /= N_Identifier then
         Error_Msg
           ("argument for pragma% must be identifier",
             Sloc (Expression (Arg)));
         raise Error_Resync;
      end if;
   end Check_Arg_Is_Identifier;

   ------------------------------------
   -- Check_Arg_Is_Library_Unit_Name --
   ------------------------------------

   procedure Check_Arg_Is_Library_Unit_Name (Arg : Node_Id) is
      Argx : constant Node_Id := Expression (Arg);

   begin
      if Nkind (Argx) /= N_Identifier
        and then (Nkind (Argx) /= N_Selected_Component
                   or else Nkind (Selector_Name (Argx)) /= N_Identifier)
      then
         Error_Msg
           ("argument for pragma% must be library unit name", Sloc (Argx));
         raise Error_Resync;
      end if;
   end Check_Arg_Is_Library_Unit_Name;

   -----------------------------
   -- Check_Arg_Is_Local_Name --
   -----------------------------

   --  LOCAL_NAME ::=
   --    DIRECT_NAME
   --  | DIRECT_NAME'ATTRIBUTE_DESIGNATOR
   --  | library_unit_NAME

   procedure Check_Arg_Is_Local_Name (Arg : Node_Id) is
      Argx : constant Node_Id    := Expression (Arg);
      Loc  : constant Source_Ptr := Sloc (Arg);

   begin
      if Nkind (Argx) not in N_Direct_Name
        and then (Nkind (Argx) /= N_Selected_Component
                   or else Nkind (Selector_Name (Argx)) /= N_Identifier)
        and then (Nkind (Argx) /= N_Attribute_Reference
                   or else Present (Expressions (Argx))
                   or else Nkind (Prefix (Argx)) /= N_Identifier)
      then
         Error_Msg ("argument for pragma% must be local name", Loc);
         raise Error_Resync;
      end if;
   end Check_Arg_Is_Local_Name;

   ----------------------------
   -- Check_Arg_Is_On_Or_Off --
   ----------------------------

   procedure Check_Arg_Is_On_Or_Off (Arg : Node_Id) is
      Argx : constant Node_Id := Expression (Arg);

   begin
      Check_Arg_Is_Identifier (Arg);

      if Chars (Argx) /= Name_On and then Chars (Argx) /= Name_Off then
         Error_Msg_Name_2 := Name_On;
         Error_Msg_Name_3 := Name_Off;

         Error_Msg
           ("argument for pragma% must be% or%", Sloc (Argx));
         raise Error_Resync;
      end if;
   end Check_Arg_Is_On_Or_Off;

   ---------------------------------
   -- Check_Arg_Is_String_Literal --
   ---------------------------------

   procedure Check_Arg_Is_String_Literal (Arg : Node_Id) is
   begin
      if Nkind (Expression (Arg)) /= N_String_Literal then
         Error_Msg
           ("argument for pragma% must be string literal",
             Sloc (Expression (Arg)));
         raise Error_Resync;
      end if;
   end Check_Arg_Is_String_Literal;

   ---------------------------------
   -- Check_At_Least_One_Argument --
   ---------------------------------

   procedure Check_At_Least_One_Argument is
   begin
      if Arg_Count = 0 then
         Error_Msg ("pragma% requires at least one argument", Pragma_Sloc);
         raise Error_Resync;
      end if;
   end Check_At_Least_One_Argument;

   -------------------------------------
   -- Check_External_And_Or_Link_Name --
   -------------------------------------

   procedure Check_External_And_Or_Link_Name (A1 : Node_Id; A2 : Node_Id) is
   begin
      if No (A1) then
         return;

      elsif Present (A2) then
         Check_Optional_Identifier (A1, Name_External_Name);
         Check_Optional_Identifier (A2, Name_Link_Name);

      elsif Chars (A1) /= Name_Link_Name then
         Check_Optional_Identifier (A1, Name_External_Name);
      end if;

   end Check_External_And_Or_Link_Name;

   -------------------------------
   -- Check_Library_Unit_Pragma --
   -------------------------------

   procedure Check_Library_Unit_Pragma is
   begin
      Check_Ada_83_Warning;

      if Arg_Count /= 0 then
         Check_No_Identifiers;
         Check_Arg_Count (1);
         Check_Arg_Is_Library_Unit_Name (Arg1);
      end if;
   end Check_Library_Unit_Pragma;

   -------------------------
   -- Check_No_Identifier --
   -------------------------

   procedure Check_No_Identifier (Arg : Node_Id) is
   begin
      if Chars (Arg) /= No_Name then
         Error_Msg_N ("pragma% does not permit named arguments", Arg);
         raise Error_Resync;
      end if;
   end Check_No_Identifier;

   --------------------------
   -- Check_No_Identifiers --
   --------------------------

   procedure Check_No_Identifiers is
   begin
      if Arg_Count > 0 then
         Arg_Node := Arg1;

         while Present (Arg_Node) loop
            Check_No_Identifier (Arg_Node);
            Arg_Node := Next (Arg_Node);
         end loop;
      end if;
   end Check_No_Identifiers;

   -------------------------------
   -- Check_Optional_Identifier --
   -------------------------------

   procedure Check_Optional_Identifier (Arg : Node_Id; Id : Name_Id) is
   begin
      if Present (Arg) and then Chars (Arg) /= No_Name then
         if Chars (Arg) /= Id then
            Error_Msg_Name_2 := Id;
            Error_Msg_N ("pragma% argument expects identifier%", Arg);
            raise Error_Resync;
         end if;
      end if;
   end Check_Optional_Identifier;

   ----------
   -- Prag --
   ----------

begin
   Error_Msg_Name_1 := Pragma_Name;

   --  Count number of arguments. This loop also checks if any of the arguments
   --  are Error, indicating a syntax error as they were parsed. If so, we
   --  simply return, because we get into trouble with cascaded errors if we
   --  try to perform our error checks on junk arguments.

   Arg_Count := 0;

   if Present (Pragma_Argument_Associations (Pragma_Node)) then
      Arg_Node := Arg1;

      while Arg_Node /= Empty loop
         Arg_Count := Arg_Count + 1;

         if Expression (Arg_Node) = Error then
            return Error;
         end if;

         Arg_Node := Next (Arg_Node);
      end loop;
   end if;

   --  Remaining processing is pragma dependent

   case Get_Pragma_Id (Pragma_Name) is

      ------------
      -- Ada_83 --
      ------------

      --  This pragma must be processed at parse time, since we want to set
      --  the Ada 83 and Ada 95 switches properly at parse time to recognize
      --  Ada 83 syntax or Ada 95 syntax as appropriate.

      when Pragma_Ada_83 =>
         Ada_83 := True;
         Ada_95 := False;

      ------------
      -- Ada_95 --
      ------------

      --  This pragma must be processed at parse time, since we want to set
      --  the Ada 83 and Ada_95 switches properly at parse time to recognize
      --  Ada 83 syntax or Ada 95 syntax as appropriate.

      when Pragma_Ada_95 =>
         Ada_83 := False;
         Ada_95 := True;

      ------------------
      -- Debug (GNAT) --
      ------------------

      --  pragma Debug (PROCEDURE_CALL_STATEMENT);

      --  Syntax check: one argument which must be of the form of a procedure
      --  call, parsed either as a name or as a function call. It is then
      --  converted to the corresponding procedure call.

      when Pragma_Debug =>
         Check_No_Identifiers;
         Check_Arg_Count (1);

         declare
            Expr : constant Node_Id := New_Copy (Expression (Arg1));

         begin
            if Nkind (Expr) /= N_Indexed_Component
              and then Nkind (Expr) /= N_Function_Call
              and then Nkind (Expr) /= N_Identifier
              and then Nkind (Expr) /= N_Selected_Component
            then
               Error_Msg
                 ("argument of pragma% is not procedure call", Sloc (Expr));
               raise Error_Resync;
            else
               Set_Debug_Statement
                 (Pragma_Node, P_Statement_Name (Expr));
            end if;
         end;

      ------------------------
      -- Elaborate (10.2.1) --
      ------------------------

      --  pragma Elaborate (library_unit_NAME {, library_unit_NAME});

      --  Syntax check: at least one argument, all arguments of the form
      --   of either identifiers, or selected components with the selector
      --   name being an identifier.

      when Pragma_Elaborate =>
         Check_No_Identifiers;
         Check_At_Least_One_Argument;

         Arg_Node := Arg1;

         while Present (Arg_Node) loop
            Check_Arg_Is_Library_Unit_Name (Arg_Node);
            Arg_Node := Next (Arg_Node);
         end loop;

      ----------------------------
      -- Elaborate_All (10.2.1) --
      ----------------------------

      --  pragma Elaborate_All (library_unit_NAME {, library_unit_NAME});

      --  Syntax check: at least one argument, all arguments of the form
      --   of either identifiers, or selected components with the selector
      --   name being an identifier.

      when Pragma_Elaborate_All =>
         Check_Ada_83_Warning;
         Check_No_Identifiers;
         Check_At_Least_One_Argument;

         Arg_Node := Arg1;
         while Present (Arg_Node) loop
            Check_Arg_Is_Library_Unit_Name (Arg_Node);
            Arg_Node := Next (Arg_Node);
         end loop;

      -----------------------------
      -- Elaborate_Body (10.2.1) --
      -----------------------------

      --  pragma Elaborate_Body [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name

      when Pragma_Elaborate_Body =>
         Check_Library_Unit_Pragma;

      ------------------
      -- Export (B.1) --
      ------------------

      --  pragma Export (
      --    [Convention =>]    convention_IDENTIFIER,
      --    [Entity =>]        LOCAL_NAME
      --  [,[External_Name =>] static_string_EXPRESSION]]
      --  [,[Link_Name =>]     static_string_EXPRESSION]] );

      --  Syntax check: 2-4 arguments. 1st argument must be a
      --  convention, 2nd argument must be of the form of a local name

      when Pragma_Export =>
         Check_Ada_83_Warning;

         if Arg_Count in 3 .. 4 then
            Check_External_And_Or_Link_Name (Arg3, Arg4);
         else
            Check_Arg_Count (2);
         end if;

         Check_Arg_Is_Convention (Arg1);
         Check_Arg_Is_Local_Name (Arg2);
         Check_Optional_Identifier (Arg2, Name_Entity);

      -----------------------------
      -- Error_Monitoring (GNAT) --
      -----------------------------

      --  pragma Error_Monitoring (ON | OFF, [STRING_LITERAL])

      --  This pragma must be processed at parse time, since it may be used
      --  to monitor syntax errors in parse only mode wih semantics off.

      --  Note: at the current time, Error_Monitoring does not work for
      --  syntax errors, but this will be fixed some time ???


      when Pragma_Error_Monitoring =>
         Check_Ada_83_Warning;
         Check_No_Identifiers;
         Check_Arg_Is_On_Or_Off (Arg1);

         if Arg_Count > 1 then
            Check_Arg_Count (2);
            Check_Arg_Is_String_Literal (Arg2);
         end if;

      ------------------
      -- Import (B.1) --
      ------------------

      --  pragma Import (
      --    [Convention =>]    convention_IDENTIFIER,
      --    [Entity =>]        LOCAL_NAME
      --  [,[External_Name =>] static_string_EXPRESSION]]
      --  [,[Link_Name =>]     static_string_EXPRESSION]] );

      --  Syntax check: 2-4 arguments. 1st argument must be a convention,
      --  2nd argument must be of the form of a local name

      when Pragma_Import =>
         Check_Ada_83_Warning;

         if Arg_Count in 3 .. 4 then
            Check_External_And_Or_Link_Name (Arg3, Arg4);
         else
            Check_Arg_Count (2);
         end if;

         Check_Arg_Is_Convention (Arg1);
         Check_Arg_Is_Local_Name (Arg2);
         Check_Optional_Identifier (Arg2, Name_Entity);

      --------------------
      -- Inline (6.3.2) --
      --------------------

      --  pragma Inline (NAME {, NAME});

      --  Syntax check: at least one argument, and the arguments are either
      --  of the form of identifiers, or of selected components.

      when Pragma_Inline =>
         Check_No_Identifiers;
         Check_At_Least_One_Argument;

         Arg_Node := Arg1;
         while Present (Arg_Node) loop
            Expr_Node := Expression (Arg_Node);

            if Nkind (Expr_Node) /= N_Identifier
              and then Nkind (Expr_Node) /= N_Selected_Component
              and then Nkind (Expr_Node) /= N_Operator_Symbol
            then
               Error_Msg
                 ("argument of pragma% is not subprogram name",
                   Sloc (Expr_Node));
            end if;

            Arg_Node := Next (Arg_Node);
         end loop;

      ------------------------
      -- Interface (Ada 83) --
      ------------------------

      --  pragma Interface (convention_IDENTIFIER, LOCAL_NAME);

      --  Syntax check: two arguments, first is a convention name

      when Pragma_Interface =>
         Check_No_Identifiers;
         Check_Arg_Count (2);
         Check_Arg_Is_Convention (Arg1);
         Check_Arg_Is_Local_Name (Arg2);

      ---------------------------
      -- Interface_Name (GNAT) --
      ---------------------------

      --  pragma Interface_Name (
      --      [Entity =>]         LOCAL_NAME
      --    [,[External_Name =>]  static_string_EXPRESSION]]
      --    [,[Link_Name =>]      static_string_EXPRESSION]] );

      --  Syntax check: two or three arguments, first is of the form of a
      --  local name.

      when Pragma_Interface_Name =>

         if Arg_Count /= 3 then
            Check_Arg_Count (2);
         end if;

         Check_External_And_Or_Link_Name (Arg2, Arg3);
         Check_Arg_Is_Local_Name (Arg1);

      ----------------
      -- List (2.8) --
      ----------------

      --  pragma List (Off | On)

      --  The processing for pragma List must be done at parse time,
      --  since a listing can be generated in parse only mode.

      when Pragma_List =>
         Check_No_Identifiers;
         Check_Arg_Count (1);
         Check_Arg_Is_On_Or_Off (Arg1);

         --  We unconditionally make a List_On entry for the pragma, so that
         --  in the List (Off) case, the pragma will print even in a region
         --  of code with listing turned off (this is required!)

         List_Pragmas.Increment_Last;
         List_Pragmas.Table (List_Pragmas.Last) :=
           (Ptyp => List_On, Ploc => Sloc (Pragma_Node));

         --  Now generate the list off entry for pragma List (Off)

         if Chars (Expression (Arg1)) = Name_Off then
            List_Pragmas.Increment_Last;
            List_Pragmas.Table (List_Pragmas.Last) :=
              (Ptyp => List_Off, Ploc => Semi);
         end if;

      ----------------
      -- Page (2.8) --
      ----------------

      --  pragma Page;

      --  Processing for this pragma must be done at parse time, since a
      --  listing can be generated in parse only mode with semantics off.

      when Pragma_Page =>
         Check_No_Identifiers;
         Check_Arg_Count (0);
         List_Pragmas.Increment_Last;
         List_Pragmas.Table (List_Pragmas.Last) := (Page, Semi);

      ---------------------------
      -- Preelaborate (10.2.1) --
      ---------------------------

      --  pragma Preelaborate [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name

      when Pragma_Preelaborate =>
         Check_Library_Unit_Pragma;

      -------------------
      -- Pure (10.2.1) --
      -------------------

      --  pragma Pure [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name.

      when Pragma_Pure =>
         Check_Library_Unit_Pragma;

      -----------------------------------
      -- Remote_Call_Interface (E.2.3) --
      -----------------------------------

      --  Pragma Remote_Call_Interface [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name

      when Pragma_Remote_Call_Interface =>
         Check_Library_Unit_Pragma;

      --------------------------
      -- Remote_Types (E.2.2) --
      --------------------------

      --  Pragma Remote_Types [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name

      when Pragma_Remote_Types =>
         Check_Library_Unit_Pragma;

      ----------------------------
      -- Shared_Passive (E.2.1) --
      ----------------------------

      --  pragma Shared_Passive [(library_unit_NAME)];

      --  Syntax check: at most one argument, which, if present, is the
      --  current compilation unit name

      when Pragma_Shared_Passive =>
         Check_Library_Unit_Pragma;

      -----------------------------
      -- Source_Reference (GNAT) --
      -----------------------------

      --  pragma Source_Reference
      --    (INTEGER_LITERAL [, STRING_LITERAL] );

      --  Processing for this pragma must be done at parse time, since error
      --  messages needing the proper line numbers can be generated in parse
      --  only mode with semantic checking turned off, and indeed we usually
      --  turn off semantic checking anyway if any parse errors are found.

      when Pragma_Source_Reference =>
         Check_No_Identifiers;

         if Arg_Count /= 1 then
            Check_Arg_Count (2);
            Check_Arg_Is_String_Literal (Arg2);

            declare
               S : constant String_Id := Strval (Expression (Arg2));
               C : Char_Code;

            begin
               Name_Len := 0;

               for J in 1 .. String_Length (S) loop
                  C := Get_String_Char (S, J);

                  if In_Character_Range (C) then
                     Name_Len := Name_Len + 1;
                     Name_Buffer (Name_Len) := Get_Character (C);
                  else
                     Store_Encoded_Character (Get_String_Char (S, J));
                  end if;
               end loop;

               Set_Reference_Name (Current_Source_File, Name_Find);
            end;
         end if;

         if Nkind (Expression (Arg1)) /= N_Integer_Literal then
            Error_Msg
              ("argument for pragma% must be integer literal",
                Sloc (Expression (Arg1)));
            raise Error_Resync;

         else
            Set_Line_Offset
              (Current_Source_File,
               UI_To_Int (Intval (Expression (Arg1))) - 2);
         end if;

      ---------------------
      -- Suppress (11.5) --
      ---------------------

      --  pragma Suppress (IDENTIFIER [, [On =>] NAME]);

      --  Syntax check: first argument must be an identifier which is a
      --  valid check name. Second argument must be named On if name given.

      --  Note: pragma Unsuppress shares the same processing

      when Pragma_Suppress | Pragma_Unsuppress =>
         Check_No_Identifier (Arg1);
         Check_Optional_Identifier (Arg2, Name_On);
         Check_At_Least_One_Argument;
         Check_Arg_Is_Identifier (Arg1);

         if not Is_Check_Name (Chars (Expression (Arg1))) then
            Error_Msg
              ("argument of pragma% is not valid check name",
                Sloc (Expression (Arg1)));
         end if;

      -----------------------
      -- Unsuppress (GNAT) --
      -----------------------

      --  pragma Unsuppress (IDENTIFIER [, [On =>] NAME]);

      --  Syntax check: first argument must be an identifier which is a
      --  valid check name. Second argument must be named On if name given.

      --  processing for Unsuppress shares the pragma Suppress circuit

      ----------------------
      -- All Oher Pragmas --
      ----------------------

      --  For all other pragmas, checking and processing is handled
      --  entirely in Sem_Prag, and no further checking is done by Par.

      when Pragma_Abort_Defer             |
           Pragma_All_Calls_Remote        |
           Pragma_Annotate                |
           Pragma_Asynchronous            |
           Pragma_Atomic                  |
           Pragma_Atomic_Components       |
           Pragma_Assert                  |
           Pragma_Attach_Handler          |
           Pragma_Controlled              |
           Pragma_Convention              |
           Pragma_CPP_Class               |
           Pragma_CPP_Constructor         |
           Pragma_CPP_Destructor          |
           Pragma_CPP_Virtual             |
           Pragma_CPP_Vtable              |
           Pragma_Discard_Names           |
           Pragma_Inspection_Point        |
           Pragma_Interrupt_Handler       |
           Pragma_Interrupt_Priority      |
           Pragma_Linker_Options          |
           Pragma_Locking_Policy          |
           Pragma_Normalize_Scalars       |
           Pragma_Machine_Attribute       |
           Pragma_Memory_Size             |
           Pragma_Optimize                |
           Pragma_Pack                    |
           Pragma_Priority                |
           Pragma_Queuing_Policy          |
           Pragma_Restrictions            |
           Pragma_Reviewable              |
           Pragma_Shared                  |
           Pragma_Storage_Size            |
           Pragma_Storage_Unit            |
           Pragma_System_Name             |
           Pragma_Task_Dispatching_Policy |
           Pragma_Unimplemented_Unit      |
           Pragma_Volatile                |
           Pragma_Volatile_Components  =>
         null;

   end case;

   return Pragma_Node;

   --------------------
   -- Error Handling --
   --------------------

exception
   when Error_Resync =>
      return Error;

end Prag;
