------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                  O P T                                   --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.62 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains global switches set by the initialization
--  routine from the command line and referenced throughout the compiler,
--  the binder or gnatmake. The comments indicate which options are used by
--  the compiler which are used by the binder or gnatmake, and which are
--  common to these programs.

with Types;          use Types;
with System.WCh_Con; use System.WCh_Con;

package Opt is

   --------------------------------------------
   -- Switches Common to Compiler and Binder --
   --------------------------------------------

   Brief_Output : Boolean := False;
   --  Force brief error messages to standard error, even if verbose mode is
   --  set (so that main error messages go to standard output).

   GNAT_Mode : Boolean := False;
   --  True if compiling in GNAT system mode (-g switch set)

   Identifier_Character_Set : Character;
   --  This variable indicates the character set to be used for identifiers.
   --  The possible settings are:
   --    '1'  Latin-1
   --    '2'  Latin-2
   --    '3'  Latin-3
   --    '4'  Latin-4
   --    'p'  PC
   --    'f'  Full upper set (all distinct)
   --    'n'  No upper characters (Ada/83 rules)
   --    'w'  Latin-1 plus wide characters allowed in identifiers
   --
   --  The setting affects the set of letters allowed in identifiers and the
   --  upper/lower case equivalences. It does not affect the interpretation of
   --  character and string literals, which are always stored using the actual
   --  coding in the source program. This variable is initialized to the
   --  default value appropriate to the system (in Osint.Initialize), and then
   --  reset if a command line switch is used to change the setting.

   Maximum_Errors : Int := 9999;
   --  Maximum number of errors before compilation is terminated

   Maximum_File_Name_Length : Int;
   --  Maximum number of characters allowed in a file name, not counting the
   --  extension, as set by the appropriate switch. If no switch is given,
   --  then this value is initialized by Osint to the appropriate value.

   Suppress_Options : Suppress_Record;
   --  Flags set True to suppress corresponding check, i.e. add an implicit
   --  pragma Suppress at the outer level of each unit compiled. Note that
   --  these suppress actions can be overridden by the use of the Unsuppress
   --  pragma. This variable is initialized by Osint.Initialize.

   Verbose_Mode : Boolean := False;
   --  Set to True to get verbose mode (full error message text and location
   --  information sent to standard output, also header, copyright and summary)

   type Warning_Mode_Type is (Suppress, Normal, Treat_As_Error);
   Warning_Mode : Warning_Mode_Type := Normal;
   --  Controls treatment of warning messages. If set to Suppress, warning
   --  messages are not generated at all. In Normal mode, they are generated
   --  but do not count as errors. In Treat_As_Error mode, warning messages
   --  are generated and are treated as errors.

   ----------------------------------------
   -- Switches Used Only By The Compiler --
   ----------------------------------------

   --  All switches can be used with GNAT1 or GNATF, but some of them are only
   --  relevant to one or other of these programs as indicated in the comments.

   Ada_83_Switch : Boolean := False;
   --  This is the value of the command line switch for Ada 83 mode. At the
   --  start of compiling a unit, Ada_95 and Ada_83 are set from this value
   --  but then they can be subsequently modified by pragmas Ada_83, Ada_95.

   Ada_95 : Boolean := True;
   --  Set True if operating in Ada 95 mode
   --  Set False if operating in Ada 83 mode

   Ada_83 : Boolean := False;
   --  Set True if operating in Ada 83 mode
   --  Set False if operating in Ada 95 mode

   All_Errors_Mode : Boolean := False;
   --  Flag set to force display of multiple errors on a single line and
   --  also repeated error messages for references to undefined identifiers.

   Assertions_Enabled : Boolean := False;
   --  Enable assertions made using pragma Assert. Used only by GNATF, ignored
   --  by GNAT1 (actually the pragmas are still processed, but if no code is
   --  generated, they have no effect).

   type Distribution_Stub_Mode_Type is
     (No_Stubs,
      --  Normal mode, no generation/compilation of distribution stubs

      Compile_Receiver_Stub_Spec,
      --  The unit being compiled is the body of the receiver stubs, and
      --  the corresponding spec will be generated appropriately.

      Compile_Caller_Stub_Spec,
      --  The unit being compiled is the body of the caller stubs, and
      --  the corresponding spec will be generated appropriately

      Generate_Receiver_Stub_Body,
      --  The unit being compiled is the RCI spec, and the compiler will
      --  generate the body for the receiver stubs.

      Generate_Caller_Stub_Body);
      --  The unit being compiled is the RCI spec, and the compiler will
      --  generate the body for the caller stubs.

   Stub_Mode : Distribution_Stub_Mode_Type := No_Stubs;
   --  This enumeration variable indicates the five states of distribution
   --  annex stub generation/compilation.

   Full_List : Boolean := False;
   --  Set True to generate full source listing with embedded errors

   Immediate_Errors : Boolean := False;
   --  If set to True, then error messages are output as soon as they are
   --  detected (useful for navigating around compiler error situations)

   Inline_Active : Boolean := True;
   --  Set False to deactive pragma Inline processing across modules. Default
   --  to inline across module boundaries, as specified in the RM. Used only
   --  by GNAT1, ignored by GNATF.

   List_Units : Boolean := False;
   --  List units in the active lbrary

   type Mode_Type is (Check_Syntax, Check_Semantics, Generate_Code);
   Operating_Mode : Mode_Type := Generate_Code;
   --  Indicates the operating mode of the compiler. The default is generate
   --  code, which runs the parser, semantics and backend. Switches can be
   --  used to set syntax checking only mode, or syntax and semantics checking
   --  only mode. Operating_Mode can also be modified as a result of detecting
   --  errors during the compilation process. In particular if any error is
   --  detected then this flag is reset from Generate_Code to Check_Semantics
   --  after generating an error message. For GNATF, Generate_Code is treated
   --  as equivalent to Check_Semantics.

   Software_Overflow_Checking : Boolean;
   --  Set to True by Osint.Initialize if the target requires the software
   --  approach to integer arithmetic overflow checking (i.e. the use of
   --  double length arithmetic followed by a range check). Set to False
   --  if the target implements hardware overflow checking. Used only by
   --  GNAT1, not used by GNATF.

   Style_Check : Boolean := False;
   --  Set True to perform style checks. Activates checks carried out
   --  in package Style (see body of this package for details of checks)

   Tree_Output : Boolean := False;
   --  Set True to generate output tree file

   Try_Semantics : Boolean := False;
   --  Flag set to force attempt at semantic analysis, even if parser errors
   --  occur. This will probably cause blowups at this stage in the game. On
   --  the other hand, most such blowups will be caught cleanly and simply
   --  say compilation abandoned.

   RM_Column_Check : Boolean := False;
   --  Flag set to cause column alignment to be taken into account in
   --  determining legality of various constructs, using the layout rules
   --  specified in the RM.

   Wide_Character_Encoding_Method : WC_Encoding_Method := WCEM_None;
   --  Method used for encoding wide characters in the source program. See
   --  description of type in unit System.WCh_Con for a list of the methods
   --  that are currently supported. The default mode (WCEM_None) prohibits
   --  wide character encodings from appearing in the program completely.

   Upper_Half_Encoding : Boolean := False;
   --  Normally set False, indicating that upper half ASCII characters are
   --  used in the normal way to represent themselves. If the wide character
   --  encoding method uses the upper bit for this encoding, then this flag
   --  is set True, and upper half characters in the source indicate the
   --  start of a wide character sequence.

   Xref_Analyze : Boolean := False;
   --  This flag is used to indicate to semantic analyzer that the current
   --  compilation is done for GNATF. So the expander mustn't be called.

   Xref_Flag_1 : Boolean := False;
   --  Set to generate warning messages for unused with clauses. Used only
   --  by GNATF, ignored by GNAT1.

   Xref_Flag_2 : Boolean := False;
   --  Set to generate warning messages for unused entities (including
   --  unused with clauses). Used only by GNATF, ignored by GNAT1.

   Xref_Flag_3 : Boolean := False;
   --  Set to generate cross-reference file listing all references in the
   --  compiled files (also generates warning messages described above).
   --  Used only by GNATF, ignored by GNAT1.

   Xref_Flag_4 : Boolean := False;
   --  Set to include in the reference list all informations about entities
   --  declared in bodies if the corresponding spec declares inlined
   --  subprograms or generics. Includes effects of Xref_Flag_1,2,3).
   --  Used only by GNATF, ignored by GNAT1.

   Xref_Flag_5 : Boolean := False;
   --  Set to include all information in cross-reference listing.
   --  (includes effects of Xref_Flag_1,2,3 described above). Used
   --  only by GNATF, ignored by GNAT1.

   Xref_Flag_6 : Boolean := False;
   --  Same thing as Xef_Flag_5 except that a global xref file is generated

   Xref_Flag_9 : Boolean := False;
   --  Set to generate a cross-reference listing of Ada 95 features used. This
   --  listing is sorted by category and output to the standard output file.
   --  Unlike other Xref flags, used by both GNAT1 and GNATF.

   Xref_Flag_B : Boolean := False;
   --  If set, cross-reference file includes information on required interfaces
   --  for library unit bodies. Used only by GNATF, ignored by GNAT1.

   Xref_Flag_S : Boolean := False;
   --  If set, cross-reference file includes information on required interfaces
   --  for library package specs. Used only by GNATF, ignored by GNAT1.

   --------------------------------------
   -- Switches Used Only By The Binder --
   --------------------------------------

   All_Sources : Boolean := False;
   --  Set to True to require all source files to be present. This flag is
   --  directly modified by gnatmake to affect the shared binder routines.

   Bind_Main_Program : Boolean := True;
   --  Set to False if not binding main Ada program.

   Check_Only : Boolean := False;
   --  Set to True to do checks only, no output of binder file.

   Check_Source_Files : Boolean := True;
   --  Set to True to enable consistency checking for any source files that
   --  are present (i.e. date must match the date in the library info file).
   --  Set to False for object file consistency check only. This flag is
   --  directly modified by gnatmake, to affect the shared binder routines.

   Elab_Dependency_Output : Boolean := False;
   --  Set to True to output complete list of elaboration constraints

   Elab_Order_Output : Boolean := False;
   --  Set to True to output chosen elaboration order

   Ignore_Time_Stamp_Errors : Boolean := False;
   --  Ignore time stamp mismatch errors (treat as warnings only)

   Output_Filename_Present : Boolean := False;
   --  Set to True when the output C filename is given with option -o

   ------------------------------------
   -- Switches Used Only By Gnatmake --
   ------------------------------------

   Compile_Only : Boolean := False;
   --  Set to True to skip bind and link step.

   Force_Compilations : Boolean := False;
   --  Set to force recompilations even when the objects are up-to-date .

   Dont_Execute : Boolean := False;
   --  Set to True if no actual compilations should be undertaken.

   Quiet_Output : Boolean := False;
   --  Set to True if the list of compilation commands should not be output.

   Smart_Compilations : Boolean := False;
   --  Set to True if smart recompilations requested.

   Generate_Debug : Boolean := False;
   --  Set to True if the -g switch has to be set for gcc and gnatbl

   -----------------------
   -- Tree I/O Routines --
   -----------------------

   procedure Tree_Read;
   --  Reads switch settings from current tree file using Tree_Read

   procedure Tree_Write;
   --  Writes out switch settings to current tree file using Tree_Write

end Opt;
