------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              I T Y P E S                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.14 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;    use Alloc;
with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Stand;    use Stand;
with Table;

package body Itypes is

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id);
   --  Add the Itype E at the end of implicit types list attached to N. If
   --  E is the head of an implicit type list, the full list is appended.

   procedure Insert_In_Tree
     (Assoc_Node  : Node_Id;
      Insert_Node : Node_Id;
      Insert_List : List_Id);
   --  This is the shared processing for Insert_Action and Insert_Actions.
   --  Either Insert_Node or Insert_List is set, with the other being Empty
   --  or No_List respectively.

   ------------------
   -- Append_Itype --
   ------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id) is
      Ityp  : Entity_Id;
      Itypn : Node_Id;

   begin
      pragma Assert (Nkind (N) not in N_Entity and Nkind (E) in N_Entity);

      if Debug_Flag_B then
         Itypn := Make_Implicit_Types (Sloc (E));
         Set_Next_Itype (Itypn, Ityp);

         Insert_Action (Itypn, N);
         return;
      end if;

      if No (First_Itype (N)) then
         Set_First_Itype (N, E);

         if Nkind (N) in N_Subexpr then
            Set_Cannot_Be_Constant (N);
         end if;

      else
         Ityp := First_Itype (N);
         while Present (Next_Itype (Ityp)) loop
            Ityp := Next_Itype (Ityp);
         end loop;

         Set_Next_Itype (Ityp, E);
      end if;

      Ityp := E;

      loop
         --  This is commented out because it blows up something to do with
         --  private type entity exchanges, which needs better understanding???
         --  Set_Parent (Ityp, N);
         Ityp := Next_Itype (Ityp);
         exit when No (Ityp);
      end loop;

   end Append_Itype;

   ---------------------
   -- Attach_Itype_To --
   ---------------------

   procedure Attach_Itype_To (N : Node_Id; E : Entity_Id) is
   begin
      pragma Assert (Next_Itype (E) = Empty);
      Set_Is_Itype (E, True);
      Append_Itype (N, E);
   end Attach_Itype_To;

   -------------------
   -- Insert_Action --
   -------------------

   procedure Insert_Action (Action : Node_Id; Node : Node_Id) is
   begin
      Insert_In_Tree (Node, Action, No_List);
   end Insert_Action;

   --------------------
   -- Insert_Actions --
   --------------------

   procedure Insert_Actions (Actions  : List_Id; Node : Node_Id) is
   begin
      Insert_In_Tree (Node, Empty, Actions);
   end Insert_Actions;

   --------------------
   -- Insert_In_Tree --
   --------------------

   procedure Insert_In_Tree
     (Assoc_Node  : Node_Id;
      Insert_Node : Node_Id;
      Insert_List : List_Id)
   is
      N : Node_Id;
      P : Node_Id;

      procedure Insert_Actions_Before (B : Node_Id);
      --  Inserts the action or actions before node B

      function Make_Actions return List_Id;
      --  Make a list (a new list if necessary) of the required actions

      procedure Insert_Actions_Before (B : Node_Id) is
      begin
         if Present (Insert_Node) then
            Insert_Before (B, Insert_Node);
         else
            Insert_List_Before (B, Insert_List);
         end if;
      end Insert_Actions_Before;

      function Make_Actions return List_Id is
      begin
         if Present (Insert_Node) then
            return New_List (Insert_Node);
         else
            return Insert_List;
         end if;
      end Make_Actions;

   --  Start of processing for Insert_In_Tree

   begin
      pragma Assert (Nkind (Assoc_Node) in N_Subexpr);

      loop
         P := Parent (N);
         pragma Assert (Present (P));

         case Nkind (P) is

            --  Statement case

            when N_Statement                |
                 N_Accept_Statement         |
                 N_Procedure_Call_Statement =>

            begin
               if Is_List_Member (P) then
                  Insert_Actions_Before (P);
                  return;
               end if;
            end;

            --  While loop and elsif cases

            when N_Iteration_Scheme |
                 N_Elsif_Part       =>

            begin
               if N = Condition (P) then
                  if No (Condition_Actions (P)) then
                     Set_Condition_Actions (P, Make_Actions);
                  else
                     Insert_Actions_Before (First (Condition_Actions (P)));
                  end if;

                  return;
               end if;
            end;

            --  Right operand of AND THEN or ELSE IF

            when N_And_Then |
                 N_Or_Else  =>
            begin
               if N = Right_Opnd (P) then
                  if No (Actions (P)) then
                     Set_Actions (P, Make_Actions);
                  else
                     Insert_Actions_Before (First (Actions (P)));
                  end if;

                  return;
               end if;
            end;

            --  Then or Else operand of conditional expression

            when N_Conditional_Expression =>
            declare
               Then_Expr : constant Node_Id := Next (First (Expressions (N)));
               Else_Expr : constant Node_Id := Next (Then_Expr);

            begin
               if N = Then_Expr then
                  if No (Then_Actions (P)) then
                     Set_Then_Actions (P, Make_Actions);
                  else
                     Insert_Actions_Before (First (Then_Actions (P)));
                  end if;

                  return;

               elsif N = Else_Expr then
                  if No (Else_Actions (P)) then
                     Set_Else_Actions (P, Make_Actions);
                  else
                     Insert_Actions_Before (First (Else_Actions (P)));
                  end if;

                  return;
               end if;
            end;

            --  In all other cases, keep climbing up the tree

            when others => null;
         end case;

         N := P;
      end loop;

   end Insert_In_Tree;

   ----------------------
   -- Prepend_Itype_To --
   ----------------------

   procedure Prepend_Itype_To (N : Node_Id; E : Entity_Id) is
   begin
      pragma Assert (Nkind (N) not in N_Entity and Nkind (E) in N_Entity);
      pragma Assert (Next_Itype (E) = Empty);
      Set_Is_Itype (E, True);

      if No (First_Itype (N)) then
         Append_Itype (N, E);

      else
         Set_Next_Itype (E, First_Itype (N));
         Set_First_Itype (N, E);
      end if;

   end Prepend_Itype_To;

   ---------------
   -- New_Itype --
   ---------------

   function New_Itype
     (Ekind        : Entity_Kind;
      In_Node      : Node_Id;
      Related_Id   : Entity_Id   := Empty;
      Suffix       : Character   := ' ';
      Suffix_Index : Nat         := 0;
      Scope_Id     : Entity_Id   := Current_Scope)
     return         Entity_Id
   is
      Typ : constant Entity_Id :=
        New_Itype_Not_Attached (Ekind, Sloc (In_Node),
          Related_Id, Suffix, Suffix_Index, Scope_Id);

   begin
      Attach_Itype_To (In_Node, Typ);
      return Typ;
   end New_Itype;

   ---------------------------
   -- New_Itype_Not_Attched --
   ---------------------------

   function New_Itype_Not_Attached
     (Ekind        : Entity_Kind;
      Loc          : Source_Ptr;
      Related_Id   : Entity_Id := Empty;
      Suffix       : Character := ' ';
      Suffix_Index : Nat       := 0;
      Scope_Id     : Entity_Id := Current_Scope)
      return         Entity_Id
   is
      Typ : Entity_Id;

   begin
      if Related_Id = Empty then
         Typ := New_Internal_Entity (Ekind, Scope_Id, Loc, 'T');

         Set_Public_Status (Typ);
         --  This must surely be a bug, how can an internal name be public???

      else
         Typ := New_External_Entity
           (Ekind, Scope_Id, Loc, Related_Id, Suffix, Suffix_Index, 'T');
      end if;

      Set_Etype (Typ, Any_Type);
      return Typ;
   end New_Itype_Not_Attached;

   ---------------------
   -- Transfer_Itypes --
   ---------------------

   procedure Transfer_Itypes (From : Node_Id; To : Node_Id) is
   begin
      pragma Assert (Nkind (From) not in N_Entity
                       and Nkind (To) not in N_Entity);

      if From /= To
        and then Nkind (From) in N_Has_Itypes
        and then Present (First_Itype (From))
      then
         Append_Itype (To, First_Itype (From));
         Set_First_Itype (From, Empty);

         if Has_Dynamic_Itype (From) then
            Set_Has_Dynamic_Itype (To,   True);
            Set_Has_Dynamic_Itype (From, False);
         end if;
      end if;
   end Transfer_Itypes;

end Itypes;
