------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                F N A M E                                 --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.16 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the association between source file names and
--  unit names as defined in package Uname.

with Types; use Types;

package Fname is

   --  Note: this package spec does not depend on the Uname spec in the Ada
   --  sense, but the comments and description of the semantics do depend on
   --  the conventions established by Uname, which explains the "with Uname".

   ---------------------------
   -- File Name Conventions --
   ---------------------------

   --  GNAT requires that there be a one to one correspondence between source
   --  file names (as used in the Osint package interface) and unit names as
   --  defined by the Uname package. This correspondence is defined by the
   --  two subprograms defined here in the Fname package.

   --  The body of this package is potentially system dependent, since file
   --  naming conventions do differ from operating system to operating system.
   --  However, the code in the body of Fname does not typically require any
   --  operating system interface, and furthermore, we choose a convention
   --  that is likely to be widely implementable, and certainly is one that
   --  can be shared between Unix, DOS, NT, Mac OS and OS/2.

   --  Since we do expect this convention to be followed widely, and since
   --  Osint depends on the convention, it is described here in the Spec.
   --  However, no unit (other than Osint) in any way depends on the choices
   --  described here.

   --  Unit names are the Ada names, with all lower case letters (except for
   --  the use of upper case letters for encoding and for internal names,
   --  see package Namet for further details), and a suffix that is either
   --  that is either %b or %s for bodies and specs respectively. This is the
   --  convention described and implemented in package Uname.

   --  Source file names are obtained by taking the decoded unit name (i.e.
   --  with Uhh and Whhhh sequences decoded to the ESC sequence or literal
   --  upper half character), excluding the %b or %s, and replacing the
   --  periods with minus signs. The extension is either .ads for a spec,
   --  or .adb for a body (or subunit).

   --  Examples of these rules are:

   --    Unit                           Unit name           File name

   --    Packge_Scan (spec)             packge_scan%s      packge_scan.ads
   --    Packge_Scan (body)             packge_scan%b      packge_scan.adb
   --    Scn.Nlit (subunit)             scn.nlit%b         scn-nlit.adb
   --    Child.Pkg (child unit spec)    child.pkg%s        child-pkg.ads
   --    Child.Pkg (child unit body)    child.pkg%b        child-pkg.adb
   --    Xyz.Arg.Lms (child subunit)    xyz.arg.lms%b      xyz-arg-lms.adb
   --    Accent?d (spec)                accentUc1d         accent?d.ads

   --  In the last example, ? stands for the graphic character that is
   --  displayed for the character UC_A_Acute (i.e. an upper case accented A).

   --  Note that the file name does *not* include the directory name. The
   --  management of directories is provided by Osint, and full file names
   --  are used only for error message purposes within GNAT itself.

   -------------------------
   -- File Name Crunching --
   -------------------------

   --  The rules described above give the file names that are generated if
   --  there is no restriction on the length of file names. However, the
   --  Get_File_Name routine will, if necessary according to the value in
   --  Opt.Maximum_File_Name_Length, crunch these file names down to this
   --  maximum value. For details of the crunching algorithm, see Krunch.

   -----------------
   -- Subprograms --
   -----------------

   type Expected_Unit_Type is (Expect_Body, Expect_Spec, Unknown);
   --  Return value from Get_Expected_Unit_Type

   function Get_Expected_Unit_Type
     (Fname : File_Name_Type)
      return  Expected_Unit_Type;
   --  If possible, determine whether the given file name corresponds to a unit
   --  that is a spec or body (e.g. by examining the extension). If this cannot
   --  be determined with the file naming conventions in use, then the returned
   --  value is set to Unknown.

   function Get_File_Name (Uname : Unit_Name_Type) return File_Name_Type;
   --  This function returns the file name that corresponds to a given unit
   --  name. The caller is responsible for ensuring that the unit name meets
   --  the requirements given in package Uname and described above.

   function Is_Language_Defined_Unit (Fname : File_Name_Type) return Boolean;
   --  This function determines if the given file name (which must be a simple
   --  file name with no directory information) is the file name for one of
   --  the predefined library units.

   function Is_File_Name (Name : Name_Id) return Boolean;
   --  Returns True if Name is the name of an ada source file (i.e. it abides
   --  to gnat file naming conventions).

   function File_Name_Of_Spec (Name : Name_Id) return File_Name_Type;
   --  Returns the file name that corresponds to the spec of a given unit
   --  name. Name does not end in "%s". If it does use Get_File_Name above.

   function File_Name_Of_Body (Name : Name_Id) return File_Name_Type;
   --  Returns the file name that corresponds to the body of a given unit
   --  name. Name does not end in "%b". If it does use Get_File_Name above.

end Fname;
