/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
                 2000, 2001 Werner Trobin <trobin@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef __koffice_filter_manager_h__
#define __koffice_filter_manager_h__

#include <qobject.h>
#include <qmap.h>
#include <koFilterChain.h>

class KoDocument;

/**
 *  This class manages all filters for a KOffice application. Normally
 *  you won't have to use it, since KoMainWindow takes care of loading
 *  and saving documents.
 *
 *  @short The class managing all the filters.
 *  @ref KoFilter
 *
 *  @author Kalle Dalheimer <kalle@kde.org>
 *  @author Torben Weis <weis@kde.org>
 *  @author Werner Trobin <trobin@kde.org>
 *  @version $Id: koFilterManager.h,v 1.56 2003/06/29 14:22:09 lukas Exp $
 */
class KoFilterManager : public QObject
{
    Q_OBJECT
public:
    /**
     * This enum is used to distinguish the import/export cases
     */
    enum Direction { Import = 1,  Export = 2 };

    /**
     * Create a filter manager for a document
     */
    KoFilterManager( KoDocument* document );
    /**
     * Create a filter manager for a filter which wants to embed something.
     * The url it passes is the file to convert, obviously. You can't use
     * the @ref import() method -- use @ref exp0rt() to convert the file to
     * the destination mimetype you prefer.
     *
     * @param url The file you want to export
     * @param mimetypeHint The mimetype of the file you want to export. You have
     *        to specify this information only if the automatic detection will
     *        fail because e.g. you saved an embedded stream to a *.tmp file.
     *        Most likely you don't have to care about that.
     * @param parentChain The parent filter chain of this filter manager. Used
     *        to allow embedding for filters. Most likely you don't have to care.
     */
    KoFilterManager( const QString& url, const QCString& mimetypeHint = "",
                     KoFilterChain* const parentChain = 0 );

    virtual ~KoFilterManager();

    /**
     * Imports the passed URL and returns the resultant filename
     * (most likely some file in /tmp).
     * The status vaiable signals the success/error of the conversion
     * If the QString which is returned isEmpty() and the status is OK,
     * then we imported the file directly into the document.
     */
    QString import( const QString& url, KoFilter::ConversionStatus& status );
    /**
     * Exports the given file/document to the specified URL/mimetype.
     * If mimetype.isEmpty() then the closest matching KOffice part is searched
     * and when the method returns "mimetype" contains this mimetype.
     * Oh, well, export is a C++ keyword ;)
     */
    KoFilter::ConversionStatus exp0rt( const QString& url, QCString& mimeType );


    // ### Static API ###
    /**
     * Suitable for passing to @ref KFileDialog::setMimeFilter. The default mime
     * gets set by the "users" of this method, as we don't have enough
     * information here.
     */
    static QStringList mimeFilter( const QCString& mimetype, Direction direction );

    /**
     * The same method as above but suited for KoShell.
     * We don't need the mimetype, as we will simply use all available
     * KOffice mimetypes. The Direction enum is omitted, as we only
     * call this for importing. When saving from KoShell we already
     * know the KOffice part we're using.
     */
    static QStringList mimeFilter();

    /**
     * Method used to check if that filter is available at all.
     * Note: Slow, but cached
     */
    static bool filterAvailable( KoFilterEntry::Ptr entry );

signals:
    void sigProgress( int );

private:
    // ### API for KoFilterChains ### (internal)
    // The friend methods are private in KoFilterChain and
    // just forward calls to the methods here. Should be
    // pretty save.
    friend QString KoFilterChain::filterManagerImportFile() const;
    QString importFile() const { return m_importUrl; }
    friend QString KoFilterChain::filterManagerExportFile() const;
    QString exportFile() const { return m_exportUrl; }
    friend KoDocument* KoFilterChain::filterManagerKoDocument() const;
    KoDocument* document() const { return m_document; }
    friend int KoFilterChain::filterManagerDirection() const;
    int direction() const { return static_cast<int>( m_direction ); }
    friend KoFilterChain* const KoFilterChain::filterManagerParentChain() const;
    KoFilterChain* const parentChain() const { return m_parentChain; }

    // Private API
    KoFilterManager( const KoFilterManager& rhs );
    KoFilterManager &operator=( const KoFilterManager& rhs );

    void importErrorHelper( const QString& mimeType, const bool suppressDialog = false );

    static const int s_area;

    KoDocument* m_document;
    KoFilterChain* const m_parentChain;
    QString m_importUrl, m_exportUrl;
    QCString m_importUrlMimetypeHint;  // suggested mimetype
    KOffice::Graph m_graph;
    Direction m_direction;

    // A static cache for the availability checks of filters
    static QMap<QString, bool> m_filterAvailable;

    // unused right now
    class Private;
    Private *d;
};

#endif  // __koffice_filter_manager_h__
