

/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>
#include <limits.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <libxfce4util/util.h>

#include "glade_callbacks.h"
#include "glade_gui.h"
#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "add_node_contents.h"
#include "dummies.h"
#include "entry.h"
#include "gdir.h"
#include "icons.h"
#include "misc.h"
#include "monitor.h"
#include "tubo.h"
#include "widgets.h"

#define HISTORY_LIMIT 20
#define GLOB "fgr"
#define FIND "find"

#define D(x)

extern int stop;

static history_combo_info_t find_combo_info={
	NULL,NULL,NULL,NULL,NULL
};
static short int findCount;	/* how many files found */
static short int fileLimit = 64;
static gboolean cancelled;
static pid_t Gpid;

static GList *path_list = NULL;
static GList *filter_list = NULL;
static GList *grep_list = NULL;
static GList *type_list = NULL;
static GList *find_results_list = NULL;
static gdir_t find_gdir;
static char *find_path;

static GtkTreeView *find_treeview;
static GtkTreeIter results_iter;

static void destroy(GtkWidget * widget, gpointer data);
static void on_find_clicked(GtkButton * button, gpointer user_data);
static void on_help_filter(GtkToggleButton * button, gpointer user_data);
static void on_help_grep(GtkToggleButton * button, gpointer user_data);
static char *ftypes[] = {
    N_("Any kind"),
    N_("Regular"),
    N_("Directory"),
    N_("Symlink"),
    N_("Socket"),
    N_("Block device"),
    N_("Character device"),
    N_("FIFO"),
    NULL
};
static char *ft[] = {
    "any",
    "f",
    "d",
    "l",
    "s",
    "b",
    "c",
    "p",
    NULL
};

static char *find_ft[] = {
    "any",
    "reg",
    "dir",
    "sym",
    "sock",
    "blk",
    "chr",
    "fifo",
    NULL
};

static gint on_key_press(GtkWidget * entry, GdkEventKey * event, gpointer data)
{
    void ok_input(GtkTreeView * treeview);
    GtkTreeView *treeview = (GtkTreeView *) data;
    GtkEditable *editable=(GtkEditable *)entry;

    /*printf("DBG(1):got key= 0x%x\n",event->keyval);*/
 if (event->keyval == GDK_BackSpace && (event->state&GDK_CONTROL_MASK)){
	gtk_editable_delete_text (editable,0,-1);
	return TRUE;
 }
    if(event->keyval == GDK_Escape)
    {
	destroy(lookup_widget((GtkWidget *) entry, "find_dialog"),(GtkWidget *) treeview);
	gtk_widget_grab_focus((GtkWidget *) treeview);
	return (TRUE);
    }

    return (FALSE);
}


static void save_ff_text(char *p)
{				
    char fname[_POSIX_PATH_MAX];
    if (!p ) return;
    xfce_get_userfile_r(fname, _POSIX_PATH_MAX-1, "xffm%cxffm.fflist.2.dbh",
		    G_DIR_SEPARATOR); 
    save_to_history(fname,p);
}

static char *load_ff_list(GList **g)
{
    static char fname[_POSIX_PATH_MAX];
    xfce_get_userfile_r(fname, _POSIX_PATH_MAX-1, "xffm%cxffm.fflist.dbh",
		    G_DIR_SEPARATOR);
    unlink(fname);
    xfce_get_userfile_r(fname, _POSIX_PATH_MAX-1, "xffm%cxffm.fflist.2.dbh",
		    G_DIR_SEPARATOR);
    get_history_list(g,fname,NULL);
    return fname;
}

static void find_over(void)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(find_treeview);
    tree_details_t *tree_details = get_tree_details(find_treeview);
    GtkTreePath *treepath;
    GtkTreeIter iter;
    tree_entry_t *en;

    unset_load_wait(&tree_details);

    get_find_root(find_treeview, &iter, &en);
    treepath = gtk_tree_model_get_path(treemodel, &iter);
    gtk_tree_view_expand_row(find_treeview, treepath, FALSE);
    erase_dummy(find_treeview, &iter);
    gtk_tree_path_free(treepath);

    treepath = gtk_tree_model_get_path(treemodel, &results_iter);
    gtk_tree_view_expand_row(find_treeview, treepath, FALSE);

    /* scroll is in callbacks.c, open_dir(), 
     * but only called if preference enabled */
    gdk_flush();
    gtk_tree_view_scroll_to_cell(find_treeview, treepath, NULL, TRUE, 0.0, 0.0);
    gtk_tree_path_free(treepath);
    if(!findCount)
	reset_dummy(find_treeview, &results_iter, 4);

}

GtkTreeIter add_find_results(GtkTreeView * treeview, char *name)
{
    GtkTreeIter iter;
    tree_entry_t *en, *c_en;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    /*tree_details_t *tree_details=get_tree_details(treeview); */

    get_find_root(treeview, &iter, &en);

    c_en = mk_entry(en->type);
    c_en->path = g_strdup(name);
    SET_XF_FND(c_en->type);
    gtk_tree_store_prepend((GtkTreeStore *) treemodel, &results_iter, &iter);
    gtk_tree_store_set((GtkTreeStore *) treemodel,&results_iter, 
	ENTRY_COLUMN, c_en, 
	NAME_COLUMN, my_utf_string(abreviate(name)), -1);
    set_icon(treeview, &results_iter);

    add_dummy(treeview, &results_iter);

    return results_iter;
}

void add_find_results_content(void)
{
    GList *tmp;
    int i;
    tree_entry_t *en;
    GtkTreeIter iter, child;
    GtkTreePath *treepath;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(find_treeview);
    tree_details_t *tree_details = get_tree_details(find_treeview);


    gtk_tree_model_get(treemodel, &results_iter, ENTRY_COLUMN, &en, -1);

    if(findCount)
    {
	find_gdir.pathc = findCount;
	find_gdir.gl = (dir_t *) malloc(findCount * sizeof(dir_t));
	if(!find_gdir.gl)
	    g_assert_not_reached();
	for(i = 0; i < find_gdir.pathc; i++)
	    find_gdir.gl[i].pathv = NULL;

	tmp = find_results_list;
	for(i = 0; i < find_gdir.pathc; i++)
	{
	    char *name;
	    if(!tmp)
		g_assert_not_reached();

	    if(!strchr((char *)tmp->data, '/'))
		g_assert_not_reached();
	    name = strrchr((char *)tmp->data, '/');
	    if(strlen(name) == 1)
		name = "/";
	    else
		name++;
	    find_gdir.gl[i].pathv = g_strdup(name);
	    find_gdir.gl[i].en = stat_entry((char *)tmp->data, en->type);
	    if(!find_gdir.gl[i].en){
		/* file has dissappeared ! */  
		find_gdir.pathc--;
		g_free(find_gdir.gl[i].pathv);
		find_gdir.gl[i].pathv=NULL;
	        i--;
		/*g_assert_not_reached();*/
	    }
	    g_free((char *)tmp->data);
	    tmp->data=NULL;
	    tmp = tmp->next;
	}
	if(find_results_list)
	    g_list_free(find_results_list);
	find_results_list = NULL;
	
	hide_stop(tree_details->window);

    	if (find_gdir.pathc){
	  add_node_contents(find_treeview, &results_iter, &find_gdir);
	  get_find_root(find_treeview, &iter, &en);
	  erase_dummy(find_treeview, &iter);
	  /* collapse all */
	  if(gtk_tree_model_iter_children(treemodel, &child, &iter))
	  {
	    do
	    {
		treepath = gtk_tree_model_get_path(treemodel, &child);
		gtk_tree_view_collapse_row(find_treeview, treepath);
		gtk_tree_path_free(treepath);
	    }
	    while(gtk_tree_model_iter_next(treemodel, &child));
	  }
	}
	
	gdirfree(&find_gdir);
    }
}



/****************************************************************/

static void abort_because_of_limit(GtkWidget * widget, gpointer data)
{
    GtkTreeView *treeview = (GtkTreeView *) data;
    tree_details_t *tree_details = get_tree_details(treeview);
    if(tree_details->tubo_object)
    {
	char *message;
	char m[32];
	sprintf(m, "%d\n", Gpid);
	print_diagnostics(find_treeview, "xf_WARNING_ICON", _("Subprocess aborted, pid="), m, NULL);
	TuboCancel(tree_details->tubo_object, NULL);
	tree_details->tubo_object = NULL;
	if(Gpid)
	    kill(Gpid, SIGTERM);
	Gpid = 0;
	message = (char *)malloc(strlen(_("Results limit reached")) + 64);
	sprintf(message, "%s (%d)\n", _("Results limit reached"), fileLimit);
	print_status(treeview, "xf_WARNING_ICON", message, NULL);
	g_free(message);
	message=NULL;
	cancelled = TRUE;
	hide_stop(tree_details->window);
	if(findCount)
	    add_find_results_content();
	cursor_reset(find_treeview);
	D("find over by abort limit\n") find_over();
    }
}

static gint watch_stop(gpointer data)
{
    tree_details_t *tree_details = get_tree_details(find_treeview);

    if(!tree_details->tubo_object)
	return FALSE;
    if(stop)
    {
	char m[32];
	sprintf(m, "%d\n", Gpid);
	print_diagnostics(find_treeview, "xf_WARNING_ICON", _("Subprocess aborted, pid="), m, NULL);
	cancelled = TRUE;
	TuboCancel(tree_details->tubo_object, NULL);
	if(Gpid)
	    kill(Gpid, SIGHUP);	/*hack! */
	stop = FALSE;
	tree_details->tubo_object = NULL;
	print_diagnostics(find_treeview, "xf_WARNING_ICON", _("Search interrupted."), "\n", NULL);
	print_status(find_treeview, "xf_INFO_ICON", _("Search done"), NULL);
	hide_stop(tree_details->window);
	Gpid = 0;
	if(findCount)
	    add_find_results_content();
	cursor_reset(find_treeview);
	D("find over by stop\n") find_over();
	return FALSE;
    }
    set_progress(find_treeview, -1, -1);
    return TRUE;
}

static int operate_stderr(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics(find_treeview, "xf_ERROR_ICON", line, NULL);
    return TRUE;
}

static int operate_stdout(int n, void *data)
{
    char *line;
    char *filename;

    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;



    if(stop)
    {
	return (TRUE);
    }
    if(line[0] == '/')
    {
	if(findCount >= fileLimit)
	    abort_because_of_limit(NULL, (gpointer) find_treeview);
	else
	{
	    char *path, *linecount = NULL, cuenta[32];
	    struct stat st;

	    path = line;
	    if(strstr(path, "\n"))
		path = strtok(path, "\n");
	    if(strstr(path, ":"))
	    {
		/* some filenames with semicolons exist! */
		if(lstat(path, &st) < 0)
		{
		    /* but not this one */
		    linecount = strrchr(path, ':') + 1;
		    *(strrchr(path, ':')) = 0;
		    if(strcmp(linecount, "0") == 0)
		    {
			linecount = NULL;
			return TRUE;
		    }
		}
	    }


	    findCount++;
	    find_results_list = g_list_append(find_results_list, g_strdup(path));
	    if(linecount)
		sprintf(cuenta, "%d (%s %s)", findCount, linecount, _("lines"));
	    else
		sprintf(cuenta, "%d", findCount);
	    filename = strrchr(path, '/') + 1;

	    /* path was zapped to directory 5 lines above */
	    print_diagnostics(find_treeview, NULL, path, "\n", NULL);
	}
    }
    else
    {
	print_diagnostics(find_treeview, "xf_WARNING_ICON", line, NULL);
    }

    return TRUE;
}


static void fork_finished_function(pid_t pid)
{
    char m[32];
    int status;
    tree_details_t *tree_details = get_tree_details(find_treeview);
    sprintf(m, "%d\n", pid);
    print_diagnostics(find_treeview, "xf_INFO_ICON", _("Subprocess done, pid="), m, NULL);

    tree_details->tubo_object = NULL;
    hide_stop(tree_details->window);
    cursor_reset(find_treeview);
    /* another wait just in case the one in tubo misses 
     * (shouldn't happen, but it does)*/
    waitpid(pid, &status, WNOHANG);

    if(findCount)
    {
	char mess[32];
	sprintf(mess, "%d ", findCount);
	print_diagnostics(find_treeview, "xf_INFO_ICON", _("Files found="), mess, " ", (findCount >= fileLimit) ? _("(limit reached)") : " ", "\n", NULL);
	print_status(find_treeview, "xf_INFO_ICON", _("Search done"), NULL);
	add_find_results_content();

    }
    else
    {
	print_diagnostics(find_treeview, "xf_INFO_ICON", _("Nothing found...\n"), NULL);
	print_status(find_treeview, "xf_INFO_ICON", _("Search done"), NULL);
    }
    D("find over by fork_finished_function\n") find_over();
}

gchar *get_combo_entry(GtkEntry * entry)
{
    int i, j;
    static gchar *s = NULL;
    if(s){
	g_free(s);
	s=NULL;
    }
    s = g_strdup((char *)gtk_entry_get_text(entry));
    while((s[0] == ' ') || (s[0] == '\t'))
	s++;
    j = strlen(s) - 1;
    for(i = j; i >= 0; i--)
    {
	if((s[i] == ' ') || (s[i] == '\t'))
	    s[i] = (char)0;
	else
	    break;
    }
    return s;
}

static void do_find(GtkWidget * widget)
{
    GtkWidget *find_dialog;
    GtkTreeView *treeview = get_treeview(widget);
    tree_details_t *tree_details = (tree_details_t *) get_tree_details(treeview);
    tree_entry_t *en;
    GtkTreeIter iter;
    const gchar *path;

    en = get_selected_entry(treeview, &iter);
    if(en) path=en->path;
    else path=g_get_home_dir();
    if (!path) path="/";
    
    if(tree_details->tubo_object)
    {
	print_status(treeview, "xf_ERROR_ICON", _("A subprocess is already running."), NULL);
	return;
    }
    find_treeview = treeview;
    find_dialog = create_find_dialog();
    find_combo_info.combo=lookup_widget(find_dialog, "filter_combo");
    gtk_widget_set_size_request(find_dialog, 555, 333);
    gtk_widget_show(find_dialog);
    gtk_window_set_transient_for(GTK_WINDOW(find_dialog), GTK_WINDOW(tree_details->window));

    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "find_button")), "clicked", G_CALLBACK(on_find_clicked), (gpointer) treeview);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "togglebutton2")), "toggled", G_CALLBACK(on_help_filter), NULL);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "togglebutton3")), "toggled", G_CALLBACK(on_help_grep), NULL);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "path_entry")), "activate", G_CALLBACK(on_find_clicked), (gpointer) treeview);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "filter_entry")), "activate", G_CALLBACK(on_find_clicked), (gpointer) treeview);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "grep_entry")), "activate", G_CALLBACK(on_find_clicked), (gpointer) treeview);

    g_signal_connect(G_OBJECT(find_dialog), "destroy_event", G_CALLBACK(destroy), (gpointer) treeview);
    g_signal_connect(G_OBJECT(find_dialog), "delete_event", G_CALLBACK(destroy), (gpointer) treeview);
    gtk_combo_disable_activate((GtkCombo *) lookup_widget(find_dialog, "path_combo"));
    gtk_combo_disable_activate((GtkCombo *) find_combo_info.combo);
    gtk_combo_disable_activate((GtkCombo *) lookup_widget(find_dialog, "grep_combo"));
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "path_entry")), "key_press_event", 
		    G_CALLBACK(on_key_press), (gpointer) treeview);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "filter_entry")), "key_press_event", 
		    G_CALLBACK(on_key_press_history), (gpointer) &find_combo_info);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "grep_entry")), "key_press_event", 
		    G_CALLBACK(on_key_press), (gpointer) treeview);


    if(path)
    {
	GList *tmp;
	gchar *dpath;
       	
	if (en && IS_DIR(en->type)) dpath=g_strdup(en->path);
        else dpath= g_path_get_dirname(path);	
	tmp = path_list;
	while(tmp)
	{
	    if(strcmp(dpath, (char *)tmp->data) == 0)
		break;
	    tmp = tmp->next;
	}
	if(tmp)
	{
	    path_list = g_list_remove(path_list, tmp->data);
	    g_free((char *)tmp->data);
	    tmp->data=NULL;
	}
	path_list = g_list_prepend(path_list, dpath);
    }

    if(path_list)
	gtk_combo_set_popdown_strings((GtkCombo *) lookup_widget(find_dialog, "path_combo"), path_list);
    
    find_combo_info.active_dbh_file=load_ff_list(&filter_list);
    find_combo_info.list=filter_list;
    set_limited_combo(&find_combo_info,NULL);
/*    gtk_combo_set_popdown_strings((GtkCombo *) lookup_widget(find_dialog, "filter_combo"), filter_list);*/
    if(!grep_list)
	grep_list = g_list_prepend(grep_list, "");
    gtk_combo_set_popdown_strings((GtkCombo *) lookup_widget(find_dialog, "grep_combo"), grep_list);
    if(!type_list)
    {
	int i;
	for(i = 0; ftypes[i] != NULL; i++)
	{
	    type_list = g_list_append(type_list, _(ftypes[i]));
	}

    }
    gtk_combo_set_popdown_strings((GtkCombo *) lookup_widget(find_dialog, "file_type_combo"), type_list);
    gtk_label_set_text((GtkLabel *) lookup_widget(find_dialog, "filter_help"), _("Basic rules:\n" "\n" "*  Will match any character zero or more times.\n" "?  Will match any character exactly one time\n"));
    gtk_label_set_text((GtkLabel *) lookup_widget(find_dialog, "regexp_help"),
		       _("Reserved characters for extended regexp are\n"
			 ". ^ $ [ ] ? * + { } | \\ ( ) : \n"
			 "In  basic regular expressions the metacharacters\n"
			 "?, +, {, |, (, and ) lose their special meaning.\n"
			 "\n"
			 "The  period  .   matches  any  single  character.\n"
			 "The caret ^ matches at the start of line.\n"
			 "The dollar $ matches at the end of line.\n" "\n"
			 "Characters within [ ] matches any single \n"
			 "       character in the list.\n"
			 "Characters within [^ ] matches any single\n"
			 "       character *not* in the list.\n"
			 "Characters inside [ - ] matches a range of\n"
			 "       characters (ie [0-9] or [a-z]).\n" "\n"
			 "A regular expression may be followed by one\n"
			 "       of several repetition operators:\n"
			 "?      The preceding item is optional and matched\n"
			 "       at most once.\n"
			 "*      The preceding item will be matched zero\n"
			 "       or more times.\n"
			 "+      The preceding item will be matched one or\n"
			 "       more times.\n"
			 "{n}    The preceding item is matched exactly n times.\n"
			 "{n,}   The preceding item is matched n or more times.\n" "{n,m}  The preceding item is matched at least n times,\n" "       but not more than m times.\n" "\n" "To match any reserved character, precede it with \\. \n" "\n" "Two regular expressions may be joined by the logical or\n" "       operator |.\n" "Two regular expressions may be concatenated.\n" "\n" "More information is available by typing \"man grep\"\n" "       at the command prompt.\n"));

}



static GList *put_item(GList ** list, char **r, GtkCombo * combo)
{
    GList *tmp = *list;
    while(tmp)
    {
	if(strcmp(*r, (char *)tmp->data) == 0)
	    break;
	tmp = tmp->next;
    }
    if(!tmp)
    {
	tmp = *list;
	tmp = g_list_prepend(tmp, *r);
    }
    else
	tmp = *list;
/*  if (tmp) {
	  char *d=(char *)tmp->data;
	  
	  tmp=g_list_remove(tmp,tmp->data);
	  g_free(d);
  } else tmp=*list;

  tmp=g_list_prepend(tmp,*r);*/
    return tmp;
}

/******  internal callbacks   *******/

static void destroy(GtkWidget * widget, gpointer data)
{
    tree_details_t *tree_details = get_tree_details(find_treeview);

    if(tree_details->tubo_object)
	TuboCancel(tree_details->tubo_object, NULL);
    tree_details->tubo_object = NULL;
    if(Gpid)
	kill(Gpid, SIGHUP);
    Gpid = 0;
    cursor_reset(find_treeview);
    gtk_widget_destroy(widget);
}


static void on_find_clicked(GtkButton * button, gpointer data)
{
    short int i, j;
    char *s;
    char *argument[MAX_ARGS];
    GtkWidget *w = (GtkWidget *) button;
    char *filter = NULL, *path = NULL, *token = NULL;
    static char *find_filter = NULL;
    GtkTreeView *treeview = (GtkTreeView *) data;
    tree_details_t *tree_details = get_tree_details(treeview);

    gtk_widget_grab_focus((GtkWidget *) treeview);

    if(!set_load_wait(&tree_details))
    {
	g_assert_not_reached();
    }
    /* get options */

    cancelled = FALSE;

    if(tree_details->tubo_object)
	TuboCancel(tree_details->tubo_object, NULL);
    tree_details->tubo_object = NULL;
    if(Gpid)
	kill(Gpid, SIGTERM);
    Gpid = 0;

    findCount = 0;
/* get the parameters set by the user... *****/

/* limit */
    fileLimit = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(w, "spinbutton1")));

/* the rest */

    s = get_combo_entry((GtkEntry *) (lookup_widget(w, "path_entry")));
    if(!s || strlen(s) == 0)
	s = "/";
    /* tilde expansion */
    else if(s[strlen(s) - 1] == '~')
	s = "~/";
    /* environment variables */
    else if(s[0] == '$')
    {
	char *p = getenv(s + 1);
	if(p)
	    s = p;
	else
	    s = "/";
    }

    find_path = path = g_strdup(s);
    path_list = put_item(&path_list, &path, (GtkCombo *) lookup_widget(w, "path_combo"));

    s = get_combo_entry((GtkEntry *) lookup_widget(w, "filter_entry"));
    if(s)
    {
	filter = g_strdup(s);
	filter_list = g_list_prepend(filter_list, g_strdup(filter));
	filter_list = put_item(&filter_list, &filter, (GtkCombo *) lookup_widget(w, "filter_combo"));
        save_ff_text(filter);
    }

    s = get_combo_entry((GtkEntry *) lookup_widget(w, "grep_entry"));
    if(s && strlen(s))
    {
	token = g_strdup(s);
	grep_list = put_item(&grep_list, &token, (GtkCombo *) lookup_widget(w, "grep_combo"));
    }
    /* select list */
    s = get_combo_entry((GtkEntry *) lookup_widget(w, "file_type_entry"));

    i = 0;
    /* special case find: if no grep token needed, best to use FIND */
    if((!token || !strlen(token)) && (filter && !strchr(filter, '?') && !strchr(filter, '*')))
    {
	argument[i++] = FIND;
	/* first argument to FIND must be the path */
	argument[i++] = path;
	if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "recursive")))
	    argument[i++] = "-prune";
	/* no filter if none specified  */
	if(filter && strlen(filter))
	{
	    if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "case_sensitive")))
		argument[i++] = "-ipath";
	    else
		argument[i++] = "-path";

	    if(find_filter){
		g_free(find_filter);
		find_filter=NULL;
	    }
	    find_filter = (char *)malloc(strlen(filter) + 3);
	    if(!find_filter)
		g_assert_not_reached();
	    sprintf(find_filter, "*/%s", filter);


	    argument[i++] = find_filter;
	}
	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "xdev")))
	    argument[i++] = "-xdev";

	/* OjO: "any" type is equal to no type specified in FIND 
	 *      so begin with j=1 */
	for(j = 1; ftypes[j] != NULL; j++)
	{
	    if(strcmp(s, _(ftypes[j])) == 0)
	    {
		argument[i++] = "-type";
		if (argument[0] == FIND && strcmp("reg",find_ft[j])==0) argument[i++] = "file";
		else argument[i++] = find_ft[j];
		break;
	    }
	}
	print_diagnostics(treeview, "xf_WARNING_ICON", _("Searching for"), " ", filter, " ", _("at"), " ", path, "\n", NULL);
	print_status(treeview, "xf_WARNING_ICON", _("Search begun"), NULL);

    }
    else
    {
	argument[i++] = GLOB;

	/*argument[i++] = "-v"; (verbose output from glob for debugging) */
	/*argument[i++] = "-P"; */

	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "recursive")))
	    argument[i++] = "-r";
	if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "case_sensitive")))
	    argument[i++] = "-i";
	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "xdev")))
	    argument[i++] = "-a";
	if(token)
	{
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "ext_regexp")))
		argument[i++] = "-E";
	    else
		argument[i++] = "-e";
	    argument[i++] = token;


	    /* options for grep: ***** */
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "no_look_in_binaries")))
		argument[i++] = "-I";
	    /*
	       if (gtk_toggle_button_get_active(
	       (GtkToggleButton *)lookup_widget(w,"output_count")))
	       argument[i++] = "-c"; */
	    /* radio buttons */
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "anywhere")))
		;
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_words")))
		argument[i++] = "-w";
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_lines")))
		argument[i++] = "-x";
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_no_match")))
		argument[i++] = "-L";
	}

	for(j = 0; ftypes[j] != NULL; j++)
	{
	    if(strcmp(s, _(ftypes[j])) == 0)
	    {
		argument[i++] = "-t";
		argument[i++] = ft[j];
		break;
	    }
	}


	/* apply wildcard filter if not specified  */
	argument[i++] = "-f";
	if(filter && strlen(filter))
	{
	    argument[i++] = filter;
	}
	else
	{
	    argument[i++] = "*";
	}

	/* last argument to GLOB must be the path */
	argument[i++] = path;
	{
	    char mess[_POSIX_PATH_MAX];
	    sprintf(mess, "%s %s %s %s %s %s\n", _("Searching for"), (token) ? token : "", (token) ? _("in") : "", (filter) ? filter : "", _("at"), path);

	    print_diagnostics(treeview, "xf_WARNING_ICON", mess, NULL);
	    print_status(treeview, "xf_WARNING_ICON", _("Search begun"), NULL);
	}
    }

    argument[i] = (char *)0;

    /*for (j=0;j<i;j++) printf ("%s ",argument[j]);printf ("\n"); */


    cursor_wait(find_treeview);
    print_diagnostics(treeview, "xf_INFO_ICON", NULL);
    for(j = 0; j < i; j++)
    {
	print_diagnostics(treeview, NULL, argument[j], " ", NULL);
    }
    print_diagnostics(treeview, NULL, "\n", NULL);

    stop = FALSE;

    /*printf("DBG: show stop at find.c\n"); */

    show_stop(tree_details->window);

    { /* map the find branch if its not shown */
      GtkTreeIter iter;
      tree_entry_t *en;
      get_find_root(treeview, &iter, &en);
    }

    tree_details->tubo_object = Tubo(
		    fork_function, 
		    (void *)argument, 
		    fork_finished_function, 
		    FALSE, 
		    operate_stdout, operate_stderr);
    if (!tree_details->tubo_object){
            gtk_widget_destroy(lookup_widget(w, "find_dialog"));
	    print_diagnostics(treeview,"xf_ERROR_ICON","Could not fork!\n",
			    NULL);
	    return;
    }
    g_timeout_add_full(0,260, (GtkFunction) watch_stop, (gpointer) treeview,NULL);
    Gpid = TuboPID(tree_details->tubo_object);

    gtk_widget_destroy(lookup_widget(w, "find_dialog"));
    {
	int size = 5;
	char *name;

	size += ((path) ? strlen(path) : 0);
	size += ((filter) ? strlen(filter) : 0);
	size += ((token) ? strlen(token) : 0);

	name = (char *)malloc(size);
	sprintf(name, "%s", path);
	if(strcmp(path, "/") != 0)
	    strcat(name, "/");
	if(filter && strlen(filter))
	    strcat(name, filter);
	if(token && strlen(token))
	{
	    strcat(name, "(");
	    strcat(name, token);
	    strcat(name, ")");
	}
	/*printf("DBG:size=%d, name=%s\n",size,name); */

	add_find_results(treeview, name);

	g_free(name);
	name=NULL;
    }

}

void on_find_close_clicked(GtkButton * button, gpointer user_data)
{
    destroy(lookup_widget((GtkWidget *) button, "find_dialog"), NULL);
}

static void on_help_filter(GtkToggleButton * button, gpointer user_data)
{
    if(gtk_toggle_button_get_active(button))
	showit((GtkWidget *) button, "scrolledwindow8");
    else
	hideit((GtkWidget *) button, "scrolledwindow8");

}

static void on_help_grep(GtkToggleButton * button, gpointer user_data)
{
    if(gtk_toggle_button_get_active(button))
	showit((GtkWidget *) button, "scrolledwindow9");
    else
	hideit((GtkWidget *) button, "scrolledwindow9");

}


/*********   callbacks   *********/

void on_find_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    do_find((GtkWidget *) menuitem);
}


void on_view_find1_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    do_find((GtkWidget *) menuitem);
}

void tb_find(GtkButton * button, gpointer user_data)
{
    do_find((GtkWidget *) button);
}




void on_remove_from_results_activate(GtkMenuItem * menuitem, gpointer user_data)
{

    GtkTreeIter iter, parent;
    tree_entry_t *en;
    GtkTreeView *treeview = get_selected_treeview((GtkWidget *) menuitem);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);

    /*get selection */
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    if(gtk_tree_selection_get_selected(selection, &treemodel, &iter))
    {
	gtk_tree_store_remove((GtkTreeStore *) treemodel, &iter);
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
    get_find_root(treeview, &parent, &en);
    if(!gtk_tree_model_iter_children(treemodel, &iter, &parent))
    {
	add_dummy(treeview, &parent);
	reset_dummy(treeview, &parent, 2);
    }
}


void on_clear_all_results_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeIter iter, parent;
    tree_entry_t *en;
    GtkTreeView *treeview = get_treeview((GtkWidget *) menuitem);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);

    get_find_root(treeview, &parent, &en);
    while(gtk_tree_model_iter_children(treemodel, &iter, &parent))
    {
	gtk_tree_store_remove((GtkTreeStore *) treemodel, &iter);
    }
    add_dummy(treeview, &parent);
    reset_dummy(treeview, &parent, 2);
}
