/* kastaskitem.cpp
**
** Copyright (C) 2001-2003 Richard Moore <rich@kde.org>
** Contributor: Mosfet
**     All rights reserved.
**
** KasBar is dual-licensed: you can choose the GPL or the BSD license.
** Short forms of both licenses are included below.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
** MA 02111-1307, USA.
*/

/*
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
#include <qbitmap.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmetaobject.h>
#include <qpainter.h>
#include <qregexp.h>
#include <qtabwidget.h>
#include <qtextview.h>
#include <qtimer.h>
#include <qvbox.h>

#include <kdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <dcopclient.h>
#include <kapplication.h>

#include <taskmanager.h>
#include <taskrmbmenu.h>

#include "kastasker.h"
#include "kastaskpopup.h"
#include "kastaskitem.h"

#undef ENABLE_NEW_STUFF

/* XPM */
static const char *tiny_floppy[]={
    "10 10 4 1",
    ". c None",
    "# c #000000",
    "b c #a0a0a0",
    "a c #ffffff",
    ".########.",
    ".#aaaaaa#.",
    ".#aaaaaa#.",
    ".#aaaaaa#.",
    ".########.",
    ".########.",
    ".##bbbb##.",
    ".##bbbb##.",
    ".##bbbb##.",
    ".........."};

KasTaskItem::KasTaskItem( KasTasker *parent, Task *task )
    : KasItem( parent ),
      task_(0),
      thumbTimer(0)
{
    task_ = task;
    connect( task, SIGNAL( changed() ), this, SLOT( update() ) );
    connect( task, SIGNAL( activated() ), this, SLOT( startAutoThumbnail() ) );
    connect( task, SIGNAL( deactivated() ), this, SLOT( stopAutoThumbnail() ) );
    connect( task, SIGNAL( iconChanged() ), this, SLOT( iconChanged() ) );
}

KasTaskItem::~KasTaskItem()
{
}

KasTasker *KasTaskItem::kasbar() const
{
    return static_cast<KasTasker *> (KasItem::kasbar());
}

QPixmap KasTaskItem::icon()
{
    usedIconLoader = false;

    switch( kasbar()->itemSize() ) {
	case KasBar::Small:
	    return task_->bestIcon( KIcon::SizeSmall, usedIconLoader );
	    break;
	case KasBar::Medium:
	    return task_->bestIcon( KIcon::SizeMedium, usedIconLoader );
	    break;
	case KasBar::Large:
	    return task_->bestIcon( KIcon::SizeLarge, usedIconLoader );
	    break;
    }

    return KGlobal::iconLoader()->loadIcon( "error",
					    KIcon::NoGroup,
					    KIcon::SizeSmall );
}

void KasTaskItem::iconChanged()
{
    iconHasChanged = true;
    update();
}

void KasTaskItem::paint( QPainter *p )
{
    setActive( task_->isActive() );
    setText( task_->visibleIconicName() );

    paintFrame( p );
    paintLabel( p );
    paintBackground( p );

    //
    // Draw icon
    //
    p->drawPixmap(4, 16, icon() );

    //
    // Overlay the small icon if the icon has changed, we have space,
    // and we are using a KIconLoader icon rather than one from the NET props.
    // This only exists because we are almost always using the icon loader for
    // large icons.
    //
    if ( usedIconLoader && iconHasChanged && ( kasbar()->itemSize() == KasBar::Large ) ) {
	p->drawPixmap(34, 18, task_->pixmap() );
    }

    //
    // Draw window state.
    //
    QString deskStr;
    if ( task_->isOnAllDesktops() )
	deskStr = i18n( "All" );
    else
	deskStr.setNum( task_->desktop() );

    KasTasker *kas = kasbar();
    p->setPen( task_->isActive() ? kasbar()->activePenColor() : kasbar()->inactivePenColor() );

    if ( kas->itemSize() != KasBar::Small ) {
	// Medium and Large modes
	p->drawText( extent()-fontMetrics().width(deskStr)-3,
		     15+fontMetrics().ascent(), deskStr );

	if( task_->isIconified() )
	    p->drawPixmap(extent()-11, extent()-11, *(kas->minIcon()) );
	else if ( task_->isShaded() )
	    p->drawPixmap(extent()-11, extent()-11, *(kas->shadeIcon()) );
	else
	    p->drawPixmap(extent()-11, extent()-11, *(kas->maxIcon()) );

#ifdef ENABLE_NEW_STUFF
	// Progress graph
	QRegExp reg( "(1?[0-9][0-9])%" );
	if ( -1 != reg.search( task_->visibleIconicName() ) ) {
	    int perc = reg.cap(1).toInt();
	    paintProgress( p, perc );
	}
#endif
    }
    else {
	// Small mode
	p->drawText( extent()-fontMetrics().width(deskStr)-2,
		     13+fontMetrics().ascent(), deskStr );

	if( task_->isIconified() )
	    p->drawPixmap( extent()-9, extent()-9,
			   *(kas->microMinIcon()) );
	else if ( task_->isShaded() )
	    p->drawPixmap( extent()-9, extent()-9,
			   *(kas->microShadeIcon()) );
	else
	    p->drawPixmap(extent()-9, extent()-9,
			  *(kas->microMaxIcon()) );
    }

    //
    // Draw document state.
    //
    if ( kasbar()->showModified() && (!( kasbar()->itemSize() == KasBar::Small ) ) ) {
	if ( task_->isModified() ) {
	    QPixmap floppy( tiny_floppy );
	    p->drawPixmap(extent()-12, extent()-22, floppy );
	}
    }
}

void KasTaskItem::mousePressEvent( QMouseEvent *ev )
{
    hidePopup();

    if ( ev->button() == LeftButton ) {
        if ( task_->isActive() && task_->isShaded() ) {
            task_->setShaded( false );
        }
        else {
            task_->activateRaiseOrIconify();
        }
    }
    else if ( ev->button() == RightButton ) {
        showWindowMenuAt( ev->globalPos() );
    }
    else {
        refreshThumbnail();
    }
}

KasPopup *KasTaskItem::createPopup()
{
    KasPopup *pop = new KasTaskPopup( this );
    return pop;
}

void KasTaskItem::dragOverAction()
{
    if ( !task_->isOnCurrentDesktop() )
	task_->toCurrentDesktop();
    if ( task_->isShaded() )
	task_->setShaded( false );
    if ( task_->isIconified() )
	task_->restore();
    if ( !task_->isActive() )
	task_->activate();
}

void KasTaskItem::startAutoThumbnail()
{
    if ( thumbTimer )
	return;
    if ( !kasbar()->thumbnailsEnabled() )
	return;

    thumbTimer = new QTimer( this, "thumbTimer" );
    connect( thumbTimer, SIGNAL( timeout() ),
	     this, SLOT( refreshThumbnail() ) );

    if ( kasbar()->thumbnailUpdateDelay() > 0 )
	thumbTimer->start( kasbar()->thumbnailUpdateDelay() * 1000 );

    QTimer::singleShot( 200, this, SLOT( refreshThumbnail() ) );
}

void KasTaskItem::stopAutoThumbnail()
{
    if ( !thumbTimer )
	return;

    delete thumbTimer;
    thumbTimer = 0;
}

void KasTaskItem::refreshThumbnail()
{
    if ( !kasbar()->thumbnailsEnabled() )
	return;
    if ( !task_->isActive() )
	return;

    // TODO: Check if the popup obscures the window
    KasItem *i = kasbar()->itemUnderMouse();
    if ( i && i->isShowingPopup() ) {
	QTimer::singleShot( 200, this, SLOT( refreshThumbnail() ) );
	return;
    }

    task_->setThumbnailSize( kasbar()->thumbnailSize() );
    task_->updateThumbnail();
}

void KasTaskItem::showWindowMenuAt( QPoint p )
{
    KPopupMenu *km = new KPopupMenu;
    km->insertItem( i18n("&About Kasbar"), kasbar(), SLOT( showAbout() ) );
    km->insertItem( i18n("&Preferences..."), kasbar(), SLOT( showPreferences() ) );

    TaskRMBMenu *tm = new TaskRMBMenu( task_, kasbar()->taskManager(), kasbar() );
    tm->insertSeparator();
    tm->insertItem( i18n("&Kasbar"), km );
    tm->insertItem( i18n("&Properties" ), this, SLOT( showPropertiesDialog() ) );

    mouseLeave();
    kasbar()->updateMouseOver();

    tm->exec( p );
    delete tm;
}

void KasTaskItem::showPropertiesDialog()
{
    //
    // Create Dialog
    //
    QDialog *dlg = new QDialog( /*kasbar()*/0L, "task_props", false );

    //
    // Title
    //
    QString s(i18n("Task '%1' - Properties").arg(task_->visibleName()));

    KPopupTitle *title = new KPopupTitle( dlg, "title" );
    title->setText( s );
    title->setIcon( icon() );

    dlg->setCaption( s );
    dlg->setIcon( icon() );

    //
    // Tabbed View
    //
    QTabWidget *tabs = new QTabWidget( dlg );
    tabs->addTab( createX11Props( tabs ), i18n("General") );
    tabs->addTab( createTaskProps( tabs ), i18n("Task") );
#if 0
    tabs->addTab( createNETProps( tabs ), i18n("NET") );
#endif

    //
    // Layout Dialog
    //
    QVBoxLayout *vbl = new QVBoxLayout( dlg, KDialog::marginHint(), KDialog::spacingHint() );
    vbl->addWidget( title );
    vbl->addWidget( tabs );

    dlg->show();

}

QWidget *KasTaskItem::createTaskProps( QWidget *parent )
{
    QVBox *vb = new QVBox( parent );
    vb->setSpacing( KDialog::spacingHint() );
    vb->setMargin( KDialog::marginHint() );

    // Create List View
    KListView *taskprops = new KListView( vb, "props_view" );
    taskprops->setResizeMode( QListView::LastColumn );
    taskprops->addColumn( i18n("Property"), 0 );
    taskprops->addColumn( i18n("Type"), 0 );
    taskprops->addColumn( i18n("Value") );

    // Create List Items
    QMetaObject *mo = task_->metaObject();
    for ( int i = 0; i < mo->numProperties(); i++ ) {
	const QMetaProperty *p = mo->property(i);

	(void) new KListViewItem( taskprops,
				  p->name(), p->type(),
				  task_->property( p->name() ).toString() );
    }

    return vb;
}

QString KasTaskItem::expandMacros( const QString &format, QObject *data )
{
    QString s = format;
    QRegExp re("\\$(\\w+)");

    int pos = 0;
    while ( pos >= 0 ) {
        pos = re.search( s, pos );
        if ( pos >= 0 ) {
	    QVariant val = data->property( re.cap(1).latin1() );
	    QString v = val.asString();
	    s.replace( pos, re.matchedLength(), v );
            pos = pos + v.length();
        }
    }

    return s;
}

QWidget *KasTaskItem::createX11Props( QWidget *parent )
{
    QVBox *vb2 = new QVBox( parent );
    vb2->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Preferred );
    vb2->setSpacing( KDialog::spacingHint() );
    vb2->setMargin( KDialog::marginHint() );

    // Create View
    new QLabel( i18n("General"), vb2, "view" );
    QTextView *tv = new QTextView( vb2 );

    QString fmt = i18n(
	"<html>"
	"<body>"
	"<b>Name</b>: $name<br>"
	"<b>Visible Name</b>: $visibleName<br>"
	"<br>"
	"<b>Iconified</b>: $iconified<br>"
	"<b>Minimized</b>: $minimized<br>"
	"<b>Maximized</b>: $maximized<br>"
	"<b>Shaded</b>: $shaded<br>"
	"<br>"
	"<b>Desktop</b>: $desktop<br>"
	"<b>All Desktops</b>: $onAllDesktops<br>"
	"<br>"
	"<b>Iconic Name</b>: $iconicName<br>"
	"<b>Iconic Visible Name</b>: $iconicVisibleName<br>"
	"</body>"
	"</html>"
	);

    tv->setText( expandMacros( fmt, task_ ) );

    return vb2;
}

QWidget *KasTaskItem::createNETProps( QWidget *parent )
{
    QVBox *vb3 = new QVBox( parent );
    vb3->setSpacing( KDialog::spacingHint() );
    vb3->setMargin( KDialog::marginHint() );

    // Create View
    new QLabel( i18n("NET WM Specification Info"), vb3, "view" );
    new QTextView( vb3 );

    return vb3;
}

#include "kastaskitem.moc"
