/*  This file is part of the KDE project
    Copyright (C) 2003 Matthias Kretz <kretz@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.

*/

#include "ksettings/componentsdialog.h"
#include <klocale.h>
#include <qlayout.h>
#include <klistview.h>
#include <qlabel.h>
#include <qheader.h>
#include <kplugininfo.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kseparator.h>

namespace KSettings
{

class ComponentsDialog::ComponentsDialogPrivate
{
    public:
        KListView * listview;
        QFrame * infowidget;
        QLabel * iconwidget;
        QLabel * commentwidget;
        QLabel * descriptionwidget;
        QMap<QCheckListItem*, KPluginInfo*> plugininfomap;
        QValueList<KPluginInfo*> plugininfolist;
};

ComponentsDialog::ComponentsDialog( QWidget * parent, const char * name )
    : KDialogBase( parent, name, false, i18n( "Select Components" ) )
, d( new ComponentsDialogPrivate )
{
    QWidget * page = new QWidget( this );
    setMainWidget( page );
    ( new QHBoxLayout( page, 0, KDialog::spacingHint() ) )->setAutoAdd( true );
    d->listview = new KListView( page );
    d->listview->setMinimumSize( 200, 200 );
    d->infowidget = new QFrame( page );
    d->infowidget->setMinimumSize( 200, 200 );
    ( new QVBoxLayout( d->infowidget, 0, KDialog::spacingHint() ) )->setAutoAdd( true );
    d->iconwidget = new QLabel( d->infowidget );
    ( void )new KSeparator( d->infowidget );
    d->commentwidget = new QLabel( d->infowidget );
    d->commentwidget->setAlignment( Qt::WordBreak );
    d->descriptionwidget = new QLabel( d->infowidget );
    d->descriptionwidget->setAlignment( Qt::WordBreak );

    d->listview->addColumn( QString::null );
    d->listview->header()->hide();
    d->listview->setRootIsDecorated( true );
    d->listview->setSorting( -1 );
    d->listview->setAcceptDrops( false );
    d->listview->setSelectionModeExt( KListView::Single );
    d->listview->setAllColumnsShowFocus( true );

    connect( d->listview, SIGNAL( pressed( QListViewItem * ) ), this,
            SLOT( executed( QListViewItem * ) ) );
    connect( d->listview, SIGNAL( spacePressed( QListViewItem * ) ), this,
            SLOT( executed( QListViewItem * ) ) );
    connect( d->listview, SIGNAL( returnPressed( QListViewItem * ) ), this,
            SLOT( executed( QListViewItem * ) ) );
    connect( d->listview, SIGNAL( selectionChanged( QListViewItem * ) ), this,
            SLOT( executed( QListViewItem * ) ) );
}

ComponentsDialog::~ComponentsDialog()
{
}

void ComponentsDialog::setPluginInfos( const QMap<QString, KPluginInfo*> &
        plugininfos )
{
    for( QMap<QString, KPluginInfo*>::ConstIterator it = plugininfos.begin();
            it != plugininfos.end(); ++it )
    {
        d->plugininfolist.append( it.data() );
    }
}

void ComponentsDialog::show()
{
    // construct the treelist
    for( QValueList<KPluginInfo*>::ConstIterator it = d->plugininfolist.begin();
            it != d->plugininfolist.end(); ++it )
    {
        ( *it )->load();
        QCheckListItem * item = new QCheckListItem( d->listview, ( *it )->name(),
                QCheckListItem::CheckBox );
        if( ! ( *it )->icon().isEmpty() )
            item->setPixmap( 0, SmallIcon( ( *it )->icon(), IconSize( KIcon::Small ) ) );
        item->setOn( ( *it )->isPluginEnabled() );
        d->plugininfomap[ item ] = ( *it );
    }
    KDialogBase::show();
}

void ComponentsDialog::executed( QListViewItem * item )
{
    kdDebug( 704 ) << k_funcinfo << endl;
    if( item == 0 )
        return;
    if( item->rtti() != 1 ) // check for QCheckListItem
        return;

    QCheckListItem * citem = static_cast<QCheckListItem *>( item );
    bool checked = citem->isOn();

    kdDebug( 704 ) << "it's a " << ( checked ? "checked" : "unchecked" )
        << " QCheckListItem" << endl;

    KPluginInfo * info = d->plugininfomap[ citem ];
    info->setPluginEnabled( checked );
    //checkDependencies( info );
    // show info about the component on the right
    d->iconwidget->setPixmap( SmallIcon( info->icon(), KIcon::SizeLarge ) );
    d->commentwidget->setText( info->comment() );
    //d->descriptionwidget->setText( info->description() );
}

void ComponentsDialog::savePluginInfos()
{
    for( QValueList<KPluginInfo*>::ConstIterator it = d->plugininfolist.begin();
            it != d->plugininfolist.end(); ++it )
    {
        if( ( *it )->config() )
        {
            ( *it )->save();
            ( *it )->config()->sync();
        }
    }
}

void ComponentsDialog::slotOk()
{
    savePluginInfos();
    KDialogBase::slotOk();
}

void ComponentsDialog::slotApply()
{
    savePluginInfos();
    KDialogBase::slotApply();
}

} //namespace

#include "componentsdialog.moc"
// vim: sw=4 sts=4 et
