/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * dxcc.c - dxcc lookups and creation of the hashtable
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dxcc.h"
#include "gc.h"
#include "support.h"
#include "types.h"

extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;
dxcc_data *dxcc;
GHashTable *prefixes;
gchar *searchpx = NULL;
gint excwaz, excitu;

static gchar *
findpfx (gchar * pfx)
{
  gchar *end, *j;

  g_strstrip (pfx);
  end = pfx + strlen (pfx);
  for (j = pfx; j < end; ++j)
    {
      switch (*j)
	{
	case '(':
	case '[':
	case ';':
	  *j = '\0';
	  break;
	}
    }
  return pfx;
}

static gchar *
replace (gchar *callsign, gint area)
{
  gchar *end, *j;

  end = callsign + strlen (callsign);
  for (j = callsign; j < end; ++j)
    {
      switch (*j)
	{
	case '0' ... '9':
	  if ((j - callsign) > 1) 
            *j = area + 48;
	  break;
	}
    }
  
  return(g_strdup(callsign));
}

static gchar *
getpx (gchar *checkcall)
{

  gchar *pxstr = NULL, **split;

  /* characters after '/' might contain a country */
  if (strchr(checkcall, '/'))
  {
    split = g_strsplit(checkcall, "/", 2);
    if (split[1]) /* we might be typing */
    {
      if ((strlen(split[1]) > 1) && 
                (strlen(split[1]) < strlen(split[0])))
      /* this might be a candidate */
      {
      if ((g_strcasecmp(split[1], "AM") == 0) ||
                (g_strcasecmp(split[1], "MM") == 0))
          pxstr = NULL; /* don't know location */
        else if (g_strcasecmp(split[1], "QRP") == 0)
          pxstr = g_strdup(split[0]);
        else pxstr = g_strdup(split[1]);
      }
      else if ((strlen(split[1]) == 1) && atoi(split[1]))
      /* callsign area changed */
      { 
        pxstr = replace(split[0], atoi(split[1]));
      }
      else pxstr = g_strdup(split[0]);
    }
    else pxstr = g_strdup(split[0]);
    g_strfreev(split);
  }
  else pxstr = g_strdup(checkcall);

  return(pxstr);
}


static gchar *findexc(gchar *exc)
{
  gchar *end, *j;

  g_strstrip(exc);
  end = exc + strlen(exc);
  for (j = exc; j < end; ++j)
  {
    switch (*j)
    {
      case '(':
        if (*(j+2) == 41)
          excwaz = *(j+1) - 48;
        else if (*(j+3) == 41)
          excwaz = ((*(j+1) - 48) * 10) + (*(j+2) - 48);
      case '[':
        if (*(j+2) == 93)
          excitu = *(j+1) - 48;
        else if (*(j+3) == 93)
          excitu = ((*(j+1) - 48) * 10) + (*(j+2) - 48);
      case ';':
        *j = '\0';
      break;
    }
  }
  return exc;
}

static gint
lookup_dxcc (gchar * callsign)
{
  gint ipx, country = 0;
  gchar *px;

  px = getpx(callsign);
  if (px)
  {
    for (ipx = strlen(px); ipx > 0; ipx--)
    {
      searchpx = g_strndup (px, ipx);
      country = GPOINTER_TO_INT (g_hash_table_lookup (prefixes, searchpx));
      if (country > 0) break;
    }
    g_free (px);
  }
  else country = 0;
  if (country == 0) searchpx = NULL;
  
  return (country);
}


gint
readctydata (void)
{

  gchar buf[4096], *cty_location, *pfx;
  gint ichar = 0, dxccitem = 0, ipfx = 0, ch = 0;
  FILE *fp;
  gchar **split, **pfxsplit;

  cty_location = g_strconcat (PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S, "dxcc",
			      G_DIR_SEPARATOR_S, "cty.dat", NULL);

  if ((fp = fopen (cty_location, "r")) == NULL)
    {
      g_free (cty_location);
      return (1);
    }

  dxcc = g_new0 (dxcc_data, 400);
  prefixes = g_hash_table_new (g_str_hash, g_str_equal);

  /* first field in case hash_table_lookup returns NULL */
  dxcc[0].countryname = g_strdup ("");
  dxcc[0].waz = -1;
  dxcc[0].itu = -1;
  dxcc[0].continent = g_strdup ("");
  dxcc[0].latitude = 0;
  dxcc[0].longitude = 0;
  dxcc[0].timezone = 0;
  dxcc[0].px = g_strdup ("");
  dxcc[0].exceptions = g_strdup ("");

  state.countries = 1;

  while (!feof (fp))
    {
      while (ch != 59)
	{
	  ch = fgetc (fp);
	  if (ch == EOF)
	    break;
	  buf[ichar++] = ch;
	}
      if (ch == EOF)
	break;
      buf[ichar] = '\0';
      ichar = 0;
      ch = 0;

      /* split up the first line */
      split = g_strsplit (buf, ":", 9);
      for (dxccitem = 0; dxccitem < 9; dxccitem++)
	g_strstrip (split[dxccitem]);
      /* split up the second line */
      pfxsplit = g_strsplit (split[8], ",", 0);

      dxcc[state.countries].countryname = g_strdup (split[0]);
      dxcc[state.countries].waz = atoi (split[1]);
      dxcc[state.countries].itu = atoi (split[2]);
      dxcc[state.countries].continent = g_strdup (split[3]);
      dxcc[state.countries].latitude = (gint)(atof(split[4]) * 100);
      dxcc[state.countries].longitude = (gint)(atof(split[5]) * 100);
      dxcc[state.countries].timezone = (gint)(atof(split[6]) * 10);
      dxcc[state.countries].px = g_strdup (split[7]);
      dxcc[state.countries].exceptions = g_strdup (split[8]);

      /* official prefix */
      g_strup(split[7]); /* some chars are lowercase */
      g_hash_table_insert (prefixes, g_strdup (split[7]),
          GINT_TO_POINTER (state.countries));

      /* exception list */
      for (ipfx = 0;; ipfx++)
	{
	  if (!pfxsplit[ipfx])
	    break;
	  pfx = findpfx (pfxsplit[ipfx]);
          if (g_strcasecmp(pfx, split[7]) != 0)
            g_hash_table_insert (prefixes, g_strdup (pfx),
              GINT_TO_POINTER (state.countries));
	}

      g_strfreev (pfxsplit);
      g_strfreev (split);
      state.countries++;
    }
  fclose (fp);
  g_free (cty_location);
  return (0);
}

gint
readxlogdata (void)
{
  gchar *dat_location, buf[4096], **split;
  FILE *fp;
  gint ichar = 0, ch = 0, country, newcountry;

  dat_location = g_strconcat (PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S, "dxcc",
			      G_DIR_SEPARATOR_S, "xlog.dat", NULL);

  if ((fp = fopen (dat_location, "r")) == NULL)
    {
      g_free (dat_location);
      return (1);
    }

  while (!feof (fp))
    {
      while (ch != 10)
	{
	  ch = fgetc (fp);
	  if (ch == EOF)
	    break;
	  buf[ichar++] = ch;
	}
      buf[ichar] = '\0';
      ichar = 0;
      ch = 0;

      /* ignore empty lines and comments */
      if ((g_strncasecmp(buf, "#", 1) != 0) && 
        (g_strncasecmp(buf, "\n", 1) != 0))
      {
        g_strdelimit(buf, "\n", ' ');
        split = g_strsplit (buf, " ", 2);
        g_strup(split[1]);
        country = GPOINTER_TO_INT (g_hash_table_lookup (prefixes, split[1]));;
        if (country > 0)
        { /* search and replace */
          g_strup(split[0]);
          newcountry = GPOINTER_TO_INT (g_hash_table_lookup (prefixes, split[0]));
          if (newcountry == 0)
          {
            g_hash_table_insert (prefixes, g_strdup (split[0]),
              GINT_TO_POINTER (country));
          }
          else
          {
            g_hash_table_remove (prefixes, split[0]);
            g_hash_table_insert (prefixes, g_strdup (split[0]),
              GINT_TO_POINTER (country));
          }
        }
        g_strfreev (split);
      }

      if (ch == EOF) break;
    }

  g_free (dat_location);

  return (0);
}


void
updatedxccframe (gchar * callsign)
{
  GtkWidget *dxcclabel1, *dxcclabel2, *dxcclabel3, *dxcclabel4,
    *dxcclabel5, *dxccframe;
  gchar *labeltext1, *labeltext2, *labeltext3, *labeltext4, *gcresult;
  gint country = -1, iexc;
  gchar **excsplit, *exc;
  gboolean exception = FALSE;

  dxcclabel1 = lookup_widget (mainwindow, "dxcclabel1");
  dxcclabel2 = lookup_widget (mainwindow, "dxcclabel2");
  dxcclabel3 = lookup_widget (mainwindow, "dxcclabel3");
  dxcclabel4 = lookup_widget (mainwindow, "dxcclabel4");
  dxcclabel5 = lookup_widget (mainwindow, "dxcclabel5");
  dxccframe = lookup_widget (mainwindow, "dxccframe");
  labeltext1 = g_strdup ("");
  labeltext2 = g_strdup ("");
  labeltext3 = g_strdup ("");
  labeltext4 = g_strdup ("");
  gcresult = g_strdup ("");
  gtk_frame_set_label (GTK_FRAME (dxccframe), "DXCC");

  if (g_strcasecmp (callsign, "") != 0)
    {
      country = lookup_dxcc (callsign);
      gtk_frame_set_label (GTK_FRAME (dxccframe), callsign);
    }

  if (country > 0)
    {
      labeltext1 = g_strdup (dxcc[country].countryname);
      /* look for exceptions */ 
      if (strchr(dxcc[country].exceptions, '(') || 
        strchr(dxcc[country].exceptions, '['))
      {
        excsplit = g_strsplit(dxcc[country].exceptions, ",", 0);
        for (iexc = 0;; iexc++)
        {
          if (!excsplit[iexc]) break;
          exc = findexc(excsplit[iexc]);
          if (g_strcasecmp(searchpx, exc) == 0)
          { 
            labeltext2 = g_strdup_printf ("%s - ITU %d - CQ %d", 
              dxcc[country].continent, excitu, excwaz);
            exception = TRUE;
          }
        }
        g_strfreev(excsplit);
      }

      if (! exception)
      labeltext2 = g_strdup_printf ("%s - ITU %d - CQ %d", 
        dxcc[country].continent, dxcc[country].itu, dxcc[country].waz);
      labeltext3 = g_strdup_printf (_("Timezone: %+1.1f"), (gdouble)dxcc[country].timezone/10);
      labeltext4 = g_strdup_printf (_("Location: %+1.2f %+1.2f"), 
        (gdouble)dxcc[country].latitude/100, (gdouble)dxcc[country].longitude/100);
      gcresult = g_strdup (gc(preferences.units, state.mylocation,
                    (gdouble)dxcc[country].latitude/100, 
                    (gdouble)dxcc[country].longitude/100));
    }
  else
    {
      labeltext1 = g_strdup ("");
      labeltext2 = g_strdup ("");
    }
  gtk_label_set_text (GTK_LABEL (dxcclabel1), labeltext1);
  gtk_label_set_text (GTK_LABEL (dxcclabel2), labeltext2);
  g_free (labeltext1);
  g_free (labeltext2);

  gtk_label_set_text (GTK_LABEL (dxcclabel3), labeltext3);
  gtk_label_set_text (GTK_LABEL (dxcclabel4), labeltext4);
  gtk_label_set_text (GTK_LABEL (dxcclabel5), gcresult);
  g_free (labeltext3);
  g_free (labeltext4);
  g_free (gcresult);
}
