/*#define DEBUG*/
/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <limits.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <regex.h>
#include <dirent.h>

#include <X11/Xlib.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <dbh.h>

#include "glade_callbacks.h"
#include "glade_gui.h"
#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "add_file.h"
#include "add_node_contents.h"
#include "misc.h"
#include "entry.h"
#include "icons.h"
#include "treeview.h"

extern root_t root[];

static gboolean diagnostics_disabled=FALSE;
extern gboolean double_treeview;

static gboolean correct_selection;
static void query_selection(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    GtkTreeView *treeview = (GtkTreeView *) data;
    if(correct_selection) return;
    correct_selection=TRUE;
    return;
}

GtkTreeView *get_selected_treeview(GtkWidget *w){
    GtkTreeView *treeview;
    GtkTreeSelection *selection;
    correct_selection=FALSE;
    treeview=(GtkTreeView *)lookup_widget(w,"treeview");
    selection = gtk_tree_view_get_selection(treeview);
    gtk_tree_selection_selected_foreach(selection, query_selection, (gpointer) treeview);
    if (correct_selection) return treeview;
    if (double_treeview){
      treeview=(GtkTreeView *)lookup_widget(w,"treeview2");
      selection = gtk_tree_view_get_selection(treeview);
      gtk_tree_selection_selected_foreach(selection, query_selection, (gpointer) treeview);
      if (correct_selection) return treeview;
    }
    return (GtkTreeView *)lookup_widget(w,"treeview");	
}

/* this function is to workaround a gtk bug in treeviews, where a 
 * while(gtk_events_pending()) will not break... 
 * Let's say 256 iterations suffice... */
void process_pending_gtk(){
  int i;
  for (i=0;i<256;i++){
	  if (!gtk_events_pending()) break;
	  else gtk_main_iteration();
  }
}

void disable_diagnostics(void){diagnostics_disabled=TRUE;};
void enable_diagnostics(void){diagnostics_disabled=FALSE;};

const gchar *get_xffm_home(void){
    static gchar *home=NULL;
    g_free(home); home=NULL;    
    if (getenv("XFFM_HOME") && strlen( getenv("XFFM_HOME"))){
	    home=g_strdup(getenv("XFFM_HOME"));
	    if (access(home,X_OK)!=0){g_free(home);home=NULL;}
    }
    if (!home) home=g_strdup(g_get_home_dir());
    if (!home) home=g_strdup("/");
    return home;
}

/* this could also delete tmp files in use by another xffm instance 
 * (very slight possibility)*/
void cleanup_tmpfiles(void)
{
    DIR *directory;
    char *fullpath;
    struct dirent *d;
    char *path="/tmp";
    directory = opendir(path);
    if(!directory) return;
    while((d = readdir(directory)) != NULL)
    {
	if(strncmp(d->d_name, "xffm",strlen("xffm"))==0){
	   fullpath = mk_path(path, d->d_name);
	   unlink(fullpath);
	}
    }
    closedir(directory);
    return;
}

void fork_function(void *data)
{
    char **argument = (char **)data;

    execvp(argument[0], argument);
    fprintf(stderr, "CHILD could not execvp: this should not happen");
    fprintf(stderr, "Do you have %s in your path?", argument[0]);
    fflush(NULL);
    usleep(500000);
    _exit(123);
}

void set_entry_tag(GtkTreeView * treeview, tree_entry_t * en, long long unsigned tama)
{
    int hcount;
    tree_details_t *tree_details = get_tree_details(treeview);

    hcount = count_hidden_files(en->path);
    if(hcount)
    {
	SET_HAS_HIDDEN(en->type);
    }
    if(SHOWS_HIDDEN(en->type))
	sprintf(en->tag, "%s (%s %s)", FILENAME(en), sizetag(tama, en->count), _("Showing hidden."));
    else
    {
	if(hcount)
	    sprintf(en->tag, "%s (%s %d %s)", FILENAME(en), sizetag(tama, en->count), hcount, _("hidden."));

	else
	    sprintf(en->tag, "%s (%s %s)", FILENAME(en), sizetag(tama, en->count), _("No hidden."));
    }
}



gboolean get_selectpath_iter(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{

    tree_details_t *tree_details = (tree_details_t *) get_tree_details(treeview);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(tree_details->treeview);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    /*GtkTreePath *selectpath; */
    GtkTreeIter parent;
    tree_entry_t *c_en, *p_en = NULL;
    int caso = 0;

    if(!tree_details)
	return FALSE;
    if(!tree_details->window)
	return FALSE;


    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    if(!gtk_tree_selection_get_selected(selection, &treemodel, iter))
    {
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
	return FALSE;
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &c_en, -1);
    if(gtk_tree_model_iter_parent(treemodel, &parent, iter))
    {
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);
    }

    if(IS_PATH(c_en->type) && IS_DIR(c_en->type))
	caso = 1;

    else if(IS_FIND_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && IS_FILE(c_en->type) && IS_XF_FND(p_en->type))
	    caso = 0;
	else if(p_en && IS_FILE(c_en->type) && IS_DIR(p_en->type))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_LOCAL_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_BOOKMARK_TYPE(c_en->type))
    {
	if(IS_DIR(c_en->type) || IS_ROOT_TYPE(c_en->type) ||
	   IS_NETDIR(c_en->subtype) || IS_XF_NETSHARE(c_en->subtype) ||
	   IS_XF_NETWS(c_en->subtype)  )
	    caso = 1;
	else if (p_en && 
		 (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)
		  || IS_NETFILE(c_en->subtype))
		) caso = 2;
	else
	    caso = 0;
    }
    else if(IS_TRASH_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_NETWORK_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type) || 
	   IS_XF_NETWG(c_en->subtype) ||
	   IS_XF_NETWS(c_en->subtype) ||
	   IS_NETDIR(c_en->subtype) ||
	   IS_XF_NETSHARE(c_en->subtype))
	    caso = 1;
	else if (p_en )
	    caso = 2;
	else
	    caso = 0;
    }
    else if(IS_APP_TYPE(c_en->type))
    {
	get_apps_root(tree_details->treeview, &parent, &p_en);
	if (p_en )
	    caso = 2;
	else
	    caso = 0;
    }
    else if (IS_FSTAB_TYPE(c_en->type))
    {
	if(IS_ROOT_TYPE(c_en->type)||IS_DIR(c_en->type))
	    caso = 1;
	else if(p_en && (IS_FILE(c_en->type) || IS_DUMMY_TYPE(c_en->type)))
	    		caso = 2;
	else caso = 0;
    }
	    
    if(caso)
    {
	if(caso == 2)
	{
	    *en = p_en;
	    *iter = parent;
	}
	else
	    *en = c_en;
    }
    else
	return FALSE;

    return TRUE;
}

static int get_the_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    const gchar *g;
    if(!gtk_tree_model_get_iter_first(treemodel, iter)){
new_root:
#ifdef DEBUG
	   printf("DBG(misc.c):create_root_element(treeview,iter,which,g)\n");
#endif
	   g=get_xffm_home();
	   create_root_element(treeview,iter,which,g);
   	   gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);
	   return TRUE;
    }
	   
	    
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);

    while((((*en)->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, iter))
	   goto new_root;
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, en, -1);
    }
    return TRUE;
}

gboolean find_root(GtkTreeView * treeview, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    const gchar *g;
    GtkTreeIter iter;
    tree_entry_t *en;
    if(!gtk_tree_model_get_iter_first(treemodel, &iter)) return FALSE;
	   
	    
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);

    while(((en->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, &iter))
	   return FALSE;
	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    }
    return TRUE;
}

static void erase_root(GtkTreeModel * treemodel, GtkTreeIter * target)
{
    GtkTreeIter iter;
    tree_entry_t *en;



    while(gtk_tree_model_iter_children(treemodel, &iter, target))
    {
	erase_children(treemodel, &iter);

	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
	destroy_entry(en);

	gtk_tree_store_remove((GtkTreeStore *) treemodel, &(iter));
    }
    gtk_tree_model_get(treemodel, target, ENTRY_COLUMN, &en, -1);
    destroy_entry(en);
    gtk_tree_store_remove((GtkTreeStore *) treemodel, target);
    
    return;
}

static int erase_the_root(GtkTreeView * treeview, int which)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeIter target;
    tree_entry_t *en;

    if(!gtk_tree_model_get_iter_first(treemodel, &target))
	    return FALSE;
    gtk_tree_model_get(treemodel, &target, ENTRY_COLUMN, &en, -1);

    while(((en->type) & __TREE_MASK) != root[which].type)
    {
	if(!gtk_tree_model_iter_next(treemodel, &target)){
	   return FALSE;
	}
	gtk_tree_model_get(treemodel, &target, ENTRY_COLUMN, &en, -1);
    }
    erase_root(treemodel, &target);
    return TRUE;
}
int erase_local_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_FILES);
}
int erase_find_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_FIND);
}
int erase_bookmark_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_BOOKMARKS);
}
int erase_trash_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_TRASH);
}
int erase_apps_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_APP);
}
int erase_network_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_NETWORK);
}
#if defined(HAVE_GETMNTENT) || defined(HAVE_GETFSENT) || defined(HAVE_GETVFSENT)
int erase_fstab_root(GtkTreeView * treeview){
	return erase_the_root(treeview,ROOT_FSTAB);
}
#endif

int get_bookmark_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_BOOKMARKS);
}

int get_local_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_FILES);
}
int get_network_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_NETWORK);
}

int get_apps_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_APP);
}

int get_find_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en,ROOT_FIND);
}


int get_trash_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_TRASH);
}


#if defined(HAVE_GETMNTENT) || defined(HAVE_GETFSENT) || defined(HAVE_GETVFSENT)
int get_fstab_root(GtkTreeView * treeview, GtkTreeIter * iter, tree_entry_t ** en)
{
    return get_the_root(treeview, iter, en, ROOT_FSTAB);
}
#endif

char *abreviate(char *path)
{
    int i;
    static char *label = NULL;

    if(strlen(path) <= 21)
	return path;

    if(label)
	g_free(label);
    label = (char *)malloc(strlen(path) + 1);
    if(!label)
	return path;
    strcpy(label, path);
    label[8] = '~';
    for(i = 9; i <= 21; i++)
	label[i] = label[strlen(label) - (21 - i)];
    return label;
}


static char *plain_sizetag(long long tama, long long tamal)
{
    char *tag = "KB";
    char *tagl = "KB";
    static char buf[64];

    buf[0] = 0;
    if(tama >= (unsigned long long)1024 * 1024)
    {
	(unsigned long long)tama /= (unsigned long long)1024 *1024;
	tag = "GB";
    }
    else if(tama >= 1024)
    {
	(unsigned long long)tama /= 1024;
	tag = "MB";
    }

    if(tamal >= (unsigned long long)1024 * 1024)
    {
	(unsigned long long)tamal /= (unsigned long long)1024 *1024;
	tagl = "GB";
    }
    else if(tamal >= 1024)
    {
	(unsigned long long)tamal /= 1024;
	tagl = "MB";
    }
    sprintf(buf, "%llu %s / %llu %s", tama, tag, tamal, tagl);

    return buf;
}

static void set_progress_generic(GtkTreeView * treeview, int count, int total, int caso)
{
    static int smallcount = 1;
    tree_details_t *tree_details = get_tree_details(treeview);

    gfloat fraction;

    GtkWidget *progress;

    if(!tree_details || !tree_details->window)
	return;
    progress = lookup_widget(tree_details->window, "progressbar1");
    if(!progress)
	return;
    if(count == -1)
    {
	/*printf("DBG:updating pulse\n"); */
	if(total < 0 || (smallcount++ & (1 << 7)))
	{
	    gtk_progress_bar_pulse((GtkProgressBar *) progress);
    process_pending_gtk();
	    smallcount = 1;
	}
    }
    else
    {
	char texto[_POSIX_PATH_MAX];
	if(!total)
	    fraction = 0.0;
	else
	    fraction = (float)count / (float)total;
	if(fraction < 0 || fraction > 1)
	    return;
	gtk_progress_bar_set_fraction((GtkProgressBar *) progress, fraction);
	if(count >= total)
	    texto[0] = 0;
	else
	    switch (caso)
	    {
		case 1:
		    sprintf(texto, "%d / %d", count, total);
		    break;
		case 0:
		default:
		    sprintf(texto, "%s", plain_sizetag(count, total));
		    break;
	    }
	gtk_progress_bar_set_text((GtkProgressBar *) progress, texto);
    }
}

void set_progress(GtkTreeView * treeview, int count, int total)
{
    set_progress_generic(treeview, count, total, 1);
}

void set_progress_bytes(GtkTreeView * treeview, int count, int total)
{
    set_progress_generic(treeview, count, total, 0);
}


void cursor_wait(GtkTreeView * treeview)
{
    static GdkCursor *cursor = NULL;
    tree_details_t *tree_details = get_tree_details(treeview);
    if(!tree_details->window)
	return;
    if(!cursor)
	cursor = gdk_cursor_new(GDK_WATCH);
    gdk_window_set_cursor(tree_details->window->window, cursor);
    /*animation(treeview,TRUE); */
    gdk_flush();
}

void cursor_reset(GtkTreeView * treeview)
{
    tree_details_t *tree_details = get_tree_details(treeview);
    if(!tree_details->window)
	return;
    gdk_window_set_cursor(tree_details->window->window, NULL);
    /*animation(treeview,FALSE); */

}

#if 0
void set_font(GtkTreeView * treeview, GtkTreeIter * iterator)
{
    PangoFontDescription *desc;
    int size;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_entry_t *en = get_entry(treeview, iterator);

    if(!en)
	return;

    if(IS_ROOT_TYPE(en->type))
    {
	if(IS_SIZE_XL(en->type))
	    size = 18 * 1000;
	else if(IS_SIZE_L(en->type))
	    size = 16 * 1000;
	else if(IS_SIZE_M(en->type))
	    size = 14 * 1000;
	else
	    size = 12 * 1000;
    }
    else
    {
	if(IS_SIZE_XL(en->type))
	    size = 16 * 1000;
	else if(IS_SIZE_L(en->type))
	    size = 14 * 1000;
	else if(IS_SIZE_M(en->type))
	    size = 12 * 1000;
	else
	    size = 10 * 1000;
    }

    desc = pango_font_description_new();

#if 10
    if(IS_ROOT_TYPE(en->type))
	pango_font_description_set_family(desc, "times");
#endif

    pango_font_description_set_size(desc, size);

    /*printf("font size is=%d\n",pango_font_description_get_size(desc)); */

    gtk_tree_store_set((GtkTreeStore *) treemodel, iterator, FONT_COLUMN, desc, -1);
    pango_font_description_free(desc);
}
#endif

tree_details_t *get_tree_details(GtkTreeView * treeview)
{
    if(!treeview)
	return NULL;
    return g_object_get_data(G_OBJECT(treeview), "tree_details");
}

GtkTreeView *get_treeview(GtkWidget * widget)
{
    if(!widget)
	return NULL;
    return (GtkTreeView *) lookup_widget(widget, "treeview");
}


void hideit(GtkWidget * parent, gchar * widget_name)
{
    GtkWidget *widget;
    if(!parent)
	return;
    if((widget = lookup_widget(parent, widget_name)) != NULL)
    {
	if(GTK_WIDGET_VISIBLE(widget))
	    gtk_widget_hide_all(widget);
    }

}
void showit(GtkWidget * parent, gchar * widget_name)
{
    GtkWidget *widget;
    if(!parent)
	return;
    if((widget = lookup_widget(parent, widget_name)) != NULL)
    {
	if(!GTK_WIDGET_VISIBLE(widget))
	    gtk_widget_show_all(widget);
    }

}

void show_text(GtkWidget * parent)
{
    GtkWidget *vpaned;
    if(!parent)
	return;
    /*printf("DBG:show_text\n");*/
    vpaned = lookup_widget(parent, "vpaned1");
    if (gtk_paned_get_position((GtkPaned *)vpaned) > vpaned->allocation.height * 0.80)
       gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height * 0.80);
}

void hide_text(GtkWidget * parent)
{
    GtkWidget *vpaned;
    vpaned = lookup_widget(parent, "vpaned1");
    gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height);
}

gchar *utf_2_local_string(char *t){
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;

    
    g_get_charset(&fc);
    if (fc) from_codeset = g_strdup(fc);
    else from_codeset = g_strdup("ISO-8859-1");
    
    
    if (strcmp(from_codeset,"UTF-8")==0 || strcmp(from_codeset,"ASCII")==0){
	    g_free(from_codeset);
	    from_codeset = g_strdup("ISO-8859-1");
    }    
#ifdef DEBUG    
    printf("DBG: codeset for system is: %s\n",from_codeset);    
#endif
    
    if(s) g_free(s);
#ifdef DEBUG    
    printf("DBG: setting to local codeset-> %s\n",t); 
#endif
    s = g_convert (t,strlen(t),from_codeset,"UTF-8",&r_bytes, &w_bytes, &error);
    g_free(from_codeset);

    if(error){
	g_error_free(error);
    }
    return s;
}

gchar *my_utf_string(char *t)
{
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;
    
    if(!t) g_assert_not_reached();
    
    if (g_utf8_validate (t,-1,NULL)) {
#ifdef DEBUG    
    printf("DBG: valid utf8: %s\n",t);    
#endif
	    return t;
    }
    
    /* so we got a non-UTF-8 */
    
    if (getenv("SMB_CODESET") && strlen( getenv("SMB_CODESET"))){
	    from_codeset=g_strdup(getenv("SMB_CODESET"));
    }
    else {
    	g_get_charset(&fc);
    	if (fc) from_codeset = g_strdup(fc);
    	else from_codeset = g_strdup("ISO-8859-1");
    }
    
/* hey, rsh will give you ANSI_X3.4-1968, so let's limit conversion with ISO */
/*    if (strcmp(from_codeset,"UTF-8")==0 || strcmp(from_codeset,"ASCII")==0){*/
    if (!strcmp(from_codeset,"ISO-")){
	    g_free(from_codeset);
	    from_codeset = g_strdup("ISO-8859-1");
    }    

#ifdef DEBUG    
    printf("DBG: codeset for system is: %s\n",from_codeset);    
#endif
    
    if(s) g_free(s);
#ifdef DEBUG    
    printf("DBG: utfing %s\n",t); 
#endif
    for(c = (unsigned char *)t; *c != 0; c++)
	if(*c < 32 && *c != '\n')
	    *c = ' ';
    s = g_convert (t,strlen(t),"UTF-8",from_codeset,&r_bytes, &w_bytes, &error);
    g_free(from_codeset);

    if(!s)
    {
	s=g_strdup(t);
	for(c = s; *c != 0; c++) if(*c > 128) *c = '?';
    }
    if(error){
        printf("DBG: %s. Codeset for system is: %s\n",
			error->message,from_codeset);
        printf("DBG: You should set the environment variable SMB_CODESET to ISO-8859-1\n");
	g_error_free(error);
    }
    return s;
}

static void insert_string(GtkTextBuffer * buffer, gchar *s)
{
    GtkTextIter start, end;
    if(s)
    {
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_insert(buffer, &end, s, -1);
    }
    return;
}

gboolean is_number(char *p){
   char *c=p;
   if (!c || !strlen(c)) return FALSE;
   for (;*c;c++) {
	if (*c < '0' || *c > '9') return FALSE;
   }
   return TRUE;   
}

gchar * my_cut_utf_string(gchar *s){
	static  gchar *u=NULL;
	gchar *v=g_strdup(my_utf_string(s));
	int max_len=48;
	if (getenv("XFFM_STATUS_LINE_LENGTH") && strlen( getenv("XFFM_STATUS_LINE_LENGTH"))) {
		if (is_number(getenv("XFFM_STATUS_LINE_LENGTH")))
		   max_len=(atoi(getenv("XFFM_STATUS_LINE_LENGTH")));
	}
	if (u) g_free(u);
        if (g_utf8_strlen(s,-1) > max_len){
	    u=g_utf8_offset_to_pointer (v,max_len-4);
	    *u=0;
	    u=g_strjoin ("",v,"...",NULL);
	    g_free(v);
       } else u=v;
       return u;
} 

void print_status_tmp(GtkTreeView * treeview, GdkPixbuf * icono, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(tree_details->status);

    if(!tree_details->window)
	return;
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    if(icono)
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, "  ", -1);
    va_start(var_args, icono);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}



void print_status(GtkTreeView * treeview, char *id, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(tree_details->status);

    if(!tree_details->window)
	return;
    gtk_text_buffer_set_text(buffer, "", -1);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(id && icon_tell(0, id))
	gtk_text_buffer_insert_pixbuf(buffer, &end, icon_tell(0, id));
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, "  ", -1);
    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){	
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}

void print_diagnostics(GtkTreeView * treeview, char *id, ...)
{
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(tree_details->diagnostics);

    if(!tree_details->window || diagnostics_disabled)
	return;
    show_text((GtkWidget *)tree_details->window);
    /*printf("DBG:print_diagnostics\n");*/
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(id && icon_tell(0, id))
	gtk_text_buffer_insert_pixbuf(buffer, &end, icon_tell(0, id));

    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);

    gtk_text_buffer_get_bounds(buffer, &start, &end);
    mark = gtk_text_buffer_create_mark(buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark(tree_details->diagnostics, mark, 0.2,	/*gdouble within_margin, */
				 FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buffer, mark);
  gdk_flush();
}
void print_diagnostics_tmp(GtkTreeView * treeview,GdkPixbuf * icono, ...)
{
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTextBuffer *buffer = gtk_text_view_get_buffer(tree_details->diagnostics);

    if(!tree_details->window || diagnostics_disabled)
	return;
    /*printf("DBG:print_diagnostics_tmp\n");*/
    show_text((GtkWidget *)tree_details->window);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    if(icono)
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
 
    va_start(var_args, icono);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);

    gtk_text_buffer_get_bounds(buffer, &start, &end);
    mark = gtk_text_buffer_create_mark(buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark(tree_details->diagnostics, mark, 0.2,	/*gdouble within_margin, */
				 FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buffer, mark);
  gdk_flush();
}


void clear_diagnostics(GtkTreeView * treeview)
{
    GtkTextIter start, end;
    GtkTextBuffer *buffer;
    tree_details_t *tree_details = get_tree_details(treeview);
    if(!tree_details->window)
	return;
    buffer = gtk_text_view_get_buffer(tree_details->diagnostics);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    hide_text(tree_details->window);
}

void show_stop(GtkWidget * widget)
{
    hideit(widget, "clear_text");
    showit(widget, "stop");
    /*printf("DBG: showing stop\n"); */

}

void hide_stop(GtkWidget * widget)
{
    hideit(widget, "stop");
    showit(widget, "clear_text");
    /*printf("DBG: hiding stop\n"); */
}

/*XXX put these separate lib */


#define HISTORY_ITEMS 13
static GList **the_list;

typedef struct history_dbh_t{
	int hits;
	time_t last_hit;
	char path[256];
}history_dbh_t;

static int history_compare (gconstpointer a,gconstpointer b){
	history_dbh_t *da=(history_dbh_t *)a;
	history_dbh_t *db=(history_dbh_t *)b;
	if (db->hits != da->hits) return db->hits - da->hits;
	if (db->last_hit != da->last_hit) return db->last_hit - da->last_hit;
	return (strcmp(da->path,db->path));
}

static void history_mklist(DBHashTable * d)
{
    history_dbh_t *history_mem = (history_dbh_t *)malloc(sizeof(history_dbh_t));
    if (!history_mem) g_assert_not_reached();
    memcpy(history_mem,DBH_DATA(d),sizeof(history_dbh_t));
    if (!the_list) g_assert_not_reached(); 
    *the_list = g_list_insert_sorted(*the_list,history_mem,history_compare);
}


static void clean_history_list(GList **list){
	GList *tmp;
	if (!*list) return ;
	for (tmp=*list;tmp;tmp=tmp->next){
		/*printf("freeing %s\n",(char *)tmp->data);*/
		g_free(tmp->data);
	}
	g_list_free(*list);
	*list=NULL;
	return;
}

/* if top==NULL, the top entry is left blank,
 * if top=="", the top entry is the one with the greatest access time for last hit
 * if top=="anything", the entry is set to "anything"
 * */
void get_history_list(GList **in_list,char *dbh_file,char *top){
   DBHashTable *d;
   GList *tmp;
   char *first=NULL;
   time_t last_hit=0;
   the_list=in_list;
   clean_history_list(the_list);	
   if ((d=DBH_open(dbh_file))!=NULL){
	DBH_foreach_sweep(d,history_mklist);	
	DBH_close(d);
   }
   /* leave only strings in the history list: */
   for (tmp=*the_list;tmp;tmp=tmp->next){
 	history_dbh_t *history_mem=(history_dbh_t *)tmp->data;
	gchar *p=g_strdup(history_mem->path);
	if (history_mem->last_hit >= last_hit) {
		first=p;
		last_hit=history_mem->last_hit;
	}
#ifdef DEBUG
	printf("DBG:%s, hits=%d\n",history_mem->path,history_mem->hits);
#endif
	tmp->data=p;
	g_free(history_mem);
   }
   
   
   if (!top) *the_list=g_list_prepend(*the_list,g_strdup(""));
   if (top && strlen(top)) *the_list=g_list_prepend(*the_list,g_strdup(top));
	
   if (*the_list==NULL) {
	   *the_list=g_list_prepend(*the_list,g_strdup(""));
   }
   return ;
}


void save_to_history(char *dbh_file,char *path2save){
    	GString *gs;
	DBHashTable *d;
	history_dbh_t *history_dbh;
	int size;
	
	if (strlen(path2save) > 255) return;
	if ((d=DBH_open(dbh_file))==NULL){
		if ((d=DBH_create(dbh_file, 11))==NULL) return;
	}
	gs = g_string_new(path2save);
    	sprintf((char *)DBH_KEY(d), "%10u", g_string_hash(gs));
   	g_string_free(gs, TRUE);

	history_dbh = (history_dbh_t *)DBH_DATA(d);
	if (DBH_load(d)){
		history_dbh->hits++;
	} else {
		strncpy(history_dbh->path,path2save,255);
		history_dbh->hits=1;
	}
	history_dbh->last_hit=time(NULL);
	/* don't write more to disk than that which is necessary: */
	size= sizeof(history_dbh_t) + strlen(history_dbh->path) - 255;
	DBH_set_recordsize(d, size);
	DBH_update(d);
    	DBH_close(d);
}

