/*
 *  xfce4-volstatus-icon
 *
 *  Copyright (c) 2006 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include <gtk/gtk.h>

#include <ghal/ghal.h>
#include <exo-hal/exo-hal.h>

#include "xfce-volstatus-common.h"
#include "xfce-volstatus-icon.h"


static void
sighandler(int sig)
{
    gint i, level = gtk_main_level();
    
    if(!level)
        exit(EXIT_SUCCESS);
    
    for(i = 0; i < level; ++i)
        gtk_main_quit();
}

static void
xfce_volstatus_firstrun_check()
{
    XfceRc *rcfile;
    gchar *filename;
    gboolean already_run = FALSE;
    
    filename = xfce_resource_lookup(XFCE_RESOURCE_CONFIG,
                                    "xfce4/volstatus/settings.rc");
    if(filename) {
        rcfile = xfce_rc_simple_open(filename, TRUE);
        g_free(filename);
        
        if(rcfile) {
            if(xfce_rc_has_group(rcfile, "global")) {
                xfce_rc_set_group(rcfile, "global");
                already_run = xfce_rc_read_bool_entry(rcfile,
                                                      "initial_setup_done",
                                                      FALSE);
            }
            xfce_rc_close(rcfile);
        }
        
        if(already_run)
            return;
    }
    
    filename = xfce_resource_save_location(XFCE_RESOURCE_CONFIG,
                                           "xfce4/volstatus/settings.rc",
                                           TRUE);
    if(!filename)
        return;  /* something is very very wrong */
    
    rcfile = xfce_rc_simple_open(filename, FALSE);
    g_free(filename);
    
    if(!rcfile)
        return;  /* something is very very wrong */
    
    xfce_rc_set_group(rcfile, "global");
    already_run = xfce_rc_read_bool_entry(rcfile, "initial_setup_done", FALSE);
    if(already_run) {
        xfce_rc_close(rcfile);
        return;
    }
    
    if(GTK_RESPONSE_ACCEPT == xfce_message_dialog(NULL, _("Removable Volumes"),
                                               GTK_STOCK_DIALOG_QUESTION,
                                               _("This is the first time you've run the Xfce Volstatus Icon application"),
                                               _("Would you like Xfce Volstatus to run automatically each time your session begins?  " \
                                                 "You can disable this behavior later by using the \"Autostarted Applications\" settings panel."),
                                               GTK_STOCK_NO, GTK_RESPONSE_NO,
                                               GTK_STOCK_YES, GTK_RESPONSE_ACCEPT,
                                               NULL))
    {
        XfceRc *autostart = xfce_rc_config_open(XFCE_RESOURCE_CONFIG,
                                                "autostart/xfce4-volstatus-icon.desktop",
                                                FALSE);
        if(!autostart) {
            xfce_message_dialog(NULL, _("Removable Volumes"),
                                GTK_STOCK_DIALOG_ERROR,
                                _("Failed to write autostart file"),
                                _("For some reason, the autostart location could not be found or written to.  This location is usually in your home directory, in ~/.config/autostart/."),
                                GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
                                NULL);
        } else {
            xfce_rc_set_group(autostart, "Desktop Entry");
            xfce_rc_write_entry(autostart, "Version", "1.0");
            xfce_rc_write_entry(autostart, "Type", "Application");
            xfce_rc_write_entry(autostart, "Name", _("Xfce Volstatus Icon"));
            xfce_rc_write_entry(autostart, "Comment",
                                _("Safe removal of removable volumes"));
            xfce_rc_write_entry(autostart, "Icon", "drive-removable-media");
            xfce_rc_write_entry(autostart, "Exec", "xfce4-volstatus-icon");
            xfce_rc_write_bool_entry(autostart, "StartupNotify", FALSE);
            xfce_rc_write_bool_entry(autostart, "Terminal", FALSE);
            xfce_rc_write_bool_entry(autostart, "Hidden", FALSE);
            xfce_rc_flush(autostart);
            xfce_rc_close(autostart);
        }
    }
    
    xfce_rc_write_bool_entry(rcfile, "initial_setup_done", TRUE);
    xfce_rc_flush(rcfile);
    xfce_rc_close(rcfile);
}

static void
xfce_volstatus_device_added(GHalContext *context,
                            GHalDevice *device,
                            gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    
    TRACE("entering(, %s)", ghal_device_peek_udi(device));
    
    if(!GHAL_IS_DRIVE(device))  /* is the device a drive? */
        return;
    
    if(ghal_drive_is_hotpluggable(GHAL_DRIVE(device))
       || ghal_drive_uses_removable_media(GHAL_DRIVE(device)))
    {
        xfce_volstatus_icon_add_drive(icon, GHAL_DRIVE(device));
    }
}

static void
xfce_volstatus_device_removed(GHalContext *context,
                              GHalDevice *device,
                              gpointer user_data)
{
    XfceVolstatusIcon *icon = user_data;
    
    TRACE("entering(, %s)", ghal_device_peek_udi(device));
    
    if(!GHAL_IS_DRIVE(device))
        return;
    
    xfce_volstatus_icon_remove_drive(icon, GHAL_DRIVE(device));
}

int
main(int argc,
     char **argv)
{
    const gchar *icon_name;
    DBusGConnection *dbus_gconn;
    GHalContext *context;
    GError *error = NULL;
    GtkStatusIcon *icon;
    GList *drives, *l;
    
    signal(SIGPIPE, SIG_IGN);
    signal(SIGINT, sighandler);
    signal(SIGQUIT, sighandler);
    signal(SIGTERM, sighandler);
    
    xfce_textdomain(PACKAGE, LOCALEDIR, "UTF-8");
    
    gtk_init(&argc, &argv);
    
    xfce_volstatus_firstrun_check();
    
    icon_name = xfce_volstatus_get_icon_name();
    if(icon_name)
        gtk_window_set_default_icon_name(icon_name);
    
    icon = xfce_volstatus_icon_new(NULL);
    gtk_status_icon_set_visible(icon, FALSE);
    
    dbus_gconn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if(!dbus_gconn) {
        gchar *secondary;
        
        if(error)
            secondary = error->message;
        else
            secondary = _("An unknown error occurred.");
        
        xfce_message_dialog(NULL, _("Xfce Volstatus"), GTK_STOCK_DIALOG_ERROR,
                            _("Unable to connect to D-Bus system bus daemon"),
                            secondary, GTK_STOCK_QUIT, GTK_RESPONSE_ACCEPT,
                            NULL);
        if(error)
            g_error_free(error);
        return EXIT_FAILURE;
    }
    
    context = ghal_context_get(dbus_gconn, &error);
    if(!context) {
        xfce_message_dialog(NULL, _("Xfce Volstatus"), GTK_STOCK_DIALOG_ERROR,
                            _("Unable to connect to HAL"), error->message,
                            GTK_STOCK_QUIT, GTK_RESPONSE_ACCEPT, NULL);
        g_error_free(error);
        return EXIT_FAILURE;
    }
    
    g_signal_connect(G_OBJECT(context), "device-added",
                     G_CALLBACK(xfce_volstatus_device_added), icon);
    g_signal_connect(G_OBJECT(context), "device-removed",
                     G_CALLBACK(xfce_volstatus_device_removed), icon);
    
    drives = ghal_context_find_device_by_capability(context, "storage",
                                                    &error);
    if(error) {
        xfce_message_dialog(NULL, _("Xfce Volstatus"), GTK_STOCK_DIALOG_ERROR,
                            _("Unable to get device list from HAL"),
                            error->message,
                            GTK_STOCK_QUIT, GTK_RESPONSE_ACCEPT, NULL);
        g_error_free(error);
        return EXIT_FAILURE;
    }
    
    for(l = drives; l; l = l->next) {
        /* just try to add it.  the device added function will make sure
         * it's a removable drive */
        xfce_volstatus_device_added(context, GHAL_DEVICE(l->data), icon);
    }
    g_list_foreach(drives, (GFunc)g_object_unref, NULL);
    g_list_free(drives);
    
    gtk_main();
    
    g_object_unref(G_OBJECT(icon));
    g_object_unref(G_OBJECT(context));
    
    return EXIT_SUCCESS;
}
